# Copyright (c) 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI Netstart.pm,v 1.9 1997/12/18 00:13:38 sanders Exp

package AdminWEB::Netstart;

# NOTE: This manages the "current" host configuration, not the data
# in /etc/netstart.  The problem is that it would be virtually
# impossible to "do the right thing" in all cases if we just edited
# /etc/netstart and configured new interfaces.  It's far simpler
# to just manage the host configuration and let /etc/netstart reflect
# the current state of things to the best of our ability.

use AdminWEB::Descriptions;

BEGIN {
    $NETSTART = "/etc/netstart";
    $NETSTART_TMP = "$NETSTART.tmp";
    $NETSTART_PROTO = "/etc/netstart.proto";
    @SINGLES = ("ether", "token_ring", "fddi");
}

sub new {
    my $self = bless { }, shift;

    my @QUERY_LIST = (@_ ? @_ : @SINGLES);
    $self->{'QUERY_LIST'} = \@QUERY_LIST;
    $self->read_netstart;
    $self->query_interfaces;
    $self;
}

sub get_interface {
    my $self = shift;

    my $iface = shift;

    $self->{'iface'} = $iface;

    # Parse the choices
    my @c = @choices = split('%', $media_choices{$iface});
    # Parse the options into a hash
    foreach $_ (split(' ', $media_options{$iface})) {
	$options{$_} = $_;
    }
    # Propagate the options onto the choices
    foreach $choice (@c) {
	foreach $option (keys %options) {
	    push(@choices, $choice . " " . $option);
	}
    }

    # Remove all choices that involve forcing ``full_duplex'' as
    # that could trash the network.  There is no point in trying
    # ``nomedia''!
    @choices = grep(! /\bfull_duplex\b/ && ! /\bnomedia\b/, @choices);

    ### Figure out our current configuration
    my @addrs = split('%', $inet{$iface});
    $_ = pop(@addrs);
    if (/^\s*([\d.]+)\s+netmask\s+([\d.]+)\s+broadcast\s+([\d.]+)\s?$/) {
	$laddr = $1;
	$netmask = $2;
	$broadcast = $3;
	if (&same_subnet($laddr, $3, $netmask) &&
	    &same_host($3, "255.255.255.255", $netmask)) {
	    undef $broadcast;  
	}
    }
    elsif (/^\s*([\d.]+)\s+destination\s+([\d.]+)\s+netmask\s+([\d.]+)\s?$/) {    
	$laddr = $1;
	$destination = $2;
	$netmask = $3;
    }

    $self;
}

sub primary {
    my $self = shift;
    my $primary = shift;
    my $old = $netstart_misc{'primary'};
    $netstart_misc{'primary'} = $primary if defined $primary;
    return $old;
}

sub get_misc_data {
    my $self = shift;
    return $netstart_misc{$_[0]};
}

sub set_misc_data {
    my $self = shift;
    $netstart_misc{$_[0]} = $_[1];
    return $self;
}

sub list_media_choices {
    my $self = shift;
    return \@choices;
}

sub networks {
    my $self = shift;
    return @networks;
}

sub default_media_choice {
    my $self = shift;
    # what is currently selected...
    return $media_current{$self->{'iface'}} if $media_current{$self->{'iface'}} ne '';
    # else some nice default
    return 'auto' if grep(/\bauto\b/, @choices);
    return 'manual' if grep(/\bmanual\b/, @choices);
    return $choices[0];
}

sub interface_description {
    my $self = shift;
    my ($type) = $self->{'iface'} =~ /^(.*)[1-9]*[0-9]$/;
    return $self->{'iface'} . ' -- ' . $desc{$type};
}

sub read_netstart {
    my $self = shift;

    open (TMP, "< $NETSTART") || die "$NETSTART: $!\n";
    @netstart = <TMP>;		# yet another global
    close (TMP);

    my $netstart_ok = 0;

    foreach $_ (@netstart) {
	if (m/^#\s*DO NOT DELETE THIS LINE \(V3\.0\)/) {
	    $netstart_ok = 1;
	    last;
	}
	chomp $_;
	# skip blank or comment lines or known data
	if (m/^\s*$/ || m/^#/ || m/^interfaces=/
		    || m/^(ipaddr|netmask|linkarg_additional)_/) {
	    $_ = undef;
	    next;
	}
	# look like something we are interested in keeping around, like:
	#     hostname="foo"     primary="bat"     defroute="baz"
	if (m/^([a-zA-Z_]+)="?([^"]*)"?$/) {
	    $netstart_misc{$1} = $2;
	}
	$_ = undef;
    }

    die "Sorry, we were unable to process your /etc/netstart file.\n" .
	"It is apparently in an old format or you have made changes to it.\n"
		unless $netstart_ok;
    $self;
}

sub write_netstart {
    my $self = shift;
    my($iface, $laddr, $netmask, $destination, $router);
    my(@proto);

    # Query interfaces again
    $self->query_interfaces;

    open (TMP, ">$NETSTART_TMP") || 
	die "$0: can't open $NETSTART_TMP: $!\n";
    print TMP <<ETX;
#
# netstart - configure network daemons, interfaces, and routes
#
# Do not change lines above the `DO NOT DELETE THIS LINE' line below.
# They are generated by MaxIM.  You can change them by accessing
# MaxIM at http://localhost:880/.
#

ETX

    # dump all the misc stuff (mainly hostname, primary and defroute)
    foreach $_ (sort keys %netstart_misc) {
	printf TMP qq/%s="%s"\n/, $_, $netstart_misc{$_};
    }
    print TMP "\n";

    # dump network configuration
    printf TMP qq/interfaces="%s"\n/, join(' ', @configured_networks);

    foreach $iface (@configured_networks) {
	my(@addrs);

	@addrs = split('%', $inet{$iface});
	$_ = pop(@addrs);
	
	if (/^\s*([\d.]+)\s+netmask\s+([\d.]+)\s+broadcast\s+([\d.]+)\s?$/) {
	    $laddr = $1;
	    $netmask = $2;
	    $broadcast = $3;
	    if (&same_subnet($laddr, $3, $netmask) &&
		&same_host($3, "255.255.255.255", $netmask)) {
		undef $broadcast;
	    }
	} elsif (/^\s*([\d.]+)\s+destination\s+([\d.]+)\s+netmask\s+([\d.]+)\s?$/) {
	    $laddr = $1;
	    $destination = $2;
	    $netmask = $3;
	}

	printf TMP qq/# %s::\n/, $iface;
	printf TMP qq/ipaddr_%s="%s"\n/, $iface, $laddr;
	printf TMP qq/netmask_%s="%s"\n/, $iface, $netmask;

	if ($media_current{$iface} ne '') {
	    printf TMP qq/linkarg_%s="media %s"\n/, $iface, 
		&xlate_media($media_current{$iface});
	} else {
	    printf TMP qq/linkarg_%s=""\n/, $iface;
	}

	if (defined($broadcast)) {
	    printf TMP qq/additional_%s="broadcast %s"\n/, $iface, $broadcast;
	} elsif (defined($destination)) {
	    printf TMP qq/additional_%s="destination %s"\n/, $iface, $destination;
	} else {
	    printf TMP qq/additional_%s=\n/, $iface;
	}
    }
    print TMP "\n";

    # @netstart contains the rest of the real /etc/netstart, starting with
    # the line about not deleting this line.
    print TMP @netstart;
    close TMP;
    rename ("$NETSTART_TMP", "$NETSTART") ||
    	die "$0: can't rename $NETSTART_TMP into place: $!\n";

    $self;
}

sub query_interfaces {
    my $self = shift;
    my($i, $iface, $match, %linktypes);

    @networks = ();
    @configured_networks = ();
    %inet = ();
    %linktype = ();
    %media_active = ();
    %media_choices = ();
    %media_current = ();
    %media_options = ();
    %media_status = ();

    foreach $_ (@{$self->{'QUERY_LIST'}}) {
	$linktypes{$_} = $_;
    }

    if (open(IFCONFIG, "ifconfig -am|") == 0) {
	die "Unable to determine network interfaces present " .
	    "(ifconfig -am failed).\n";
    }
    while (<IFCONFIG>) {
	if (/^([a-zA-Z]+[0-9]+): flags=/) {
	    $iface = $1;
	} elsif (/^\s+link\s+type\s+(\w+)\s+.*$/) {

	    # If link types specified, only return those
	    if (%linktypes && !defined($linktypes{$1})) {
		undef $iface;
		next;
	    }

	    push(@networks, $iface);
	    $linktype{$iface} = $1;
	} elsif (defined($iface)) {
	    if (/^\s+media\s+options\s+(.*)\s*$/) {
		# Ignore choices with loopback
		my(@options) = grep(!/^loopback$/, split(/\s+/, $1));
		$media_options{$iface} = 
		    join(' ', @options,
			 split(' ', $media_options{$iface}));
	    } elsif (/^\s+media\s+choice\s+(.*)\s*$/) {
		$match = $1;
		# Ignore choices with loopback
		if ($match =~ /\bloopback\b/) {
		    next;
		}
		if (defined($media_choices{$iface})) {
		    $media_choices{$iface} .= '%' . $1;
		} else {
		    $media_choices{$iface} = $1;
		}
	    } elsif (/^\s+media\s+([^\(]+)(\(([^\)]+)\))?\s+status\s+(.*)$/) {
		$media_current{$iface} = join(' ', split(/\s+/, $1));
		if ($3 ne '') {
		    $media_active{$iface} = join(' ', split(/\s+/, $3));
		} else {
		    $media_active{$iface} = undef;
		}
		$media_status{$iface} = join(' ', split(/\s+/, $4));
	    } elsif (/^\s+media\s+([^\(]+)(\(([^\)]+)\))?$/) {
		$media_current{$iface} = join(' ', split(/\s+/, $1));
		if ($3 ne '') {
		    $media_active{$iface} = join(' ', split(/\s+/, $3));
		} else {
		    $media_active{$iface} = undef;
		}
		$media_status{$iface} = undef;
	    } elsif (/^\s+inet\s+(.*)$/) {
		$match = $1;
		$match =~ s/-->/destination/;
		if (defined($inet{$iface})) {
		    $inet{$iface} .= '%' . $match;
		} else {
		    $inet{$iface} = $match;
		    $inet{$iface} =~ /^([^\s]+)\s/;
		    if ($netutil{"LADDR"} eq undef) {
		    	$netutil{"LADDR"} = $match;
		    }
		    if ($netutil{"IFACE"} eq undef) {
		    	$netutil{"IFACE"} = $iface;
		    }
		}
	    }
	}
    }
    close(IFCONFIG);

    # If at least one interface is configured with an IP
    # address life is alot simpler.
    @configured_networks = grep(defined($inet{$_}), @networks);

    $self;
}

sub inet_aton {
    local($a) = @_;

    if ($a =~ /^0x[0-9a-fA-F]{8}$/) { 
        return hex($a);
    } elsif ($a =~ /^([0-9]*)\.([0-9]*)\.([0-9]*)\.([0-9]*)$/) {
        return (($1 << 24) | ($2 << 16) | ($3 << 8) | $4);
    } elsif ($a =~ /^([0-9]*)\.([0-9]*)\.([0-9]*)$/) {          
        return (($1 << 24) | ($2 << 16) | $3);  
    } elsif ($a =~ /^([0-9]*)\.([0-9]*)$/) {          
        return (($1 << 24) | $2);  
    } elsif ($a =~ /^([0-9]*)$/) {          
        return ($1);             
    }
}

# XXX: turns out that Socket::inet_aton really calls inet_addr which
# cannot distinquish between INADDR_NONE and errors (sigh).
# Don't use until inet_aton is fixed...
#
# use Socket();		# use it, but don't import anything
# sub inet_aton {
#     my($a) = @_;
# 
#     # call the real inet_aton via the Socket package.
#     return unpack('L', Socket::inet_aton($a));
# }

sub same_subnet {
    my($laddr, $raddr, $netmask) = @_;

    return (((&inet_aton($laddr) ^ &inet_aton($raddr)) 
	     & &inet_aton($netmask)) == 0);
}

sub same_host {
    my($laddr, $raddr, $netmask) = @_;

    return (((&inet_aton($laddr) ^ &inet_aton($raddr)) 
	     & ~&inet_aton($netmask)) == 0);
}

sub xlate_media {
    my($media) = @_;

    return join(',', split(/\s+/, $media));
}

sub unconfigure_interface {
    my $self = shift;
    my($iface, $laddr, $netmask, $media, $router) = @_;
    my @cmd;

    if ($router ne '') {
	@cmd = ("route", "-n", "delete", "default");
	if (fork() == 0) {
	    open(STDOUT, "> /dev/null");
	    open(STDERR, "> /dev/null");
	    exec @cmd;
	}
    }

    if ($iface ne '') {
	@cmd = ("ifconfig", $iface, "-remove");
	if (fork() == 0) {
	    open(STDOUT, "> /dev/null");
	    open(STDERR, "> /dev/null");
	    exec @cmd;
	}
    }
    $self;
}

sub configure_interface {
    my $self = shift;
    my($iface, $laddr, $netmask, $media, $router) = @_;

    &unconfigure_interface($iface, $laddr, $netmask, $media, $router);

    my @cmd = ("ifconfig", $iface, "set", $laddr);
    push(@cmd, "netmask", $netmask) if $netmask ne '';
    push(@cmd, "media", &xlate_media($media)) if $media ne '';
    system @cmd;
    die "Unable to configure interface $iface\n" unless (($?>>8)&0xff) == 0;
	
    if ($router ne '') {
	@cmd = ("route", "-n", "add", "default", $router);
	system @cmd;
	die "Unable to add default route\n" unless (($?>>8)&0xff) == 0;
    }
    $self;
}

1;
