# Copyright (c) 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#	BSDI Support.pm,v 1.21 1999/09/27 22:45:49 polk Exp
#
# AdminWEB Support Functions

package AdminWEB::Support;
require AdminWEB::Status;		# standard format status object
use AdminWEB::Paths;			# site-wide paths
use vars qw($STANDARD_ACTION_HELP);

BEGIN {
    require Exporter;
    @ISA = qw(Exporter);
    @EXPORT = qw(
	get_account
	put_account
	accounts
	bookmark
	cgi_header
	confirmation
	copy_file
	encodehtml
	encodeurl
	help_url
	get_groups
	get_login_conf
	get_shells
	install_skel
	ishelp
	list_views
	next_available_uid
	encrypt_passwd
	setuser_group
	parsegroup
	writegroup
	pidkill
	subview_bookmark
	run_backend
	viewpath
	actionURL
	PhaseURL
	initmode
	affirmative
	nextphase
	nextphaseURL
	crop
	$STANDARD_ACTION_HELP
    );

    $STANDARD_ACTION_HELP = "
Choosing <B>Submit</B> will submit the form data to the system for
processing.";
# XXX:
#<P>
#Choosing <B>Bookmark</B> will allow you to save your current form
#data via your browsers standard bookmarking function.  This feature
#can be very handy for saving data or selections on screens that
#are often used.";

    # %escapes is a global used by the encode* routines
    for (0..255) {
	$escapes{chr($_)} = sprintf("%%%02X", $_);
    }

}

# returns undef on failure, else reference to account array
sub get_account {
    my $user = shift;
    my @account = getpwnam($user);
    return undef unless @account;
    # rearrange into BSDI standard format (sigh)
    @account = ( 
        $account[0],			# name
        $account[1],			# pwd
        $account[2],			# uid
        $account[3],			# gid
        $account[5],			# class
        $account[4],			# change
        $account[9],			# expire
        $account[6],			# gecos
        $account[7],			# home
        $account[8],			# shell
    );
    return \@account;
}

sub put_account {
    my $account = shift;
    my @cmd = ($_PATH_CHPASS, '-a', join(":", @$account));
    # returns ($status, \@output) from run_backend
    run_backend(@cmd);
}

# returns an array ref to a sorted list of all accounts on the system
sub accounts {
    my ($line, @accounts);
    local(*PWD);
    open(PWD, "</etc/passwd") || return [ 'root' ];
    while (defined($line = <PWD>)) {
	$line =~ m/^(.*?):/;
	push(@accounts, $1);
    }
    @accounts = sort @accounts;
    return \@accounts;
}

### $html = encodehtml($string)
###
### escapes & < and >
sub encodehtml {
    my $string = shift;
    $string =~ s/\&/\&amp\;/g;
    $string =~ s/\</\&lt\;/g;
    $string =~ s/\>/\&gt\;/g;
    $string;
}

### $url = encodeurl($string)
###
### encodes a string for use in a URL using %## escapes
sub encodeurl {
    my $string = shift;
    $string =~ s/([^0-9A-Za-z\$\-\_\@\.\&])/$escapes{$1}/xg;
    $string;
}

### get_groups($group_filter_regexp)
###
### Returns a ref to the sorted list of groups currently defined
### on the system; minus any that match $group_filter_regexp.
sub get_groups {
    my $group_filter = shift;
    my ($group, @groups);
    setgrent();
    while ($group = (getgrent())[0]) {
	# filter out system groups
	next if $group_filter && $group =~ /$group_filter/;
	push(@groups, $group);
    }
    endgrent();
    @groups = sort @groups;
    return \@groups;
}

### get_shells()
###
### Returns a ref to the list of shells defined in /etc/shells.
sub get_shells {
    my $class_filter = shift;
    my ($shell, @shells);
    open(SHELL, "< $_PATH_SHELLS");
    while ($shell = <SHELL>) {
	next unless $shell =~ m/^\//;	# skip comments, random junk, etc
	$shell =~ s/[\s\r\n]+$//;	# strip trailing white space
	push(@shells, $shell);
    }
    close(SHELL);
    return \@shells;
}

### get_login_conf($class_filter)
###
### Returns a ref to the sorted list of classes from /etc/login.conf
sub get_login_conf {
    my $class_filter = shift;
    my ($class, @classes);
    open(LOGIN_CONF, "< $_PATH_LOGIN_CONF");
    while ($class = <LOGIN_CONF>) {
	next unless $class =~ m/^([a-zA-Z_-]*):/;
	$class = $1;
	next if $class_filter && $class =~ /$class_filter/;
	push(@classes, $class);
    }
    close(LOGIN_CONF);
    @classes = sort @classes;
    return \@classes;
}

### install_skel ($skel, $where, $user, $group, $class, $mode)
###
### We just have to run a make in the correct directory and it
### installs the skel files for us.
sub install_skel {
    my ($skel, $where, $user, $group, $class, $mode) = @_;
    my ($status, @output, $pid);
    local(*CMD);
    my @cmd = ('make', "HOMEDIR=$where",
		      "OWNER=$user",
		      "GROUP=$group",
		      "LOGIN_CLASS=$class",
		      "DIRMODE=$mode", 'install');

    $pid = open(CMD, '-|');
    die "pipe: $!\n" unless defined $pid;

    if ($pid == 0) {
	### child
	open(STDERR, '>& STDOUT');
	chdir($skel) || die "chdir: $skel: $!\n";
	exec @cmd;
	die "exec @cmd: $!\n";
    }
    else {
	### parent
	### collect @output and status from child
	@output = <CMD>;
	close(CMD);
	$status = ($? >> 8) & 0xff;
    }
    die join('', @output) unless $status == 0;
}

### copy_file ($from, $to)
###
### Copies a file from $from to $to.
### Returns an error string on failure, `undef' on success.
sub copy_file {
    my ($from, $to) = @_;
    local (*IN, *OUT);
    
    $from =~ s#^(\s)#./$1#;		# protect leading spaces
    open(IN,  "< $from\0") || return undef;
    $to   =~ s#^(\s)#./$1#;		# protect leading spaces
    open(OUT, "> $to\0") || do {
	close(IN);
	return undef;
    };
    while (<IN>) { print OUT; };
    close(IN);
    close(OUT);
    return 1;
}

# Parse group file
#     my ($group, $gid, $gpasswd, $members) = parsegroup($_PATH_GROUP);
#     ...
#     writegroup($_PATH_GROUP, $group, $gid, $gpasswd, $members);
sub parsegroup {
    my ($GROUP) = @_;
    my ($hgroup, $hgid, $hgpasswd, $hmembers) = ( {}, {}, {}, {} );
    my ($group, $passwd, $gid, $members);
    local($_);
    print "Reading current group file information\n" if $opt_v;
    open (GROUP, "$GROUP") || die "$0: can't open $GROUP\n";
    while (<GROUP>) {
	chop;
	next if /^$/;
	($group, $passwd, $gid, $members) = split(/:/);
	$hgroup->{$group} = $gid;
	$hgid->{$gid} = $group;
	$hgpasswd->{$group} = $passwd;
	$hmembers->{$group} = $members;
    }
    close(GROUP);
    ($hgroup, $hgid, $hgpasswd, $hmembers);
}

# Write out new group file
sub writegroup {
    my ($GROUP, $hgroup, $hgid, $hgpasswd, $hmembers) = @_;
    my (@tmp, $name);
    @tmp=();
    for $i (keys(%$hgid)) {
	next if !defined($hgid->{$i});
	push(@tmp, $i);
    }
    @tmp = sort {$a <=> $b} @tmp;
    print "Writing new group file..." if $opt_v;
    open (GROUP, ">${GROUP}.tmp") || 
	die "\n$0: can't open ${GROUP}.tmp for writing\n";
    for $i (@tmp) {
	$name = $hgid->{$i};
	print GROUP "$name:$hgpasswd->{$name}:$i:$hmembers->{$name}\n";
    }
    close(GROUP);
    print "DONE.\n" if $opt_v; 
    print "Renaming new group file into place.\n" if $opt_v;
    rename("$GROUP.tmp", "$GROUP") ||
	die "$0: can't rename temporary file to $GROUP\n";
    1;
}

### setuser_group($login, @gids)
###
### Makes sure the specified user is in all the groups in @gids.
### $login is textual and @gids are numeric.
sub setuser_group {
    my ($login, @gids) = @_;
    # XXX: make sure this locks the group file
    my ($group, $gid, $gpasswd, $members) = parsegroup($_PATH_GROUP);
    ### we've already looked up all the groups in @gids
    my ($group_id, $name);
    foreach $group_id (@gids) {
	$name = $gid->{$group_id};
	if ($name && $members->{$name} !~ /\b$login\b/) {
	    $members->{$name} =~ s/\s+//g;
	    $members->{$name} .= ',' unless $members->{$name} eq '';
	    $members->{$name} .= $login;
	}
    }
    writegroup($_PATH_GROUP, $group, $gid, $gpasswd, $members);
}

### next_available_uid($properties)
###
### Gets the next available UID.
### $properties is a propery database object like DB::Properties
### with properties: This_User_ID_Next This_User_ID_Start This_User_ID_Max
# XXX: should do it's own locking
sub next_available_uid {
    my $properties = shift;
    my $start = $properties->value('This_User_ID_Start');
    my $MAX_UID = $properties->value('This_User_ID_Max');
    my $uid = $properties->value('This_User_ID_Next') || $start;

    ### search while under the limit and the uid is in use
    for (; $uid < $MAX_UID && scalar(getpwuid($uid)); $uid++) { }

    if ($uid == $MAX_UID) {
	### start over next time
	$properties->set('This_User_ID_Next', $start);
	die "Could not allocate a new user ID.\n";
    }
    else {
	### We save our place at the found uid in case it ends
	### up not getting used for some reason and it's not a
	### big deal to look up the one extra UID each time
	### since on BSD/OS pwents come out of a DB.
	$properties->set('This_User_ID_Next', $uid);
    }

    return $uid;
}

use IPC::Open2;

sub encrypt_passwd {
    my $class = shift;
    my $pwd = shift;

    local(*READER, *WRITER);
    my $pid = open2(\*READER, \*WRITER, '/usr/libexec/crypt_passwd');
    die "open2: $!\n" unless defined $pid;

    print WRITER $class,"\n";
    print WRITER $pwd,"\n";
    @output = <READER>;
    close(READER);
    close(WRITER);
    waitpid($pid, 0);
    $status = ($? >> 8) & 0xff;

    die join('', @output) unless $status == 0;
    chomp(@output);
    return join(' ', @output);
}

### pidkill($cgi, $stat, $DEBUG, $sig, $pidfile,
###     $msg_nopidfile, $msg_ok, $msg_failed)
###
### interface builder for sending signals to processes with a pidfile
sub pidkill {
    my ($cgi, $stat, $DEBUG, $sig, $pidfile,
	$msg_nopidfile, $msg_ok, $msg_failed) = @_;

    my $pid;
    if (-f $pidfile) {
	open(PID, "< $pidfile") ||
	    $stat->error("open($pidfile): $!");
	chomp($pid = <PID>);
	close(PID);
    }

    $stat->error($msg_nopidfile)
	unless -f $pidfile			# file exists
	&& defined $pid				# $pid OK
	&& kill(0, $pid);			# $pid is running

    if ($stat->errors) {
	$stat->ReportErrors("System Error(s)");
	print $stat->HTML;
	return;
    }

    ### do the work

    my $msg;
    if ($DEBUG) {
	$msg = "DEBUG Output";
	$output = [ "kill -$sig $pid\n" ];
    }
    elsif (kill($sig,$pid)) {
	$msg = "Command Successful";
	$output = [ $msg_ok ];
    }
    else {
	$msg = "Failed";
	$output = [ $msg_failed ];
    }

    $stat->Report($msg, $output);
    print $stat->HTML;
}

### run_backend(@cmd)
###
### Runs the command catching any output and returning the exit status.
sub run_backend {
    my @cmd = @_;

    ### command status info
    my ($status, @output);

    ### run the command to add the user
    local(*CMD);
    my $pid = open(CMD, '-|');
    if (! defined $pid) {
	### error
	$status = 1;
	@output = "Opening Pipe: $!\n";
    }
    elsif ($pid == 0) {
	### child
	open(STDERR, '>& STDOUT');
	exec @cmd;
	die "exec @cmd: $!\n";
    }
    else {
	### parent
	### collect @output and status from child
	@output = <CMD>;
	close(CMD);
	$status = ($? >> 8) & 0xff;
    }

    return ($status, \@output);
}

### bookmark($cgi)
###
### Returns a bookmark'able URL for the current view.
sub bookmark {
    my $cgi = shift;

    my @tmp = $cgi->param('Action');
    $cgi->delete('Action');

    my $script_name = $cgi->script_name;
    $script_name =~ s#/([^/]*)$##;
    $script_name .= '/' . $cgi->param('View');

    my $url = 'http://' . $cgi->server_name;
    $url .= ':' . $cgi->server_port unless $cgi->server_port == 80;
    $url .= $_NAME_VIEW . $script_name;
    $url .= $cgi->path_info if $cgi->path_info;		# likely not used
    $url .= '?' . $cgi->query_string if $cgi->query_string;

    if (@tmp && defined $tmp[0]) {
	$cgi->param('Action', @tmp);
    } else {
	$cgi->delete('Action');
    }

    $url;
}

### subview_bookmark($cgi)
###
### Returns a bookmark'able URL for a subview (usually handled by /View)
sub subview_bookmark {
    my $cgi = shift;

    my @tmp = $cgi->param('SubView');
    $cgi->delete('SubView');

    my $script_name = $cgi->path_info;
    $script_name =~ s#/([^/]*)$##;
    $script_name =~ s#/([^/]*)$##;
    $script_name .= '/' . $cgi->param('View');

    my $url = 'http://' . $cgi->server_name;
    $url .= ':' . $cgi->server_port unless $cgi->server_port == 80;
    $url .= $_NAME_VIEW . $script_name;
    $url .= '?' . $cgi->query_string if $cgi->query_string;

    if (@tmp && defined $tmp[0]) {
	$cgi->param('SubView', @tmp);
    } else {
	$cgi->delete('SubView');
    }
    return $url;
}

### goto_url
###
### Goto a specific URL and preserve our state.
sub goto_url {
    my $cgi = shift;

    # XXX: fix this
}

### confirmation($cgi)
###
### Returns a URL for the current View data with Action set to Confirmed.
sub confirmation {
    my $cgi = shift;

    my @tmp = $cgi->param('Action');
    $cgi->param('Action', 'Confirmed');

    my $confirmation = $cgi->self_url;

    if (@tmp && defined $tmp[0]) {
	$cgi->param('Action', @tmp);
    } else {
	$cgi->delete('Action');
    }

    $confirmation;
}

### ishelp($cgi)
###
### returns true if we are in helpmode
sub ishelp {
    my $cgi = shift;
    $cgi->param('Help');
}

### help_url($cgi[, @extra_path_args])
###
### Returns a Help URL for the current view
sub help_url {
    my $cgi = shift;
    my @helps = $cgi->param('Help');
    $cgi->param('Help', 1);

    my $url = 'http://' . $cgi->server_name;
    $url .= ':' . $cgi->server_port unless $cgi->server_port == 80;
    $url .= $_NAME_HELP;
    $url .= $cgi->path_info if $cgi->path_info;
    $url .= join('', @_) if @_;
    $url .= '?' . $cgi->query_string;

    if (@helps && defined $helps[0]) {
	$cgi->param('Help', @helps);
    } else {
	$cgi->delete('Help');
    }

    $url;
}

### affirmative($expression)
###
### return true if the expression appears to be affirmative
sub affirmative {
    my $thing = lc(shift);
    return 1 if defined $thing && $thing ne '' &&
	($thing eq 'yes' || $thing eq 'true' || $thing eq 'on'
	    || $thing eq 'enable' || $thing eq 'enabled' || $thing > 0);
    return undef;
}

### initmode($cgi)
###
### returns true if were called from Initialize (or should behave as if)
sub initmode {
    my $cgi = shift;
    return (affirmative($cgi->param('Initialize'))
	|| $cgi->path_info =~ m#/Initialize/#) ? 1 : undef;
}

### actionURL($cgi, $handler)
###
### Builds an ACTION URL for forms from the cgi environment
### to point to a $handler CGI script in the "current directory"
### (determined from $PATH_INFO).  This is designed to be called
### from the *.view context.
sub actionURL {
    my $cgi = shift;
    my $handler = shift;
    my $url = $cgi->path_info;
    $url =~ s#/[^/]*$#/$handler#;	# /*.view -> /$handler
    $url;
}

### PhaseURL($cgi, $phase)
###
### Used to make HREF's to the next phase of a multiphase interface.
### For example: PhaseURL($cgi, '../PhaseIII');
sub PhaseURL {
    my $cgi = shift;
    my $phase = shift;		# e.g., 'PhaseII', '../PhaseIII'

    my @tmp = $cgi->param('Action');
    $cgi->delete('Action');

    my $script_name = $cgi->path_info;
    $script_name =~ s#/([^/]*)$##;		# remove view
    while ($phase =~ s/^\.\.\///) {
	$script_name =~ s#/([^/]*)$##;		# ../
    }
    $script_name .= '/' . $phase . '/' . $cgi->param('View');

    my $url = $_NAME_VIEW . $script_name;
    $url .= '?' . $cgi->query_string if $cgi->query_string;

    if (@tmp && defined $tmp[0]) {
        $cgi->param('Action', @tmp);
    } else {
        $cgi->delete('Action');
    }
    return $url;
}   

### nextphaseURL($cgi)
###
### Returns the URL of the next phase in a multi-phase interface.
### Currently for use by the initialization interfaces.
sub nextphaseURL($) {
    my $cgi = shift;
    my $script = $_NAME_VIEW . $cgi->script_name;
    if ($script !~ /Step/) {
	$script = $_NAME_VIEW . $cgi->path_info;
    }
    else {
	$script =~ s#/[^/]*$#'/' . $cgi->param('View')#e;
    }
    # remove our multiphase info so we link to the top of the next interface
    $script =~ s#/PhaseI*##;
    # increment the step number
    $script =~ s/Step(\d+)/'Step' . ($1+1)/e;
    $script;
}

### nextphase($cgi)
###
### Generate HTML to link to the next phase of a multi-phase interface.
### Currently for use by the initialization interfaces.
sub nextphase($) {
    my $cgi = shift;
    my $url = nextphaseURL($cgi);
    "<H1><A HREF=\"$url\">Proceed to the next stage</A></H1>";
}

### cgi_header($content_type[, $cacheok])
###
### Generate the CGI return header *if* not already generated (state
### is kept in the global $cgi_header_generated).  This statefullness
### is mainly used for reporting errors since an error could happen at
### any time and HTTP headers in the middle of your data look pretty
### cheezy (IMHO, it is a major bug in HTTP that it does have any
### mid-stream error handling capabilities).
###
### The caller must do the actual output to the client.  If $cacheok 
### is passed then ``Pragma: nocache'' is not included in the output.
### This can be used on static pages but is not the default since
### most of the AdminWEB pages are dynamic.
sub cgi_header {
    return '' if $cgi_header_generated; $cgi_header_generated = 1;
    my $headers;
    $headers = "Pragma: nocache\r\n" unless defined $_[1];
    $headers .= "Content-type: $_[0]\r\n\r\n";
    $headers;
}

### ($path, $view) = viewpath(cgi_path_info)
sub viewpath {
    my $path = shift;
    $path =~ s#/([^/]*\.view)##;
    ($path, $1);
}

### list_views($cgi, $properties)
###
### Generate a nice list of other views for this interface.
sub list_views {
    my ($cgi, $properties) = @_;
    my (@views, $pretty, $view);

    my ($path, $curview) = viewpath($cgi->path_info);

    if (ishelp($cgi)) {
	$curview = 'Help';
	push(@views, "<I>$curview</I>");
    }
    else {
	push(@views, join('', '<A HREF="', help_url($cgi), '">Help</A>'));
    }

    foreach $view (split(/ /, $properties->value('View_List'))) {
	$pretty = $view;
	$pretty =~ s/\.view$//;
	if ($view eq $curview) {
	    push (@views, "<I>$pretty</I>");
	}
	else {
	    push (@views, join('', '<A HREF="', $_NAME_VIEW,
		$path, '/', $view,
		($cgi->query_string eq '' ? '' : ('?', $cgi->query_string)),
		'">', $pretty, '</A>'));
	}
    }

    if (initmode($cgi)) {
	push(@views, '<A HREF="' . nextphaseURL($cgi) . '">SKIP THIS STEP</A>');
    }

    "<P>\n<B>View(s):</B> " . join(", ", @views) . "\n";
}

sub crop {
    my ($value, $min, $max) = @_;
    return ($value <= $min ? $min : ($value > $max ? $max : $value));
}

1;
