# Copyright (c) 1995, 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#	BSDI View.pm,v 1.9 1998/06/05 15:46:42 sanders Exp
#
# AdminWEB Interface View Parser and HTML Generator
#
# Reads the view datafile and builds an HTML interface from it.
#
# Should probably be broken into several classes and cleaned up.

package AdminWEB::View;
require HTML::Markup::HTML2;
use AdminWEB::Support;
use AdminWEB::Paths;
use Carp;

BEGIN {
    ### ``method name'' to color map
    %colors = (
	'Create' => '#0000c0',
	'Submit' => '#0000c0',
	'Test'   => '#007000',
	'Help'   => '#c000c0',
	'Reset'  => '#c00000',
	'Bookmark'	=> '#007000',
    );

    ### Misc configuration items
    $DEF_PACKAGE = 'main';

    ### Contexts for evaluate()
    $STRING = 0;
    $ARRAY  = 1;
    $HASH   = 2;

    ### eval types
    # $LIST   = ord('[');			# ]
    # $CODE   = ord('{');			# }

    ### $format array indicies for Parser()
    $CONTEXT = 0;
    $VALUE = 1;
}

### STATIC METHOD: Create($cgi, $navigator, $properties)
###
### Builds an object of this type.
sub Create {
    my $self = bless { }, shift;

    my $cgi = $self->{'CGI'} = shift;			# CGI
    $self->{'Navigator'} = shift;			# DB::Properties
    my $properties = $self->{'Properties'} = shift;	# DB::Properties 

    $self->{'ScriptName'} = $cgi->url . $cgi->path_info;	# Ourself
    $self->{'Query'} = $cgi->query_string;			# CGI state
    $self->{'HelpMode'} = $cgi->script_name eq $_NAME_HELP;	# Help boolean
    $self->{'HeaderNote'} = undef;

    my $path;
    ($path, $self->{'View'}) = viewpath($cgi->path_info);
    $self;
}

### METHOD: HTML()
###
### Returns an HTML string built from the state of the object.
sub HTML {
    my $self = shift;
    return ($self->{'HTMLObject'})->realize;
}

### METHOD: create_standard_form()
###
### Builds the interface HTML object structure.
### The details are filled in by load().
sub create_standard_form {
    my $self = shift;

    my $html = $self->{'HTMLObject'} = Create HTML::Markup::HTML2;

    my $head = $self->{'HTMLObject.HEAD'} = $html->HEAD;

    $self->{'HTMLObject.TITLE'} =
	$head->TITLE->Text('BSDI MaxIM: ');

    my $body = $html->BODY($self, 'HTMLObject.BODY');
    $body->attribute('bgcolor', "a5dbc6");
    $body->attribute('text', "000000");
    $body->attribute('link', "006600");
    $body->attribute('vlink', "996633");

    my $form = $body->FORM($self, 'HTMLObject.FORM')->method('POST');

    ### encode the current View state in a hidden field
    $form->INPUT->name('View')->type('hidden')->value($self->{'View'});

    my $debug = $self->{'CGI'}->param('Debug')
	|| $self->{'Properties'}->value('Debug')
	|| $self->{'Navigator'}->value('Debug');
    $form->INPUT->name('Debug')->type('hidden')->value($debug) if $debug;

    my $initp = initmode($self->{'CGI'});
    $form->INPUT->name('Initialize')->type('hidden')->value($initp) if $initp;

    ### top table with action buttons
    my $table = $form->BANNER->TABLE->width('100%')->border(0)->cellspacing(4);
    my $tr = $table->TR;

    my $td = $tr->TD($self, 'HTMLObject.HEADER')
	->valign($self->{'HelpMode'} ? 'top' : 'center');

    $tr->TD($self, 'HTMLObject.ACTIONS');
    $self->{'HTMLObject.ACTIONS'}->align('right')->nowrap(1)
	unless $self->{'HelpMode'};

    ### Main interface table
    $table = $form->TABLE->width('100%')->border(2)->cols(4);
    $table->cellpadding($self->{'HelpMode'} ? 6 : 4);

    my $navigator = $self->{'Navigator'};
    my $a = $form;
    if ($navigator->value('Navigator_ViewIcon')) {
	$a = $form->A->href($navigator->value('Navigator_ViewIconURL'))
	    if $navigator->value('Navigator_ViewIconURL');
	$a->IMG->alt($navigator->value('Navigator_ViewIconAlt'))
		->src($navigator->value('Navigator_ViewIcon'))
		->align('right')->border(0)
    }
    $form->A->href('/')->IMG->alt('[Navigator]')
	->src('/Icons/NavButton.gif')->border(0)->align('right');

    ### init interface table data structures
    push(@{$self->{'HTMLObject.TABLE'}}, $table);
    push(@{$self->{'HTMLObject.TR'}}, $table->TR);
    push(@{$self->{'HTMLObject.I'}}, 0);		# current column index
    push(@{$self->{'HTMLObject.Elements'}}, 0);		# 0 = infinite
    push(@{$self->{'HTMLObject.Columns'}}, 4);		# number of columns

    $self;
}

### abort($msg[, $viewline])
###
### Reports errors during loading of the view file.
### Expects $viewline and $viewfile in scope.
sub abort {
    no strict 'vars';

    my $msg = shift;
    my $line = shift || $viewline;
    die "ERROR: line $line of $viewfile\n$msg\n";
}

### METHOD: load($view)
###
### Load ``$view'' which should be a viewfile
sub load {
    my $self = shift;
    local $viewfile = shift;
    local $viewline = 0;
    local *VIEWFILE;
    my ($type, $field, $value, $eval, @record);

    $self->create_standard_form();

    # Read the data file one record at a time into our
    # data structure and call the handler for each Type:
    # Records are processed in order.

    open(VIEWFILE, "< $viewfile\0") || die "$viewfile: $!\n";
    local($/, $_) = "\n";

    ### init parser state
    @record = ();
    $type = $field = undef;
    while ($_ = <VIEWFILE>) {
	$viewline++;

	### skip over *between record* comments and blank lines
	next if (! defined $field) && (/^\s*#/ || /^\s*$/);

	### process line
	if (/^\s*#/ || /^\s*$/) {
	    ### do nothing, comments get ignored, end of record processed later
	}
	elsif (s/^(\s*>|\s+)//) {		# continuation
	    abort("No previous field for continuation") unless $field;
	    chop;			# not chomp, we really want chop
	    ### append to the previous entry in @record
	    $record[$#record]->{'value'} .= "\n" . $_;	# append to previous
	}
	elsif (m/^(\S+):\s*(.*)/) {
	    ### field
	    $field = $1;
	    $value = $2;
	    $eval = ($value =~ m/^([\[\{])/ ? ord($1) : 0);	# }]
	    # A literal prefix escapes eval's, remove it now
	    $value =~ s/^>//;

	    ### Type: is handled here as a special case because we need
	    ### it to call the correct routine to handle the record.
	    ### It is not made available in the record itself because
	    ### the routine already knows it's own type and would just
	    ### have to have extra code to ignore it.
	    if ($field eq 'Type') {
		abort("Duplicate Type: entry") if defined $type;
		$type = $self->evaluate($STRING,
		    { 'value' => $value, 'eval' => $eval, 'line', $viewline });
		abort("No such View record type: $type")
		    unless defined &{"Parse_" . $type};
	    }
	    else {
		### create entry in @record
		push(@record, {
		    'field' => $field,
		    'value' => $value,
		    'eval' => $eval,
		    'line' => $viewline,
	       });
	    }
	}
	else {
	    abort("Not a valid field or continuation line");
	}

	### Process end of record
	if (/^\s*$/ || eof(VIEWFILE)) {
	    abort("Previous record did not have a Type: field")
		unless defined $type;
	    eval qq{ \$self->Parse_${type}(\\\@record) };
	    if ($@) {
		my($a, $msg, $line, $file);
		if ($@ =~ /^ERROR:/) {
		    ($a, $msg) = split(/\n/, $@, 2);
		    ($line, $file) = $a =~ m/^ERROR: line (\d+) of (.*)/;
		}
		else {
		    ($msg, $file, $line) = $@ =~ m/(.*) at (\S+) line (\d+)\.$/;
		    $msg = "Unknown error ($@)" unless defined $msg;
		    $file = "Unknown file" unless defined $file;
		    $line = "Unknown line" unless defined $line;
		}

		print "Content-Type: text/html\r\n\r\n";

		my $html = Create HTML::Markup::HTML2;
		$html->HEAD->TITLE->Text("MaxIM: View Error -- $file");
		my $body = $html->BODY;
		$body->H1->Text("Error in View: Line $line");
		$body->P->Text("Processing entry of type: $type");
		$body->P->PRE->Text($msg);

		print $html->realize;
		exit(1);
	    }

	    ### reset parser state
	    @record = ();
	    $type = $field = undef;
	}
    }

    close(VIEWFILE);

    $self;
}

### METHOD: evaluate($context, \%record)
###
### Evaluates the given record (HASH REF) in the
### ($STRING, $ARRAY, or $HASH) $context.
### %record should have the following elements:
###     'field' => not used here but should be the field name
###     'value' => the value of the field
###     'eval'  => true if we need to eval
###     'line'  => line number for reporting errors
sub evaluate {
    local $self = shift;
    my ($context, $record) = @_;
    my $value;

    ### unset items evaluate to undef
    return undef unless (defined $record && defined $record->{'value'});

    $value = $record->{'value'};

    # 'eval' is $ARRAY or $CODE but we don't really need to know that
    if ($record->{'eval'}) {
	my $package = $self->{'View::Package'} || $DEF_PACKAGE;
	${$package . '::cgi'} = $self->{'CGI'};
	${$package . '::navigator'} = $self->{'Navigator'};
	${$package . '::properties'} = $self->{'Properties'};
	$value = eval qq{
	    package $package;
	    $value
	};
	abort("$@", $record->{'line'}) if ($@);
    }

    $context == $STRING && do {
	return join(' ', @$value) if ref $value eq 'ARRAY';
	abort("Cannot use HASH in STRING context", $record->{'line'})
	    if ref $value eq 'HASH';
	return "$value";		# stringify it
    };
    $context == $ARRAY && do {
	return $value if ref $value eq 'ARRAY';
	abort("Cannot use HASH in ARRAY context", $record->{'line'})
	    if ref $value eq 'HASH';
	return [ ] if $value eq '';		# handle empty value
	return [ $value ];
    };
    $context == $HASH && do {
	return { @$value } if ref $value eq 'ARRAY';
	return $value if ref $value eq 'HASH';
	return { } if $value eq '';		# handle empty value
	abort("LITERAL or unknown value in HASH context", $record->{'line'});
    };

    abort("Programming Error: Invalid Context: $context", $record->{'line'});
}

### METHOD: Parser($format, $record)
###
### Parses a record given the record data structure.
sub Parser {
    my $self = shift;
    my ($format, $record) = @_;
    my ($i, $field, $what);

    while ($i = shift(@$record)) {
	$field = $i->{'field'};
	$what = $format->{$field};
	abort("Invalid keyword ``" . $i->{'field'} . "''", $i->{'line'})
		unless defined $what;
	$what->[$VALUE] = $self->evaluate($what->[$CONTEXT], $i);
    }
}

### METHOD: Required($format, @list)
###
### Make sure that the @list entries in $format have a value.
sub Required {
    my $self = shift;
    my $format = shift;
    my $i;

    foreach $i (@_) {
	abort("Programming Error: Invalid field $i")
	    unless defined $format->{$i};
	abort("Missing required field $i")
	    unless $format->{$i}->[$VALUE] ne '';
    }
}

### METHOD: Values($format, @list)
###
### Returns an array of values for the entries in @list.
sub Values {
    my $self = shift;
    my $format = shift;
    my ($i, @values);

    foreach $i (@_) {
	abort("Programming Error: Invalid field $i")
	    unless defined $format->{$i};
	push (@values, $format->{$i}->[$VALUE]);
    }
    @values;
}

### METHOD: Parse_View(\@record)
###
### Parses records of type View.
sub Parse_View {
    my $self = shift;
    my $record = shift;
    my $i;
    my @actions;

    abort ("Only one record type=View allowed") if $self->{'View::Initialized'};
    $self->{'View::Initialized'} = 1;

    while ($i = shift(@$record)) {
	$i->{'field'} eq 'Package'	&& do {
	    $self->{'View::Package'} = $self->evaluate($STRING, $i);
	    next;
	};
	$i->{'field'} eq 'Initialize'	&& do {
	    $self->evaluate($STRING, $i);	# for its side-effects only
	    next;
	};
	$i->{'field'} eq 'Dynamic' && do {
	    my $s = $self->evaluate($STRING, $i);
	    if (affirmative($s)) {
		# force browser to reload page as often as possible
		$self->{'HTMLObject.HEAD'}->META
		    ->attribute('http-equiv', 'Expires')
		    ->content('Thu, 01 Jan 1970 00:00:00 GMT');
	    }
	    next;
	};
	$i->{'field'} eq 'Head'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    $self->{'HTMLObject.HEAD'}->Literal($s);
	    next;
	};
	$i->{'field'} eq 'TitleBar'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    ($self->{'HTMLObject.TITLE'})->append($s);
	    next;
	};
	$i->{'field'} eq 'Header'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    my $head = $self->{'HTMLObject.HEADER'}->H1;
	    if ($self->{'HelpMode'}) {
		$head->Text($s);
	    }
	    else {
		$head->A->href(help_url($self->{'CGI'}))->Text($s);
	    }
	    next;
	};
	$i->{'field'} eq 'HeaderNote'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    $self->{'HeaderNote'} = $s;
	    next;
	};
	# JavaScript Support
	$i->{'field'} eq 'Reset_onClick'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    $self->{'Reset_onClick'} = $i;
	    next;
	};
	$i->{'field'} eq 'Action_onClick'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    $self->{'Action_onClick'} = $i;
	    next;
	};
	$i->{'field'} eq 'Actions'	&& do {
	    my $s = $self->evaluate($ARRAY, $i);
	    push(@actions, @{$s});
	    next;
	};
	$i->{'field'} eq 'ActionURL'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    $self->{'HTMLObject.ActionURL'} = $s;
	    next;
	};
	$i->{'field'} eq 'Trailer'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    ($self->{'HTMLObject.FORM'})->Literal($s);
	    next;
	};
	$i->{'field'} eq 'Help'		&& do {
	    my $s = $self->evaluate($STRING, $i);
	    ### View help text replaces the submit buttons
	    if ($self->{'HelpMode'}) {
		($self->{'HTMLObject.ACTIONS'})->width('45%')->Literal($s);
	    }
	    next;
	};
	# JavaScript Support
	$i->{'field'} eq 'onLoad'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    $self->{'HTMLObject.BODY'}->onLoad($s);
	    next;
	};
	$i->{'field'} eq 'onUnload'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    $self->{'HTMLObject.BODY'}->onUnload($s);
	    next;
	};
	$i->{'field'} eq 'onSubmit'	&& do {
	    my $s = $self->evaluate($STRING, $i);
	    $self->{'HTMLObject.FORM'}->onSubmit($s);
	    next;
	};
	# specify netscape TARGET=... for form submit
	$i->{'field'} eq 'Target'	&& do {
	    my $t = $self->evaluate($STRING, $i);
	    $self->{'HTMLObject.FORM'}->target($t);
	    next;
	};
	abort("Invalid keyword ``" . $i->{'field'} . "''", $i->{'line'});
    }

    # TODO: need to Require: TitleBar, Header, Actions, and ActionURL

    ### Process Parse_View
    $self->{'HTMLObject.FORM'}->action($self->{'HTMLObject.ActionURL'});

    $self->Actions(@actions);

    $self->{'HTMLObject.HEADER'}->P->Literal($self->{'HeaderNote'})
	if defined $self->{'HeaderNote'};

    $self;
}

### METHOD: Actions(@list)
###
### Support method for building the list of submit buttons.
### Called by Parse_View().
sub Actions {
    my $self = shift;
    my $submit = $self->{'Properties'}->value('View_Submit') || 'Action';
    my $i;

    return $self if $self->{'HelpMode'};

    # %colors is package global

    my $td = $self->{'HTMLObject.ACTIONS'};
    my $input;

    foreach $i (@_) {
	# XXX: convert "Bookmark" into "Save Defaults"
	next if $i eq 'Bookmark';	# bookmark is special-cased
	if ($colors{$i}) {
	    $input = $td->FONT->color($colors{$i})->INPUT;
	} else {
	    $input = $td->INPUT;
	}
	$input->type('submit')->name($submit)->value($i);
	$input->onClick($self->{'Action_onClick'})
	    if $self->{'Action_onClick'} ne '';
    }

    # Every form gets a Reset Button
    my $reset = $td->FONT->color($colors{'Reset'})
	->INPUT->type('reset')->value('Reset');
    $reset->onClick($self->{'Reset_onClick'})
	    if $self->{'Reset_onClick'} ne '';

    $td->A->href(help_url($self->{'CGI'}))->IMG->alt('[*--HELP--*]')
	->src('/Icons/Help.gif')->border(0)->align('absmiddle');

    $self;
}

### METHOD: Parse_Header(\@record)
###
### Parses records of type Header.
sub Parse_Header {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Attrs'		=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Label');

    $self->header($self->Values($format, 'Label', 'Attrs'));

    $self;
}

### METHOD: Parse_Const(\@record)
###
### Parses records of type Const (constant text literal)
sub Parse_Const {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Name'		=> [ $STRING, undef ],
	'Value'		=> [ $STRING, undef ],
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    # constant text literal
    my ($name, $value, $helptext) =
	$self->Values($format, 'Name', 'Value', 'Help');
    my $tr = $self->newElement();

    if ($self->{'HelpMode'}) { $tr->TD->Literal($helptext); }
    else {
	my $td = $tr->TD;
	$td->string_attributes($self->Values($format, 'Attrs'));
	$td->Literal($value);
	$td->INPUT->type('hidden')->name($name)->value($value) if $name ne '';
    }

    $self;
}

### METHOD: Parse_Text(\@record)
###
### Parses records of type Text.
sub Parse_Text {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Name'		=> [ $STRING, undef ],
	'Size'		=> [ $STRING, undef ],
	'Value'		=> [ $STRING, undef ],
	'MaxLength'	=> [ $STRING, undef ],
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
	'onBlur'	=> [ $STRING, undef ],
	'onChange'	=> [ $STRING, undef ],
	'onFocus'	=> [ $STRING, undef ],
	'onSelect'	=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Name');

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    $self->input('text', $self->Values($format, 'Name', 'Size', 'Value',
	'MaxLength', 'Help', 'Attrs',
	'onBlur', 'onChange', 'onFocus', 'onSelect'));

    $self;
}

### METHOD: Parse_TextArea(\@record)
###
### Parses records of type TextArea.
sub Parse_TextArea {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Name'		=> [ $STRING, undef ],
	'Rows'		=> [ $STRING, undef ],
	'Cols'		=> [ $STRING, undef ],
	'Wrap'		=> [ $STRING, undef ],
	'Value'		=> [ $STRING, undef ],
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
	'onBlur'	=> [ $STRING, undef ],
	'onChange'	=> [ $STRING, undef ],
	'onFocus'	=> [ $STRING, undef ],
	'onSelect'	=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Name');

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    $self->textarea($self->Values($format, 'Name', 'Rows', 'Cols',
	'Wrap', 'Value', 'Help', 'Attrs',
	'onBlur', 'onChange', 'onFocus', 'onSelect'));

    $self;
}

### METHOD: Parse_Password(\@record)
###
### Parses records of type Password.
sub Parse_Password {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Name'		=> [ $STRING, undef ],
	'Size'		=> [ $STRING, undef ],
	'Value'		=> [ $STRING, undef ],
	'MaxLength'	=> [ $STRING, undef ],
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
	'onBlur'	=> [ $STRING, undef ],
	'onChange'	=> [ $STRING, undef ],
	'onFocus'	=> [ $STRING, undef ],
	'onSelect'	=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Name');

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    my $tr = $self->newElement();
    if ($self->{'HelpMode'}) { $tr->TD->Literal($format->{'Help'}->[$VALUE]); }
    else {
        my $td = $tr->TD->string_attributes($attrs);
	my $count;
	for($count = 1; $count <= 2; $count++) {
	    my $thing = $td->INPUT;
	    $thing->type('password');
	    $thing->name($self->Values($format, 'Name'))
		if defined($self->Values($format, 'Name'));
	    $thing->size($self->Values($format, 'Size'))
		if defined($self->Values($format, 'Size'));
	    $thing->value($self->Values($format, 'Value'))
		if defined($self->Values($format, 'Value'));
	    $thing->maxlength($self->Values($format, 'MaxLength'))
		if defined($self->Values($format, 'MaxLength'));
	    # JavaScript Extensions
	    $thing->onBlur($self->Values($format, 'onBlur'))
		if defined($self->Values($format, 'onBlur'));
	    $thing->onChange($self->Values($format, 'onChange'))
		if defined($self->Values($format, 'onChange'));
	    $thing->onFocus($self->Values($format, 'onFocus'))
		if defined($self->Values($format, 'onFocus'));
	    $thing->onSelect($self->Values($format, 'onSelect'))
		if defined($self->Values($format, 'onSelect'));
	    $td->BR;
	}
    }

    $self;
}

### METHOD: Parse_Hidden(\@record)
###
### Parses records of type Hidden.
sub Parse_Hidden {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Name'		=> [ $STRING, undef ],
	'Value'		=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Name');

    if (! $self->{'HelpMode'}) {
	my $thing = $self->{'HTMLObject.FORM'}->INPUT;
	$thing->type('hidden');
	$thing->name($self->Values($format, 'Name'));
	$thing->value($self->Values($format, 'Value'));
    }

    $self;
}

### METHOD: Parse_Empty(\@record)
###
### Parses records of type Empty.
sub Parse_Empty {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);

    my $tr = $self->newElement();
    if ($self->{'HelpMode'} && $format->{'Help'}->[$VALUE]) {
	$tr->TD->Literal($format->{'Help'}->[$VALUE]);
    }
    else {
	$tr->TD->string_attributes($self->Values($format, 'Attrs'))->BR;
    }

    $self;
}

### METHOD: Parse_Select(\@record)
###
### Parses records of type Select.
sub Parse_Select {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Name'		=> [ $STRING, undef ],
	'Size'		=> [ $STRING, undef ],
	'List'		=> [ $ARRAY,  undef ],
	'Selected'	=> [ $STRING, undef ],
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
	'onBlur'	=> [ $STRING, undef ],
	'onChange'	=> [ $STRING, undef ],
	'onFocus'	=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Name', 'List');

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    $self->nlist(
	$self->Values($format, 'Name'),
	0,
	$self->Values($format, 'Size', 'List'),
	[ $self->Values($format, 'Selected') ],
	$self->Values($format, 'Help', 'Attrs',
	    'onBlur', 'onChange', 'onFocus'));

    $self;
}

### METHOD: Parse_Select_Multi(\@record)
###
### Parses records of type Select_Multi.
sub Parse_Select_Multi {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Name'		=> [ $STRING, undef ],
	'Size'		=> [ $STRING, undef ],
	'List'		=> [ $ARRAY,  undef ],
	'Selected'	=> [ $ARRAY,  undef ],
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
	'onBlur'	=> [ $STRING, undef ],
	'onChange'	=> [ $STRING, undef ],
	'onFocus'	=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Name', 'List');

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    $self->nlist(
	$self->Values($format, 'Name'),
	1,
	$self->Values($format, 'Size', 'List'),
	$self->Values($format, 'Selected'),
	$self->Values($format, 'Help', 'Attrs',
	    'onBlur', 'onChange', 'onFocus'));

    $self;
}

### METHOD: Parse_Select_Popup(\@record)
###
### Parses records of type Select_Popup.
sub Parse_Select_Popup {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Name'		=> [ $STRING, undef ],
	'List'		=> [ $ARRAY,  undef ],
	'Selected'	=> [ $STRING, undef ],
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
	'onBlur'	=> [ $STRING, undef ],
	'onChange'	=> [ $STRING, undef ],
	'onFocus'	=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Name', 'List');

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    $self->nlist(
	$self->Values($format, 'Name'),
	0, 0,
	$self->Values($format, 'List'),
	[ $self->Values($format, 'Selected') ],
	$self->Values($format, 'Help', 'Attrs',
	    'onBlur', 'onChange', 'onFocus'));

    $self;
}

### METHOD: Parse_Select_Radio(\@record)
###
### Parses records of type Select_Radio.
sub Parse_Select_Radio {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Name'		=> [ $STRING, undef ],
	'Border'	=> [ $STRING, '2' ],
	'Fill'		=> [ $STRING, undef ],
	'Columns'	=> [ $STRING, undef ],
	'List'		=> [ $ARRAY,  undef ],
	'Links'		=> [ $HASH,   undef ],
	'Checked'	=> [ $STRING, undef ],
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
	'onClicks'	=> [ $ARRAY,  undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Name', 'Columns', 'Links');

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    $self->col_major_table(
	$self->Values($format, 'Columns', 'Name', 'Border'),
	'radio', [ $self->Values($format, 'Checked') ],
	$self->Values($format, 'List', 'Links', 'Help', 'Attrs', 'onClicks'));

    $self;
}

### METHOD: Parse_Select_Checkbox(\@record)
###
### Parses records of type Select_Checkbox.
sub Parse_Select_Checkbox {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=> [ $STRING, undef ],
	'Name'		=> [ $STRING, undef ],
	'Border'	=> [ $STRING, '2' ],
	'Fill'		=> [ $STRING, undef ],
	'Columns'	=> [ $STRING, undef ],
	'List'		=> [ $ARRAY,  undef ],
	'Links'		=> [ $HASH,   undef ],
	'Checked'	=> [ $ARRAY,  undef ],
	'Attrs'		=> [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=> [ $STRING, undef ],
	'onClicks'	=> [ $ARRAY,  undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Name', 'Columns', 'Links');

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    $self->col_major_table(
	$self->Values($format, 'Columns', 'Name', 'Border'),
	'checkbox', $self->Values($format, 'Checked'),
	$self->Values($format, 'List', 'Links', 'Help', 'Attrs', 'onClicks'));

    $self;
}

### METHOD: Parse_Select_Input(\@record)
###
### Parses records of type Select_Input.
sub Parse_Select_Input {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=>  [ $STRING, undef ],
	'Select_Name'	=>  [ $STRING, undef ],
	'Select_List'	=>  [ $ARRAY,  undef ],
	'Select_Selected'=> [ $STRING, undef ],
	'Input_Label'	=>  [ $STRING, undef ],
	'Input_Name'	=>  [ $STRING, undef ],
	'Input_Size'	=>  [ $STRING, undef ],
	'Input_Value'	=>  [ $STRING, undef ],
	'Input_MaxLength'=> [ $STRING, undef ],
	'Attrs'		=>  [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=>  [ $STRING, undef ],
	'Select_onBlur'	=>  [ $ARRAY,  undef ],
	'Select_onChange'	=>  [ $ARRAY,  undef ],
	'Select_onFocus'	=>  [ $ARRAY,  undef ],
	'Input_onBlur'	=>  [ $STRING, undef ],
	'Input_onChange'	=>  [ $STRING, undef ],
	'Input_onFocus'	=>  [ $STRING, undef ],
	'Input_onSelect'	=>  [ $STRING, undef ],
    };
    $self->Parser($format, $record);
    $self->Required($format, 'Select_Name', 'Select_List', 'Input_Name');

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    $self->list_or_input(
	$self->Values($format, 'Select_Name', 'Select_List',
	    'Select_Selected', 'Input_Label'),
	'text',
	$self->Values($format, 'Input_Name', 'Input_Size', 'Input_Value',
	    'Input_MaxLength', 'Help', 'Attrs', 'Input_onBlur',
	    'Input_onChange', 'Input_onFocus', 'Input_onSelect',
	    'Select_onBlur', 'Select_onChange', 'Select_onFocus'));

    $self;
}

### METHOD: Parse_Literal(\@record)
###
### Lets the user specify their own HTML for the entry.
sub Parse_Literal {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Label'		=>  [ $STRING, undef ],
	'HTML'		=>  [ $STRING, undef ],
	'Attrs'		=>  [ $STRING, 'ALIGN="center" VALIGN="center"' ],
	'Help'		=>  [ $STRING, undef ],
    };
    $self->Parser($format, $record);

    $self->header($self->Values($format, 'Label'))
	if $format->{'Label'}->[$VALUE] ne '';

    my $tr = $self->newElement();

    if ($self->{'HelpMode'}) { $tr->TD->Literal($format->{'Help'}->[$VALUE]); }
    else {
	$tr->TD->string_attributes($format->{'Attrs'}->[$VALUE])
	    ->Literal($format->{'HTML'}->[$VALUE]);
    }

    $self;
}

### METHOD: Parse_Span(\@record)
###
### Parses records of type Span.
sub Parse_Span {
    my $self = shift;
    my $record = shift;

    my $format = {
	'Attrs'		=> [ $STRING, undef ],
	'Rowspan'	=> [ $STRING, 1 ],
	'Colspan'	=> [ $STRING, 1 ],
	'Elements'	=> [ $STRING, undef ],
	'Columns'	=> [ $STRING, undef ],
	'Literal'	=> [ $STRING, undef ],
    };
    $self->Parser($format, $record);

    if ($format->{'Literal'}->[$VALUE]) {
	abort("Columns cannot be used with Literal in Span")
	    if $format->{'Columns'}->[$VALUE];
	abort("Elements cannot be used with Literal in Span")
	    if $format->{'Elements'}->[$VALUE];

	my $outer_tr = $self->newElement();
	my $new_table = $outer_tr->TD->
	    rowspan($format->{'Rowspan'}->[$VALUE])->
	    colspan($format->{'Colspan'}->[$VALUE])->
	    TABLE;
	if (defined $format->{'Attrs'}->[$VALUE]) {
	    $new_table->string_attributes($format->{'Attrs'}->[$VALUE]);
	}
	else {
	    $new_table->width('100%');
	}
	$new_table->Literal($format->{'Literal'}->[$VALUE]);

	return $self;
    }

    $self->Required($format, 'Elements', 'Columns');

    # create subtable
    my $outer_tr = $self->newElement();
    my $new_table = $outer_tr->TD->
	    rowspan($format->{'Rowspan'}->[$VALUE])->
	    colspan($format->{'Colspan'}->[$VALUE])->
	    TABLE;
    if (defined $format->{'Attrs'}->[$VALUE]) {
	$new_table->string_attributes($format->{'Attrs'}->[$VALUE]);
    }
    else {
	$new_table->width('100%');
    }

    # XXX: Yikes, we need to account for the rows and columns used in
    # XXX: the span.  That's going to be a major, major pain.
    # XXX: ---
    # XXX: A simple stack of "column inits" that default to 1 when empty
    # XXX: would work for a single Span in a row -- but handling multiple
    # XXX: spans in a row is harder.

    # increment the current I by the number of columns we are spanning
    my $i = $self->{'HTMLObject.I'};
    $i->[$#$i] += $format->{'Colspan'}->[$VALUE];

    ### push down a table structure
    push(@{$self->{'HTMLObject.TABLE'}}, $new_table);
    push(@{$self->{'HTMLObject.TR'}}, $new_table->TR);
    push(@{$self->{'HTMLObject.I'}}, 0);
    push(@{$self->{'HTMLObject.Elements'}}, $format->{'Elements'}->[$VALUE]+1);
    push(@{$self->{'HTMLObject.Columns'}}, $format->{'Columns'}->[$VALUE]);

    $self;
}

###
### Interface Building Support Methods
###

### METHOD: newElement()
###
### Prepares the object structure for a new table element to be added and
### returns the HTML::Markup::HTML::TR object.  Handles recursive tables.
sub newElement {
    my $self = shift;

    ### these are all ARRAY REF's
    my $table = $self->{'HTMLObject.TABLE'};
    my $tr = $self->{'HTMLObject.TR'};
    my $i = $self->{'HTMLObject.I'};
    my $ele = $self->{'HTMLObject.Elements'};
    my $cols = $self->{'HTMLObject.Columns'};

    # XXX: Need to handle spans better

    ### If we have run out of elements then pop the stacks.
    while ($ele->[$#$ele] && --$ele->[$#$ele] == 0) {
	pop(@$table); pop(@$tr); pop(@$i); pop(@$ele); pop(@$cols);
    }

    ### Increment the column index, if we're out of columns start a new row.
    # should never be > except for my kludge above to make spans sort of work
    if ($i->[$#$i]++ >= $cols->[$#$cols]) {
	$i->[$#$i] = 1;
	$tr->[$#$tr] = $table->[$#$table]->TR;
    }

    return $tr->[$#$tr];
}

### METHOD: header($header_text, $attrs)
###
### Creates HTML TH (TableHeader) element with the approriate
### anchor and text; conditional on help state.
sub header {
    my $self = shift;
    my $label = shift;
    my $attrs = shift;
    my $anchor = $label;

    ### fixup the anchor a bit
    $anchor =~ y/A-Z/a-z/;
    $anchor =~ s/ /_/g;

    my $tr = $self->newElement();

    if ($self->{'HelpMode'}) {
	$tr->TH->A->name($anchor)->Literal($label);
    }
    else {
	my $th = $tr->TH;
	$th->string_attributes($attrs);
	### XXX: Select the Help button now
	# $th->A->href(help_url($self->{'CGI'}, '#', $anchor))->Literal($label);
	$th->Literal($label);
    }

    $self;
}

### METHOD: input($type, $name, $size, $value, $helptext, $attrs,
###		$onBlur, $onChange, $onFocus, $onSelect)
###
### Create HTML TD (TableData) element with either an input field
### or help text; conditional on help state.
sub input {
    my $self = shift;
    my ($type, $name, $size, $value, $maxlen, $helptext, $attrs,
	$onBlur, $onChange, $onFocus, $onSelect) = @_;

    my $tr = $self->newElement();

    if ($self->{'HelpMode'}) { $tr->TD->Literal($helptext); }
    else {
	my $td = $tr->TD->string_attributes($attrs);
	my $thing = $td->INPUT;
	$thing->type($type) if defined($type);
	$thing->name($name) if defined($name);
	$thing->size($size) if defined($size);
	$thing->value($value) if defined($value);
	$thing->maxlength($maxlen) if defined($maxlen);
	# JavaScript Extensions
	$thing->onBlur($onBlur) if defined($onBlur);
	$thing->onChange($onChange) if defined($onChange);
	$thing->onFocus($onFocus) if defined($onFocus);
	$thing->onSelect($onSelect) if defined($onSelect);
    }

    $self;
}

### METHOD: textarea($name, $rows, $cols, $wrap, $value, $helptext,
###		$attrs, $onBlur, $onChange, $onFocus, $onSelect)
###
### Create HTML TD (TableData) element with a textarea input
### or help text; conditional on help state.
sub textarea {
    my $self = shift;
    my ($name, $rows, $cols, $wrap, $value, $helptext,
	$attrs, $onBlur, $onChange, $onFocus, $onSelect) = @_;

    my $tr = $self->newElement();

    if ($self->{'HelpMode'}) { $tr->TD->Literal($helptext); }
    else {
	my $td = $tr->TD->string_attributes($attrs);
	my $thing = $td->TEXTAREA;
	$thing->name($name) if defined($name);
	$thing->rows($rows) if defined($rows);
	$thing->cols($cols) if defined($cols);
	$thing->wrap($wrap) if defined($wrap);
	$thing->Literal($value) if defined($value);
	# JavaScript Extensions
	$thing->onChange($onChange) if defined($onChange);
	$thing->onFocus($onFocus) if defined($onFocus);
	$thing->onBlur($onBlur) if defined($onBlur);
	$thing->onClick($onclick) if defined($onclick);
    }

    $self;
}

### METHOD: list_or_input($listname, \@list, $selected,
###        $label, $type, $name, $size, $value, $helptext,
###        $attrs, $onBlur, $onChange, $onFocus, $onSelect,	# for text
###	   $sel_onBlur, $sel_onChange, $sel_onFocus)
###
### Create HTML TD (TableData) element with either a list|input
### type interface or help text; conditional on help state.
sub list_or_input {
    my $self = shift;
    my ($listname, $list, $selected, $label,
	$type, $name, $size, $value, $maxlen, $helptext,
	$attrs, $onBlur, $onChange, $onFocus, $onSelect,	# for text
	$sel_onBlur, $sel_onChange, $sel_onFocus) = @_;

    my $tr = $self->newElement();

    if ($self->{'HelpMode'}) {
	$tr->TD->Literal($helptext);
    }
    else {
	my ($thing, $item, $opt);
	my $td = $tr->TD->string_attributes($attrs);
	    my $sel = $td->SELECT->name($listname);
	    $sel->onBlur($sel_onBlur) if $sel_onBlur;
	    $sel->onChange($sel_onChange) if $sel_onChange;
	    $sel->onFocus($sel_onFocus) if $sel_onFocus;
	    foreach $item (@$list) {
		$opt = $sel->OPTION;
		$opt->selected if $item eq $selected;
		$opt->Text($item);
	    }
	    $td->BR;
	    $td->B->Text('OR');
	    $td->Text(' ' . $label);
	    $thing = $td->INPUT;
	    $thing->type($type) if defined($type);
	    $thing->name($name) if defined($name);
	    $thing->size($size) if defined($size);
	    $thing->value($value) if defined($value);
	    $thing->maxlength($maxlen) if defined($maxlen);
	    # JavaScript Extensions
	    $thing->onBlur($onBlur) if defined($onBlur);
	    $thing->onChange($onChange) if defined($onChange);
	    $thing->onFocus($onFocus) if defined($onFocus);
	    $thing->onSelect($onSelect) if defined($onSelect);
    }

    $self;
}

### METHOD: col_major_table($ncolumns, $name, $border, $type, \@checked,
###     \@list, \%links, $helptext, $attrs, $onClicks)
###
### Generating col-major tables is a pain so this function can really help.
### It is designed for building simple lists of radio or checkboxes and
### has the ability of hypertext linking the data via %links.
### Any entries matching @$checked will be checked.
sub col_major_table {
    my $self = shift;
    my ($ncolumns, $name, $border, $type, $checked,
	$list, $links, $helptext, $attrs, $onClicks) = @_;
    my ($item, $index, $row, $col, $rows, $cols, @rows);
    my @onClicks = @$onClicks;

    my $tr = $self->newElement();

    if ($self->{'HelpMode'}) {
	$tr->TD->Literal($helptext);
	return $self;
    }

    my $td = $tr->TD;
    $td->string_attributes($attrs);
    my $table = $td->TABLE->border($border)->width('100%');

    $cols = $ncolumns;
    # compute number of rows needed for a $cols column table
    $rows = int($#$list / $cols) + 1;
    # build all the table row objects first
    for ($row = 0; $row < $rows; $row++) {
	$rows[$row] = $table->TR;
    }
    # place elements in col-major order
    COLUMN:
    for ($col = 0; $col < $cols; $col++) {
	for ($row = 0; $row < $rows; $row++) {
	    $index = $col*$rows + $row;
	    last COLUMN unless defined($list->[$index]);
	    $item = $list->[$index];
	    $td2 = $rows[$row]->TD;
	    $i = $td2->INPUT->name($name)->type($type)->value($item);
	    $i->checked() if grep($item eq $_, @$checked);
	    my $onClick = pop @onClicks;
	    $i->onClick($onClick) if $onClick ne '';

	    if ($links->{$item}) {
		$td2->A->href($links->{$item})->Text($item);
	    }
	    else {
		$td2->Text($item);
	    }
	} # for $row
    } # for $col

    $self;
}

### METHOD: nlist($name, $multi_bool, $size, $list_ref, $selected, $helptext,
###              $attrs, $onBlur, $onChange, $onFocus)
###
### Generates a one-of-n or n-of-n (if $multi_bool is true)
### selection box input.  If $size == 0 and $multi_bool is false
### a plain SELECT (popup in Netscape) is generated.
sub nlist {
    my $self = shift;
    my ($name, $multi_bool, $size, $list, $selected, $helptext,
	$attrs, $onBlur, $onChange, $onFocus) = @_;
    my ($td, $sel, $item, $opt);

    my $tr = $self->newElement();

    if ($self->{'HelpMode'}) {
	$tr->TD->Literal($helptext);
	return;;
    }

    $td = $tr->TD->string_attributes($attrs);
    $sel = $td->SELECT->name($name);
    $sel->multiple() if $multi_bool;
    $sel->size($size) if $size > 0;
    $sel->onBlur($onBlur) if $onBlur;
    $sel->onChange($onChange) if $onChange;
    $sel->onFocus($onFocus) if $onFocus;
    foreach $item (@$list) {
	$opt = $sel->OPTION;
	$opt->selected if $selected && grep($item eq $_, @$selected);
	$opt->Text($item);
    }

    return $self;
}

1;
