# Copyright (c) 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI DomainDB.pm,v 1.4 1999/10/22 22:17:19 polk Exp
#
# Object-oriented interface to DNS Domain files.

# TODO: we don't preserve the order of A or NS records, does that matter?
# TODO: $INCLUDE needs to resolve paritial paths.
#       probably relative to $Conf->directory

package	FileFormat::DNS::DomainDB;

use FileFormat::DNS::Utils;
use FileFormat::DNS::Paths;
use Carp;
use IO::LockingFile;

# XXX: debug print
sub Dprint {
    my @stuff = caller(0);
    # print STDERR join(":", @stuff[2]), "-->", @_;
}

BEGIN {
    $timeout = 15;
}

sub new { ## new($domain, $dbfile_path, $nolock);
    my $self = bless { }, shift;
    $self->Initialize(@_);
}

# cleans up the object removing any cross-references so everything is freed
sub close {
    my $self = shift;
    return if $self->{'CLOSED'};
    $self->{'CLOSED'} = 1;

    eval { $self->write_dbfile } if $self->{'CHANGED'} && ! $self->{'READONLY'};
    print STDERR $@ if $@;
    $self->{'FH'}->close;

    # tear down the object
    my $obj;
    foreach $obj ($self->owners) { $obj->{'parent'} = undef; }
    $self->{'owners'} = { };
    $self->{'types'} = { };
    $self->{'iostack'} = '';
    $self->{'READONLY'} = 0;
    $self->{'CHANGED'} = 0;
}

sub Initialize {
    my $self = shift;
    my ($domain, $dbfile_path, $nolock) = @_;

    $self->{'FILE'} = $dbfile_path;
    $self->{'READONLY'} = 0;
    $self->{'CHANGED'} = 0;
    $self->{'CREATED'} = 0;

    $self->{'soa_domain'} = $domain;
    $self->{'owners'} = { };	# we store resource record objects by name
    $self->{'types'} = { };	# ...and by type
    $self->{'iostack'} = '';	# input stack for io functions

    $self->{'FH'} = new IO::LockingFile($dbfile_path, O_RDONLY|O_CREAT);
    die "$dbfile_path: $!\n" unless defined $self->{'FH'};
    $self->{'FH'}->exclusive($timeout) or die "$dbfile_path: $!\n"
	unless $nolock;
    $self->read_dbfile($domain, $domain) or $self->{'CREATED'} = 1;
    $self->{'READONLY'} = 1 if $nolock;
    $self->{'CHANGED'} = 0;

    $self;
}

sub created { $_[0]->{'CREATED'}; }

sub DESTROY {
    my $self = shift;
    $self->close;
}

sub readonly {
    my $self = shift;
    $self->{'READONLY'} = 1;
}

sub writable {
    my $self = shift;
    $self->{'READONLY'} = 0;
}

### This class handles Resource Records grouped by domain name (owner)
{
    package FileFormat::DNS::DomainDB::RR;

# XXX: debug print
sub Dprint {
    my @stuff = caller(0);
    # print STDERR join(":", @stuff[2]), "-->", @_;
}

    use Carp;

    sub new { ## $class->new(name);
	my $self = bless { }, shift;
	$self->{'parent'} = shift;
	my $name = shift;
	$self->{'name'} = $name;
	$name =~ m/^([^.]+)\.(.*)$/;
	$self->{'host'} = $1;
	$self->{'domain'} = $2;
	$self->changed;
	$self;
    }

    # clears out a RR object
    sub clean {
	my $self = shift;
	$self->changed;
	# clear SOA NS A MX CNAME PTR HINFO TXT WKS ANY records
	# applications can clear records like so:
	#     $self->delete_SOA if $self->has_SOA;
	#     $self->delete_A($self->A) if $self->has_A;
	# but we can do it much faster:
	$self->{'SOA'}	= [ ];
	$self->{'NS'}	= { };
	$self->{'A'}	= { };
	$self->{'MX'}	= { };
	$self->{'CNAME'}= { };
	$self->{'PTR'}	= { };
	$self->{'HINFO'}= [ ];
	$self->{'TXT'}	= [ ];
	$self->{'WKS'}	= [ ];
	$self->{'ANY'}	= [ ];
    }

    sub changed { $_[0]->{'parent'}->{'CHANGED'} = 1; }
    sub name { $_[0]->{'name'}; }
    sub host { $_[0]->{'host'}; }
    sub domain { $_[0]->{'domain'}; }

    sub ANY {
	my $self = shift;
	if (@_ == 1) {
	    split("\t", $self->{'ANY'}->[shift()], 2);
	}
	elsif (@_) {
	    $self->changed;
	    push(@{$self->{'ANY'}}, join("\t", @_));
	}
	else {
	    0..$#{@{$self->{'ANY'}}};
	}
    }
    sub delete_ANY {
	my $self = shift;
	my $item = shift;
	splice(@{$self->{'ANY'}}, $item, 1);
    }
    sub has_ANY {
	scalar(@{$_[0]->{'ANY'}});
    }

    sub A {
	my $self = shift;
	if (@_ == 1) {
	    my $key = shift;
	    ($self->{'A'}->{$key}, $key);	# ttl ip.address
	}
	elsif (@_) {
	    croak "usage: A ttl ip.address" unless @_ == 2;
	    $self->changed;
	    $self->{'A'}->{$_[1]} = $_[0];
	}
	else {
	    keys %{$self->{'A'}};
	}
    }
    sub delete_A {
	my $self = shift;
	my $ip = shift;
	croak "A $ip doesn't exist in " . $self->name
	    unless exists $self->{'A'}->{$ip};
	delete $self->{'A'}->{$ip};
    }
    sub has_A {
	scalar(keys %{$_[0]->{'A'}});
    }

    sub NS {
	my $self = shift;
	if (@_ == 1) {
	    my $key = shift;
	    ($self->{'NS'}->{$key}, $key);	# ttl nameserver
	}
	elsif (@_) {
	    croak "usage: NS ttl nameserver" unless @_ == 2;
	    $self->changed;
	    $self->{'NS'}->{$_[1]} = $_[0];
	}
	else {
	    keys %{$self->{'NS'}};
	}
    }
    sub delete_NS {
	my $self = shift;
	my $ns = shift;
	croak "NS $ns doesn't exist in " . $self->name
	    unless exists $self->{'NS'}->{$ns};
	delete $self->{'NS'}->{$ns};
    }
    sub has_NS {
	scalar(keys %{$_[0]->{'NS'}});
    }

    sub CNAME {
	my $self = shift;
	if (@_ == 1) {
	    my $key = shift;
	    ($self->{'CNAME'}->{$key}, $key);	# ttl cname
	}
	elsif (@_) {
	    croak "usage: CNAME ttl cname" unless @_ == 2;
	    $self->changed;
	    $self->{'CNAME'}->{$_[1]} = $_[0];
	}
	else {
	    keys %{$self->{'CNAME'}};
	}
    }
    sub delete_CNAME {
	my $self = shift;
	my $cname = shift;
	croak "CNAME $cname doesn't exist in " . $self->name
	    unless exists $self->{'CNAME'}->{cname};
	delete $self->{'CNAME'}->{$cname};
    }
    sub has_CNAME {
	scalar(keys %{$_[0]->{'CNAME'}});
    }

    sub PTR {
	my $self = shift;
	if (@_ == 1) {
	    my $key = shift;
	    ($self->{'PTR'}->{$key}, $key);	# ttl ptr
	}
	elsif (@_) {
	    croak "usage: PTR ttl ptr" unless @_ == 2;
	    $self->changed;
	    $self->{'PTR'}->{$_[1]} = $_[0];
	}
	else {
	    keys %{$self->{'PTR'}};
	}
    }
    sub delete_PTR {
	my $self = shift;
	my $ptr = shift;
	croak "PTR $ptr doesn't exist in " . $self->name
	    unless exists $self->{'PTR'}->{$ptr};
	delete $self->{'PTR'}->{$ptr};
    }
    sub has_PTR {
	scalar(keys %{$_[0]->{'PTR'}});
    }

    sub TXT {
	my $self = shift;
	if (@_ == 1) {
	    split(':', $self->{'TXT'}->[shift()], 2);	# ttl:"txt"
	}
	elsif (@_) {
	    croak "usage: TXT ttl item" unless @_ == 2;
	    $self->changed;
	    push(@{$self->{'TXT'}}, join(':', @_));
	}
	else {
	    0..$#{@{$self->{'TXT'}}};
	}
    }
    sub delete_TXT {
	my $self = shift;
	my $item = shift;
	splice(@{$self->{'TXT'}}, $item, 1);
    }
    sub has_TXT {
	scalar(@{$_[0]->{'TXT'}});
    }

    sub MX {
	my $self = shift;
	if (@_ == 1) {
	    my $key = shift;
	    (@{$self->{'MX'}->{$key}}, $key);	# ttl pref mxhost
	}
	elsif (@_) {
	    croak "usage: MX ttl pref mxhost" unless @_ == 3;
	    $self->changed;
	    $self->{'MX'}->{$_[2]} = [ $_[0], $_[1] ];	# mhhost = ttl pref
	}
	else {
	    keys %{$self->{'MX'}};
	}
    }
    sub delete_MX {
	my $self = shift;
	my $mx = shift;
	croak "MX $mx doesn't exist in " . $self->name
	    unless exists $self->{'MX'}->{$mx};
	delete $self->{'MX'}->{$mx};
    }
    sub has_MX {
	scalar(keys %{$_[0]->{'MX'}});
    }

    sub HINFO {
	my $self = shift;
	if (@_ == 1) {
	    @{$self->{'HINFO'}->[shift()]};	# ttl hw sw
	}
	elsif (@_) {
	    croak "usage: HINFO ttl hw sw" unless @_ == 3;
	    $self->changed;
	    push(@{$self->{'HINFO'}}, [ $_[0], $_[1], $_[2] ] );
	}
	else {
	    0..$#{@{$self->{'HINFO'}}};
	}
    }
    sub delete_HINFO {
	my $self = shift;
	my $item = shift;
	splice(@{$self->{'HINFO'}}, $item, 1);
    }
    sub has_HINFO {
	scalar(@{$_[0]->{'HINFO'}});
    }

    sub WKS {
	my $self = shift;
	if (@_ == 1) {
	    @{$self->{'WKS'}->[shift()]};	# ttl ...
	}
	elsif (@_) {
	    croak "usage: WKS ttl serv proto \@list" unless @_ >= 4;
	    $self->changed;
	    push(@{$self->{'WKS'}}, [ @_ ] );
	}
	else {
	    0..$#{@{$self->{'WKS'}}};
	}
    }
    sub delete_WKS {
	my $self = shift;
	my $item = shift;
	splice(@{$self->{'WKS'}}, $item, 1);
    }
    sub has_WKS {
	scalar(@{$_[0]->{'WKS'}});
    }

    sub SOA {
	my $self = shift;
	if (@_) {
	    Dprint "SOA ", join(':', @_), "\n";
	    croak "usage: SOA ttl host contact serial refresh retry expire minimum" unless @_ == 8;
	    $self->changed;
	    $self->{'SOA'} = [ @_ ];
	}
	else {
	    @{$self->{'SOA'}};		# ttl host contact ...
	}
    }
    sub delete_SOA {
	my $self = shift;
	my $item = shift;
	$self->{'SOA'} = [ ];
    }
    sub has_SOA {
	scalar(@{$_[0]->{'SOA'}});
    }

    # print out SOA NS A MX CNAME PTR HINFO TXT WKS ANY records
    sub stringify {
	my $self = shift;
	my $mode = shift;		# @ or .
	my ($s, $ttl, @list) = '';

	# starts out as host, gets set to "\t" after the first use
	# XXX: risky???
	my $d = sprintf("%-14s ", $mode eq '@' ? '@' : $self->host);

	if ($self->has_SOA) {
	    ($ttl, @list) = $self->SOA;
	    my ($h, $c, $se, $rf, $rt, $e, $m) = @list;
	    $s .= "${d}$ttl\tIN\tSOA\t$h  $c (\n";
	    $s .= "\t\t\t\t    $se\t; serial\n";
	    $s .= "\t\t\t\t    $rf\t; refresh\n";
	    $s .= "\t\t\t\t    $rt\t; retry\n";
	    $s .= "\t\t\t\t    $e\t; expire\n";
	    $s .= "\t\t\t\t    $m\t; minimum\n";
	    $s .= "\t\t\t\t)\n";
	    $d = "\t";
	}

	map {
	    ($ttl, @list) = $self->NS($_);
	    $s .= "${d}$ttl\tIN\tNS\t@list\n";
	    $d = "\t";
	} $self->NS;

	map {
	    ($ttl, @list) = $self->A($_);
	    $s .= "${d}$ttl\tIN\tA\t@list\n";
	    $d = "\t";
	} $self->A;

	map {
	    ($ttl, @list) = $self->MX($_);
	    $s .= "${d}$ttl\tIN\tMX\t@list\n";
	    $d = "\t";
	} sort { ($self->MX($a))[1] <=> ($self->MX($b))[1]} $self->MX;

	map {
	    ($ttl, @list) = $self->CNAME($_);
	    $s .= "${d}$ttl\tIN\tCNAME\t@list\n";
	    $d = "\t";
	} $self->CNAME;

	map {
	    ($ttl, @list) = $self->PTR($_);
	    $s .= "${d}$ttl\tIN\tPTR\t@list\n";
	    $d = "\t";
	} $self->PTR;

	map {
	    ($ttl, @list) = $self->HINFO($_);
	    $list[0] =~ s/[;"\n\(\)\\]/\\$&/gm;
	    $list[0] =~ s/^[\000-\377]*$/\"$&\"/m;
	    $list[1] =~ s/[;"\n\(\)\\]/\\$&/gm;
	    $list[1] =~ s/^[\000-\377]*$/\"$&\"/m;
	    $s .= "${d}$ttl\tIN\tHINFO\t@list\n";
	    $d = "\t";
	} $self->HINFO;

	map {
	    ($ttl, @list) = $self->TXT($_);
	    $list[0] =~ s/[;"\n\(\)\\]/\\$&/gm;
	    $list[0] =~ s/^[\000-\377]*$/\"$&\"/m;
	    $s .= "${d}$ttl\tIN\tTXT\t@list\n";
	    $d = "\t";
	} $self->TXT;

	map {
	    ($ttl, @list) = $self->WKS($_);
	    $s .= "${d}$ttl\tIN\tWKS\t@list\n";
	    $d = "\t";
	} $self->WKS;

	map {
	    ($ttl, @list) = $self->ANY($_);
	    # TODO: I really need to make sure I avoid any decoding on
	    # things that I don't understand
	    $s .= "${d}$ttl\tIN\t@list\n";
	    $d = "\t";
	} $self->ANY;

	return $s;
    }
}

# owner returns an object for the named object,
# it allocates a new (empty) object if needed
sub owner { ## PUBLIC
    my $self = shift;
    my $name = shift;

    return $self->{'owners'}->{$name}
	if defined $self->{'owners'}->{$name};

    $self->{'CHANGED'} = 1;
    $self->{'owners'}->{$name} =
	new FileFormat::DNS::DomainDB::RR($self, $name);
}
sub owners { ## PUBLIC
    my $self = shift;
    return values %{$self->{'owners'}};
}
sub owners_names { ## PUBLIC
    my $self = shift;
    return keys %{$self->{'owners'}};
}

# type returns all the objects that have resources records of a specific type
sub type { ## PUBLIC
    my $self = shift;
    my $type = shift;
    my ($obj, @objs);
    foreach $obj ($self->owners) {
	push @objs, $obj if eval qq{ \$obj->has_$type };
    }
    @objs;
}

# TODO: sub types { } returns types defined for an object?

#############################################################################

# IO Access Functions
sub ungetc { ## PRIVATE
    $_[0]->{'iostack'} = $_[1] . $_[0]->{'iostack'};
}

sub getc { ## PRIVATE
    $_[0]->{'iostack'} = $_[0]->{'FH'}->getline if $_[0]->{'iostack'} eq '';
    $_[0]->{'iostack'} =~ s/^([\000-\377])//;
    $1;
}

sub getline { ## PRIVATE
    return $_[0]->{'FH'}->getline if $_[0]->{'iostack'} eq '';
    Dprint "DEBUG: iostack wasn't empty: (",
	$_[0]->{'iostack'}, ")\n";
    $_[0]->{'iostack'} =~ s/^([^\n]*\n?)//;
    $1;
}

sub getword { ## PRIVATE
    my $self = shift;
    my $preserve = shift;
    my $tok = '';
    my $c;

    while (1) {
	### the IO functions have been inlined here and tuned for speed
	# make sure the buffer is full
	$self->{'iostack'} = $self->{'FH'}->getline
	    if $self->{'iostack'} eq '';
	last unless defined $self->{'iostack'};

	# grab the next whole token
	Dprint "DEBUG: iostack before tokenization ($1): (",
	    $self->{'iostack'}, ")\n";
	$tok .= $1 if ($self->{'iostack'} =~ s/^\s*([^\"\;\n\s\(\)\\]+)//);
	Dprint "DEBUG: getword grabbed a ``$1''\n";
	Dprint "DEBUG: token is now $tok\n";

	# Get the magic character that stopped us and process it.
	# We don't have to fill the buffer because we know it has at
	# least \n.
	Dprint "DEBUG: iostack before inlined getchar: (",
		$self->{'iostack'}, ")\n";
	$self->{'iostack'} =~ s/^([\000-\377])//;	# inlined getc
	Dprint "DEBUG: getword got a ``$1'' off the stack\n";
	last unless defined($c = $1);
	#
	### End of endlined IO functions

	if ($c eq ';') {		# comment, skip rest of line
	    Dprint "DEBUG: getword skipping comment line",
		$self->{'iostack'}, " -- ", $self->getline, "\n";
	    $c = "\n";
	    # fall through to next if
	}
	if ($c eq "\n") {
	    # leave \n for next pass if it's a token boundry
	    $tok eq '' ? $lineno++ : $self->ungetc($c);
	    last;
	}
	if ($c eq '"') {		# quoted string
	    while (defined($c = $self->getc) && $c ne '"' && $c ne "\n") {
		if ($c eq '\\') {
		    $c = '\\' unless defined($c = $self->getc);	# EOF
		    $tok .= '\\' if $preserve && ($c eq '\\' || $c eq '.');
		    $lineno++ if $c eq "\n";
		}
		$tok .= $c;
	    }

	    if ($c eq "\n") {
		$tok .= ' ';
		$lineno++;
		next;
	    }

	    # Sample following character, check for terminator.
	    $self->ungetc($c) if defined($c = $self->getc);
	    return $tok if ! defined $c || $c =~ /\s/;
	    next;
	} # end if(quoted string)

	if ($c eq '\\') {			# do escape processing
	    $c = '\\' unless defined($c = $self->getc);	# EOF
	    $tok .= '\\' if $preserve && ($c eq '\\' || $c eq '.');
	    $lineno++ if $c eq "\n";
	} elsif ($c =~ /\s/) {
	    # Blank of some kind.  Skip run
	    while (defined($c = $self->getc) && $c =~ /\s/ && $c ne "\n") { 1; }
	    $self->ungetc($c);
	    last if $tok ne '';			# Trailing \s (token boundry)
	    next;				# Leading \s, keep looking
	}
	$tok .= $c;
    }
    return undef if $tok eq '';		# end of line or file
    return $tok;
}

#############################################################################

BEGIN {
    # parser tokens
    $CURRENT = 1;
    $DOT     = 2;
    $AT      = 3;
    $DNAME   = 4;
    $INCLUDE = 5;
    $ORIGIN  = 6;
    $TTL     = 7;

    %CLASS = ( "IN", 1, "CHAOS", 1, "CH", 1, "HS", 1, "CSNET", 1, );
    # valid RR types (and a "preserve" flag)
    %RR = (
        "A", 0, "NS", 1, "CNAME", 1, "SOA", 1, "MB", 1, "MG", 1,
        "MR", 1, "NULL", 0, "WKS", 0, "PTR", 1, "HINFO", 0, "MINFO", 1,
        "MX", 0, "UINFO", 0, "TXT", 0, "RP", 1, "AFSDB", 0, "X25", 0,
        "ISDN", 0, "RT", 0, "NSAP", 0, "NSAP_PTR", 0, "UID", 0, "GID", 0,
        "PX", 0, "AAAA", 0, "LOC", 0, "UNSPEC", 0,
    );
}

sub getwords { ## PRIVATE
    my $self = shift;
    my @preserve = @_;		# set true for the fields that need to
				# preserve \, short list is ok

    my ($cont, $word, @words) = 0;
CONTINUATION:
    while (defined($word = $self->getword(shift @preserve)) || $cont) {
	Dprint "DEBUG: getwords gota ", $word, "\n";
	if (! defined $word || $word eq '') {
	    last if $self->{'FH'}->eof;		# can't go any further
	    next;				# skip EOL indication
	}
	if (! $cont && $word eq '(') {		#) continuation?
	    my $c;
	    while (defined($c = $self->getc)) {
		if ($c eq "\n") {
		    Dprint "DEBUG: in continuation\n";
		    $cont++;			# yeap, continuation
		    next CONTINUATION;
		}
		elsif ($c =~ /\s/) {
		    next;			# dunno yet
		}
		else {
		    push @words, $word;		# nope, move on
		    $self->ungetc($c);
		    last;
		}
	    }
	}
	elsif ($cont && $word eq ')') {
	    $cont = 0;				# end of continuation
	}
	else {
	    push @words, $word;
	}
    }
    @words;
}

sub gettoken { ## PRIVATE
    my $self = shift;
    my $op;
    while (1) {
	return undef if $self->{'FH'}->eof;
	local($_) = $self->getc;
	if ($_ eq '$') {
	    $op = lc($self->getword);
	    return $ORIGIN if $op eq 'origin';
	    return $INCLUDE if $op eq 'include';
	    return $TTL if $op eq 'ttl';
	    croak "Invalid \$ operator: $op";
	}
	elsif ($_ eq ' ' || $_ eq "\t") {
	    return $CURRENT;
	}
	elsif ($_ eq '.') {
	    return $DOT;
	}
	elsif ($_ eq '@') {
	    return $AT;
	}
	elsif ($_ eq ';') {
	    $self->getline;
	    $lineno++;
	    next;
	}
	elsif ($_ eq "\n") {
	    $lineno++;
	    next;
	}
	else {
	    $self->ungetc($_);
	    return $DNAME;
	}
    }
}

sub makename { ## PRIVATE function($name, $origin)
    my ($name, $origin) = @_;
    return undef if ! defined $name || $name eq '';
    return $origin if $name eq '@';
    return CanonicalizeDomain($name) if ($name =~ m/\.$/);
    return CanonicalizeDomain($name . '.' . $origin);
}

sub read_dbfile { ## PRIVATE
    my $self = shift;
    my $domain = shift;
    my $origin = shift;
    my $incfile = shift;		# undef initially XXX: implement it

    return undef if $self->{'FH'}->eof;

    $origin = CanonicalizeDomain($origin || '.');
    $domain = makename($domain, $origin);

    local $lineno = 0;			# DB File line counter
    local $_;

    my $dtok;
    my $token;
    my $class;
    my $RRtype;

    # $incfile must we prefix named.boot directory???

    while (defined($token = $self->gettoken)) {
	if ($token == $INCLUDE) {
	    croak "recursive include in domain file" if defined $incfile;
	    my $file = $self->getword;
	    my $tok;
	    $tok = $self->getword(1) if defined $file;
	    my $incorigin = makename(($tok || '@'), $origin);
	    $self->getline if defined $tok;	# trash the rest of the line
	    $self->read_dbfile($domain, $incorigin, $file);
	    next;
	}
	elsif ($token == $ORIGIN) {
	    my $new = $self->getword(1);
	    $origin = makename($new, $origin);
	    $self->getline if defined $new;	# trash the rest of the line
	    next;
	}
	elsif ($token == $TTL) {
	    $xttl = $self->getword(1);
	    $self->getline if defined $xttl;	# trash the rest of the line
	    next;
	}
	elsif ($token == $DNAME) {
	    $domain = makename(($dtok = $self->getword(1)), $origin);
	}
	elsif ($token == $DOT) {
	    $dtok = $domain = '.';
	}
	elsif ($token == $AT) {
	    $dtok = '@';
	    $domain = $origin;
	}
	elsif ($token == $CURRENT) {
	    # $domain = $domain;
	    $dtok = "\t";
	}
	# fall through: process $domain record

	# defaults if error
	$ttl    = '';
	$class  = 'in' unless $class;	# inherit
	$RRtype = 'unspec';

	# Parse: ttl
	my $tok = $self->getword;
	next unless defined $tok;	# no data, just a name, toss it
	if ($tok =~ /^\d+$/) {
	    $ttl = $tok;
	    $tok = $self->getword;
	    next unless defined $tok;	# got ttl but nothing else, toss it
	}

	$tok = uc($tok);		# class or RR, either is UC

	# Parse: class
	if (exists $CLASS{$tok}) {
	    $class = $tok;
	    if ($class ne 'IN') {
		# unsupported class type
		my $rest = $self->getline || '';
		$self->owner($domain)->ANY($ttl, $class, $rest);
		next;
	    }
	    $tok = uc($self->getword);
	    next unless defined $tok;	# host, ttl, class; still toss it
	}
	else {
	    if (exists $RR{$tok} && defined $class && $class ne '') {
		# if it's a known RR type and we have a $class use it
		1;
	    }
	    else {
		# Unknown line
		Dprint "DEBUG: unknown record: $tok\n";
		my $rest = $self->getline || '';
		$self->owner($domain)->ANY($ttl, $tok, $rest);
		next;
	    }
	}

	# Parse: RR type
	if (! exists $RR{$tok}) {
	    Dprint "DEBUG: unknown record: $tok\n";
	    my $rest = $self->getline || '';
	    $self->owner($domain)->ANY($ttl, $class, $tok, $rest);
	    next;
	}
	$RRtype = $tok;

	# So far we have: $domain $ttl $class $RRtype
	# now we must parse the specific arguments

	if ($RRtype eq 'A') {
	    # TODO: $words[0] = canonicalize_ip($words[0]);
	    $self->owner($domain)->A($ttl, $self->getwords(0));
	    next;
	}
	elsif ($RRtype eq 'NS') {
	    $words[0] = makename($words[0], $origin);
	    $self->owner($domain)->NS($ttl, $self->getwords(1));
	    next;
	}
	elsif ($RRtype eq 'CNAME') {
	    $words[0] = makename($words[0], $origin);
	    $self->owner($domain)->CNAME($ttl, $self->getwords(1));
	    next;
	}
	elsif ($RRtype eq 'SOA') {
	    $words[0] = makename($words[0], $origin);
	    $words[1] = makename($words[1], $origin);
	    $self->owner($domain)->SOA($ttl, $self->getwords(1, 1));
	    next;
	}
	elsif ($RRtype eq 'PTR') {
	    $words[0] = makename($words[0], $origin);
	    $self->owner($domain)->PTR($ttl, $self->getwords(1));
	    next;
	}
	elsif ($RRtype eq 'MX') {
	    $words[1] = makename($words[1], $origin);
	    $self->owner($domain)->MX($ttl, $self->getwords(0, 1));
	    next;
	}
	elsif ($RRtype eq 'TXT') {
	    $self->owner($domain)->TXT($ttl, join(' ', $self->getwords(0)));
	    next;
	}
	elsif ($RRtype eq 'WKS') {
	    $self->owner($domain)->WKS($ttl, $self->getwords(0));
	    next;
	}
	elsif ($RRtype eq 'HINFO') {
	    $self->owner($domain)->HINFO($ttl, $self->getwords(0));
	    next;
	}
	# Unknown, push it onto the unknown stack.
	Dprint "DEBUG: unknown record: $RRtype\n";
	my $rest = $self->getline || '';
	$self->owner($domain)->ANY($ttl, $class, $RRtype, $rest);
	next;
    }

    1;
}

# for sorting resource records
sub byRR {
    my ($a1, $b1, $a2, $b2);

    # force subdomain delgation to the bottom
    $a1 = ($a->has_NS ? ' .' : '') . $a->name;
    $b1 = ($b->has_NS ? ' .' : '') . $b->name;

    # count the dots
    $a2 = $a1 =~ s/\././g;
    $b2 = $b1 =~ s/\././g;

    # put subdomains (more dots) at the bottom
    return $a2 <=> $b2 unless ($a2 <=> $b2) == 0;

    # sorted by host
    return $a1 cmp $b1;
}

sub write_dbfile { ## PRIVATE
    my $self = shift;

    # write tmp file and then move into place for atomic updates
    my $newfile = $self->{'FILE'} . '.new';
    unlink($newfile);
    my $fh = new IO::LockingFile($newfile, O_WRONLY|O_CREAT|O_EXCL);
    croak "$newfile: $!" unless defined $fh;

    chop(my $date = `date`);
    print $fh <<"END_OF_HEADER";
;DNS-Format-Version-1
;
; This file was generated by the AdminWEB DNS configuration utility on:
;     $date
;
; It should be ok to hand edit this file, however, AdminWEB *will*
; strip out all comments and reorder the contents the next time it
; writes it (I wish I didn't have to do that but the format of DNS
; domain files makes it very hard to preserve comments in any sane way).
;
; Due to the complexity of domain configuration files it
; would be prudent to save any changes you make here and test them
; with the AdminWEB DNS tool just in case.  If you have a valid DNS
; file that isn't properly handled please do submit a bug report giving
; as much detail as possible so we can improve this tool.
;
; Domains defined in this file:
END_OF_HEADER

    # output list of domains
    {
	my %origins;
	map { $origins{$_->{'domain'}} = 1 } $self->owners;
	# undo the big cheese domain
	delete $origins{$self->owner($self->{'soa_domain'})->{'domain'}};
	map { print $fh ";     $_\n" } sort keys %origins;
	print $fh ";\n";
    }

    # Handle SOA as a special case
    my $origin = $self->{'soa_domain'};
    my $obj = $self->owner($origin);
    croak "No SOA record in DNS Domain: ``$origin''" unless $obj->has_SOA;

    # update the serial number
    {
	my @list = $obj->SOA;
	my $t = time;
	$list[3] > $t ? $list[3]++ : ($list[3] = $t);
	$obj->SOA(@list);
    }

    $xttl = 28800 if $xttl == 0;
    print $fh '$TTL ', $xttl, "\n";
    print $fh '$ORIGIN ', $origin, "\n";
    print $fh $obj->stringify('@');

    my $do_soa = 1;
    my $string = '.';
    foreach $obj (sort byRR $self->owners) {
	$string = '.';
	if ($do_soa && $obj->name eq $self->{'soa_domain'}) {
	    # this will almost always happen first so we can avoid
	    # lots of compares by testing a boolean first.
	    $do_soa = 0;		# avoid needless compares
	    next;			# already did SOA
	}
	if ($obj->has_NS) {
	    # handle subdomain delegations
	    print $fh ";\n";
	    print $fh "; subdomain delegation\n";
	    print $fh '$ORIGIN ', ($origin = $obj->name), "\n";
	    # XXX: make stringify special with an @
	    $string = '@';
	}
	elsif ($origin ne $obj->domain) {
	    print $fh '$ORIGIN ', ($origin = $obj->domain), "\n";
	}
	print $fh $obj->stringify($string);
    }

    # swap newfile for oldfile
    $fh->close or die "$newfile: $!\n"; undef $fh;
    rename $newfile, $self->{'FILE'} or
	 die "rename $newfile " . $self->{'FILE'} . ": $!\n";
}

1;
