# Copyright (c) 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI NamedBoot.pm,v 1.9 1999/11/19 01:42:51 polk Exp
#
# Object-oriented interface to /etc/named.boot

package	FileFormat::DNS::NamedBoot;
use FileFormat::DNS::Paths;
use FileFormat::DNS::Utils;
use FileFormat::DNS::DomainDB;
use Sys::Hostname;
use IO::LockingFile;
use Carp;

use FileFormat::DNS::Utils qw(CanonicalizeDomain bydomain);

BEGIN {
    # tells the AUTOLOAD'er how to build the access functions
    # this means you can add new named.boot datatypes without
    # having to write a lot of functions for them, just add them
    # here, write the parser stub in read_namedboot and write
    # the code in write_namedboot to format the data.
    %loader = (
	# attributes and options
	'limit',      'HASH',
	'options',    'HASH',

	# simple lists of things
	'bogusns',    'ARRAY',
	'sortlist',   'ARRAY',
	'forwarders', 'ARRAY',
	'xfrnets',    'ARRAY',

	# Highlander's [there can be only one]
	'directory',  'SCALAR',
	'domain',     'SCALAR',

	# things hashed by domain
	'cache',      'HASH',
	'primary',    'HASH',
	'secondary',  'HASH',
    );
}

sub new { ## CLASS->new($named_path, $nolock);
    my $self = bless { }, shift;
    # print STDERR "loading $_[0]\n";
    $self->Initialize(@_);
}

# undefines the object (must remove any self-references first)
sub close {
    my $self = shift;
    return if $self->{'CLOSED'};
    $self->{'CLOSED'} = 1;
    map { $_->close } values %{$self->{'opened'}};
    %{$self->{'opened'}} = ();
    $self->write_namedboot if $self->{'CHANGED'} && ! $self->{'READONLY'};
    $self->{'FH'} = undef;
    $self->{'CHANGED'} = 0;
}

sub Initialize {
    my $self = shift;
    my ($named_path, $nolock, $nocheck) = @_;

    # XXXXXX HACK HACK HACK
    # Here we check to see if named.conf either doesn't exist or is 
    # an unmodified version of one we generated.  If it is, we allow
    # things to proceed, otherwise, we abort with an error.
    # The errors look kind of ugly, but it's far simpler to do this
    # here than attempt to make MaxIM understand in all the places...
    if (! $nocheck) {
    	my $named_conf = $named_path;
    	$named_conf =~ s/.boot/.conf/;
    	if ( -f $named_path ) {
	    if ( -f $named_conf ) {
	    	my $bootsig = `/usr/contrib/bin/md5 -q $named_conf`;
	    	my $confsig = `/usr/sbin/named-bootconf < $named_path | /usr/contrib/bin/md5 -q`;
	    	if ($bootsig ne $confsig) {
    			croak "It appears that you have a BIND 8 style named.conf file\n($named_conf) which has been edited or was not generated from the\ncorresponding named.boot file.  MaxIM may not be used with customized\nnamed.conf files.\n\nAborting";
	    	}
	    }
    	}
    	elsif ( -f $named_conf ) {
		croak "It appears that you have a BIND 8 style named.conf file\n($named_conf) but you do not have the named.boot configuration from which\nit was generated.  MaxIM may not be used with customized named.conf files.\n\nAborting";
	}
    }
	
    $self->{'FILE'} = $named_path;

    $self->{'CHANGED'}			= 0;
    $self->{'CREATED'}			= 0;
    $self->{'READONLY'}			= 0;

    # initialize the data structure to empty
    $self->{'opened'} = { };

    my $name;
    foreach $name (keys %loader) {
	if ($loader{$name} eq 'SCALAR') {
	    $self->{$name} = undef;
	    $self->{$name . '_comment'} = undef;
	}
	elsif ($loader{$name} eq 'ARRAY') {
	    $self->{$name} = [ ];
	    $self->{$name . '_comment'} = undef;
	}
	elsif ($loader{$name} eq 'HASH') {
	    $self->{$name} = { };
	    $self->{$name . '_comment'} = undef;
	}
    }
    $self->{'UNKNOWN'}			= [ ];

    # setup IO handle for $named_path
    $self->{'FH'} = new IO::LockingFile($named_path, O_RDONLY|O_CREAT);
    croak "$named_path: $!" unless defined $self->{'FH'};
    $self->{'FH'}->exclusive(15) or croak "$self->{FILE}: $!"
	unless $nolock;
    $self->read_namedboot or $self->{'CREATED'} = 1;
    $self->{'READONLY'} = 1 if $nolock;
    $self->{'CHANGED'} = 0;

    $self;
}

sub created { $_[0]->{'CREATED'}; }

sub DESTROY { ## PRIVATE
    my $self = shift;
    $self->close;
}

sub readonly {
    my $self = shift;
    $self->{'READONLY'} = 1;
}

sub writable {
    my $self = shift;
    $self->{'READONLY'} = 0;
}

# loads data access functions on the fly from a template
sub AUTOLOAD { ## PRIVATE
    my $name = $AUTOLOAD;
    $name =~ s/^.*:://;
    $name =~ s/_comment$//;

    croak "Undefined subroutine \&$AUTOLOAD" unless exists $loader{$name};

    # loads both feature() and feature_comment() functions
    GEN_feature($loader{$name}, $name);
    goto &$AUTOLOAD;
}

# This guy builds the functions needed to access the data
# based on their types in %loader.
sub GEN_feature { ## PRIVATE
    my $type = shift;		# SCALAR ARRAY HASH
    my $name = shift;		# directory, primary, etc
    my $eval;			# what we return

    # print STDERR "loading $name, $type\n";
    if ($type eq 'SCALAR') {
	### BUILDS SCALAR ACCESS FUNCTION:
	#        $value = $obj->FEATURE();
	#   $prev_value = $obj->FEATURE('new_value');
	#      $comment = $obj->FEATURE_comment();
	# $prev_comment = $obj->FEATURE_comment('comment');
	$eval = q{
	    sub FEATURE {
		my $self = shift;
		my $value = $self->{'FEATURE'};
		if (@_) {
		    $self->{'CHANGED'} = 1;
		    $self->{'FEATURE'} = shift;
		}
		$value;
	    }
	    sub FEATURE_comment {
		my $self = shift;
		my $value = $self->{'FEATURE_comment'};
		if (@_) {
		    $self->{'CHANGED'} = 1;
		    $self->{'FEATURE_comment'} = shift;
		}
		$value;
	    }
	}; # END SCALAR EVAL
    }
    elsif ($type eq 'ARRAY') {
	### BUILDS ARRAY ACCESS FUNCTION:
	#       @values = $obj->FEATURE();
	#  @prev_values = $obj->FEATURE(@values);
	#      $comment = $obj->FEATURE_comment();
	# $prev_comment = $obj->FEATURE_comment('comment');
	$eval = q{
	    sub FEATURE {
		my $self = shift;
		my @values = @{$self->{'FEATURE'}};
		if (@_) {
		    $self->{'CHANGED'} = 1;
		    if (defined $_[0]) {
			@{$self->{'FEATURE'}} = @_;
		    }
		    else {
			@{$self->{'FEATURE'}} = ();
		    }
		}
		@values;
	    }
	    sub FEATURE_comment {
		my $self = shift;
		my $value = $self->{'FEATURE_comment'};
		if (@_) {
		    $self->{'CHANGED'} = 1;
		    $self->{'FEATURE_comment'} = shift;
		}
		$value;
	    }
	}; # END ARRAY EVAL
    }
    elsif ($type eq 'HASH') {
	### BUILDS HASH ACCESS FUNCTION:
	#         @keys = $obj->FEATURE();
	#        $value = $obj->FEATURE('key');
	#   $prev_value = $obj->FEATURE('key', 'new_value');
	#      $comment = $obj->FEATURE_comment('key');
	# $prev_comment = $obj->FEATURE_comment('key', 'comment');
	$eval = q{
	    sub FEATURE {
		my $self = shift;
		# print STDERR "DEBUG: hash FEATURE dying\n" if @_ > 2;
		die "usage: FEATURE('key'[, 'value'])\n" if @_ > 2;
		my $key = shift;
		if (! defined $key) {
		    return keys %{$self->{'FEATURE'}};
		}
		my $value = $self->{'FEATURE'}->{$key};
		if (@_) {
		    $self->{'CHANGED'} = 1;
		    $self->{'FEATURE'}->{$key} = shift;
		    delete $self->{'FEATURE'}->{$key}
			unless defined $self->{'FEATURE'}->{$key};
		}
		$value;
	    }
	    sub FEATURE_comment {
		my $self = shift;
		die "usage: FEATURE_comment('key'[, 'comment'])\n"
		    if @_ == 0 || @_ > 2;
		my $key = shift;
		my $value = defined $self->{'FEATURE_comment'}->{$key} ?
		    $self->{'FEATURE_comment'}->{$key} : undef;
		if (@_) {
		    $self->{'CHANGED'} = 1;
		    $self->{'FEATURE_comment'}->{$key} = shift;
		}
		$value;
	    }
	}; # END HASH EVAL
    }
    else {
	croak "cannot generate function of type $type for $name";
    }
    $eval =~ s/FEATURE/$name/g;
    eval $eval;
    die $@ if $@;
}

# opens a FileFormat::DNS::DomainDB object for the given primary
sub primary_open {
    my $self = shift;
    my ($domain) = @_;
    my $dbfile = $self->primary($domain);
    croak "primary $domain doesn't exit" unless defined $dbfile;
    $dbfile = $self->resolve_path($dbfile);

    # we return the opened db object from out cache if we have one
    my $db = $self->{'opened'}->{$domain};
    if (! defined $db) {
	$db = new FileFormat::DNS::DomainDB($domain, $dbfile);
	$self->{'opened'}->{$domain} = $db;
    }
    $db;
}

sub find_reverse {
    my $self = shift;
    my $iphost = (ip_2_inaddrarpa($_[0]))[0];	# ##.##.##.##.in-addr.arpa

    # check to see if we are primary for the "class C" equivalent
    $iphost =~ /^(\d{1,3})\.(.*)$/;
    return ($iphost, $2) if defined $self->primary($2);

    # check to see if we are primary for the "class B" equivalent
    $iphost =~ /^(\d{1,3}\.\d{1,3})\.(.*)$/;
    return ($iphost, $2) if defined $self->primary($2);

    # check to see if we are primary for the "class A" equivalent
    $iphost =~ /^(\d{1,3}\.\d{1,3}\.\d{1,3})\.(.*)$/;
    return ($iphost, $2) if defined $self->primary($2);

    return ($iphost, undef);
}

sub set_reverse {
    my $self = shift;
    my ($ip, $host) = @_;
    my ($iphost, $ipdomain) = $self->find_reverse($ip);
    return undef unless defined $ipdomain;

    my $db = $self->primary_open($ipdomain);
    $db->owner($iphost)->PTR('', $host);
    return 1;
}

sub unset_reverse {
    my $self = shift;
    my ($host, $domain) = @_;
    return undef unless defined $self->primary($domain);
    my $owner = $self->primary_open($domain)->owner($host);
    my ($ip, $db, $ipdomain, $iphost);
    foreach $ip ($owner->A) {
	# if we are primary for $ip's reverse domain then remove it.
	$ip = canonicalize_ip($ip);
	next if ! defined $ip || $ip eq '127.0.0.1';

	($iphost, $ipdomain) = $self->find_reverse($ip);
	next unless defined $ipdomain;		# nope, not ours

	$db = $self->primary_open($ipdomain);
	$db->owner($iphost)->delete_PTR($host)
	   if defined $db->owner($iphost)->PTR($host);
    }
    return 1;
}

sub split_line { ## PRIVATE
    my $line = shift;
    my ($comment) = $line =~ m/\s*;\s*(.*)\s*$/;
    $line =~ s/\s*;.*//; $line =~ s/\s+/ /; $line =~ s/^\s+//;
    my @args = split(' ', $line); shift @args;
    ($comment, @args);
}

sub resolve_path { ## PRIVATE
    my $self = shift;
    my $path = shift;
    return $path if $path =~ /^\//;
    my $dir = ($self->directory || $_PATH_DNS_DBDIR);
    $path = $dir . '/' . $path if defined $dir;
    $path;
}

sub shorten_path { ## PRIVATE
    my $self = shift;
    my $path = shift;
    my $dir = ($self->directory || $_PATH_DNS_DBDIR);
    $path =~ s#^\Q$dir\E/## if defined $dir;
    $path;
}

sub read_namedboot { ## PRIVATE
    my $self = shift;
    my $fh = $self->{'FH'};

    # indicate an empty/uninitalized file
    return undef if eof($fh);

    while (<$fh>) {
	chomp;

	### Entry type...

	# STUFF WE IGNORE
	if (/^\s*;\s*directory where\b/i	# directory comment
	||  /^\s*;\s*$/				# empty comment
	||  /^\s*;\s*this hosts domain\b/i	# domain comment
	||  /^\s*;\s*named\(8\) boot file\b/i	# leader note
	||  /^\s*;\s*type\s*domain\b/i		# primary/secondary header
	||  /^\s*;\s*-----/			# header markers
	||  /^\s*;\s*domains for which\b/i	# primary/secondary labels
	||  /^\s*;\s*-*\s*DNS Server Options\b/i	# options label
	||  /^\s*$/) {				# blank lines
	    next;
	}
	elsif (/^\s*suffixes\b/i) {
	    # comment out obsolete suffixes entry
	    push(@{$self->{'UNKNOWN'}}, ';[obsolete entry] ' . $_);
	}
	elsif (/^\s*;/) {
	    # XXX: ignore existing comments ?
	    push(@{$self->{'UNKNOWN'}}, $_);
	}
	elsif (/^\s*#/) {
	    # XXX: ignore existing comments ?
	    # illegal #-style comment, fix it (sigh)
	    s/#/;/;
	    push(@{$self->{'UNKNOWN'}}, $_);
	}
	elsif (/^\s*directory\s+(\S+)/i) {		# Highlander
	    die "duplicate directory entries\n" if defined $self->directory;
	    my $directory = $1;
	    my ($comment) = m/\s*;\s*(.*)\s*$/;
	    $self->directory($directory);
	    $self->directory_comment($comment) if defined $comment;
	}
	elsif (/^\s*domain\s+(\S+)/i) {			# Highlander
	    # domain is obsolete but we keep it anyway
	    die "duplicate domain entries\n" if defined $self->domain;
	    my $domain = $1;
	    my ($comment) = m/\s*;\s*(.*)\s*$/;
	    $self->domain($domain);
	    $self->domain_comment($comment) if defined $comment;
	}
	elsif (/^\s*cache\b/i) {			# HASH by domain
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args == 2;
	    $args[0] = CanonicalizeDomain($args[0]);
	    $self->cache(@args);			# store file
	    $self->cache_comment($args[0], $comment) if defined $comment;
	}
	elsif (/^\s*primary\b/i) {			# HASH by domain
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args == 2;
	    $args[0] = CanonicalizeDomain($args[0]);
	    $self->primary(@args);			# store source file
	    $self->primary_comment($args[0], $comment) if defined $comment;
	}
	elsif (/^\s*secondary\b/i) {			# HASH by domain
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args > 1;
	    my $domain = CanonicalizeDomain(shift @args);
	    my $file = pop @args;
	    $self->secondary($domain, [ $file, @args ] );# store ARRAYREF
	    $self->secondary_comment($args[0], $comment) if defined $comment;
	}
	elsif (/^\s*limit\b/i) {			# HASH (attributes)
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args == 2;
	    $self->limit(@args);
	    $self->limit_comment($args[0], $comment) if defined $comment;
	}
	elsif (/^\s*max-fetch\b/i) {			# HASH (into limit)
	    ### CONVERTER
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args == 1;
	    $self->limit('transfers-in', @args);
	    $self->limit_comment('transfers-in', $comment) if defined $comment;
	}
	elsif (/^\s*options\b/i) {			# HASH (attributes)
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args > 0;
	    map { $self->options($_, 1) } @args;
	    $self->options_comment($args[0], $comment) if defined $comment;
	}
	elsif (/^\s*slave\b/i) {			# HASH (into options)
	    ### CONVERTER
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args == 0;
	    $self->options('forward-only', 1);
	    $self->options_comment('forward-only', $comment) if defined $comment;
	}
	elsif (/^\s*sortlist\b/i) {			# ARRAY (simple list)
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args > 0;
	    $self->sortlist($self->sortlist, @args);
	    $self->sortlist_comment($comment) if defined $comment;
	}
	elsif (/^\s*forwarders\b/i) {			# ARRAY (simple list)
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args > 0;
	    $self->forwarders($self->forwarders, @args);
	    $self->forwarders_comment($comment) if defined $comment;
	}
	elsif (/^\s*xfrnets\b/i || /^\s*tcplist\b/i) {	# ARRAY (simple list)
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args > 0;
	    $self->xfrnets($self->xfrnets, @args);
	    $self->xfrnets_comment($comment) if defined $comment;
	}
	elsif (/^\s*bogusns\b/i) {			# ARRAY (simple list)
	    my ($comment, @args) = split_line($_);
	    die "illegal syntax: $_\n" unless @args > 0;
	    $self->bogusns($self->bogusns, @args);
	    $self->bogusns_comment($comment) if defined $comment;
	}
	elsif (/^\s*include\b/i) {
	    # XXX: need to support recursion (sigh)
	    # I could even maintain included files by tagging
	    # cache, primary, and secondary with the file they came from
	    # options and such would get pulled out but that's not
	    # what include files are mostly used for
	    die "Sorry, `include' not supported yet: $_\n";
	}
	else {
	    # else unknown, save as is
	    push(@{$self->{'UNKNOWN'}}, $_);
	}
    }
    1;
}

sub formatline { ## PRIVATE
    my $line = sprintf("%-10s %-24s %-20s %s", @_);
    $line =~ s/\s+$//;
    $line;
}
sub headerline { ## PRIVATE
    ';' . '-' x 9 . ' ' . '-' x 24 . ' ' . '-' x 20 . ' ' . '-' x 10;
}

sub write_namedboot { ## PRIVATE
    my $self = shift;
    my $key;

    # write tmp file and then move into place for atomic updates
    my $newfile = $self->{'FILE'} . '.new';
    unlink($newfile);
    my $fh = new IO::LockingFile($newfile, O_WRONLY|O_CREAT|O_EXCL);
    croak "$newfile: $!" unless defined $fh;

    print $fh ";\n; named(8) boot file for ", hostname(), "\n;\n";

    if (defined $self->directory) {
	print $fh "; directory where cache files are stored\n",
		  "directory ", $self->directory;
	print $fh "\t\t; ", $self->directory_comment
	    if defined $self->directory_comment;
	print $fh "\n\n";
    }

    if (defined $self->domain) {
	print $fh "; this hosts domain [obsolete]\n",
		  "domain ", $self->domain;
	print $fh "\t\t; ", $self->domain_comment
	    if defined $self->domain_comment;
	print $fh "\n\n";
    }

    print $fh &formatline(';type', 'domain', 'source (ip/file)', 'backup file');
    print $fh "\n", &headerline(), "\n";

    foreach $key (sort bydomain $self->cache) {
	print $fh &formatline('cache', $key, '', $self->cache($key));
	print $fh "\t; ", $self->cache_comment($key)
	    if defined $self->cache_comment($key);
	print $fh "\n";
    }
    print $fh "\n";

    print $fh "; domains for which we are the primary DNS server\n";
    foreach $key (sort bydomain $self->primary) {
	# XXX: should we chop the trailing `.' of $key?
	my $domain = $key; $domain =~ s/\.+$//;
	print $fh &formatline('primary', $domain, $self->primary($key), '');
	print $fh "\t; ", $self->primary_comment($key)
	    if defined $self->primary_comment($key);
	print $fh "\n";
    }
    print $fh "\n";

    print $fh "; domains for which we are just a secondary/backup DNS server\n";
    foreach $key (sort bydomain $self->secondary) {
	# XXX: should we chop the trailing `.' of $key?
	my ($file, @list) = @{$self->secondary($key)};
	my $domain = $key; $domain =~ s/\.+$//;
	print $fh &formatline('secondary', $domain,
	    join(' ', @list), $file);
	print $fh "\t; ", $self->secondary_comment($key)
	    if defined $self->secondary_comment($key);
	print $fh "\n";
    }
    print $fh "\n";

    print $fh "; ------ DNS Server Options ------\n";

    foreach $key ($self->options) {
	next unless defined $self->options($key) && $self->options($key);
	# value isn't used for options, it's just a boolean
	print $fh 'options ', $key;
	print $fh "\t\t; ", $self->options_comment($key)
	    if defined $self->options_comment($key);
	print $fh "\n";
    }

    if ($self->forwarders) {
	print $fh 'forwarders ', join(' ', $self->forwarders);
	print $fh "\t\t; ", $self->forwarders_comment
	    if defined $self->forwarders_comment;
	print $fh "\n";
    }

    if ($self->sortlist) {
	print $fh 'sortlist ', join(' ', $self->sortlist);
	print $fh "\t\t; ", $self->sortlist_comment
	    if defined $self->sortlist_comment;
	print $fh "\n";
    }

    if ($self->xfrnets) {
	print $fh 'xfrnets ', join(' ', $self->xfrnets);
	print $fh "\t\t; ", $self->xfrnets_comment
	    if defined $self->xfrnets_comment;
	print $fh "\n";
    }

    if ($self->bogusns) {
	print $fh 'bogusns ', join(' ', $self->bogusns);
	print $fh "\t\t; ", $self->bogusns_comment
	    if defined $self->bogusns_comment;
	print $fh "\n";
    }

    foreach $key ($self->limit) {
	print $fh 'limit ', $key, ' ', $self->limit($key);
	print $fh "\t\t; ", $self->limit_comment($key)
	    if defined $self->limit_comment($key);
	print $fh "\n";
    }

    if (@{$self->{'UNKNOWN'}}) {
	print $fh "\n; ------ Entries not processed ------\n";
	print $fh join("\n", @{$self->{'UNKNOWN'}}),"\n";
    }

    # swap newfile for oldfile
    $fh->close or croak "$newfile: $!"; undef $fh;
    rename $newfile, $self->{'FILE'} or
         die "rename $newfile " . $self->{'FILE'} . ": $!\n";

    # XXXXXX HACK HACK HACK
    # This is the other half of the named.conf hack.  Here we 
    # run named-bootconf to convert the newly written .boot file
    # to a .conf file...
    my $boot = $self->{'FILE'};
    my $conf = $self->{'FILE'};
    $conf =~ s/.boot/.conf/;
    system("/usr/sbin/named-bootconf < $boot > $conf");
    die "named-bootconf $boot to $conf failed\n" if $? ne 0;
}

### quick setup for a new primary domain
sub new_primary {
    my $self = shift;
    my ($domain, $host, $mailhost, @nameservers) = @_;

    # build the ``primary domain source_file'' entry
    $self->primary($domain, $domain . 'db')
	unless defined $self->primary($domain);
    # open it as a DomainDB
    my $db = $self->primary_open($domain);
    # build the root entry with the basic SOA, NS and MX records
    my $obj = $db->owner($domain);
    $obj->SOA('', $host, 'hostmaster.'.$host, time,
	qw(28800 7200 604800 86400));

    map { $obj->NS('', $_) if defined $_ } @nameservers;
    $obj->MX('', 100, $mailhost) if defined $mailhost;
    # XXX: double check these SOA defaults
    $db;
}

### synthesize a default named.boot file for this system
sub init_namedboot {
    my $self = shift;
    my ($host, $fwddomain) = @_;

    # set default directory
    $self->directory($_PATH_DNS_DBDIR);

    # XXX: install a root.cache file or expect that it exists?
    $self->cache('.', 'root.cache');

    my $domain = '127.in-addr.arpa.';
    my $db = $self->new_primary($domain, $host, undef, $host);
    $db->owner('1.0.0.' . $domain)->PTR('', 'localhost.' . $fwddomain);
    1;
}

=head1 NAME

FileFormat::DNS::NamedBoot -- Object-oriented interface to named.boot files

=head1 SYNOPSIS

    use FileFormat::DNS::NamedBoot;
    $conf = '/etc/named.boot';
    $named = new FileFormat::DNS::NamedBoot($conf);
    die "$conf: $!\n" unless defined $named;
    # [ $named->readonly or $named->writable ]
    $named->METHOD;
    $named->close;

=head1 DESCRIPTION

C<new FileFormat::DNS::NamedBoot($CONF_FILE)>
creates an object to manage the data contained in the C<$CONF_FILE>.
C<$CONF_FILE> is created and initalized if it doesn't exist.

C<$named-E<gt>close>
undef's the calling object which causes the object to cleanup and
write itself out (if write is enabled).   Further use of the object
after calling close will result in an error.

C<$named-E<gt>readonly>
sets the object's state to readonly so that changes are not written
back out to the original file.

C<$named-E<gt>writable>
sets the object's state to be writable.

C<$named-E<gt>directory>
with no argument, returns the directory setting.  Otherwise it sets
directory to the argument.

             $directory = $named->directory;
    $previous_directory = $named->directory('/etc/namedb');

C<$named-E<gt>domain>
with no argument, returns the domain setting.  Otherwise it sets
domain to the argument.

             $domain = $named->domain;
    $previous_domain = $named->domain('bsdi.com.');

C<$named-E<gt>cache>
with no argument returns the domains for which cache entries exist.
With one argument (a fully-qualified domain name, usually just ".")
it returns the cache filename for that domain.  With two arguments
it sets the cache filename for the given domain to the second
argument and returns the previous value (if any).

    @cached_domains = $named->cache;
        $root_cache = $named->cache('.');
    $previous_cache = $named->cache('.', 'root.cache');

C<$named-E<gt>primary>
with no argument returns the list of domains for which we are listed
as the primary DNS server.  With one argument (a fully-qualified
domain name) it returns the filename that defines that domain.
With two arguments it sets the filename for the given domain to
the second argument and returns the previous value (if any).

     @primary_domains = $named->primary;
        $primary_file = $named->primary('bsdi.com.');
    $previous_primary = $named->primary('bsdi.com.', 'bsdi.com.db');

C<$named-E<gt>primary_open($domain)>
returns an opened C<FileFormat::DNS::DomainDB> object for the given domain.

C<$named-E<gt>secondary>
with no argument returns the list of domains for which we are listed
as a secondary DNS server.  With one argument (a fully-qualified
domain name) it returns an ARRAYREF to a list that contains the IP
addresses of servers to contact to download current copies of the
specified domain and the last argument in the list is the backup
file name for local storage.  With two arguments it sets the list
(primary servers IP addresses and the backup filename) for the
given domain to the second argument (which must be an ARRAYREF to the list)
and returns the previous value (if any).

     @secondary_domains = $named->secondary;
        $secondary_info = $named->secondary('sub.bsdi.com.');
    $secondary_info->[-1] eq 'sub.bsdi.com.bak';
    $previous_secondary = $named->secondary('sub.bsdi.com.',
				    ['10.0.0.1', 'sub.bsdi.com.bak']);

C<$named-E<gt>limit>
with no argument returns the list of limits that are currently set.
With one argument (the name of a limit) it returns the current
setting for that limit.  With two arguments it sets the limit
to the specified value and returns the previous value (if any).

            @limits = $named->limit;
             $limit = $named->limit('datasize');
    $previous_limit = $named->limit('datasize', '64M');

C<$named-E<gt>options>
with no argument returns the list of options that are currently
set.  With one argument (the name of an option) it returns the
current setting for that option (true if set, false if not set).
With two arguments it sets the option to the specified value (true
or false, therefore enabling or disabling the option) and returns
the previous setting.

            @options = $named->options;
             $option = $named->options('forward-only');
    $previous_option = $named->options('forward-only', 1);	# enable
    $previous_option = $named->options('forward-only', 0); # disable

C<$named-E<gt>sortlist>
with no argument returns sortlist.  If arguments are passed
sortlist is set (not appended) to the given list.

             @sortlist = $named->sortlist;
    @previous_sortlist = $named->sortlist(@new_sortlist);

To append to the list you can use:

    $named->sortlist($named->sortlist, @new_sortlist);

C<$named-E<gt>forwarders>
with no argument returns forwarders.  If arguments are passed
forwarders is set (not appended) to the given list.
    
             @forwarders = $named->forwarders; 
    @previous_forwarders = $named->forwarders(@new_forwarders);

To append to the list you can use:

    $named->forwarders($named->forwarders, @new_forwarders);


C<$named-E<gt>xfrnets>
with no argument returns xfrnets.  If arguments are passed
xfrnets is set (not appended) to the given list.
    
             @xfrnets = $named->xfrnets; 
    @previous_xfrnets = $named->xfrnets(@new_xfrnets);

To append to the list you can use:

    $named->xfrnets($named->xfrnets, @new_xfrnets);


C<$named-E<gt>bogusns>
with no argument returns bogusns.  If arguments are passed
bogusns is set (not appended) to the given list.
    
             @bogusns = $named->bogusns; 
    @previous_bogusns = $named->bogusns(@new_bogusns);

To append to the list you can use:

    $named->bogusns($named->bogusns, @new_bogusns);

C<$named-E<gt>new_primary($domain, $host, $mailhost, @secondaries)>
sets up C<named.boot> and the C<$domain.db> file as a primary.
C<$mailhost> and C<@secondaries> can be C<undef> to leave them
unset.

C<$named-E<gt>init_namedboot($domain, $host)>
initializes the C<named.boot> and C<127.in-addr.arpa.db> files
from scrach.  This should only be done if C<$named->created>
and you want to setup a default configuration.

C<$named-E<gt>find_reverse($ip)>
given an ip address it will find if this server is a primary for
the reverse domain.  It returns C<($iphost, $ipdomain)> where
C<$iphost> is the complete reverse name for this ip address and
C<$ipdomain> is the domain name for which we are primary that
includes the given ip address.

C<$named-E<gt>set_reverse($ip, $host)>
Sets a PTR record in the reverse domain for C<$ip) if we are primary for
the reverse domain.

C<$named-E<gt>unset_reverse($host, $domain)>
Removes the PTR records for all reverse domains for which we are
primary using the A records for $host in $domain (C<$host> must
be fully qualified but you must also supply the correct C<$domain>).

=head1 HISTORY

The FileFormat::DNS::NamedBoot module appeared in BSD/OS V3.0.
Still under development.

=head1 COPYRIGHT

Copyright 1996 Berkeley Software Design, Inc.

    http://www.bsdi.com/

=cut

1;
