# Copyright (c) 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI Resolver.pm,v 1.2 1997/04/17 21:14:59 sanders Exp
#
# Object-oriented interface to /etc/resolv.conf

package	FileFormat::DNS::Resolver;
use FileFormat::DNS::Utils;
use IO::LockingFile;
use Carp;

# new($resolver_path, $nolock);
sub new {
    my $self = bless { }, shift;
    $self->Initialize(@_);
}

# undefines the object (must remove any self-references first)
sub close {
    my $self = shift;
    return if $self->{'CLOSED'};
    $self->{'CLOSED'} = 1;

    $self->write_resolver if $self->{'CHANGED'} && ! $self->{'READONLY'};
    $self->{'FH'} = undef;
    $self->{'CHANGED'} = 0;
}

sub Initialize {
    my $self = shift;
    my ($resolver_path, $nolock) = @_;

    $self->{'FILE'} = $resolver_path;
    $self->{'CREATED'} = 0;
    $self->{'READONLY'} = 0;
    $self->{'CHANGED'} = 0;

    $self->{'domain'} = undef;
    $self->{'nameserver'} = [];
    $self->{'search'} = [];
    $self->{'sortlist'} = [];
    $self->{'options'} = [];
    $self->{'UNKNOWN'} = [];

    # if resolver exists we read it, if not we get a default one

    $self->{'FH'} = new IO::LockingFile($resolver_path, O_RDONLY|O_CREAT);
    die "$resolver_path: $!\n" unless defined $self->{'FH'};
    $self->{'FH'}->exclusive(15) or die "$self->{FILE}: $!\n"
	unless $nolock;
    $self->read_resolver or $self->{'CREATED'} = 1;
    $self->{'READONLY'} = 1 if $nolock;
    $self->{'CHANGED'} = 0;

    $self;
}

sub created { $_[0]->{'CREATED'}; }

sub DESTROY {
    my $self = shift;
    $self->close;
}

sub readonly {
    my $self = shift;
    $self->{'READONLY'} = 1;
}

sub writable {
    my $self = shift;
    $self->{'READONLY'} = 0;
}

sub domain {
    my $self = shift;
    my $value = $self->{'domain'};
    if (@_) {
	$self->{'CHANGED'} = 1;
	my $domain = shift;
	if (defined $domain) {
	    $self->{'domain'} =
		FileFormat::DNS::Utils::CanonicalizeDomain($domain);
	}
	else {
	    $self->{'domain'} = undef;
	}
    }
    $value;
}

sub nameserver {
    my $self = shift;
    my @value = @{$self->{'nameserver'}};
    if (@_) {
	$self->{'CHANGED'} = 1;
	if (defined $_[0]) {
	    @{$self->{'nameserver'}} = @_;
	}
	else {
	    @{$self->{'nameserver'}} = ();
	}
    }
    @value;
}

sub sortlist {
    my $self = shift;
    my @value = @{$self->{'sortlist'}};
    if (@_) {
	$self->{'CHANGED'} = 1;
	if (defined $_[0]) {
	    @{$self->{'sortlist'}} = @_;
	}
	else {
	    @{$self->{'sortlist'}} = ();
	}
    }
    @value;
}

sub options {
    my $self = shift;
    my @value = @{$self->{'options'}};
    if (@_) {
	$self->{'CHANGED'} = 1;
	if (defined $_[0]) {
	    @{$self->{'options'}} = @_;
	}
	else {
	    @{$self->{'options'}} = ();
	}
    }
    @value;
}

sub search {
    my $self = shift;
    my @value = @{$self->{'search'}};
    if (@_) {
	$self->{'CHANGED'} = 1;
	if (defined $_[0]) {
	    @{$self->{'search'}} = @_;
	}
	else {
	    @{$self->{'search'}} = ();
	}
    }
    @value;
}

sub read_resolver { ## PRIVATE
    my $self = shift;
    my $fh = $self->{'FH'};

    # indicate an empty file
    return undef if eof($fh);

    while (<$fh>) {
	chomp; s/^\s+//; s/\s+$//; s/\s+/ /;	# cleanup input
	if (/^domain\s+(\S+)/i) {
	    $self->domain($1);
	}
	elsif (/^nameserver\s+\S/i) {
	    # this isn't technically allowed but we're going to fix it up
	    my @ns = split; shift @ns;
	    $self->nameserver($self->nameserver, @ns);
	}
	elsif (/^search\s+\S/i ) {
	    my @search = split; shift @search;
	    $self->search($self->search, @search);
	}
	elsif (/^sortlist\s+\S/i ) {
	    my @sortlist = split; shift @sortlist;
	    $self->sortlist($self->sortlist, @sortlist);
	}
	elsif (/^options\s+\S/i ) {
	    my @options = split; shift @options;
	    $self->options($self->options, @options);
	}
	else {
	    push(@{$self->{'UNKNOWN'}}, $_);
	}
    }

    return 1;
}

sub write_resolver { ## PRIVATE
    my $self = shift;
    my $i;

    # write tmp file and then move into place for atomic updates
    my $newfile = $self->{'FILE'} . '.new';
    unlink($newfile);
    my $fh = new IO::LockingFile($newfile, O_WRONLY|O_CREAT|O_EXCL);
    die "$newfile: $!\n" unless defined $fh;

    if (defined $self->domain && $self->domain ne '') {
	# convert back to resolv.conf format (w/no trailing period)
	my $domain = $self->domain;
	$domain =~ s/\.*$//;
	print $fh "domain\t\t$domain\n";
    }

    map { print $fh "nameserver\t$_\n" } $self->nameserver;

    print $fh "search\t\t", join(' ', $self->search), "\n"
	if $self->search;

    print $fh "sortlist\t", join(' ', $self->sortlist), "\n"
	if $self->sortlist;

    print $fh "options\t\t", join(' ', $self->options), "\n"
	if $self->options;

    print $fh join("\n", @{$self->{'UNKNOWN'}}),"\n"
	if @{$self->{'UNKNOWN'}};

    # swap newfile for oldfile
    $fh->close or die "$newfile: $!\n"; undef $fh;
    rename $newfile, $self->{'FILE'} or
         die "rename $newfile " . $self->{'FILE'} . ": $!\n";
}

1;
