/*	BSDI elf.h,v 1.7 1998/09/08 03:35:10 torek Exp	*/

#ifndef _SYS_ELF_H_
#define _SYS_ELF_H_ 1

/*
 * Structures of ELF binary files.
 * The information here is derived from the System V ABI section 4
 * and from the SVr4 API ELF library man pages.
 * N.B.: struct tag names are nonstandard and exist only for Linux compat.
 */


/* for 32-bit ELF */
typedef unsigned int Elf32_Addr;	/* program addresses */
typedef unsigned short Elf32_Half;	/* 16 bit field */
typedef unsigned int Elf32_Off;		/* file offsets */
typedef int Elf32_Sword;		/* signed integers */
typedef unsigned int Elf32_Word;	/* 32-bit field */

/* for 64-bit ELF -- Elf64_Sword never used but is defined by Solaris (?) */
typedef unsigned long long Elf64_Addr;	/* program addresses */
typedef unsigned short Elf64_Half;
typedef unsigned long long Elf64_Off;	/* file offsets */
typedef int Elf64_Sword;
typedef long long Elf64_Sxword;		/* signed 64-bit integers */
typedef unsigned int Elf64_Word;	/* 32-bit field */
typedef unsigned long long Elf64_Xword;	/* 64-bit field */


/*
 * The ELF header.
 */

#define	EI_NIDENT	16

typedef struct __elfhdr {
	unsigned char	e_ident[EI_NIDENT];
	Elf32_Half	e_type;
	Elf32_Half	e_machine;
	Elf32_Word	e_version;
	Elf32_Addr	e_entry;
	Elf32_Off	e_phoff;	/* program header table offset */
	Elf32_Off	e_shoff;	/* section header table offset */
	Elf32_Word	e_flags;
	Elf32_Half	e_ehsize;	/* ELF header size in bytes */
	Elf32_Half	e_phentsize;	/* program header size in bytes */
	Elf32_Half	e_phnum;	/* number of program headers */
	Elf32_Half	e_shentsize;	/* section header size in bytes */
	Elf32_Half	e_shnum;	/* number of section headers */
	Elf32_Half	e_shstrndx;	/* SHT index of section name strings */
} Elf32_Ehdr;

typedef struct {
	unsigned char	e_ident[EI_NIDENT];
	Elf64_Half	e_type;
	Elf64_Half	e_machine;
	Elf64_Word	e_version;
	Elf64_Addr	e_entry;
	Elf64_Off	e_phoff;	/* program header table offset */
	Elf64_Off	e_shoff;	/* section header table offset */
	Elf64_Word	e_flags;
	Elf64_Half	e_ehsize;	/* ELF header size in bytes */
	Elf64_Half	e_phentsize;	/* program header size in bytes */
	Elf64_Half	e_phnum;	/* number of program headers */
	Elf64_Half	e_shentsize;	/* section header size in bytes */
	Elf64_Half	e_shnum;	/* number of section headers */
	Elf64_Half	e_shstrndx;	/* SHT index of section name strings */
} Elf64_Ehdr;

/*
 * Symbolic names for indices into the e_ident field.
 * Why wasn't e_ident declared as a structure?
 */
#define	EI_MAG0		0		/* the magic number */
#define	EI_MAG1		1
#define	EI_MAG2		2
#define	EI_MAG3		3
#define	EI_CLASS	4		/* 32 vs. 64 bits */
#define	EI_DATA		5		/* byte ordering */
#define	EI_VERSION	6		/* ELF header version */
#define	EI_PAD		7

#define	ELFMAG0		0x7f
#define	ELFMAG1		0x45		/* ASCII 'E' */
#define	ELFMAG2		0x4c		/* ASCII 'L' */
#define	ELFMAG3		0x46		/* ASCII 'F' */

#define	ELFMAG		"\x7f\x45\x4c\x46"
#define	SELFMAG		4

#define	ELFCLASSNONE	0
#define	ELFCLASS32	1
#define	ELFCLASS64	2

#define	ELFDATANONE	0
#define	ELFDATA2LSB	1		/* little endian */
#define	ELFDATA2MSB	2		/* big endian */

#define	EV_NONE		0
#define	EV_CURRENT	1		/* the current version */

#define	EM_NONE		0
#define	EM_M32		1		/* AT&T WE 32100 */
#define	EM_SPARC	2		/* SPARC, V8 or earlier */
#define	EM_386		3		/* Intel 80386 */
#define	EM_68K		4		/* Motorola 68000 */
#define	EM_88K		5		/* Motorola 88000 */
#define	EM_486		6		/* Intel 80486 */
#define	EM_860		7		/* Intel i860 */
#define	EM_MIPS		8		/* MIPS (<=R3000?), big-endian */
#define	EM_MIPS_RS3_LE	10		/* MIPS, little-endian */
#define	EM_RS6000	11		/* RS6000 */
#define	EM_PARISC	15		/* HP PA-RISC */
#define	EM_nCUBE	16		/* nCUBE */
#define	EM_VPP500	17		/* Fujitsu VPP500 */
#define	EM_SPARC32PLUS	18		/* SPARC32+ */
#define	EM_PPC		20		/* PowerPC */
#define	EM_SPARCV9	43		/* SPARC V9 */

#define	ET_NONE		0
#define	ET_REL		1		/* ELF relocatable file type */
#define	ET_EXEC		2		/* ELF executable file type */
#define	ET_DYN		3		/* ELF shared object file type */
#define	ET_CORE		4		/* ELF core file type */
#define	ET_LOPROC	0xff00		/* reserved for implementations ... */
#define	ET_HIPROC	0xffff		/* ... through this value */


/*
 * ELF section headers.
 */

/*
 * The section headers appear in an array;
 * the index of an element is its 'section header table index'.
 * Certain fields contain section header table indices;
 * the following reserved values may appear in those fields
 * even though they don't refer to actual sections.
 */
#define	SHN_UNDEF	0
#define	SHN_LORESERVE	0xff00
#define	SHN_LOPROC	0xff00
#define	SHN_HIPROC	0xff1f
#define	SHN_ABS		0xfff1
#define	SHN_COMMON	0xfff2
#define	SHN_HIRESERVE	0xffff

typedef struct {
	Elf32_Word	sh_name;	/* index into section header strings */
	Elf32_Word	sh_type;
	Elf32_Word	sh_flags;
	Elf32_Addr	sh_addr;	/* load address */
	Elf32_Off	sh_offset;	/* offset in file */
	Elf32_Word	sh_size;
	Elf32_Word	sh_link;	/* hack dependent on section type */
	Elf32_Word	sh_info;	/* hack dependent on section type */
	Elf32_Word	sh_addralign;
	Elf32_Word	sh_entsize;	/* size of entries in section */
} Elf32_Shdr;

typedef struct {
	Elf64_Word	sh_name;	/* index into section header strings */
	Elf64_Word	sh_type;
	Elf64_Xword	sh_flags;
	Elf64_Addr	sh_addr;	/* load address */
	Elf64_Off	sh_offset;	/* offset in file */
	Elf64_Xword	sh_size;
	Elf64_Word	sh_link;	/* hack dependent on section type */
	Elf64_Word	sh_info;	/* hack dependent on section type */
	Elf64_Xword	sh_addralign;
	Elf64_Xword	sh_entsize;	/* size of entries in section */
} Elf64_Shdr;

/*
 * Quick reminder about the sh_link and sh_info hacks...
 * SHT_DYNAMIC: sh_link = SHT index for string table
 * SHT_HASH: sh_link = SHT index for symbol table
 * SHT_REL, SHT_RELA: sh_link = SHT index for symbol table,
 *	sh_info = SHT index for relocated section
 * SHT_SYMTAB, SHT_DYNSYM: sh_link = SHT index for string table,
 *	sh_link = symbol table index of last local symbol + 1
 */

#define	SHT_NULL	0
#define	SHT_PROGBITS	1		/* information unique to the program */
#define	SHT_SYMTAB	2		/* full symbol table */
#define	SHT_STRTAB	3		/* string table */
#define	SHT_RELA	4		/* relocations with addends */
#define	SHT_HASH	5		/* symbol hash table */
#define	SHT_DYNAMIC	6		/* dynamic linking information */
#define	SHT_NOTE	7		/* information about the file */
#define	SHT_NOBITS	8		/* the empty section */
#define	SHT_REL		9		/* relocations without addends */
#define	SHT_SHLIB	10		/* reserved */
#define	SHT_DYNSYM	11		/* reduced, loaded symbol table */
#define	SHT_LOPROC	0x70000000
#define	SHT_HIPROC	0x7fffffff
#define	SHT_LOUSER	0x80000000
#define	SHT_HIUSER	0xffffffff

#define	SHF_WRITE	0x1		/* section in a writable segment */
#define	SHF_ALLOC	0x2		/* this section is loaded */
#define	SHF_EXECINSTR	0x4		/* this section is executable */
#define	SHF_MASKPROC	0xf0000000

/*
 * For reference, the names of the reserved sections...
 * A = SHF_ALLOC, W = SHF_WRITE, X = SHF_EXECINSTR
 *
 * .bss (SHT_NOBITS/AW): BSS segment, zero-filled data
 * .comment (SHT_PROGBITS): version control information
 * .data (SHT_PROGBITS/AW): the usual data segment
 * .data1: same as .data
 * .debug* (SHT_PROGBITS): DWARF symbolic debugging information
 * .dynamic (SHT_DYNAMIC/AW): the _DYNAMIC[] array
 * .dynstr (SHT_STRTAB/A): dynamic linking string table
 * .dynsym (SHT_DYNSYM/A): dynamic linking symbol table
 * .fini (SHT_PROGBITS/AX): global destructors
 * .got (SHT_PROGBITS/AW): global offset table
 * .hash (SHT_HASH/A): symbol hash table
 * .init (SHT_PROGBITS/AX): global constructors
 * .interp (SHT_PROGBITS/A?): interpreter pathname
 * .line (SHT_PROGBITS): DWARF line number debugging information
 * .note (SHT_NOTE): variable length note records
 * .plt (SHT_PROGBITS/AX): procedure linkage table
 * .rel<section-name> (SHT_REL/A?): relocation records
 * .rela<section-name> (SHT_RELA/A?): relocation records with addends
 * .rodata (SHT_PROGBITS/A): read-only data
 * .rodata1: same as .rodata
 * .shstrtab (SHT_STRTAB): section name string table
 * .strtab (SHT_STRTAB/A?): string table
 * .symtab (SHT_SYMTAB/A?): symbol table
 * .text (SHT_PROGBITS/AX): the usual text segment
 */

/*
 * The symbol table.
 */

typedef struct __elf32_sym {
	Elf32_Word	st_name;	/* index into the string table */
	Elf32_Addr	st_value;
	Elf32_Word	st_size;	/* size of the object */
	unsigned char	st_info;	/* type and binding attributes */
	unsigned char	st_other;
	Elf32_Half	st_shndx;	/* SHT index */
} Elf32_Sym;

typedef struct {
	Elf64_Word	st_name;	/* index into the string table */
	unsigned char	st_info;	/* type and binding attributes */
	unsigned char	st_other;
	Elf64_Half	st_shndx;	/* SHT index */
	Elf64_Addr	st_value;
	Elf64_Xword	st_size;	/* size of the object */
} Elf64_Sym;

/* macro for packing st_info */
#define	ELF32_ST_INFO(bind, type) \
	(((bind) << 4) | ((type) & 0xf))
/* macros for unpacking st_info */
#define	ELF32_ST_BIND(info) \
	((info) >> 4)
#define	ELF32_ST_TYPE(info) \
	((info) & 0xf)

/* these appear to be the same as for ELF32 (nlist code depends on this!) */
#define	ELF64_ST_INFO(bind, type)	ELF32_ST_INFO(bind, type)
#define	ELF64_ST_BIND(info)		ELF32_ST_BIND(info)
#define	ELF64_ST_TYPE(info)		ELF32_ST_TYPE(info)

/* values for 'bind' field */
#define	STB_LOCAL	0x0
#define	STB_GLOBAL	0x1
#define	STB_WEAK	0x2
#define	STB_LOPROC	0xd
#define	STB_HIPROC	0xf

/* values for 'type' field */
#define	STT_NOTYPE	0x0
#define	STT_OBJECT	0x1
#define	STT_FUNC	0x2
#define	STT_SECTION	0x3
#define	STT_FILE	0x4
#define	STT_LOPROC	0xd
#define	STT_HIPROC	0xf

/*
 * Relocation entries.
 * The 'rela' form is for systems that
 * don't put the addend bits at the relocated offset in the section.
 */

typedef struct __elf32_rel {
	Elf32_Addr	r_offset;
	Elf32_Word	r_info;
} Elf32_Rel;

typedef struct __elf32_rela {
	Elf32_Addr	r_offset;
	Elf32_Word	r_info;
	Elf32_Sword	r_addend;
} Elf32_Rela;

typedef struct {
	Elf64_Addr	r_offset;
	Elf64_Xword	r_info;
} Elf64_Rel;

typedef struct {
	Elf64_Addr	r_offset;
	Elf64_Xword	r_info;
	Elf64_Sxword	r_addend;
} Elf64_Rela;

/* macro for packing r_info */
#define	ELF32_R_INFO(index, type) \
	(((index) << 8) | ((type) & 0xff))
/* macros for unpacking r_info */
#define	ELF32_R_SYM(info) \
	((info) >> 8)
#define	ELF32_R_TYPE(info) \
	((info) & 0xff)

#define	ELF64_R_INFO(index, type) \
	(((Elf64_Xword)(index) << 32) | ((type) & 0xffffffff))
#define	ELF64_R_SYM(info) \
	((info) >> 32)
/*
 * ??? these may be specific to SPARC: the r_info has an 8-bit type ID
 * and a 24 bit type-ID-specific data field in the lower 32 bits, so
 * the R_TYPE is the entire lower 32 bits but we have two more macros.
 */
#define	ELF64_R_TYPE(info)		((Elf64_Word)(info))
#define	ELF64_R_TYPE_DATA(info)		((Elf64_Word)(info) >> 8)
#define	ELF64_R_TYPE_ID(info)		((Elf64_Word)(info) & 0xff)
#define	ELF64_R_TYPE_INFO(data, id)	(((data) << 8) | (id))

/*
 * ELF program headers.
 */

typedef struct __elf_phdr {
	Elf32_Word	p_type;
	Elf32_Off	p_offset;
	Elf32_Addr	p_vaddr;	/* virtual address after loading */
	Elf32_Addr	p_paddr;	/* physical address after loading */
	Elf32_Word	p_filesz;	/* size of segment in file */
	Elf32_Word	p_memsz;	/* size of segment in memory */
	Elf32_Word	p_flags;
	Elf32_Word	p_align;
} Elf32_Phdr;

typedef struct {
	Elf64_Word	p_type;
	Elf64_Word	p_flags;
	Elf64_Off	p_offset;
	Elf64_Addr	p_vaddr;	/* virtual address after loading */
	Elf64_Addr	p_paddr;	/* physical address after loading */
	Elf64_Xword	p_filesz;	/* size of segment in file */
	Elf64_Xword	p_memsz;	/* size of segment in memory */
	Elf64_Xword	p_align;
} Elf64_Phdr;

#define	PT_NULL		0
#define	PT_LOAD		1
#define	PT_DYNAMIC	2
#define	PT_INTERP	3		/* interpreter name */
#define	PT_NOTE		4
#define	PT_SHLIB	5		/* reserved, unused */
#define	PT_PHDR		6		/* PHT segment, if loaded */
#define	PT_LOPROC	0x70000000
#define	PT_HIPROC	0x7fffffff

#define	PF_X		0x1
#define	PF_W		0x2
#define	PF_R		0x4
#define	PF_MASKPROC	0xf0000000

/*
 * PT_NOTE sections.  Executables may have a version stamp (NT_VERSION);
 * core files have three notes for GDB, containing the register values
 * and so forth.
 */
typedef struct {
	Elf32_Word	n_namesz;	/* name length */
	Elf32_Word	n_descsz;	/* descriptor length */
	Elf32_Word	n_type;		/* type */
	/* followed by name and descriptor data, if any */
} Elf32_Nhdr;

typedef struct {
	Elf64_Word	n_namesz;	/* name length */
	Elf64_Word	n_descsz;	/* descriptor length */
	Elf64_Word	n_type;		/* type */
	/* followed by name and descriptor data, if any */
} Elf64_Nhdr;

/*
 * Additional note types, including note types for core dumps,
 * are in <machine/elf.h> (???)
 */
#define	NT_VERSION	1


/*
 * Elf dymamic linking structures and macros.
 */
typedef struct __dynamic {
	Elf32_Sword		d_tag;
	union {
		Elf32_Word	d_val;
		Elf32_Addr	d_ptr;
	} d_un;
} Elf32_Dyn;

typedef struct {
	Elf64_Sxword		d_tag;
	union {
		Elf64_Xword	d_val;
		Elf64_Addr	d_ptr;
	} d_un;
} Elf64_Dyn;

#define	DT_NULL		0
#define	DT_NEEDED	1
#define	DT_PLTRELSZ	2
#define	DT_PLTGOT	3
#define	DT_HASH		4
#define	DT_STRTAB	5
#define	DT_SYMTAB	6
#define	DT_RELA		7
#define	DT_RELASZ	8
#define	DT_RELAENT	9
#define	DT_STRSZ	10
#define	DT_SYMENT	11
#define	DT_INIT		12
#define	DT_FINI		13
#define	DT_SONAME	14
#define	DT_RPATH	15
#define	DT_SYMBOLIC	16
#define	DT_REL		17
#define	DT_RELSZ	18
#define	DT_RELENT	19
#define	DT_PLTREL	20
#define	DT_DEBUG	21
#define	DT_TEXTREL	22
#define	DT_JMPREL	23
#define	DT_LOPROC	0x70000000
#define	DT_HIPROC	0x7fffffff

extern Elf32_Dyn _DYNAMIC[];	/* XXX -- Elf32 only */



/*
 * An 'auxiliary vector' is passed on the stack to ELF linker/loaders.
 * An auxv_t is an element of the vector; an AT_NULL element terminates it.
 * The information comes from p 3-29 of the Power PC ABI.
 */
typedef struct {
	int	a_type;
	union {
		long	a_val;
		void	*a_ptr;
		void	(*a_fcn)();
	} a_un;
} auxv_t;

#define	AT_NULL		0
#define	AT_IGNORE	1
#define	AT_EXECFD	2		/* file descriptor for loaded file */
#define	AT_PHDR		3		/* PHT address in loaded file */
#define	AT_PHENT	4		/* PHT element size */
#define	AT_PHNUM	5		/* PHT vector element count */
#define	AT_PAGESZ	6
#define	AT_BASE		7		/* loader's 'base address' */
#define	AT_FLAGS	8
#define	AT_ENTRY	9		/* loaded file's entry point */



#include <machine/elf.h>

#ifdef KERNEL

struct text;
int exec_elf_binary __P((struct text *));

#endif

#endif
