/*-
 * Copyright (c) 1996, 1998 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI apic.h,v 2.5 2000/04/28 03:00:46 donn Exp
 */

/* Structures definition 'borrowed' from mach4 to avoid duplicating efforts */
/* Ported to NetBSD & BSD/OS (c) 1995 Stefan Grefen */

/* 
 * Copyright (c) 1994 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 *
 *      Author: Bryan Ford, University of Utah CSL
 */
#ifndef _MPS_APIC_H
#define _MPS_APIC_H

#ifndef LOCORE
typedef struct apicreg_t
{
	volatile u_int32_t r;	/* the actual register */
	u_int32_t	p[3];	/* pad to the next 128-bit boundary */
} apicreg_t;

/* 
 * make this structure padded to 4k (page size) so multiple apics
 * can be handled as an array
 */
typedef struct ioapic io_apic_t;
struct io_apic {
	union {
		struct {
		    apicreg_t	ir_select;
		    apicreg_t	ir_window;
		} io_register;
		char __io_page[4096];
	} u;
#define ia_select u.io_register.ir_select
#define ia_window u.io_register.ir_window
};

/* I/O APIC offsets */
#define APIC_IO_UNIT_ID			0x00
#define APIC_IO_VERSION			0x01
#define	APIC_IO_ARB			0x02
#define APIC_IO_REDIR_LOW(int_pin)	(0x10+(int_pin)*2)
#define APIC_IO_REDIR_HIGH(int_pin)	(0x11+(int_pin)*2)

/* APIC_IO_UNIT_ID */
#define	APIC_IO_ID_SHFT		24
#define	APIC_IO_ID_MASK		0xf

/* APIC_IO_VERSION */
#define	APIC_IO_VER_MASK	0xff
#define	APIC_IO_MRE_MASK	0xff
#define	APIC_IO_MRE_SHFT	16

typedef struct local_apic local_apic_t;
struct local_apic {
	apicreg_t	la_reserved0;
	apicreg_t	la_reserved1;
	apicreg_t	la_unit_id;
	apicreg_t	la_version;
	apicreg_t	la_reserved4;
	apicreg_t	la_reserved5;
	apicreg_t	la_reserved6;
	apicreg_t	la_reserved7;
	apicreg_t	la_tpr;
	apicreg_t	la_abtr_pri;
	apicreg_t	la_proc_pri;
	apicreg_t	la_eoi;
	apicreg_t	la_remote;
	apicreg_t	la_logical_dest;
	apicreg_t	la_dest_format;
	apicreg_t	la_spurious_vector;
	apicreg_t	la_isr[8];
	apicreg_t	la_tmr[8];
	apicreg_t	la_irr[8];
	apicreg_t	la_esr;
	apicreg_t	la_reserved29[7];
	apicreg_t	la_int_command[2];
	apicreg_t	la_timer_vector;
	apicreg_t	la_reserved33;
	apicreg_t	la_pcint_vector;
	apicreg_t	la_lint0_vector;
	apicreg_t	la_lint1_vector;
	apicreg_t	la_error_vec;
	apicreg_t	la_init_count;
	apicreg_t	la_cur_count;
	apicreg_t	la_reserved3a;
	apicreg_t	la_reserved3b;
	apicreg_t	la_reserved3c;
	apicreg_t	la_reserved3d;
	apicreg_t	la_divider_config;
	apicreg_t	la_reserved3f;
};

#endif /* !LOCORE */

/* Local APIC defs in offset form */
#define	APIC_REG_ID	0x20
#define	APIC_REG_VER	0x30
#define	APIC_REG_TPR	0x80
#define	APIC_REG_APR	0x90
#define	APIC_REG_PPR	0xa0
#define	APIC_REG_EOI	0xb0
#define	APIC_REG_RRR	0xc0
#define	APIC_REG_LDR	0xd0
#define	APIC_REG_DFR	0xe0
#define	APIC_REG_SIVR	0xf0
#define	APIC_REG_ISR	0x100
#define	APIC_REG_TMR	0x180
#define	APIC_REG_IRR	0x200
#define	APIC_REG_ERR	0x280
#define	APIC_REG_ICRL	0x300
#define	APIC_REG_ICRH	0x310
#define	APIC_REG_LVT_T	0x320	/* Performance Timer */
#define	APIC_REG_LVT_P	0x340	/* Performance Vector */
#define	APIC_REG_LVT_0	0x350
#define	APIC_REG_LVT_1	0x360
#define	APIC_REG_LVT_E	0x370
#define	APIC_REG_ICNT	0x380
#define	APIC_REG_CCNT	0x390
#define	APIC_REG_TDIV	0x3e0


/* IMCR */
#define IMCR_ADDR		0x22	      /* IMCR address register */
#define IMCR_DATA		0x23	      /* IMCR data register */

#define	IMCR_REGSEL		0x70
#define IMCR_APIC_OFF		0x00	      /* Bypass io_apic */
#define IMCR_APIC_ON		0x01	      /* Use io_apic */

/* ID register */
#define	APIC_ID_SHFT		24
#define	APIC_ID_MASK		0xf

/* Apic IPI's and ICR format */
#define APIC_VC_MASK		0x000000ff    /* Vector mask*/

#define APIC_DL_FIXED		0x00000000	  /* fixed */
#define APIC_DL_LOWPR		0x00000100    /* lowest priority */
#define APIC_DL_SMI		0x00000200    /* system managment */
#define APIC_DL_RR		0x00000300    /* remote read */
#define APIC_DL_NMI		0x00000400    /* NMI */
#define APIC_DL_INIT		0x00000500    /* INIT IPI */
#define APIC_DL_START		0x00000600    /* Start Up IPI */
#define	APIC_DL_EXTINT		0x00000700    /* ExtINT trigger */
#define APIC_DL_MASK		0x00000700    /* Delivery mode mask */

#define APIC_DM_PHYS		0x00000000	/* physical dest mode */
#define APIC_DM_LOGI		0x00000800	/* logical dest mode */

#define APIC_DS_IDLE		0x00000000	/* bus or target is idle */
#define APIC_DS_BUSY		0x00001000	/* bus or target is busy */

#define APIC_PL_HIGH		0x00000000	* Input polarity active high */
#define APIC_PL_LOW		0x00002000	/* Input polarity active low */

#define APIC_RIRR		0x00004000	/* Remote IRR status */
#define APIC_LV_MASK		0x00004000	/* level offset      */

#define APIC_TM_EDGE		0x00000000	/* only for INIT IPI  */
#define APIC_TM_LEVEL		0x00008000	/* only for INIT IPI  */

#define APIC_MS_ENABLE		0x00000000	/* Enable interrupt */
#define APIC_MS_DISABLE		0x00010000	/* Disable interrupt */

#define APIC_DH_DSTF		0x00000000	/* use dest field */
#define APIC_DH_SELF		0x00040000	/* dest self      */
#define APIC_DH_ALLS		0x00080000	/* All incl. self */
#define APIC_DH_ALLE		0x000c0000	/* All excl. self */
#define APIC_DH_MASK		0x000c0000	/* destination shorthand mask */

#define APIC_DF_MASK		0xff000000	/* Destination field mask */
#define APIC_DF_SHFT		24	        /* Destination field offs */

#define	APIC_DF_BCAST		0x0f000000

#define APIC_SV_MASK		0x000000f0	/* Spuriosvector mask     */
#define APIC_SV_ENABLE		0x00000100	/* Enable APIC		  */

#define	APIC_ESR_SCS		0x00000001	/* Send checksum */
#define	APIC_ESR_RCS		0x00000002	/* Receive checksum */
#define	APIC_ESR_SAE		0x00000004	/* Send accept error */
#define	APIC_ESR_RAE		0x00000008	/* Receive accept error */
#define	APIC_ESR_SIV		0x00000020	/* Send ill vector */
#define	APIC_ESR_RIV		0x00000040	/* Receive ill vector */
#define	APIC_ESR_IRS		0x00000080	/* Ill reg addrs */

/* ICR-centric version */
#define ICR_VEC_MASK		0x000000ff	/* Vector mask*/
#define ICR_DM_FIXED		0x00000000	/* fixed */
#define ICR_DM_LOWPR		0x00000100	/* lowest priority */
#define ICR_DM_SMI		0x00000200	/* system managment */
#define ICR_DM_RR		0x00000300	/* remote read */
#define ICR_DM_NMI		0x00000400	/* NMI */
#define ICR_DM_INIT		0x00000500	/* INIT IPI */
#define ICR_DM_START		0x00000600	/* Start Up IPI */
#define ICR_DM_MASK		0x00000700	/* Delivery mode mask */
#define	ICR_DEST_LOGICAL	0x00000800	/* Logical/Physical */
#define	ICR_BUSY		0x00001000
#define	ICR_LEVEL_ASSERT	0x00004000
#define	ICR_TRIG_MODE		0x00008000
#define	ICR_RRS_INVALID		0x00000000
#define	ICR_RRS_IN_PROG		0x00010000
#define	ICR_RRS_VALID		0x00020000
#define	ICR_RRS_MASK		0x00030000
#define	ICR_DH_MASK		0x000c0000
#define	ICR_DH_DEST		0x00000000
#define	ICR_DH_SELF		0x00040000
#define	ICR_DH_ALLS		0x00080000
#define	ICR_DH_ALLE		0x000c0000
#define	ICRH_DST_SHFT		24
#define	ICRH_DST_MASK		0xf0000000

#ifdef KERNEL

#ifndef LOCORE
#include <sys/systm.h>

extern volatile local_apic_t apic;
extern volatile io_apic_t io_apic[];

extern char apic_busy[];
extern char waitbusy_panic[];
extern char apic_rripi[];

#define APIC_ICR_REG apic.la_int_command[0].r

#define	i386_apic_waitbusy()						\
do {									\
	int __cnt;							\
									\
	if (APIC_ICR_REG & ICR_BUSY) {					\
		CTR0(KTR_SMP, apic_busy);				\
		__cnt = 1000000;		/* about a second */	\
		while (--__cnt) {					\
			if ((APIC_ICR_REG & ICR_BUSY) == 0)		\
				break;					\
			DELAY(1);					\
		}							\
		if (__cnt <= 0)						\
			panic(waitbusy_panic);				\
	}								\
} while (0)

#define i386_ipi(t, i) _i386_ipi(t, i, "send ipi->%x icr=%x " __WHERE__)
#define i386_ipi_short(i) _i386_ipi_short(i, "send short ipi icr=%x " __WHERE__)

/*
 * Send to a single target
 */
#define	_i386_ipi(targ, icr, trace)					\
do {									\
	CTR2(KTR_SMP, trace, targ, icr);				\
	i386_apic_waitbusy();						\
	apic.la_int_command[1].r = (targ) << ICRH_DST_SHFT;		\
	APIC_ICR_REG = (icr);						\
} while (0)

/*
 * Send using destination shorthand
 */
#define	_i386_ipi_short(icr, trace)					\
do {									\
	CTR1(KTR_SMP, trace, icr);					\
	i386_apic_waitbusy();						\
	APIC_ICR_REG = (icr);						\
} while (0)

extern char apic_rdreg_tr[];

/* Read a remote APIC register (can be used to check if APIC is there) */
#define	i386_apic_rdreg(id, i, val)					\
({									\
	int __cnt;							\
	int __rv = 0;							\
	int *__val = (val);						\
									\
	CTR2(KTR_SMP, apic_rdreg_tr, id, i);				\
	i386_apic_waitbusy();						\
	_i386_ipi(id, ICR_DM_RR | ICR_LEVEL_ASSERT | (i), apic_rripi);	\
	__cnt = 1000000;						\
	while (--__cnt && (APIC_ICR_REG & ICR_RRS_MASK) == ICR_RRS_IN_PROG) \
		delay(1);						\
	if ((APIC_ICR_REG & ICR_RRS_MASK) == ICR_RRS_VALID) {		\
		if (__val != NULL)					\
			*__val = apic.la_remote.r;			\
		__rv = 1;						\
	}								\
	__rv;								\
})

#endif /* !LOCORE */

#endif /* KERNEL */

/* Default physical address for local APIC */
#define	APIC_LOCAL_PA	0xfee00000
#define	APIC_IO_PA	0xfec00000	

/*
 * TPR priorities (must be lower than any possible interrupt)
 */
#define	TPR_KERNEL	0x10		/* Running with kernel lock */
#define	TPR_IDLE	0x12		/* Running idle process */
#define TPR_USER	0x14		/* Running user process */

#endif /* !_MPS_APIC_H */
