/*-
 * Copyright (c) 1993, 1995, 1996, 1998 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI inline.h,v 2.17.2.1 2001/05/04 14:32:58 donn Exp
 */

/*
 * Functions to provide access to special i386 instructions
 * or inline implementations of common operations.
 */

#ifndef __INLINE_H
#define __INLINE_H

/* prototypes are here so that they precede any macro versions */
int	inb __P((u_int port));
void	outb __P((u_int port, u_char data));
void __splx __P((void));

#ifdef	__GNUC__
#include <i386/isa/icu.h>

/* Prevent the compiler's optimizer from moving code around this barrier.  */
#define	SCHED_BARRIER	asm volatile ("# scheduling barrier")

/*
 * The following pair of macros can be used instead of splmem (aka splimp)
 * and splx, in pairs, for sections that need to block memory allocation
 * for a very short time.
 */
#define	splmem_fast()	({ extern u_int	cpl; asm volatile ("cli"); cpl; })
#define	splxmem_fast(s)	{ int __attribute__((unused))_notused = (s);	\
			    asm volatile ("sti"); }

/*
 * Using the lazy spl scheme, all we do is mark the interrupt blocked in
 * main memory. If a blocked interrupt occurs we queue it and deal with
 * it at the appropriate splx().
 */
#include <sys/ktr.h>
#if KTR_COMPILE & KTR_SPL
extern char tr_spltty[];
extern char tr_splnet[];
extern char tr_splclock[];
extern char tr_splstatclock[];
extern char tr_splhigh[];
extern char tr_splbio[];
extern char tr_splimp[];
extern char tr_splmem[];
extern char tr_splraise[];
extern char tr_splx[];
extern char tr_splxdone[];

#define	SPLRT(dsc, v)							\
	CTR4(KTR_SPL, dsc , __FILE__, __LINE__,	cpl, v);
	    
#else
#define SPLRT(dsc, v)
#endif

#define	spldraise(dsc, v) ({						\
	int ocpl;							\
	SCHED_BARRIER;							\
	ocpl = cpl;							\
	SPLRT(dsc, v);							\
	asm volatile ("lock; orl %1,cpl" : "=m" (cpl) : "ri" ((v)));	\
	ocpl;								\
})

extern u_int irq2mask[];

#define	splraise(v)	({						\
	register int tcpl = (v);					\
	if (do_pic_chk & tcpl)						\
		tcpl = irq2mask[ffs(tcpl)-1];				\
	spldraise(tr_splraise, tcpl);					\
})
#define	spltty()	spldraise(tr_spltty, ttymask)
#define	splnet() 	spldraise(tr_splnet, IRQSOFT)
#define	splclock()	spldraise(tr_splclock, highmask)
#define	splstatclock()	spldraise(tr_splstatclock, highmask)
#define	splhigh()	spldraise(tr_splhigh, highmask)
#define	splbio()	spldraise(tr_splbio, biomask)
#define	splimp()	spldraise(tr_splimp, impmask)
#define	splmem()	spldraise(tr_splmem, memmask)

/*
 * Forcibly lowers CPL to protomask (which is nonmask with soft interrupts
 * blocked).
 */
#if KTR_COMPILE & KTR_SPL
#define SPLSCT								\
	CTR4(KTR_GEN, "splsoftclock (splx) %s:%d %x -> %x", __FILE__,	\
	    __LINE__, cpl, protomask)
#define	SPLSCTE								\
	CTR2(KTR_GEN, "splsoftclock done %s:%d", __FILE__, __LINE__);
#else
#define SPLSCT
#define SPLSCTE
#endif
#define	splsoftclock()	({						\
	int ocpl = cpl;							\
	SPLSCT;								\
	SCHED_BARRIER;							\
	cpl = protomask;						\
	__splx();							\
	SPLSCTE;							\
	ocpl;								\
})

#if KTR_COMPILE & KTR_SPL
#define	SPL0T								\
	CTR3(KTR_SPL, "spl0 %s:%d, old cpl=%x", __FILE__, __LINE__, cpl);
#define	SPL0TE								\
	CTR2(KTR_SPL, "spl0 done %s:%d", __FILE__, __LINE__);
#else
#define	SPL0T
#define	SPL0TE
#endif

/*
 * Spl0 forces priority to protomask, calls any pending spl0 work, then
 * spls down to nonemask (all valid interrupts enabled).
 */
#define	spl0() ({							\
	SPL0T;								\
	SCHED_BARRIER;							\
	cpl = nonemask;							\
	__splx();							\
	SPL0TE;								\
	(void)0;							\
})


/*
 * Implement splx calls that are no-ops in line, and call __splx for others.
 * Using a value-macro and the (void)0 "return" value avoids breaking code
 * that stupidly does "(void) splx(s)".  It also handles code that might
 * use splx as the only statement in a conditional.
 */
#if KTR_COMPILE & KTR_SPL
#define	SPLXRT(x)							\
	CTR4(KTR_SPL, tr_splx, __FILE__, __LINE__,	\
	    cpl, (x));
#define	SPLXRTX								\
	CTR3(KTR_SPL, tr_splxdone, __FILE__, __LINE__, cpl);
#else
#define SPLXRT(x)
#define	SPLXRTX
#endif
extern int ipending;
#define	splx(x) ({ 							\
	extern u_int cpl;						\
	SPLXRT(x);							\
	SCHED_BARRIER;							\
	cpl = x;							\
	if (ipending & ~(x))						\
		__splx();						\
	SCHED_BARRIER;							\
	SPLXRTX;							\
	(void)0; })

#define	cli()		asm volatile ("cli")
#define	sti()		asm volatile ("sti")
#define	hlt()		asm volatile ("hlt")

#define	inb(port) \
({ \
	register int _inb_result; \
\
	asm volatile ("xorl %%eax,%%eax; inb %%dx,%%al" : \
	    "=a" (_inb_result) : "d" (port)); \
	_inb_result; \
})

#define	inw(port) \
({ \
	register int _inb_result; \
\
	asm volatile ("xorl %%eax,%%eax; inw %%dx,%%ax" : \
	    "=a" (_inb_result) : "d" (port)); \
	_inb_result; \
})

#define	inl(port) \
({ \
	register u_long _inb_result; \
\
	asm volatile ("inl %%dx,%%eax" : "=a" (_inb_result) : "d" (port)); \
	_inb_result; \
})

#define	outb(port, data) \
	asm volatile ("outb %%al,%%dx" : : "a" (data), "d" (port))
#define outw(port, data) \
	asm volatile ("outw %%ax,%%dx" : : "a" (data), "d" (port))
#define outl(port, data) \
	asm volatile ("outl %%eax,%%dx" : : "a" (data), "d" (port))

#define	ffs(mask) \
({ \
	int result = (mask); \
\
	asm volatile ("bsfl %1,%0; leal 1(%0),%0; jnz 0f; xorl %0,%0; 0:" : \
	    "=r" (result) : "0" (result)); \
	result; \
})

#define	INLINE_NTOH	/* for endian.h */

#define	ntohs(s) \
(({ \
        register int rv; \
\
	asm volatile ("xchgb %%al,%%ah" : "=a" (rv) : "0" (s)); \
	rv; \
}) & 0xffff)

#define	htons(s) ntohs(s)

#define	ntohl(l) \
({ \
        register u_long rv; \
\
	asm volatile ("xchgb %%al,%%ah; roll $16,%%eax; xchgb %%al,%%ah" : \
	    "=a" (rv) : "0" (l) ); \
	rv; \
})

#define htonl(l)	ntohl(l)


#define	INLINE_MINMAX	/* for libkern.h */

#define	imin(a, b) \
({ \
	int _a = (a), _b = (b); \
\
	_a < _b ? _a : _b; \
})

#define	imax(a, b) \
({ \
	int _a = (a), _b = (b); \
\
	_a > _b ? _a : _b; \
})

#define	min(a, b) \
({ \
	unsigned int _a = (a), _b = (b); \
\
	_a < _b ? _a : _b; \
})

#define	max(a, b) \
({ \
	unsigned int _a = (a), _b = (b); \
\
	_a > _b ? _a : _b; \
})

#define	lmin(a, b) \
({ \
	long _a = (a), _b = (b); \
\
	_a < _b ? _a : _b; \
})

#define	lmax(a, b) \
({ \
	long _a = (a), _b = (b); \
\
	_a > _b ? _a : _b; \
})

#define	ulmin(a, b) \
({ \
	unsigned long _a = (a), _b = (b); \
\
	_a < _b ? _a : _b; \
})

#define	ulmax(a, b) \
({ \
	unsigned long _a = (a), _b = (b); \
\
	_a > _b ? _a : _b; \
})

/*
 * Macros to do operations locked against other processors
 */
#define	lockd_and(targ, mask) ({					\
	asm volatile ("lock; andl %1,%0" : "=m" (targ): "ir" (mask));	\
})
#define	lockd_or(targ, bits) ({						\
	asm volatile ("lock; orl %1,%0" : "=m" (targ): "ir" (bits));	\
})
#define	lockd_xor(targ, bits) ({					\
	asm volatile ("lock; xorl %1,%0" : "=m" (targ): "ir" (bits));	\
})
#define	lockd_inc(targ) ({						\
	asm volatile ("lock; incl %0" : "=m" (targ));			\
})
#define	btc(targ, bitno) ({						\
	int rc;								\
	asm volatile ("xorl %%eax,%%eax; btc %1,%0; jnc 1f; incl %%eax;1:" \
	    : "=m" (targ), "=a" (rc): "ir" (bitno));			\
	rc;								\
})
#define	bts(targ, bitno) ({						\
	int rc;								\
	asm volatile ("xorl %%eax,%%eax; bts %1,%0; jnc 1f; incl %%eax;1:" \
	    : "=m" (targ), "=a" (rc): "ir" (bitno));			\
	rc;								\
})
#define	btr(targ, bitno) ({						\
	int rc;								\
	asm volatile ("xorl %%eax,%%eax; bts %1,%0; jnc 1f; incl %%eax;1:" \
	    : "=m" (targ), "=a" (rc): "ir" (bitno));			\
	rc;								\
})

#define rdtsc(m) asm volatile ("rdtsc; movl %%eax,%0; movl %%edx,4+%0" \
    : "=m" (m) : : "eax", "edx")

#endif	/* !__GNUC__ */

#endif /* _INLINE_H */
