/*-
 * Copyright (c) 2000, 2001 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI iprof.h,v 2.5 2001/01/17 02:29:10 jch Exp
 */
#ifndef _IPROF_H_
#define _IPROF_H_

/*
 * Intel Performance Counter
 */

#define	PC_CESR		0x11	/* P5 only */
#define	PC_CTR0		0x12	/* P5 only */
#define	PC_CTR1		0x13	/* P5 only */

#define	PC_PERFCTR0	0xc1	/* P6 and better */
#define	PC_PERFCTR1	0xc2	/* P6 and better */

#define	PC_EVNTSEL0	0x186	/* P6 and better */
#define	PC_EVNTSEL1	0x187	/* P6 and better */

typedef struct {
	u_long	pes_eventselect:8;
	u_long	pes_unitmask:8;
	u_long	pes_usr:1;		/* Count in User Mode */
	u_long	pes_os:1;		/* Count in OS Mode */
	u_long	pes_e:1;		/* Edge Detect */
	u_long	pes_pc:1;		/* Pin Control */
	u_long	pes_int:1;		/* APIC Interruypt Enable */
	u_long	pes_dur:1;		/* Reserved - Used for P5 cesr */
	u_long	pes_en:1;		/* Enable Counters (PC_EVNTSEL0 only) */
	u_long	pes_inv:1;		/* Invert Counter Mask */
	u_long	pes_countermask:8;
} evntsel_t;

typedef struct {
	u_long	cesr_es0:6;		/* Event Select 0 */
	u_long	cesr_usr0:1;		/* Count in User Mode 0 */
	u_long	cesr_os0:1;		/* Count in OS Mode 0 */
	u_long	cesr_dur0:1;		/* Count Clocks 0 */
	u_long	cesr_pc0:1;		/* Pin Control 0 */
	u_long	:6;
	u_long	cesr_es1:6;		/* Event Select 0 */
	u_long	cesr_usr1:1;		/* Count in User Mode 0 */
	u_long	cesr_os1:1;		/* Count in OS Mode 0 */
	u_long	cesr_dur1:1;		/* Count Clocks 0 */
	u_long	cesr_pc1:1;		/* Pin Control 0 */
	u_long	:6;
} cesr_t;

typedef struct {
	evntsel_t	pc_event;	/* Event to select */
	quad_t		pc_counter;	/* Value of counter */
	u_quad_t	pc_cycle;	/* approximate cycle counter */
} perfctr_t;

/*
 * Information relating to the results of the CPUID instruction
 */

#define	ISP5		(CPUID_FAMILY(cpu_id) == CPU_586)
#define	ISMMX		(cpu_feature_flags & CPUID_FEAT_MMX)
#define	HASRDPMC	((CPUID_FAMILY(cpu_id) > CPU_586) || ISMMX)
#define	HASPERFCNTR	(CPUID_FAMILY(cpu_id) >= CPU_586)
#define	HASAPIC		(CPUID_FAMILY(cpu_id) >= CPU_686 && \
    			    cpu_feature_flags & CPUID_FEAT_APIC)

/*
 * CPU_IPROF definitions.
 */
#define IPROF_INIT		1
#define IPROF_RUN		2	/* For P6's and better */
#define IPROF_PERF		3
#define IPROF_MSR		4
#define IPROF_PROCESSES		5
#define IPROF_RUN5		6	/* For P5's */

#define CPU_IPROF_NAMES { \
        { 0, 0 }, \
        { "init", CTLTYPE_STRUCT }, \
        { "run", CTLTYPE_STRUCT }, \
        { "perf", CTLTYPE_STRUCT }, \
        { "msr", CTLTYPE_NODE }, \
}  

/*
 * Structure used to define what to do with the performance counters
 */
typedef struct {
#ifdef	KERNEL
	u_long		selector0;
	u_long		selector1;
#else
	evntsel_t	sel0;
	evntsel_t	sel1;
#endif
	quad_t		counter0;
	quad_t		counter1;
	quad_t		initial0;
	quad_t		initial1;
	int		pid;
} iProfParam;

/*
 * Each sample collected is stuffed into a single iProfSample record.
 * A trick is played with the performance counter.  Initially the
 * performance counter is read into the 64 bit value that is shared
 * with either a pid or cpu number.  Once that has been done we are
 * free to load the cpu or pid.  We can do this because cycle counters
 * are only 40 bits long.
 */
typedef struct {
        u_long          ips_eip;		/* Instruction Pointer */
        u_long          ips_eflags;		/* E Flags */
	quad_t		ips_cycle;		/* current cycle counter */
	union {
	    quad_t	_ips_p0cpu;
	    struct {
		quad_t	_ips_p0:48;		/* Perf Counter 0 */
		quad_t	_ips_cpu:16;		/* CPU Number */
	    } _s0;
	} _u0;
	union {
	    quad_t	_ips_p1pid;
	    struct {
		quad_t	_ips_p1:48;		/* Perf Counter 0 */
		quad_t	_ips_pid:16;		/* Process ID */
	    } _s1;
	} _u1;
#define	ips_p0cpu	_u0._ips_p0cpu
#define	ips_p0		_u0._s0._ips_p0
#define	ips_cpu		_u0._s0._ips_cpu
#define	ips_p1pid	_u1._ips_p1pid
#define	ips_p1		_u1._s1._ips_p1
#define	ips_pid		_u1._s1._ips_pid
} iProfSample;

typedef struct iProfProc iProfProc;
typedef struct symlist_t symlist_t;

struct iProfProc {
	char		ipp_path[17];
	char		ipp_flag;
	short		ipp_pid;
	u_long		ipp_size;
	u_long		ipp_addr;
	u_long		ipp_loadcount;
	union {
	    u_long	_ipp_unloadcount;
	    symlist_t	*_ipp_symbols;
	} u;
	quad_t		ipp_tsc;
	iProfProc	*ipp_next;
};
#define	ipp_unloadcount	u._ipp_unloadcount
#define	ipp_symbols	u._ipp_symbols

typedef struct iProfSymbol iProfSymbol;

struct iProfSymbol {
	union {
	    iProfSymbol	*_ips_next;
	    char	*_ips_name;
	    u_long	_ips_nameoffset;
	} u;
	u_long		ips_addr;
	u_long		ips_length;
	u_long		ips_reserved;
};
#define	ips_next	u._ips_next
#define	ips_name	u._ips_name
#define	ips_nameoffset	u._ips_nameoffset

#define	IPP_PLATHLEN	0x1f
#define	IPP_SAMPLES	0x20

#define	iProfMAGIC	"iProf v1"

/*
 * Be nice and keep this to a multiple of 16 bytes.  It makes dumps
 * look prettier.
 */
typedef struct {
	u_char		magic[8];
	quad_t		start;
	quad_t		stop;
	struct timeval	zerotime;
	iProfProc	*ipp;
	iProfSample	*ips;
	int		samples;
	int		nproc;
	int		clks_per_sec;
	int		activeprocs;
#ifdef	KERNEL
	u_long		selector0;
	u_long		selector1;
#else
	evntsel_t	sel0;
	evntsel_t	sel1;
#endif
	quad_t		initial0;
	quad_t		initial1;
	quad_t		counter0;
	quad_t		counter1;
	int		rate;
	union {
	    iProfSymbol	*symbols;
	    int		nsymbols;
	} u;
	quad_t		overhead;
	quad_t		lasttick;
	u_char		dummy[8];
} iProfData;

#endif
