/*-
 * Copyright (c) 1996, 1998 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI pcpu.h,v 2.15 2000/04/28 03:00:48 donn Exp
 */

#ifndef _MACHINE_PCPU_H_
#define _MACHINE_PCPU_H_

#ifdef KERNEL
#include <i386/i386/cpuvar.h>
#endif

#ifndef LOCORE
/*
 * The per-cpu structure.
 * One of these contains all the state and statistics regarding
 * a single cpu.
 */

struct proc;
struct pcb;

#include <sys/ktr.h>
#include <sys/time.h>
#include <sys/vmmeter.h>
#include <sys/cpustats.h>
#include <machine/segments.h>
#include <machine/tss.h>
#include <machine/apic.h>

struct pcpu {
	/*
	 * Descriptor tables.  The GDT is unique per processor since it
	 * contains an entry pointing to the local processor's pcpu structure.
	 * The IDT and LDT are unique per process and hence per processor.
	 * Currently the IDT and LDT contain only one unique entry each,
	 * and we patch those respective entries at context switch time.
	 * We put the IDT first because that makes it easy for us to create
	 * the unique read-only IDT aliases required for the f00f bug fix.
	 */
	union descriptor idt[IDTSIZE];
	union descriptor gdt[NGDT];
	union descriptor ldt[NLDT];

	/*
	 * These fields are referenced from machine-independent code.
	 * They were previously machine-independent global variables.
	 */
	struct proc	*curproc;		/* currently running process */
	int		cpuno;			/* cpu number, 0 .. Ncpu-1 */
	int		want_resched;		/* resched() was called */
	struct cpustats	cpustats;
	struct timeval	runtime;		/* time process scheduled */
	u_char		curpriority;		/* current user priority */
	u_char		__pad1[3];		/* keep alignment */

	/*
	 * These fields are original to the pcpu structure.
	 */
	volatile int	pc_state;		/* Current CPU state */
	volatile int	pc_ktr_idx;		/* Index into trace table */
	char		*pc_ktr_buf;		/* Pointer to KTR buffer */
	int		pc_cpumask;		/* CPU number in mask form */
	int		pc_user_tpr;		/* User task priority (either
						   TPR_IDLE or TPR_USER) */
	LIST_ENTRY(pcpu) pc_allcpu;		/* All CPU's */
	int		pc_tmp;			/* currently used by resched */
	int		pc_shortcut;		/* not quite into kernel */
	volatile int	pc_clk[2];		/* used in startup/clk sync */
	volatile int	pc_trigger;		/* used by clock sync */
	int		pc_error;		/* Startup error code */

	/* LINT0/LINT1/ERROR/PCINT LVT entries, used to initialize local APIC */
	int		pc_lint[4];

	/* APIC error counters */
	int		pc_apic_err_intrs;
	int		pc_apic_send_csum;
	int		pc_apic_recv_csum;
	int		pc_apic_send_acc;
	int		pc_apic_recv_acc;
	int		pc_apic_send_ill_vec;
	int		pc_apic_recv_ill_vec;
	int		pc_apic_ill_reg_adrs;
	int		pc_apic_stray;

	/* MP related stats */
	int		pc_resched_ipi;		/* # of resched ipi's */
	int		pc_resched_ipi_n;	/* resched's w/o ASTPENDING */

	/* Debug */
	int		pc_halt_esp;		/* Saved stack top on halt */

	int		__pad2[75];

	/*
	 * The following fields were i386-specific global variables
	 * prior to the changes to add multiprocessor support.
	 * The big stuff (TSSes, emergency stack) appears at the end.
	 */
	struct proc	*prevproc;
	int		astpending;		/* ast trap needed */
	struct pcb	*curpcb;		/* current process's user pcb */
	struct proc	*npxproc;		/* process owning fpu state */
	struct proc	*idleproc;		/* idle proc for this cpu */
	struct i386tss	tss;
	struct i386tss	kdebug_tss;
	struct i386tss	panic_tss;
	char		emergency_stack[1024];

#ifdef KTR
	char		pc_ktr_buf_data[0];
#endif
};

#ifdef KERNEL

/*
 * The gdt descriptor GPCPU_SEL points at the per-cpu data
 * for the current processor, and the FS register selects this descriptor.
 *
 * This would have been a lot simpler if the %z feature worked correctly
 * in GCC's i386 code templates...  Fortunately the optimizer reduces
 * all of this C code to a single instruction in most cases.
 */

#define	__pcpu_offsetof(f) ((u_long)&((struct pcpu *)NULL)->f)

#define PCPU(field) \
({ \
	typedef typeof (((struct pcpu *)NULL)->field) __ftype; \
	__ftype __result; \
\
	if (sizeof (__result) == 1) { \
		u_char __b; \
		asm volatile ("movb %%fs:%1,%0" : \
		    "=a" (__b) : \
		    "m" (*(u_char *)(__pcpu_offsetof(field)))); \
		__result = (__ftype)(u_long)__b; \
	} else if (sizeof (__result) == 2) { \
		u_short __w; \
		asm volatile ("movw %%fs:%1,%0" : \
		    "=a" (__w) : \
		    "m" (*(u_short *)(__pcpu_offsetof(field)))); \
		__result = (__ftype)(u_long)__w; \
	} else { \
		u_long __l; \
		asm volatile ("movl %%fs:%1,%0" : \
		    "=a" (__l) : \
		    "m" (*(u_long *)(__pcpu_offsetof(field)))); \
		__result = (__ftype)__l; \
	} \
	__result; \
})
 
#define SET_PCPU(field, value) \
({ \
	typedef typeof (((struct pcpu *)NULL)->field) __ftype; \
\
	if (sizeof (__ftype) == 1) { \
		u_char __b = (u_char)(u_long)(value); \
		asm volatile ("movb %1,%%fs:%0" : \
		    "=m" (*(u_char *)(__pcpu_offsetof(field))) : \
		    "a" (__b)); \
	} else if (sizeof (__ftype) == 2) { \
		u_short __w = (u_short)(u_long)(value); \
		asm volatile ("movw %1,%%fs:%0" : \
		    "=m" (*(u_short *)(__pcpu_offsetof(field))) : \
		    "a" (__w)); \
	} else { \
		u_long __l = (u_long)(value); \
		asm volatile ("movl %1,%%fs:%0" : \
		    "=m" (*(u_long *)(__pcpu_offsetof(field))) : \
		    "a" (__l)); \
	} \
	(void)0; \
})
 
#define INC_PCPU(field) \
({ \
	typedef typeof (((struct pcpu *)NULL)->field) __ftype; \
\
	if (sizeof (__ftype) == 1) \
		asm volatile ("lock; incb %%fs:%0" : \
		    "=m" (*(u_char *)(__pcpu_offsetof(field)))); \
	else if (sizeof (__ftype) == 2) \
		asm volatile ("lock; incw %%fs:%0" : \
		    "=m" (*(u_short *)(__pcpu_offsetof(field)))); \
	else \
		asm volatile ("lock; incl %%fs:%0" : \
		    "=m" (*(u_long *)(__pcpu_offsetof(field)))); \
	(void)0; \
})

#define disseminate_pcpu_hardclock()	\
	if (numcpu > 1)			\
		IPI_PCLOCK_ALL()

LIST_HEAD(cpuhead, pcpu);
extern struct cpuhead cpuhead;
extern struct pcpu *num2cpu[MAXCPUS];

/*
 * Fixed virtual addresses for pcpu-related data
 *
 * IDTs are mapped read-only starting at IDT_VA(0), going up one page per CPU.
 * This is part of the Pentium f00f bug fix.
 *
 * I/O APIC's are mapped starting at page IO_APIC_VA(0) and going up 1 page per
 * I/O APIC address.
 */
#define	PCPU_VA		(CPUPTDI << PDRSHIFT)
#define	IDT_VA(x)	(PCPU_VA + 0x3d0000 + ctob(x))
#define	IO_APIC_VA(x)	(PCPU_VA + 0x3e0000 + ctob(x))
#define	APIC_VA		(PCPU_VA + 0x3f0000)
#define KDEBUG_VA	(PCPU_VA + 0x3f1000)		/* 2 pages */

extern int sio_mode;
extern int numcpu;		/* Number of running CPU's */

#endif		/* KERNEL */
#endif		/* !LOCORE */

/* Flags used during initialization */
#define	CPUFL_CONT	0x1
#define	CPUFL_HALT	0x2


/*
 * Cpu states
 */
#define	CPU_UNKNOWN	0	/* unitialized */
#define	CPU_UP		1	/* scheduling normally */
#define	CPU_CLKSYNC	2	/* Running clock init protocol */
#define	CPU_GETCLK	3	/* master wants clock read */
#define	CPU_SETCLK	4	/* master writing to clock */
#define	CPU_INITAP	5	/* initializing */
#define	CPU_STOPPED	6	/* stopped in cpu_halt */
#define	CPU_WAITTRIG	7	/* waiting to be triggered by master */
#define	CPU_TRIGGERED	8	/* triggered by master */
#define	CPU_REJECT	10	/* master rejected slave for startup */
#define	CPU_INITERR	11	/* failed low level AP init */
#define	CPU_PANIC_HALT	13	/* halted - panic IPI */

#endif /* !_MACHINE_PCPU_H_ */
