/*	BSDI $Id: route.h,v 2.15.6.5 2001/07/31 17:55:18 dab Exp $	*/
/*
 * This file has been slightly modified by NRL for use with IPv6+IPsec.
 * Search for INET6 and/or IPSEC to see the blocks where this happened.
 * See the NRL Copyright notice for conditions on the modifications.
 */

/*
 * Copyright (c) 1980, 1986, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)route.h	8.5 (Berkeley) 2/8/95
 */

#ifndef _ROUTE_H_
#define _ROUTE_H_

#include <sys/queue.h>

/*
 * Kernel resident routing tables.
 * 
 * The routing tables are initialized when interface addresses
 * are set by making entries for all directly connected interfaces.
 */

/*
 * A route consists of a destination address and a reference
 * to a routing entry.  These are often held by protocols
 * in their control blocks, e.g. inpcb.
 */

/*
 * We've made the route structure bigger to accomodate:
 *	struct sockaddr_in:	16 bytes
 *	struct sockaddr_in6:	28 bytes
 *	struct sockaddr_iso:	32 bytes
 * We could use a union with "sockaddr_storage", but that's 128 bytes
 * and seems like a lot of wasted space.  Since we know that we are only
 * using this with the above three structures, we can limit it to the
 * minimum size needed to accomodate the largest of them.
 */
#define NEW_STRUCT_ROUTE /* Route structure with room for all sockaddrs */

struct route {
	struct	rtentry *ro_rt;
	union {
		struct sockaddr ro_dst_sa;
		char rou_dst_ss[32];
	} ro_u;
};
#define ro_dst ro_u.ro_dst_sa

/*
 * These numbers are used by reliable protocols for determining
 * retransmission behavior and are included in the routing structure.
 */
struct rt_metrics {
	u_int32_t rmx_locks;	/* Kernel must leave these values alone */
	u_int32_t rmx_mtu;	/* MTU for this path */
	u_int32_t rmx_hopcount;	/* max hops expected */
	u_int32_t rmx_expire;	/* lifetime for route, e.g. redirect */
	u_int32_t rmx_recvpipe;	/* inbound delay-bandwith product */
	u_int32_t rmx_sendpipe;	/* outbound delay-bandwith product */
	u_int32_t rmx_ssthresh;	/* outbound gateway buffer limit */
	u_int32_t rmx_rtt;	/* estimated round trip time */
	u_int32_t rmx_rttvar;	/* estimated rtt variance */
};

struct rttimer {
	CIRCLEQ_ENTRY(rttimer) rtt_next;	
	LIST_ENTRY(rttimer) rtt_link;	/* Multiple timers on a single route */
	struct rtentry	*rtt_rt;	/* Back pointer to the route */
	void		(*rtt_func)();	/* Function to call on expiration */
	time_t		rtt_expire;	/* When this timer expires */
};

LIST_HEAD(rt_linkhead, rttimer);	/* Timer information */

/*
 * rmx_rtt and rmx_rttvar are stored as microseconds;
 * RTTTOPRHZ(rtt) converts to a value suitable for use
 * by a protocol slowtimo counter.
 */
#define	RTM_RTTUNIT	1000000	/* units for rtt, rttvar, as units per sec */
#define	RTTTOPRHZ(r)	((r) / (RTM_RTTUNIT / PR_SLOWHZ))

/*
 * We distinguish between routes to hosts and routes to networks,
 * preferring the former if available.  For each route we infer
 * the interface to use from the gateway address supplied when
 * the route was entered.  Routes that forward packets through
 * gateways are marked so that the output routines know to address the
 * gateway rather than the ultimate destination.
 */
#ifndef RNF_NORMAL
#include <net/radix.h>
#if 1 /*def RADIX_ART*/
#include <net/radix_art.h>
#endif
#endif
struct rtentry {
	struct	radix_node rt_nodes[2];	/* tree glue, and other values */
#define	rt_key(r)	((struct sockaddr *)((r)->rt_nodes->rn_key))
#define	rt_mask(r)	((struct sockaddr *)((r)->rt_nodes->rn_mask))
	struct	sockaddr *rt_gateway;	/* value */
	u_long	rt_flags;		/* up/down?, host/net */
	int	rt_refcnt;		/* # held references */
	u_quad_t rt_use;		/* raw # packets forwarded */
	struct	ifnet *rt_ifp;		/* the answer: interface to use */
	struct	ifaddr *rt_ifa;		/* the answer: interface to use */
	struct	sockaddr *rt_genmask;	/* for generation of cloned routes */
	caddr_t	rt_llinfo;		/* pointer to link level info cache */
	struct	rt_metrics rt_rmx;	/* metrics used by rx'ing protocols */
	struct	rtentry *rt_gwroute;	/* implied entry for gatewayed routes */
	struct	rt_linkhead rt_timer;	/* Timer information */

	struct	rtentry *rt_parent;	/* If cloned, parent of this route. */
	struct	socket *rt_tunsec;	/* If a tunnel route, point to */
					/* tunnel-security state info. */
};

/*
 * Following structure necessary for 4.3 compatibility;
 * We should eventually move it to a compat file.
 */
struct ortentry {
	u_int32_t rt_hash;		/* to speed lookups */
	struct	sockaddr rt_dst;	/* key */
	struct	sockaddr rt_gateway;	/* value */
	short	rt_flags;		/* up/down?, host/net */
	short	rt_refcnt;		/* # held references */
	u_int32_t rt_use;			/* raw # packets forwarded */
	struct	ifnet *rt_ifp;		/* the answer: interface to use */
};

#define	RTF_UP		0x1		/* route usable */
#define	RTF_GATEWAY	0x2		/* destination is a gateway */
#define	RTF_HOST	0x4		/* host entry (net otherwise) */
#define	RTF_REJECT	0x8		/* host or net unreachable */
#define	RTF_DYNAMIC	0x10		/* created dynamically (by redirect) */
#define	RTF_MODIFIED	0x20		/* modified dynamically (by redirect) */
#define RTF_DONE	0x40		/* message confirmed */
#define RTF_MASK	0x80		/* subnet mask present */
#define RTF_CLONING	0x100		/* generate new routes on use */
#define RTF_XRESOLVE	0x200		/* external daemon resolves name */
#define RTF_LLINFO	0x400		/* generated by ARP or ESIS */
#define RTF_STATIC	0x800		/* manually added */
#define RTF_BLACKHOLE	0x1000		/* just discard pkts (during updates) */
#define	RTF_CLONED	0x2000		/* this is a cloned route */
#define RTF_PROTO2	0x4000		/* protocol specific routing flag */
#define RTF_PROTO1	0x8000		/* protocol specific routing flag */

/*
 * New IPv6 routing flags.
 */
#define	RTF_DEFAULT	0x010000	/* Default route */
#define	RTF_ISAROUTER	0x020000	/* Neighbor is a router */
#define	RTF_TUNNEL	0x040000	/* Tunnelling bit */
/*
 * The following two bits are only useful when the TUNNEL bit is set.
 */
#define	RTF_CRYPT	0x080000	/* Encrypting tunnel bit. */
#define	RTF_AUTH	0x100000	/* Authenticating tunnel bit. */

/*
 * The following bits are used to short circuit lookups
 * The MCAST bit is currently unused.
 * The UCAST bit is only used in debug kernels to make sure that
 * the route as properly identified.
 */
#define	RTF_BCAST	0x01000000	/* route represents bcast address */
#define	RTF_MCAST	0x02000000	/* route represents mcast address */
#define	RTF_UCAST	0x04000000	/* route represents unicast address */

/*
 * Routing statistics.
 */
struct	rtstat {
	u_quad_t rts_badredirect;	/* bogus redirect calls */
	u_quad_t rts_dynamic;		/* routes created by redirects */
	u_quad_t rts_newgateway;	/* routes modified by redirects */
	u_quad_t rts_unreach;		/* lookups which failed */
	u_quad_t rts_wildcard;		/* lookups satisfied by a wildcard */
};
/*
 * Structures for routing messages.
 */
struct rt_msghdr {
	u_short	rtm_msglen;	/* to skip over non-understood messages */
	u_char	rtm_version;	/* future binary compatibility */
	u_char	rtm_type;	/* message type */
	u_short	rtm_index;	/* index for associated ifp */
	int	rtm_refcnt;	/* from rtentry */
	int	rtm_flags;	/* flags, incl. kern & message, e.g. DONE */
	int	rtm_addrs;	/* bitmask identifying sockaddrs in msg */
	pid_t	rtm_pid;	/* identify sender */
	int	rtm_seq;	/* for sender to identify action */
	int	rtm_errno;	/* why failed */
	u_quad_t rtm_use;	/* from rtentry */
	u_int32_t rtm_inits;	/* which metrics we are initializing */
	struct	rt_metrics rtm_rmx; /* metrics themselves */
};

#define RTM_VERSION	5	/* Up the ante and ignore older versions */

#define RTM_ADD		0x1	/* Add Route */
#define RTM_DELETE	0x2	/* Delete Route */
#define RTM_CHANGE	0x3	/* Change Metrics or flags */
#define RTM_GET		0x4	/* Report Metrics */
#define RTM_LOSING	0x5	/* Kernel Suspects Partitioning */
#define RTM_REDIRECT	0x6	/* Told to use different route */
#define RTM_MISS	0x7	/* Lookup failed on this address */
#define RTM_LOCK	0x8	/* fix specified metrics */
#define RTM_OLDADD	0x9	/* caused by SIOCADDRT */
#define RTM_OLDDEL	0xa	/* caused by SIOCDELRT */
#define RTM_RESOLVE	0xb	/* req to resolve dst to LL addr */
#define RTM_NEWADDR	0xc	/* address being added to iface */
#define RTM_DELADDR	0xd	/* address being removed from iface */
#define RTM_IFINFO	0xe	/* iface going up/down etc. */

#define RTV_MTU		0x1	/* init or lock _mtu */
#define RTV_HOPCOUNT	0x2	/* init or lock _hopcount */
#define RTV_EXPIRE	0x4	/* init or lock _hopcount */
#define RTV_RPIPE	0x8	/* init or lock _recvpipe */
#define RTV_SPIPE	0x10	/* init or lock _sendpipe */
#define RTV_SSTHRESH	0x20	/* init or lock _ssthresh */
#define RTV_RTT		0x40	/* init or lock _rtt */
#define RTV_RTTVAR	0x80	/* init or lock _rttvar */

/*
 * Bitmask values for rtm_addr.
 */
#define RTA_DST		0x1	/* destination sockaddr present */
#define RTA_GATEWAY	0x2	/* gateway sockaddr present */
#define RTA_NETMASK	0x4	/* netmask sockaddr present */
#define RTA_GENMASK	0x8	/* cloning mask sockaddr present */
#define RTA_IFP		0x10	/* interface name sockaddr present */
#define RTA_IFA		0x20	/* interface addr sockaddr present */
#define RTA_AUTHOR	0x40	/* sockaddr for author of redirect */
#define RTA_BRD		0x80	/* for NEWADDR, broadcast or p-p dest addr */

/*
 * Index offsets for sockaddr array for alternate internal encoding.
 */
#define RTAX_DST	0	/* destination sockaddr present */
#define RTAX_GATEWAY	1	/* gateway sockaddr present */
#define RTAX_NETMASK	2	/* netmask sockaddr present */
#define RTAX_GENMASK	3	/* cloning mask sockaddr present */
#define RTAX_IFP	4	/* interface name sockaddr present */
#define RTAX_IFA	5	/* interface addr sockaddr present */
#define RTAX_AUTHOR	6	/* sockaddr for author of redirect */
#define RTAX_BRD	7	/* for NEWADDR, broadcast or p-p dest addr */
#define RTAX_MAX	8	/* size of array to allocate */

struct rt_addrinfo {
	int	rti_addrs;
	struct	sockaddr *rti_info[RTAX_MAX];
	int	rti_flags;
	struct	ifaddr *rti_ifa;
	struct	ifnet *rti_ifp;
	struct	rt_msghdr *rti_rtm;
};

struct route_cb {
	int	ip_count;
	int	ns_count;
	int	iso_count;
	int	any_count;
};

#ifdef KERNEL
#define	RTFREE(rt) \
	if ((rt)->rt_refcnt <= 1) \
		rtfree(rt); \
	else \
		(rt)->rt_refcnt--;

/*
 * A macro to see if a timer is set on this route for a particular
 * function.  "exp" is filled in with when the timer is set to go
 * off, or zero if it doesn't exist.
 */
#define	RTT_FIND(rt, func, exp) { \
	struct rttimer *_rtt; \
	(exp) = 0; \
	for (_rtt = LIST_FIRST(&(rt)->rt_timer); \
	    _rtt != LIST_END(&(rt)->rt_timer); \
	    _rtt = LIST_NEXT(_rtt, rtt_link)) { \
		if (_rtt->rtt_func == (func)) { \
			(exp) = _rtt->rtt_expire; \
			break; \
		} \
	} \
}

struct	route_cb route_cb;
struct	rtstat	rtstat;
struct	radix_node_head *rt_tables[AF_MAX+1];

struct socket;

void	 route_init __P((void));
struct socket;
int	 route_output __P((struct mbuf *, struct socket *));
int	 route_usrreq __P((struct socket *,
	    int, struct mbuf *, struct mbuf *, struct mbuf *));
void	 rt_ifmsg __P((struct ifnet *));
void	 rt_maskedcopy __P((struct sockaddr *,
	    struct sockaddr *, struct sockaddr *));
void	 rt_missmsg __P((int, struct rt_addrinfo *, int, int));
void	 rt_newaddrmsg __P((int, struct ifaddr *, int, struct rtentry *));
int	 rt_setgate __P((struct rtentry *,
	    struct sockaddr *, struct sockaddr *));
void	 rt_setmetrics __P((u_int32_t, struct rt_metrics *, struct rtentry *));
void	 rtable_init __P((void **));
void	 rtalloc __P((struct route *));
void	 rtcalloc __P((struct route *));
struct rtentry *
	 rtalloc1 __P((struct sockaddr *, int));
void	 rtalloc_noclone __P((struct route *, int));

/*
 * Values for additional argument to rtalloc_noclone()
 */
#define	ALL_CLONING	0
#define	ONNET_CLONING	1
#define	NO_CLONING	2

void	 rtfree __P((struct rtentry *));
int	rt_getifa __P((struct rt_addrinfo *));
int	 rtinit __P((struct ifaddr *, int, int));
int	 rtioctl __P((u_long, caddr_t, struct proc *));
int	 rtredirect __P((struct sockaddr *, struct sockaddr *,
	    struct sockaddr *, int, struct sockaddr *, int));
int	 rtrequest __P((int, struct sockaddr *,
	    struct sockaddr *, struct sockaddr *, int, struct rtentry **));
int	 rtrequest1 __P((int, struct rt_addrinfo *, struct rtentry **));
int	rtdeletemsg __P((struct rtentry *));
void	rt_timer_init __P((void));

void	rt_timer_remove_all __P((struct rtentry *));
int	rt_timer_add __P((struct rtentry *, time_t,
	    void(*)(struct rtentry *, struct rttimer *), struct rttimer *));
void	rt_timer_timer __P((void *));
int	rt_rmclone __P((struct radix_node *, struct rtentry *));
#endif

#endif /* !_ROUTE_H_ */
