/*
 * Copyright (c) 1991, 1993, 1997 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI text.h,v 2.8.10.2 2001/09/10 11:46:23 geertj Exp
 */

#ifndef _SYS_TEXT_H_
#define _SYS_TEXT_H_

/*
 * Information about an executable file.
 * Some fields are used only during an exec
 * (x_flags, x_path, x_strings, x_string_size, x_veclen, x_argc, x_args).
 * XXX Text caching is not implemented yet.
 * The 'x_*' member name template comes from the (free) Reno text.h.
 */
struct text {
	int	x_flags;		/* X_PATH_SYSSPACE */
	time_t	x_timestamp;		/* used for cache invalidation */
	char	*x_path;		/* pathname of exec file */
	struct	vnode *x_vnode;		/* vnode of exec file */
	struct	vattr x_vattr;		/* attributes of exec file */
#define	x_size	x_vattr.va_size
	char	*x_header;		/* first page of exec file */
	char	*x_strings;		/* copies of arguments, environment */
	int	x_string_size;		/* space used by strings */
	u_long	x_entry;		/* initial pc */
	u_long	x_save_entry;		/* hack for emulators */
	uid_t	x_uid;			/* new uid if file is setuid */
	gid_t	x_gid;			/* new gid if file is setgid */
	struct	proc *x_proc;		/* cached proc pointer for VOP_*() */
	int	x_veclen;		/* # of pointers in argv+envp */
	int	x_argc;			/* argc to pass to user code */
	TAILQ_HEAD(x_arghead, exec_arg) x_args;
	TAILQ_HEAD(x_comhead, exec_load) x_commands;
	void	*x_aux;			/* auxiliary stack data */
	int	x_aux_len;		/* bytes of auxiliary data */
	/* vmspace parameters */
	segsz_t x_tsize;
	segsz_t x_dsize;
	segsz_t x_ssize;
	caddr_t	x_taddr;
	caddr_t	x_daddr;
	caddr_t x_saddr;
};

#define	X_PATH_SYSSPACE	0x01	/* x_path is system space pointer, not user */
#define	X_ENTRY		0x02	/* x_entry is valid */
#define	X_SET_UID	0x04	/* x_uid is valid */
#define	X_SET_GID	0x08	/* x_gid is valid */
#define	X_MACHDEP1	0x10	/* for exec_machdep.c */
#define	X_LIBRARY	0x20	/* binary is a shared library */
#define	X_MACHDEP2	0x40	/* for exec_machdep.c */

/*
 * A chain of exec_arg structures is used to build
 * the argument and environment arrays.
 * The linked list form makes it simple for exec
 * to allocate a variable number of arguments,
 * and for exec handlers (e.g. exec_interpreter)
 * to manipulate the argument list.
 */
struct exec_arg {
	TAILQ_ENTRY(exec_arg) ea_list;
	char	*ea_string;
};

/*
 * A chain of exec_load structures is used to build
 * a process's address space after an exec.
 * If the data is mapped from a vnode,
 * we keep a reference to the vnode.
 * The vnode here is not necessarily the one
 * in the associated text structure.
 */
enum exec_op { EXEC_ZERO, EXEC_MAP, EXEC_READ, EXEC_CLEAR };

struct exec_load {
	TAILQ_ENTRY(exec_load) el_list;
	enum	exec_op el_op;		/* the actual command */
	struct	vnode *el_vnode;	/* where to get data */
	off_t	el_offset;		/* offset of the data in the vnode */
	vm_offset_t	el_address;	/* address of segment to be loaded */
	vm_size_t	el_length;	/* length of segment */
	vm_prot_t	el_prot;	/* protection of segment */
	int	el_attr;		/* shared, COW, private */
};

#ifdef KERNEL
/*
 * Function declarations.
 */

/* Exists on every architecture.  */
int exec_interpreter __P((struct text *));

/* For AOUT: */
int exec_compact_demand_load_binary __P((struct text *));
int exec_demand_load_binary __P((struct text *));
int exec_shared_binary __P((struct text *));
int exec_unshared_binary __P((struct text *));

/* For COFF (only on i386): */
int exec_aout_shlib __P((struct text *));
int exec_coff_binary __P((struct text *));
int load_emulator __P((struct text *));

/* For ELF: */
int exec_elf_binary __P((struct text *));

/* Routines to load segments: */
void load_clear __P((struct text *, vm_offset_t, vm_size_t));
void load_file __P((struct text *, off_t, vm_offset_t, vm_size_t, vm_prot_t));
void load_read __P((struct text *, off_t, vm_offset_t, vm_size_t, vm_prot_t));
void load_zero __P((struct text *, vm_offset_t, vm_size_t));
int exec_clear_segment __P((struct proc *, struct exec_load *));
int exec_map_segment __P((struct proc *, struct exec_load *));
int exec_read_segment __P((struct proc *, struct exec_load *));
int exec_zero_segment __P((struct proc *, struct exec_load *));

/* Machine-dependent routines: */
int analyze_exec_header __P((struct text *));
int exec_set_state __P((struct text *, register_t *));

/* Miscellaneous other routines: */
void delete_text __P((struct text *));
void exec_close_files __P((struct proc *));
int exec_gather_arguments __P((struct text *, char **, char **));
int exec_lookup __P((struct text *));
void exec_new_pcomm __P((struct proc *, char *));
void exec_set_entry __P((struct text *, u_long));
void exec_unshare __P((struct proc *));
#endif

#endif /* !_SYS_TEXT_H_ */
