/*	BSDI cd.c,v 2.4 1997/10/21 20:12:25 ewv Exp	*/

/*
 * Copyright (c) 1988 University of Utah.
 * Copyright (c) 1990, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * the Systems Programming Group of the University of Utah Computer
 * Science Department.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * from: Utah Hdr: cd.c 1.6 90/11/28
 *
 *	@(#)cd.c	8.2 (Berkeley) 11/16/93
 */

/*
 * "Concatenated" disk driver.
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/errno.h>
#include <sys/dkstat.h>
#include <sys/buf.h>
#include <sys/malloc.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/disklabel.h>
#ifdef COMPAT_NOLABEL
#include <sys/ioctl.h>
#include <sys/fcntl.h>
#endif
#include <machine/pcpu.h>

#include <dev/cdvar.h>

#ifdef DEBUG
int cddebug = 0x03;
#define CDB_FOLLOW	0x01
#define CDB_INIT	0x02
#define CDB_IO		0x04
#endif

#define	cdunit(x)	dv_unit(x)

struct cd_saveaddr {
	void		*cd_bp;		/* was in b_saveaddr */
	int		cd_unit;		/* was in b_pfcent */
	int		cd_component;		/* was in b_pfcent */
};

#if 0
#define	getcbuf()	\
	((struct buf *)malloc(sizeof(struct buf), M_DEVBUF, M_WAITOK))
#define putcbuf(bp)	\
	free((caddr_t)(bp), M_DEVBUF)
#endif

struct cd_softc {
	int		 sc_flags;		/* flags */
	size_t		 sc_size;		/* size of cd */
	int		 sc_ileave;		/* interleave */
	int		 sc_ncdisks;		/* number of components */
	struct cdcinfo	 sc_cinfo[NCDISKS];	/* component info */
	struct cdiinfo	 *sc_itable;		/* interleave table */
	int		 sc_usecnt;		/* number of requests active */
	int		 sc_dk;			/* disk index */
};

/* sc_flags */
#define	CDF_ALIVE	0x01
#define CDF_INITED	0x02

void	cdattach __P((int));
int	cdinit __P((struct cddevice *));
char	*cddevtostr __P((dev_t));	/* XXX */
int	cdinterleave __P((struct cd_softc *));
#ifdef DEBUG
void	printiinfo __P((struct cdiinfo *));
#endif
int	cdopen __P((dev_t, int, int, struct proc *p));
void	cdstrategy __P((struct buf *));
void	cdstart __P((struct cd_softc *, struct buf *));
struct	buf *cdbuffer __P((struct cd_softc *, struct buf *, daddr_t,
			caddr_t, long));
void	cdintr __P((struct cd_softc *, struct buf *));
void	cdiodone __P((struct buf *));
int	cdread __P((dev_t, struct uio *, int));
int	cdwrite __P((dev_t, struct uio *, int));
int	cdioctl __P((dev_t, int, caddr_t, int, struct proc *));
int	cdsize __P((dev_t));

struct cfdriver cdcd =
#if 0
	{ NULL, "cd", NULL, cdattach, DV_DULL, sizeof(struct cd_softc) };
#else
	/* Pseudo-devices don't get their attach routines called this way! */
	{ NULL, "cd", NULL, NULL, DV_DULL, 0 };
#endif
struct devsw cdsw = {
	&cdcd,
	cdopen, nullclose, cdread, cdwrite, cdioctl, seltrue, nommap,
	cdstrategy, nodump, cdsize, 0,
	nostop
};

struct cd_softc *cd_softc;
int numcd;

/*
 * Since this is called after auto-configuration of devices,
 * we can handle the initialization here.
 *
 * XXX this will not work if you want to use a cd as your primary
 * swap device since swapconf() has been called before now.
 */
void
cdattach(num)
	int num;
{
	char *mem;
	register u_long size;
	register struct cddevice *cd;
	int dkn = 1; /* XXX */

	if (num <= 0)
		return;
	size = num * sizeof(struct cd_softc);
	mem = malloc(size, M_DEVBUF, M_NOWAIT);
	if (mem == NULL) {
		printf("WARNING: no memory for concatonated disks\n");
		return;
	}
	bzero(mem, size);
	cd_softc = (struct cd_softc *)mem;
	numcd = num;
	for (cd = cddevice; cd->cd_unit >= 0; cd++) {
		/*
		 * XXX
		 * Assign disk index first so that init routine
		 * can use it (saves having the driver drag around
		 * the cddevice pointer just to set up the dk_*
		 * info in the open routine).
		 */
		if (dkn < DK_NDRIVE)
			cd->cd_dk = dkn++;
		else
			cd->cd_dk = -1;
		if (cdinit(cd))
			printf("%s%d configured\n", cdcd.cd_name, cd->cd_unit);
		else if (cd->cd_dk >= 0) {
			printf("%s%d configure failed\n", cdcd.cd_name, cd->cd_unit);
			cd->cd_dk = -1;
			dkn--;
		}
	}
}

int
cdinit(cd)
	struct cddevice *cd;
{
	register struct cd_softc *cs = &cd_softc[cd->cd_unit];
	register struct cdcinfo *ci;
	register size_t size;
	register int ix;
	size_t minsize;
	dev_t dev;
	struct devsw *bsw;
	int error;
	struct proc *p = PCPU(curproc); /* XXX */

#ifdef DEBUG
	if (cddebug & (CDB_FOLLOW|CDB_INIT))
		printf("cdinit: unit %d\n", cd->cd_unit);
#endif
	cs->sc_dk = cd->cd_dk;
	cs->sc_size = 0;
	cs->sc_ileave = cd->cd_interleave;
	cs->sc_ncdisks = 0;
	/*
	 * Verify that each component piece exists and record
	 * relevant information about it.
	 */
	minsize = 0;
	for (ix = 0; ix < NCDISKS; ix++) {
		if ((dev = cd->cd_dev[ix]) == NODEV)
			break;
		ci = &cs->sc_cinfo[ix];
		ci->ci_dev = dev;
		bsw = devsw[major(dev)];
		/*
		 * Open the partition
		 */
		if (bsw->d_open &&
		    (error = (*bsw->d_open)(dev, 0, S_IFBLK, p))) {
			printf("%s%d: component %s open failed, error = %d\n",
			       cdcd.cd_name, cd->cd_unit, cddevtostr(dev), error);
			return(0);
		}
		/*
		 * Calculate size (truncated to interleave boundary
		 * if necessary).
		 */
		if (bsw->d_psize) {
			size = (size_t) (*bsw->d_psize)(dev);
			if ((int)size < 0)
				size = 0;
		} else
			size = 0;
		if (cs->sc_ileave > 1)
			size -= size % cs->sc_ileave;
		if (size == 0) {
			printf("%s%d: not configured (component %s missing)\n",
			       cdcd.cd_name, cd->cd_unit, cddevtostr(dev));
			if (bsw->d_close &&
			    (error = (*bsw->d_close)(dev, 0, S_IFBLK, p))) {
				printf("%s%d: component %s close failed, error = %d\n",
				       cdcd.cd_name, cd->cd_unit,
				       cddevtostr(dev), error);
			}
			return(0);
		}
#ifdef COMPAT_NOLABEL
		/*
		 * XXX if this is a 'c' partition then we need to mark the
		 * label area writeable since there cannot be a label.
		 */
		if (dv_subunit(dev) == 2 && bsw->d_open) {
			int i, flag;

			if (devsw[major(dev)]->d_ioctl) {
				flag = 1;
				(void)(*devsw[major(dev)]->d_ioctl)(dev,
					DIOCWLABEL, (caddr_t)&flag, FWRITE, p);
			}
		}
#endif
		if (minsize == 0 || size < minsize)
			minsize = size;
		ci->ci_size = size;
		cs->sc_size += size;
		cs->sc_ncdisks++;
	}
	/*
	 * If uniform interleave is desired set all sizes to that of
	 * the smallest component.
	 */
	if (cd->cd_flags & CDF_UNIFORM) {
		for (ci = cs->sc_cinfo;
		     ci < &cs->sc_cinfo[cs->sc_ncdisks]; ci++)
			ci->ci_size = minsize;
		cs->sc_size = cs->sc_ncdisks * minsize;
	}
	/*
	 * Construct the interleave table
	 */
	if (!cdinterleave(cs))
		return(0);
	if (cd->cd_dk >= 0)
		dk_wpms[cd->cd_dk] = 32 * (60 * DEV_BSIZE / 2);	/* XXX */
	printf("%s%d: %d components ", cdcd.cd_name, cd->cd_unit, cs->sc_ncdisks);
	for (ix = 0; ix < cs->sc_ncdisks; ix++)
		printf("%c%s%c",
		       ix == 0 ? '(' : ' ',
		       cddevtostr(cs->sc_cinfo[ix].ci_dev),
		       ix == cs->sc_ncdisks - 1 ? ')' : ',');
	printf(", %d blocks ", cs->sc_size);
	if (cs->sc_ileave)
		printf("interleaved at %d blocks\n", cs->sc_ileave);
	else
		printf("concatenated\n");
	cs->sc_flags = CDF_ALIVE | CDF_INITED;
	return(1);
}

/*
 * XXX not really cd specific.
 * Could be called something like bdevtostr in machine/conf.c.
 */
char *
cddevtostr(dev)
	dev_t dev;
{
	static char dbuf[8];
	register int i = 0;

	switch (major(dev)) {
#ifdef hp300
	case 2:
		dbuf[i++] = 'r'; dbuf[i++] = 'd';
		break;
	case 4:
		dbuf[i++] = 's'; dbuf[i++] = 'd';
		break;
	case 5:
		dbuf[i++] = 'c'; dbuf[i++] = 'd';
		break;
	case 6:
		dbuf[i++] = 'v'; dbuf[i++] = 'n';
		break;
#endif
#ifdef i386
	case 3:
		dbuf[i++] = 'w'; dbuf[i++] = 'd';
		break;
	case 9:
		dbuf[i++] = 'f'; dbuf[i++] = 'd';
		break;
	case 18:
		dbuf[i++] = 's'; dbuf[i++] = 'd';
		break;
	case 23:
		dbuf[i++] = 'm'; dbuf[i++] = 'c'; dbuf[i++] = 'd';
		break;
	case 29:
		dbuf[i++] = 'c'; dbuf[i++] = 'c'; dbuf[i++] = 'd';
		break;
#endif
	default:
		dbuf[i++] = dbuf[i++] = '?';
		break;
	}
	dbuf[i++] = dv_unit(dev) + '0';
	dbuf[i++] = dv_subunit(dev) + 'a';
	dbuf[i] = '\0';
	return (dbuf);
}

int
cdinterleave(cs)
	register struct cd_softc *cs;
{
	register struct cdcinfo *ci, *smallci;
	register struct cdiinfo *ii;
	register daddr_t bn, lbn;
	register int ix;
	u_long size;

#ifdef DEBUG
	if (cddebug & CDB_INIT)
		printf("cdinterleave(%x): ileave %d\n", cs, cs->sc_ileave);
#endif
	/*
	 * Allocate an interleave table.
	 * Chances are this is too big, but we don't care.
	 */
	size = (cs->sc_ncdisks + 1) * sizeof(struct cdiinfo);
	cs->sc_itable = (struct cdiinfo *)malloc(size, M_DEVBUF, M_WAITOK);
	bzero((caddr_t)cs->sc_itable, size);
	/*
	 * Trivial case: no interleave (actually interleave of disk size).
	 * Each table entry represent a single component in its entirety.
	 */
	if (cs->sc_ileave == 0) {
		bn = 0;
		ii = cs->sc_itable;
		for (ix = 0; ix < cs->sc_ncdisks; ix++) {
			ii->ii_ndisk = 1;
			ii->ii_startblk = bn;
			ii->ii_startoff = 0;
			ii->ii_index[0] = ix;
			bn += cs->sc_cinfo[ix].ci_size;
			ii++;
		}
		ii->ii_ndisk = 0;
#ifdef DEBUG
		if (cddebug & CDB_INIT)
			printiinfo(cs->sc_itable);
#endif
		return(1);
	}
	/*
	 * The following isn't fast or pretty; it doesn't have to be.
	 */
	size = 0;
	bn = lbn = 0;
	for (ii = cs->sc_itable; ; ii++) {
		/*
		 * Locate the smallest of the remaining components
		 */
		smallci = NULL;
		for (ci = cs->sc_cinfo;
		     ci < &cs->sc_cinfo[cs->sc_ncdisks]; ci++)
			if (ci->ci_size > size &&
			    (smallci == NULL ||
			     ci->ci_size < smallci->ci_size))
				smallci = ci;
		/*
		 * Nobody left, all done
		 */
		if (smallci == NULL) {
			ii->ii_ndisk = 0;
			break;
		}
		/*
		 * Record starting logical block and component offset
		 */
		ii->ii_startblk = bn / cs->sc_ileave;
		ii->ii_startoff = lbn;
		/*
		 * Determine how many disks take part in this interleave
		 * and record their indices.
		 */
		ix = 0;
		for (ci = cs->sc_cinfo;
		     ci < &cs->sc_cinfo[cs->sc_ncdisks]; ci++)
			if (ci->ci_size >= smallci->ci_size)
				ii->ii_index[ix++] = ci - cs->sc_cinfo;
		ii->ii_ndisk = ix;
		bn += ix * (smallci->ci_size - size);
		lbn = smallci->ci_size / cs->sc_ileave;
		size = smallci->ci_size;
	}
#ifdef DEBUG
	if (cddebug & CDB_INIT)
		printiinfo(cs->sc_itable);
#endif
	return(1);
}

#ifdef DEBUG
void
printiinfo(ii)
	struct cdiinfo *ii;
{
	register int ix, i;

	for (ix = 0; ii->ii_ndisk; ix++, ii++) {
		printf(" itab[%d]: #dk %d sblk %d soff %d",
		       ix, ii->ii_ndisk, ii->ii_startblk, ii->ii_startoff);
		for (i = 0; i < ii->ii_ndisk; i++)
			printf(" %d", ii->ii_index[i]);
		printf("\n");
	}
}
#endif

int
cdopen(dev, flags, ifmt, p)
	dev_t dev;
	int flags, ifmt;
	struct proc *p;
{
	int unit = cdunit(dev);
	register struct cd_softc *cs = &cd_softc[unit];

#ifdef DEBUG
	if (cddebug & CDB_FOLLOW)
		printf("cdopen(%x, %x)\n", dev, flags);
#endif

	if (unit >= numcd || (cs->sc_flags & CDF_ALIVE) == 0)
		return(ENXIO);
	return(0);
}

void
cdstrategy(bp)
	register struct buf *bp;
{
	register int unit = cdunit(bp->b_dev);
	register struct cd_softc *cs = &cd_softc[unit];
	register daddr_t bn;
	register int sz, s;

#ifdef DEBUG
	if (cddebug & CDB_FOLLOW)
		printf("cdstrategy(%x): unit %d\n", bp, unit);
#endif
	if ((cs->sc_flags & CDF_INITED) == 0) {
		bp->b_error = ENXIO;
		bp->b_flags |= B_ERROR;
		goto done;
	}
	bn = bp->b_blkno;
	sz = howmany(bp->b_bcount, DEV_BSIZE);
	if (bn < 0 || bn + sz > cs->sc_size) {
		sz = cs->sc_size - bn;
		if (sz == 0) {
			bp->b_resid = bp->b_bcount;
			goto done;
		}
		if (sz < 0) {
			bp->b_error = EINVAL;
			bp->b_flags |= B_ERROR;
			goto done;
		}
		bp->b_bcount = dbtob(sz);
	}
	bp->b_resid = bp->b_bcount;
	/*
	 * "Start" the unit.
	 */
	s = splbio();
	cdstart(cs, bp);
	splx(s);
	return;
done:
	biodone(bp);
}

void
cdstart(cs, bp)
	register struct cd_softc *cs;
	register struct buf *bp;
{
	register long bcount, rcount;
	struct buf *cbp;
	caddr_t addr;
	daddr_t bn;

#ifdef DEBUG
	if (cddebug & CDB_FOLLOW)
		printf("cdstart(%x, %x)\n", cs, bp);
#endif
	/*
	 * Instumentation (not real meaningful)
	 */
	cs->sc_usecnt++;
	if (cs->sc_dk >= 0) {
		dk_busy |= 1 << cs->sc_dk;
		dk_xfer[cs->sc_dk]++;
		dk_wds[cs->sc_dk] += bp->b_bcount >> 6;
	}
	/*
	 * Allocate component buffers and fire off the requests
	 */
	bn = bp->b_blkno;
	addr = bp->b_data;
	for (bcount = bp->b_bcount; bcount > 0; bcount -= rcount) {
		cbp = cdbuffer(cs, bp, bn, addr, bcount);
		rcount = cbp->b_bcount;
		(*devsw[major(cbp->b_dev)]->d_strategy)(cbp);
		bn += btodb(rcount);
		addr += rcount;
	}
}

/*
 * Build a component buffer header.
 */
struct buf *
cdbuffer(cs, bp, bn, addr, bcount)
	register struct cd_softc *cs;
	struct buf *bp;
	daddr_t bn;
	caddr_t addr;
	long bcount;
{
	register struct cdcinfo *ci;
	register struct buf *cbp;
	register daddr_t cbn, cboff;

#ifdef DEBUG
	if (cddebug & CDB_IO)
		printf("cdbuffer(%x, %x, %d, %x, %d)\n",
		       cs, bp, bn, addr, bcount);
#endif
	/*
	 * Determine which component bn falls in.
	 */
	cbn = bn;
	cboff = 0;
	/*
	 * Serially concatenated
	 */
	if (cs->sc_ileave == 0) {
		register daddr_t sblk;

		sblk = 0;
		for (ci = cs->sc_cinfo; cbn >= sblk + ci->ci_size; ci++)
			sblk += ci->ci_size;
		cbn -= sblk;
	}
	/*
	 * Interleaved
	 */
	else {
		register struct cdiinfo *ii;
		int cdisk, off;

		cboff = cbn % cs->sc_ileave;
		cbn /= cs->sc_ileave;
		for (ii = cs->sc_itable; ii->ii_ndisk; ii++)
			if (ii->ii_startblk > cbn)
				break;
		ii--;
		off = cbn - ii->ii_startblk;
		if (ii->ii_ndisk == 1) {
			cdisk = ii->ii_index[0];
			cbn = ii->ii_startoff + off;
		} else {
			cdisk = ii->ii_index[off % ii->ii_ndisk];
			cbn = ii->ii_startoff + off / ii->ii_ndisk;
		}
		cbn *= cs->sc_ileave;
		ci = &cs->sc_cinfo[cdisk];
	}
	/*
	 * Fill in the component buf structure.
	 */
#if 0
	cbp = getcbuf();
#endif
	cbp = (struct buf *)malloc(sizeof(struct buf), M_DEVBUF, M_WAITOK);
	cbp->b_flags = bp->b_flags | B_CALL;
	cbp->b_iodone = cdiodone;
	cbp->b_proc = bp->b_proc;
	cbp->b_dev = ci->ci_dev;
	cbp->b_blkno = cbn + cboff;
	cbp->b_data = addr;
	cbp->b_vp = 0;
	if (cs->sc_ileave == 0)
		cbp->b_bcount = dbtob(ci->ci_size - cbn);
	else
		cbp->b_bcount = dbtob(cs->sc_ileave - cboff);
	if (cbp->b_bcount > bcount)
		cbp->b_bcount = bcount;
	/*
	 * BSD/OS 2.0 things will look at b_iocount.  We'll just set
	 * it to b_bcount for now, until we do anything useful with
	 * chaining...
	 */
	cbp->b_iocount = cbp->b_bcount;
	cbp->b_chain = NULL;
	/*
	 * XXX context for cdiodone
	 */
	cbp->b_saveaddr = (struct cd_saveaddr *)malloc(sizeof(struct cd_saveaddr), M_DEVBUF, M_WAITOK);
	((struct cd_saveaddr *)cbp->b_saveaddr)->cd_bp = (caddr_t)bp;
	((struct cd_saveaddr *)cbp->b_saveaddr)->cd_unit = cs - cd_softc;
	((struct cd_saveaddr *)cbp->b_saveaddr)->cd_component = ci - cs->sc_cinfo;
#if 0
	cbp->saveaddr = (caddr_t)bp;
	cbp->b_pfcent = ((cs - cd_softc) << 16) | (ci - cs->sc_cinfo);
#endif
#ifdef DEBUG
	if (cddebug & CDB_IO)
		printf(" dev %x(u%d): cbp %x bn %d addr %x bcnt %d\n",
		       ci->ci_dev, ci-cs->sc_cinfo, cbp, cbp->b_blkno,
		       cbp->b_data, cbp->b_bcount);
#endif
	return(cbp);
}

void
cdintr(cs, bp)
	register struct cd_softc *cs;
	register struct buf *bp;
{

#ifdef DEBUG
	if (cddebug & CDB_FOLLOW)
		printf("cdintr(%x, %x)\n", cs, bp);
#endif
	/*
	 * Request is done for better or worse, wakeup the top half.
	 */
	if (--cs->sc_usecnt == 0 && cs->sc_dk >= 0)
		dk_busy &= ~(1 << cs->sc_dk);
	if (bp->b_flags & B_ERROR)
		bp->b_resid = bp->b_bcount;
	biodone(bp);
}

/*
 * Called by biodone at interrupt time.
 * Mark the component as done and if all components are done,
 * take a cd interrupt.
 */
void
cdiodone(cbp)
	register struct buf *cbp;
{
	register struct buf *bp = (struct buf *)((struct cd_saveaddr *)cbp->b_saveaddr)->cd_bp;/* XXX */
	register int unit = ((struct cd_saveaddr *)cbp->b_saveaddr)->cd_unit;
	int count, s;

	s = splbio();
#ifdef DEBUG
	if (cddebug & CDB_FOLLOW)
		printf("cdiodone(%x)\n", cbp);
	if (cddebug & CDB_IO) {
		printf("cdiodone: bp %x bcount %d resid %d\n",
		       bp, bp->b_bcount, bp->b_resid);
		printf(" dev %x(u%d), cbp %x bn %d addr %x bcnt %d\n",
		       cbp->b_dev,
		       ((struct cd_saveaddr *)cbp->b_saveaddr)->cd_component,
		       cbp, cbp->b_blkno, cbp->b_data, cbp->b_bcount);
	}
#endif

	if (cbp->b_flags & B_ERROR) {
		bp->b_flags |= B_ERROR;
		bp->b_error = biowait(cbp);
#ifdef DEBUG
		printf("%s%d: error %d on component %d\n", cdcd.cd_name,
		       unit, bp->b_error,
		       ((struct cd_saveaddr *)cbp->b_saveaddr)->cd_component);
#endif
	}
	count = cbp->b_bcount;
#if 0
	putcbuf(cbp);
#endif
	/* Now, free stuff... */
	free((caddr_t)cbp->b_saveaddr, M_DEVBUF);
	free((caddr_t)cbp, M_DEVBUF);

	/*
	 * If all done, "interrupt".
	 */
	bp->b_resid -= count;
	if (bp->b_resid < 0)
		panic("cdiodone: count");
	if (bp->b_resid == 0)
		cdintr(&cd_softc[unit], bp);
	splx(s);
}

int
cdread(dev, uio, flags)
	dev_t dev;
	struct uio *uio;
	int flags;
{
	register int unit = cdunit(dev);

#ifdef DEBUG
	if (cddebug & CDB_FOLLOW)
		printf("cdread(%x, %x)\n", dev, uio);
#endif
	return(physio(cdstrategy, NULL, dev, B_READ, minphys, uio));
}

int
cdwrite(dev, uio, flags)
	dev_t dev;
	struct uio *uio;
	int flags;
{
	register int unit = cdunit(dev);

#ifdef DEBUG
	if (cddebug & CDB_FOLLOW)
		printf("cdwrite(%x, %x)\n", dev, uio);
#endif
	return(physio(cdstrategy, NULL, dev, B_WRITE, minphys, uio));
}

int
cdioctl(dev, cmd, data, flag, p)
	dev_t dev;
	int cmd;
	caddr_t data;
	int flag;
	struct proc *p;
{
	int unit = cdunit(dev);
	register struct cd_softc *cs;
	struct disklabel *label;
	struct partinfo *part;
	size_t size;
	int error;

#ifdef DEBUG
	if (cddebug & CDB_FOLLOW)
		printf("cdioctl(%x, %x, %x, %x, %x): unit %d\n",
		       dev, cmd, data, flag, p, unit);
#endif
	error = suser(p->p_ucred, &p->p_acflag);
	if (error)
		return (error);
	if (unit >= numcd)
		return (ENXIO);

	cs = &cd_softc[unit];

	switch (cmd) {
		case DIOCGDINFO:	/* Get disk info (label) */
			/* Make sure the device has been inited already */
			if ((cs->sc_flags & CDF_INITED) == 0)
				return(ENXIO);
			label = (struct disklabel *)data;
			/* Round the cd's size down to an 8k boundary */
			size = cs->sc_size & ~0x1fff;
			/* Set up the bogus label... */
			label->d_magic = label->d_magic2 = DISKMAGIC;
			label->d_type = DTYPE_SCSI;
			bcopy("concatenated", label->d_typename, 13);
			label->d_flags = D_RAMDISK | D_SOFT;
			/* Bogus size atributes */
			label->d_secsize = DEV_BSIZE;
			label->d_nsectors = 256;
			label->d_ntracks = 16;
			label->d_ncylinders = size / (label->d_nsectors * label->d_ntracks);
			label->d_secpercyl = label->d_nsectors * label->d_ntracks;
			label->d_secperunit = size /*label->d_ncylinders * label->d_secpercyl*/;
			/* Bogus hardware characteristics */
			label->d_rpm = 5400;
			/* Partition info */
			label->d_npartitions = 8;
			label->d_partitions[2].p_size = size;
			label->d_partitions[2].p_offset = 0;
			label->d_partitions[2].p_fsize = 1024;
			label->d_partitions[2].p_fstype = FS_BSDFFS;
			label->d_partitions[2].p_frag = 8;
			label->d_partitions[2].p_cpg = 16;
			break;

#if 0
		case DIOCGPART:		/* Get partition info */
			break;
#endif

		case DIOCSDINFO:	/* Set disk info */
		case DIOCWDINFO:	/* Set disk info, and update disk */
			/* Write a disk label?  Hah! */
			break;

		default:
#ifdef DEBUG
			printf("cdioctl(%x, %x, %x, %x, %x): unit %d\n",
			       dev, cmd, data, flag, p, unit);
#endif
			return(ENOTTY);
	}

	return(0);
}

int
cdsize(dev)
	dev_t dev;
{
	int unit = cdunit(dev);
	register struct cd_softc *cs = &cd_softc[unit];

	if (unit >= numcd || (cs->sc_flags & CDF_INITED) == 0)
		return(-1);
	return(cs->sc_size);
}
