/*-
 * Copyright (c) 1997 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI mutex.h,v 2.9 2000/04/28 03:00:47 donn Exp
 */

#ifndef _MACHINE_MUTEX_H_
#define _MACHINE_MUTEX_H_

/*
 * This is the machine-dependent mutex implementation for i386
 * machines.
 */
#define CACHE_LINE_SIZE	32		/* size of x86 cache line (bytes) */

#define	NO_CPU	0xff

#ifndef LOCORE

typedef struct mutex mutex_t;

/*
 * When lock is free m_gate == 0, m_count == 0, and m_owner == NO_CPU
 * When lock is held, the hold count is m_count, the owner is the cpu ID
 *   of the owning CPU, and m_gate == 1.
 */
struct mutex {
	volatile int m_gate;		/* the lock itself */
	volatile int m_owner;		/* cpuno of lock owner */
	int	 m_count;		/* recursion count */
	int	 m_spare;
#ifdef SMP_DEBUG
	char	*m_name;		/* identification */
	int	m_spincnt;
#endif
};

#ifdef KERNEL

/*
 * Locks with a leading _ should be manipulated via XXX_ENTER/XXX_EXIT
 * macros, not directly!
 */
extern mutex_t _klock;	/* The giant lock */
extern mutex_t slock;	/* The switch lock */

/* Special lock flags */
#define	KLOCK	1

extern inline int
get_gate(volatile int *p)
{
	int rc;
	
	asm volatile ("movl $1,%%eax; xchgl %1,%%eax"
	    : "=a" (rc), "=m" (*p));
	return(rc);
}

#define M_SPIN	1
#define M_TRY	0

#define	mutex_enter(m, s)	\
    _mutex_enter(m, s, 0, __WHERE__)


/*
 * This code looks a bit odd in C; it is arranged to inspire the
 * compiler to generate decent output.
 *
 * m points to the mutex
 * spin == M_SPIN to wait for lock forever
 *	   M_TRY to try for lock, returns 1 if we got it, 0 otherwise
 * special is used to pass in machine dependent constants that cause
 *	special behavior for a particular lock; currently this is used to
 *	set the TPR register when we first get the lock and when we release it.
 *
 * Note: the ordering of owner/count operations is critical: the count must
 *	always be correct when the owner is changed from NO_CPU (this is
 *	to avoid races when interrupts occur right after getting the gate).
 */
extern inline  int
_mutex_enter(mutex_t *m, const int spin, const int special,
    char * const name)
{
	register int cpuno = PCPU(cpuno);
	int owner = m->m_owner;
	int rc;

	if (owner == cpuno) {
		m->m_count++;
#ifdef SMP_DEBUG
		m->m_name = name;
#endif
		return (1);
	}
	if (spin == M_SPIN) {
		for (;;
#ifdef SMP_DEBUG
		    m->m_spincnt++
#endif
		     ) {
			if (m->m_owner != NO_CPU)
				continue;
			asm volatile (
			"	movl	$1,%%eax;"
			"	pushfl;"
			"	cli;"
			"	xchgl	%0,%%eax;"
			"	testl	%%eax,%%eax;"
			"	jnz	1f;"
			"	movl	%2,4+%0;"
			"	incl	8+%0;"
			"1:	popfl;"
				: "=m" (*m), "=a" (rc): "d" (cpuno) : "cc");
			if (rc == 1)
				continue;
#ifdef SMP_DEBUG
			m->m_name = name;
#endif
			if (special == KLOCK)
				apic.la_tpr.r = TPR_KERNEL;
			return (1);
		}
	}
	if (owner != NO_CPU)
		return (0);
	asm volatile (
	"	movl	$1,%%eax;"
	"	pushfl;"
	"	cli;"
	"	xchgl	%0,%%eax;"
	"	testl	%%eax,%%eax;"
	"	jnz	1f;"
	"	movl	%2,4+%0;"
	"	incl	8+%0;"
	"1:	popfl;"
		: "=m" (*m), "=a" (rc): "d" (cpuno) : "cc");
	if (rc == 1)
		return (0);
#ifdef SMP_DEBUG
	m->m_name = name;
#endif
	if (special == KLOCK)
		apic.la_tpr.r = TPR_KERNEL;
	return (1);
}

#define mutex_exit(m)	_mutex_exit(m, 0, __WHERE__)
/*
 * Notes:
 *       - ordering of owner/count ops is critical!
 *       - the 'lock' prefix on the decl is to force the previous write out
 *         of the write posting buffer (0xff -> m_owner), it has nothing
 *	   to do with the decl itself. This is cheaper than a 'cpuid', which
 *	   trashes ebx and makes the compiler spill it to the stack.
 */
#ifdef SMP_DEBUG
#define _mutex_exit(m, special, where)					\
do {									\
    if ((m)->m_count < 1 || (m)->m_owner == NO_CPU)			\
	    panic("mutex_exit: releasing free mutex %s", where);	\
    if ((m)->m_owner != PCPU(cpuno))					\
	    panic("mutex_exit: releasing someone else's mutex %s",	\
		(where));  						\
    if ((m)->m_count == 1) {						\
	    asm volatile(						\
	    "pushfl;"							\
	    "cli;"							\
	    "movl	$0xff,4+%0;"					\
	    "lock; decl	8+%0;"						\
	    "movl	$0,%0;"						\
	    "popfl;": "=m" (*(m)): : "cc");				\
	    if ((special) == KLOCK)					\
		    apic.la_tpr.r = PCPU(pc_user_tpr);			\
    } else								\
	    (m)->m_count--;						\
} while (0)

#else

#define _mutex_exit(m, special, where)					\
do {									\
    if ((m)->m_count == 1) {						\
	    asm volatile(						\
	    "pushfl;"							\
	    "cli;"							\
	    "movl	$0xff,4+%0;"					\
	    "lock; decl	8+%0;"						\
	    "movl	$0,%0;"						\
	    "popfl;": "=m" (*(m)): : "cc");				\
	    if ((special) == KLOCK)					\
		    apic.la_tpr.r = PCPU(pc_user_tpr);			\
    } else								\
	    (m)->m_count--;						\
} while (0)

#endif

/* Locks with special handling */
#define	KLOCK_ENTER(s)		\
    _mutex_enter(&_klock, s, KLOCK, __WHERE__)
#define	KLOCK_EXIT		\
    _mutex_exit(&_klock, KLOCK, __WHERE__)

extern inline int
mutex_own(mutex_t *m)
{
	if (m->m_owner == PCPU(cpuno))
		return (1);
	return (0);
}

/* Set up a mutex, not locked */
extern inline void
mutex_init(mutex_t *m)
{
	m->m_gate = 0;
	m->m_owner = NO_CPU;
	m->m_count = 0;
#ifdef SMP_DEBUG
	m->m_name = NULL;
	m->m_spincnt = 0;
#endif
}

#define M_NOTOWNER	0x01	/* Current cpu shouldn't own it */
#define M_NOVALID	0x02	/* Don't check hold count */
#define M_OWNER		0x04	/* Current cpu should own it */
#define M_NORECURSION	0x08	/* Hold count should be < 1 */

#ifdef SMP_DEBUG
#define mutex_assert(m, what)						\
do {									\
	if (((what) & M_NOVALID) == 0)					\
		if ((m)->m_count < 0)					\
			panic("mutex_assert: negative hold count %s:%d", \
			    __FILE__, __LINE__);			\
	if ((what) & M_NOTOWNER)					\
		if ((m)->m_owner == PCPU(cpuno))			\
			panic("mutex_assert: unexpectedly owned %s:%d", \
			    __FILE__, __LINE__);			\
	if ((what) & M_OWNER)						\
		if ((m)->m_owner != PCPU(cpuno))			\
			panic("mutex_assert: not owned %s:%d",		\
			    __FILE__, __LINE__);			\
	if ((what) & M_NORECURSION)					\
		if ((m)->m_count != 1)					\
			panic("mutex_assert: unexpected recursion %s:%d", \
			    __FILE__, __LINE__);			\
} while (0)
#else
#define mutex_assert(m, what)
#endif

#endif /* KERNEL */

#else

#ifdef SMP_DEBUG
#define	MUTEX_ENTER_SETDESC(desc)					\
	.section ".rodata";						\
6:	.string	desc;							\
	.section ".text";						\
	movl	$6b,MUTEX_NAME(%ebx);

#define	SMUTEX_SETDESC(m,desc)						\
	.section ".rodata";						\
6:	.string	desc;							\
	.section ".text";						\
	movl	$6b,m+MUTEX_NAME;
#else
#define	MUTEX_ENTER_SETDESC(desc)
#define	SMUTEX_SETDESC(m,desc)
#endif

#define	SMP_LOCK	lock;

/*
 * l	- Lock
 * fail	- label to branch to if lock failed
 * desc	- Ascii string describing where we got the lock
 * take	- Code run when the lock is acquired from the unlocked state
 */
#define	MUTEX_ENTER(l, fail, desc, take)				\
	/* See if we already have the lock */				\
	movl	$l,%ebx;						\
	movl	%fs:PCPU_CPUNO,%eax;	/* Get our CPU number */	\
	cmpl	%eax,MUTEX_OWN(%ebx);	/* See if its us */		\
	je	8f;			/* Already have it... ref */    \
									\
	/* Try to get the gate */					\
	movl	$1,%ecx;						\
	xchgl	%ecx,MUTEX_GATE(%ebx);					\
	testl	%ecx,%ecx;						\
	jne	fail;							\
	take;								\
8:	incl	MUTEX_COUNT(%ebx);	/* Increment recursion count */	\
	MUTEX_ENTER_SETDESC(desc);					\
	movl	%eax,MUTEX_OWN(%ebx);	/* Say its us */

/*
 * l	- Lock
 * give	- Code run when lock is completely released
 *
 * XXX It would be nice to do the decl last, however this is outside the
 *     safety of m_gate. We must decrement the count after setting the
 *     ownership to NO_CPU to prevent races against interrupt code in the
 *     same CPU.
 *
 * As above, the lock prefix on the MUTEX_COUNT() decrement is to force the
 * NO_CPU write to memory.
 */
#define	MUTEX_EXIT(l,give)						\
	movl	$l,%ebx;						\
	cmpl	$1,MUTEX_COUNT(%ebx);					\
	je	1f;							\
	decl	MUTEX_COUNT(%ebx);					\
	jmp	9f;							\
1:	movl	$NO_CPU,MUTEX_OWN(%ebx);				\
	lock; decl MUTEX_COUNT(%ebx);					\
	movl	$0,MUTEX_GATE(%ebx);					\
	give;								\
9:

/*
 * Very simple (non-recursive) spin locks
 *
 * Suggest using these with great care if interrupts are enabled
 */
#define	SMUTEX_ENTER(m, d, r)						\
	movl	$1,r;							\
9:	xchgl	m+MUTEX_GATE,r;						\
	testl	r,r;							\
	jnz	9b;							\
	SMUTEX_SETDESC(m, d);

#define SMUTEX_EXIT(m, d)						\
	SMUTEX_SETDESC(m, d);						\
	movl	$0,m+MUTEX_GATE;

#endif /* LOCORE */

#endif /* !_MACHINE_MUTEX_H_ */
