/*-
 * Copyright (c) 1992, 1993, 1997 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI vmlayout.h,v 2.5 1999/06/02 19:54:22 jch Exp
 */

/*
 * Virtual address space arrangement.  These definitions were previously
 * scattered among "machine/param.h", "machine/vmparam.h" and locore.s.
 *
 * On 386, both user and kernel share the address space,
 * similar to the VAX.  However, unlike the VAX, the division
 * of the address space into user and kernel areas is not fixed
 * by the hardware.
 *
 * USRTEXT is the start of the user text/data space, while USRSTACK
 * is the top (end) of the user stack. Immediately above the user stack
 * is a space (kstack) previously used for double-mapping the user structure
 * and kernel stack which is UPAGES long.  This space is currently unused.
 *
 * Immediately after the stack area is the page-table map (PTmap),
 * the kernel address space, and the alternate page-table map (APTmap).
 *
 * This layout places the per-process user area first, followed
 * by the per-process kernel areas (page-table pages for
 * per-process user and kernel areas), and then the global kernel
 * addresses.  This arrangement is reflected in the page tables
 * for each process, and that in turn is reflected in the layout
 * of the page table directory (PDR), which maps the page tables.
 *
 * The page table directory is used by the system as a page-table page
 * in order to map the page tables (including itself) into virtual memory.
 * The page tables are mapped into the address space between the
 * user and kernel areas so that the per-process kernel addresses
 * including the page table area follow the per-process user addresses,
 * and the global kernel page table area falls just below the global
 * kernel area.
 *
 * The layout is defined here in terms of indices in the page table
 * directory.  Each page table directory entry (PDE) maps a page of page
 * table space, which in turn maps a 4 MB address region.
 */

#ifndef _MACHINE_VMLAYOUT_H_
#define _MACHINE_VMLAYOUT_H_

/*
 * Layout of the PTD.  These constants are indices;
 * the PTD contains 1024 entries, 0x000 through 0x3ff.
 * Entries 0 through KSTKPTDI map the current user space
 * (previously including kstack).
 * Entry PTDPTDI points to the current PTD, and thus maps the current
 * page tables.  Entries KPTDI_FIRST through KPTDI_LAST map the kernel
 * page table pages allocated at startup, and used in all processes.
 * The last two entries (APTDPTDI and CPUPTDI) map temporary page tables
 * and per-CPU areas, respectively.
 *
 * The kernel page tables are sized at boot time.  They are never extended
 * in the current implementation; extension would require that the PTD
 * for each process be updated.
 */
#ifndef KERNEL_VIRTUAL_MB
#define KERNEL_VIRTUAL_MB	248	/* 256 - 8 */
#endif
#ifndef NBPDR				/* for locore */
#define NBPDR			(1024*NBPG)	/* bytes/page dir */
#define	PDROFSET		(NBPDR-1)	/* byte offset into page dir */
#endif
#define	__KERNEL_VIRTUAL	((0 - (KERNEL_VIRTUAL_MB + 8)*1024*1024) &~ PDROFSET)
#define	__KPTDI_FIRST		((__KERNEL_VIRTUAL/NBPDR & 0x3ff))	/* start index of kernel PDEs */

#define	__KSTKPTDI		(__KPTDI_FIRST - 2)    /* PTD index ending with kstack */
#define	__PTDPTDI		(__KPTDI_FIRST - 1)    /* PTD entry that points to PTD! */
#define	KPTDI_LAST		0x3fd		 /* index of last kernel PDE */
#define	APTDPTDI		0x3fe		/* PTD entry that points to APTD */
#define CPUPTDI			0x3ff		/* PTD entry for per-cpu state */
#define	NKPTD_DFLT		4		/* default num. of kernel PDEs to use */
#define	__NKPTD			(APTDPTDI - __KPTDI_FIRST)	/* max kernel PDEs */

/*
 * user/kernel map constants; derived from PTD layout.
 * _KSTACK ends at the end of the 4 MB area for KSTKPTDI,
 * and is UPAGES long; USRSTACK ends where _KSTACK begins.
 * (KSTACK should be removed, but cpu_fork does not yet relocate
 * the kernel stack when copying.)
 */
#define	USRTEXT			0
#define	__VM_MIN_ADDRESS	0
#define	__USRSTACK		(((__KSTKPTDI + 1) << PDRSHIFT) - UPAGES * NBPG)
#define	__VM_MAXUSER_ADDRESS	__USRSTACK

/*
 * The PTD, used as a page table page, maps the page table space
 * (PTmap[], set in locore to __PTMAP).  User page tables come first.
 * User page tables end just before the (recursively mapped) PTD.
 * Kernel page tables begin after the PTD.  Finally, there is
 * an alternate pagemap area APTmap[] at _APTMAP, used to map
 * the page tables of another process temporarily, followed
 * by the per-CPU area for MP.
 */
#define	__PTMAP			(__PTDPTDI << PDRSHIFT)
#define	UPT_MIN_ADDRESS		_PTMAP
#define	UPT_MAX_ADDRESS		((vm_offset_t) (_PTMAP + (PTDPTDI << PGSHIFT)))
#define	VM_MAX_ADDRESS		UPT_MAX_ADDRESS

#define	VM_MIN_KERNEL_ADDRESS	UPT_MAX_ADDRESS
#define	KPT_MIN_ADDRESS		((vm_offset_t) (_PTMAP + (KPTDI_FIRST << PGSHIFT)))

#define	__KERNBASE		(__KPTDI_FIRST << PDRSHIFT)	/* VA of kernel */
#define	VM_MAX_KERNEL_ADDRESS	((vm_offset_t) ((KPTDI_LAST + 1) << PDRSHIFT))
#define	__APTMAP		(APTDPTDI << PDRSHIFT)

#ifdef	LOCORE
/*
 * Assembly code accesses the loader symbols directly
 */
#define	USRSTACK		_USRSTACK
#define	KERNBASE		_KERNBASE
#define	KPTDI_FIRST		_KPTDI_FIRST
#define	PTDPTDI			_PTDPTDI
#define	VM_MIN_ADDRESS		_VM_MIN_ADDRESS
#define	VM_MAXUSER_ADDRESS	_VM_MAXUSER_ADDRESS

#else /* LOCORE */
#ifdef	KERNEL
/*
 * C code needs to use the address of the symbols.
 */
#define	USRSTACK	((vm_offset_t) &_USRSTACK)
extern char _USRSTACK;

#define	KERNBASE	((vm_offset_t) &_KERNBASE)
extern char _KERNBASE;

#define	VM_MIN_ADDRESS	((vm_offset_t) &_VM_MIN_ADDRESS)
extern char _VM_MIN_ADDRESS;

#define	VM_MAXUSER_ADDRESS ((vm_offset_t) &_VM_MAXUSER_ADDRESS)
extern char _VM_MAXUSER_ADDRESS;

#define	KPTDI_FIRST	((u_int) &_KPTDI_FIRST)
extern char _KPTDI_FIRST;

#define	NKPTD		((u_int) &_NKPTD)
extern char _NKPTD;

#define	PTDPTDI		((u_int) &_PTDPTDI)
extern char _PTDPTDI;

#define	_PTMAP			((vm_offset_t) &PTmap)

#endif	/* KERNEL */
#endif	/* LOCORE */

#ifndef	KERNEL
/*
 * The only symbols we export to user space are the nlist names of the
 * exported symbols.
 */
#define	NL_USRSTACK		"_USRSTACK"
#define	NL_KERNBASE		"_KERNBASE"
#define	NL_SYSTEM		"SYSTEM"
#define	NL_VM_MIN_ADDRESS	"_VM_MIN_ADDRESS"
#define	NL_VM_MAXUSER_ADDRESS	"_VM_MAXUSER_ADDRESS"

#endif	/* KERNEL */
#endif /* _MACHINE_VMLAYOUT_H_ */
