/*	BSDI aimctl.c,v 2.4 1998/06/03 19:14:53 karels Exp	*/

/*-
 *	Copyright (c) 1993, 1994 Robert J.Dunlop. All rights reserved.
 *
 *	Portions Copyright (c) 1991, 1992, 1993 Chase Research PLC.
 *	Portions Trade Secrets of Chase Research PLC.
 *	All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, is permitted provided the following conditions are met:
 *   1.	Redistributions in source form must retain the above copyright and trade
 *	secret notices, this list of conditions and the following disclaimers.
 *   2. Redistributions in binary form must reproduce all items from condition
 *	1 in the accompanying documentation and/or other materials.
 *   3. Neither the names of Chase Research PLC and it's subsidiary companies,
 *	nor that of Robert Dunlop may be used to endorse, advertise or promote
 *	products derived from this software without specific written permission
 *	from the party or parties concerned.
 *   4.	Adaptation of this software to operating systems other than BSD/OS
 *	requires the prior written permission of Chase Research PLC.
 *   5.	Adaptation of this software for operation with communications hardware
 *	other than the Chase IOPRO and IOLITE series requires the prior written
 *	permission of Chase Research PLC.
 */

/*
 *	Chase Research IOPRO control driver (aimc)
 *
 *	This driver is used to control the interface cards and external units
 *	it does not perform user data transfers or port operations.
 *
 *	aimctl.c,v 1.11 1995/07/28 08:59:33 rjd Exp
 */

/* Only include code if we are configured in */
#include "aimc.h"
#if NAIMC > 0

/* 1.0 and 1.1 system users may need to convert these <> back to "" */
#include <cdefs.h>
#include <types.h>
#include <errno.h>
#include <fcntl.h>
#include <device.h>
#include <uio.h>
#include "isavar.h"
#include "icu.h"
#include "isa.h"
#include <bitstring.h>
#include <param.h>
#include <systm.h>
#include <kernel.h>
#include <conf.h>

#include "aimreg.h"
#include "aimvar.h"
#include "aimioctl.h"


/*
 *	Minor device number breakdown
 *	=============================
 *
 *	Other Chase drivers use a single control port and multiplex all the
 *	control data through it. This requires a smart support daemon and lots
 *	of different ioctls. Here I have tried to use lots of simple support
 *	routines and a small ioctl set, card and external unit addressing is
 *	specified by the minor device number.
 */
#define UNIT_MASK	0x07		/* Up to 8 external units per card */
#define CARD_MASK	0x38		/* Up to 8 host cards */
#define CARD_SHIFT	3
#define TYPE_MASK	0xC0		/* Type of operation performed */
#define TYPE_RAW	0x00		/* Raw data xfer to unit (download) */
#define TYPE_CONFIG	0x40		/* Config string to/from unit */
#define TYPE_CARD	0x80		/* Per card interface */
#define TYPE_ALLCARD	0xC0		/* All cards interface */

/* Extract the three components from the minor device number */
/* XXX In several places we pass a minor device number into these macros. While
 *     they pass through the minor() macro unchanged at present this might not
 *     be a good idea for the future.
 */
#define MUNIT(M)	( minor(M) & UNIT_MASK )
#define MCARD(M)	(( minor(M) & CARD_MASK ) >> CARD_SHIFT )
#define	MTYPE(M)	( minor(M) & TYPE_MASK )


/*
 *	A very small file
 *	=================
 *
 *	The ALLCARD port is used to provide information covering all the cards.
 *	This could be done via special ioctls as in the "standard" driver but I
 *	have chosen to present a readonly file for the benefit of scripts etc.
 *	The first line of the file contains the number of interface boards
 *	detected, can't think of anything else we might need at present.
 */
static char aim_info_file[] = "0\n";

#define IFILE_NCARDS_OFFSET	0
#define IFILE_SIZE		( sizeof ( aim_info_file ) - 1 )


/*
 *	Host interface cards I/O registers
 *	==================================
 */
#define AIM_IOSIZE	16		/* 16 locations in all */

#define AIM_MBS1	0		/* Memory block select reg 1 (R/W) */
#define AIM_MBS2	1		/* Memory block select reg 2 (R/W) */
#define AIM_ISA_ID_OFF	2		/* ISA signature offset ( 4 bytes RO) */
#define AIM_EISA_ID_OFF	0x80		/* EISA ID offset ( 4 bytes RO) */
#define AIM_INTEN	6		/* Interrupt enable register (WO) */
#define AIM_INTTYPE	7		/* Edge/Level Trigger register (WO) */
#define AIM_INTSTAT	8		/* Interrupt status register (RO) */
#define AIM_MEMSEL	8		/* Memory select register (WO) */

#define IO_IS_EISA(I)	((I) > IO_ISAEND )


/*
 *	Interrupts
 *	==========
 */
			/* Valid card interrupts */
#define AIM_IRQS	(IRQ3|IRQ4|IRQ5|IRQ9|IRQ10|IRQ11|IRQ12|IRQ15)

			/* Card interrupt select code ( goes in low three bits
			 * of MBS1 ) indexed by physical line number.
			 * 0xFF indicates illegal selection
			 */
static Ubyte irq_sel[] = {
	0xFF, 0xFF, 0xFF, 0,
	1,    2,    0xFF, 0xFF,
	0xFF, 3,    4,    5,
	6,    0xFF, 0xFF, 7
};

			/* Reverse translation */
static u_short irq_from_sel[] = {
	 IRQ3, IRQ4, IRQ5, IRQ9, IRQ10, IRQ11, IRQ12, IRQ15
};

			/* Determine the line # given the request bit */
			/* XXX This IMHO should be in icu.h */
#define IL_FROM_IRQ(I)	( ffs((I)) - 1 )

				/* Returns from intr routine */
				/* XXX This IMHO should also be in icu.h */
#define INTR_NOT_I	0	/* Not for this device */
#define INTR_EXPECTED	1	/* Serviced and expected */
#define INTR_UNEXPECTED	(-1)	/* Serviced and unexpected */

#define WATCHDOG_RATE	( 2 * hz )	/* How fast the lost interrupt polling
					 * loop should run.
					 */

/*
 *	Debug level
 *	===========
 */
#ifdef AIM_DEBUG
    int aim_debug = AIM_DEBUG;
#endif


/*
 *	Card detection and configuration
 *	================================
 */

/*
 *	Check for an EISA/ISA ID given the base address
 */
static Bool
check_EISA_ID ( ioaddr )
	register u_short ioaddr;
{
	/* EISA ID is also used as a "signature" for the ISA cards */
	/* Differing offsets not strictly necessary as EISA cards also respond
	 * with signature at ISA offset.
	 */
	ioaddr += IO_IS_EISA ( ioaddr ) ? AIM_EISA_ID_OFF : AIM_ISA_ID_OFF;

	/* EISA ID encoding is a little strange the following corresponds with
	 * "CHA 02 01". CHA being Chase Research, 02 the product ID and 01 the
	 * version.
	 */
	return inb ( ioaddr ) == 0x0D && inb ( ioaddr + 1 ) == 0x01
	    && inb ( ioaddr + 2 ) == 0x02 && inb ( ioaddr + 3 ) == 0x01;
}

int
aimcprobe ( parent, cf, aux )
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	u_short irq;
	caddr_t	maddr;


	/* Check the card's ID register. */
#ifdef MCA_SUPPORT
	if ( check_MCA_ID ( ia->ia_iobase ))
	    Read MCA POS registers and xfer data to ISA/EISA register format.
#endif

	if ( ! check_EISA_ID ( ia->ia_iobase ))
	    return 0;

	/* Signature is correct. Allocate irq and memory space */

	/* If user config has specified an irq check it is legal */
	if ( ia->ia_irq != IRQUNK
			&& irq_sel[ IL_FROM_IRQ ( ia->ia_irq )] == 0xFF )
	{
	    printf ("aimc%d: OS irq illegal (reassigning)\n", cf->cf_unit );
	    ia->ia_irq = IRQUNK;
	}

	/* On EISA cards read back irq and memory base and check against OS */
	if ( IO_IS_EISA ( ia->ia_iobase ))
	{
	    irq = irq_from_sel[ inb ( ia->ia_iobase + AIM_MBS1 ) & 0x07 ];
	    maddr = (caddr_t)(( inb ( ia->ia_iobase + AIM_MBS2 ) << 8 )
			| ( inb ( ia->ia_iobase + AIM_MBS1 ) & 0xF8 ));
	    (int)maddr <<= 8;

	    /* If a conflict occurs I default to using the OS values due to the
	     * fact that most EISA configurators are a load of fetid dingo's
	     * kidneys.
	     */
	    if ( ia->ia_irq == IRQUNK )
		ia->ia_irq = irq;
	    else if ( ia->ia_irq != irq )
	    {
		printf ("aimc%d: EISA irq mismatch using OS value\n"
								, cf->cf_unit );
	    }
	    if ( ia->ia_maddr == NULL )
		ia->ia_maddr = maddr;
	    else if ( ia->ia_maddr != maddr )
	    {
		printf ("aimc%d: EISA maddr mismatch using OS value\n"
								, cf->cf_unit );
	    }
	}

	/* Insert conflict checks and/or search for free locations here */

	/* If irq not yet specified grab a free slot */
	/* Need to do more work to support interrupt sharing on EISA and MCA
	 * hosts including AT cards in EISA slots.
	 */
	if ( ia->ia_irq == IRQUNK )
	{
	    if (( ia->ia_irq = isa_irqalloc ( AIM_IRQS )) == 0 )
	    {
		/* Or perhaps we should switch to polling ? */
		printf ("aimc%d: No free IRQ\n", cf->cf_unit );
		return 0;
	    }
	}

	/* By this time we must have maddr specified either by the user, the
	 * EISA/MCA configurator or the OS.
	 */
	if ( ia->ia_maddr == NULL )
	{
	    printf ("aimc%d: maddr not set\n", cf->cf_unit );
	    return 0;
	}

	/* Have found a card. Register number of I/O locations used. */
	ia->ia_iosize = AIM_IOSIZE;
	ia->ia_msize = sizeof ( struct aim_memory );

	return 1;
}

/*
 *	Disable a cards interrupts and the polling loop.
 *	This is called at system shutdown so we don't bother to clean up any
 *	of the data structures just kill the hardware.
 */
static void
aimcshutdown ( sc )
	struct aimc_softc *sc;
{
	forward void aimcpoll ( void * );

	/* Disable interrupts */
	outb ( sc->card_iobase + AIM_INTEN, 0 );

	/* Kill the safety polling loop */
	untimeout ( aimcpoll, sc );
}


void
aimcattach ( parent, self, aux )
	struct device *parent;
	struct device *self;
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	register struct aimc_softc *sc = (struct aimc_softc *) self;
	static int aim_cnt = 0;
	forward int aimcintr ( struct aimc_softc * );
	forward void aimcpoll ( void * );

	/* Link into the OS */
	isa_establish ( &sc->sc_id, &sc->sc_dev );

	/* Record card parameters */
	sc->card_iobase = ia->ia_iobase;
	/* Currently assume that dual port memory is within the ISA HOLE */
	sc->card_mem = (struct aim_memory *)ISA_HOLE_VADDR( ia->ia_maddr );
	sc->card_num = aim_cnt++;	/* Unit number for diags */

	/* Program card with memory and irq settings */
	outb ( ia->ia_iobase + AIM_MBS1, ( (int)ia->ia_maddr >> 8 ) & 0xF8
					| irq_sel[ IL_FROM_IRQ ( ia->ia_irq )]);
	outb ( ia->ia_iobase + AIM_MBS2, ( (int)ia->ia_maddr >> 16 ) & 0xFF );

	/* Enable dual port RAM and select page to sync registers */
	outb ( ia->ia_iobase + AIM_MEMSEL, 1 );
	RPAGE_SEL ( sc, 0, 0 );

	/* Enable all card interrupts (edge triggered) */
	/* NOTE: Edge triggered interrupts won't work on an MCA bus. Also we are
	 *       missing a chance to share interrupts on EISA buses.
	 */
	sc->sc_ih.ih_fun = aimcintr;
	sc->sc_ih.ih_arg = (void *)sc;
	intr_establish ( ia->ia_irq, &sc->sc_ih, DV_TTY );
	outb ( ia->ia_iobase + AIM_INTTYPE, 0 );
	/* Only enable bus error and unit interrupts as power status interrupts
	 * tend to be noisy on earlier units when powered down and the 2 second
	 * polling loop filters this quite nicely.
	 */
	outb ( ia->ia_iobase + AIM_INTEN, INTR_UNIT | INTR_BUS );

	/* Start a watchdog polling loop */
	timeout ( aimcpoll, sc, WATCHDOG_RATE );

	/* Register the shutdown routine */
	sc->card_shutdown.func = aimcshutdown;
	sc->card_shutdown.arg = (void *)sc;
	atshutdown ( &sc->card_shutdown, ATSH_ADD );

#if BSDI_RELEASE >= 200

	/* Attach the interrupt event counter */
	evcnt_attach ( &sc->sc_intr, &sc->sc_dev,"intr");

	/* Finish the output line */
	printf ("\n");
#endif
}

/* OS configuration hooks */

struct cfdriver aimccd = {

#if BSDI_RELEASE >= 200
	NULL, "aimc", aimcprobe, aimcattach, DV_DULL,
		sizeof ( struct aimc_softc ), NULL, 0
#else
	NULL, "aimc", aimcprobe, aimcattach, sizeof ( struct aimc_softc )
#endif
};



/*
 *	Support routines
 *	================
 *
 *	Wait for a card to report power good
 */
static int
wait_power_good ( sc )
	struct aimc_softc *sc;
{
	int	error = ENOERROR;

	PROTECT
	    while ( ! ( sc->card_flags & CARD_POWER ))
	    {
		sc->card_flags |= CARD_WPOWER;
		if ( error = tsleep ( (caddr_t)&sc->card_flags, MYPRI | PCATCH
								,"aimpwr", 0 ))
	    	    break;
	    }
	ALLOW

	return error;
}

/*
 *	Wait for a card to report power bad
 */
static int
wait_power_bad ( sc )
	struct aimc_softc *sc;
{
	int	error = ENOERROR;

	PROTECT
	    while ( sc->card_flags & CARD_POWER )
	    {
		sc->card_flags |= CARD_WBPOWER;
		if ( error = tsleep ( (caddr_t)&sc->card_flags, MYPRI | PCATCH
								,"aimdwn", 0 ))
	    	    break;
	    }
	ALLOW

	return error;
}

/*
 *	Find the number of units on a recently powered host interface.
 *	Must be interrupt protected.
 */
static void
probe_units ( sc )
	struct aimc_softc *sc;
{
	int unit;
	Ubyte junk;

	/* Dummy read to clear possible junk state in external ASIC */
	junk = sc->card_mem->product_id[ 0 ].hw_reg;

	/* Scan reading ID's, non-existent unit will give AIM bus error and read
	 * as 0xFF
	 */
	for ( unit = 0 ; unit < NUNITS ; ++unit )
	{
	    if ( sc->card_mem->product_id[ unit ].hw_reg == AIM_ID_IOPRO )
		sc->card_uinfo[ unit ].unit_flags |= UNIT_PRESENT;
	    else
		break;
	}
	sc->card_nunits = unit;

	/* Quosh the bus error flag */
	if ( unit < NUNITS )
	    junk = sc->card_mem->intr_status.hw_reg;

	/* Mark remaining unit as not present */
	while ( unit < NUNITS )
	    sc->card_uinfo[ unit++ ].unit_flags &= ~UNIT_PRESENT;
}


/*
 *	Interrupt handler
 *	=================
 */

/*
 *	Service a unit interrupt
 */
static void
service_unit ( sc, unit )
	struct aimc_softc *sc;
	Ubyte unit;
{
	volatile struct master_page *mp;
	Ubyte	flags[ ITCSERV_SIZE ];	/* Buffer for service flags */
	int	i;

	/* Select the units control page.
	 * Note that we must use RPAGE_SEL() as there is a small window in a
	 * PAGE_SEL() which is not interrupt protected.
	 */
	RPAGE_SEL ( sc, unit, PGNUM_MASTER );
	mp = &sc->card_mem->msp;

	/* Fast exit if nothing to do */
	if ( mp->itserv == ITSERV_NOEVENT )
	    return;

	/* Set the lock and grab the service flags */
	/* Lock is based on "Peterson's solution". See "Operating Systems.
	 * Design and Implementation" by Andrew S. Tanenbaum.
	 */
	i = 0;
	mp->ithgard = TRUE;
	mp->ithacc = TRUE;
	/* This should be a very short wait. Firmware spec max 4uS */
	while ( mp->ithacc && mp->itfgard )
	{
	    if ( ++i >= 50 )
	    {
		/* Lock failed. Bomb on regardless */
		printf ("aimc%d: FIRMWARE LOCK FAILURE\n", sc->card_num );
		break;
	    }
	}

	/* Grab and zero flags */
	mp->itserv = ITSERV_NOEVENT;
	for ( i = sc->card_uinfo[ unit ].unit_maxpage / 8 ; i >= 0 ; --i )
	{
	    flags[ i ] = mp->itcserv[ i ];
	    mp->itcserv[ i ] = 0;
	}

	/* Clear the lock */
	mp->ithgard = FALSE;

	/* Check and process master page interrupts */
	if ( bit_test ( flags, PGNUM_MASTER ))
	{
	    bit_clear ( flags, PGNUM_MASTER );

	    /* Someone waiting for a response we now have ? */
	    if (( sc->card_uinfo[ unit ].unit_flags & UNIT_WRESPONSE )
					    && mp->cfrown == CFROWN_HOST )
	    {
		sc->card_uinfo[ unit ].unit_flags &= ~UNIT_WRESPONSE;
		wakeup ( (caddr_t)&sc->card_uinfo[ unit ]);
	    }
	    /* Someone waiting to send a command ? */
	    if (( sc->card_uinfo[ unit ].unit_flags & UNIT_WCOMMAND )
					    && mp->cfcown == CFCOWN_HOST )
	    {
		sc->card_uinfo[ unit ].unit_flags &= ~UNIT_WCOMMAND;
		wakeup ( (caddr_t)&sc->card_uinfo[ unit ]);
	    }
	}

	/* Now scan remaining flags looking for channel work */
	for ( i = sc->card_uinfo[ unit ].unit_maxpage ; i >= 0 ; --i )
	{
	    if ( bit_test ( flags, i ))
		aimdata_service ( sc, unit, i );
	}
}

/*
 *	Device driver interrupt entry point
 */
int
aimcintr ( sc )
	struct aimc_softc *sc;
{
	Ubyte	status;
	Ubyte	saved_unit;
	Ubyte	saved_page;
	int	unit;
	int	safety;

	dbg ( DBG_INTR,("I"));

	/* Read interrupt status and check it was for us */
	if (( status = sc->card_mem->intr_status.hw_reg & ALL_INTRS ) == 0 )
	    return INTR_NOT_I;

	dbg ( DBG_INTR,("%x ", status ));

#if BSDI_RELEASE >= 200

	/* Count the interrupt */
	sc->sc_intr.ev_count++;
#endif

	/* First handle power status changes and AIM bus errors */
	/* Note. Power bad takes priority over power good if both set */
	if ( status & INTR_PBAD )
	{
	    /* Is this a change of power state ? */
	    if ( sc->card_flags & CARD_POWER )
	    {
		dbg ( DBG_MISC,("PBAD\n"));

		sc->card_flags &= ~CARD_POWER;
		aim_deassign_minor ( sc );

		/* Clear out the units interrupt info and mark them as not
		 * present
		 */
		for ( unit = 0 ; unit < NUNITS ; ++unit )
		{
		    sc->card_uinfo[ unit ].unit_maxpage = 0;
		    sc->card_uinfo[ unit ].unit_flags &= ~UNIT_PRESENT; 
		}

		/* Wakeup anyone waiting for power to go bad */
		if ( sc->card_flags & CARD_WBPOWER )
		{
		    sc->card_flags &= ~CARD_WBPOWER;
		    wakeup ( (caddr_t)&sc->card_flags );
		}
	    }
	    /* If power is bad none of the rest of the flags make sense */
	    return INTR_EXPECTED;
	}

	safety = 0;
	if ( status & INTR_BUS )
	{
	    /* The latentcy on processing the interrupt makes it very difficult
	     * to decide which unit has failed giving the bus error (you can't
	     * rely on sc->card_unit). In practice we will get a power fail
	     * signal shortly and can discard the lot so we ignore the problem
	     * for the moment.
	     */
	    dbg ( DBG_MISC,("aimc%d: BUS ERROR\n", sc->card_num ));
	    safety = 1;
	}

	if ( status & INTR_PGOOD )
	{
	    /* Is this a change of power state ? */
	    if ( ! ( sc->card_flags & CARD_POWER ))
	    {
		dbg ( DBG_MISC,("PGOOD\n"));

		sc->card_flags |= CARD_POWER;
		probe_units ( sc );
		/* Wakeup anyone waiting for power */
		if ( sc->card_flags & CARD_WPOWER )
		{
		    sc->card_flags &= ~CARD_WPOWER;
		    wakeup ( (caddr_t)&sc->card_flags );
		}
	    }
	    safety = 1;
	}

	/* Save the current unit and page selection for restoration after
	 * interrupt service is complete.
	 */
	saved_unit = sc->card_unit;
	saved_page = sc->card_page;

	/* Service external unit interrupts */
	while ( status & INTR_UNIT )
	{
	    dbg ( DBG_INTR,("U%x ", status ));

	    /* Acknowledge external interrupts */
	    unit = sc->card_mem->intr_ack.hw_reg;

	    /* We ignore the unit vector and scan all units. Under heavy load
	     * there is a good chance we can service several units, under
	     * light load who cares ?
	     */
	    for ( unit = 0 ; unit < NUNITS ; ++unit )
	    {
		if ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT )
		{
		    dbg ( DBG_INTR,("%d ", unit ));
		    service_unit ( sc, unit );
		}
		else
		    break;
	    }

	    /* Reread interrupt register to service any new unit events */
	    status = sc->card_mem->intr_status.hw_reg;

	    /* If we've been round this loop more than 5 times it probably
	     * means that intr_status is stuck ( or cached ). Exiting the loop
	     * should hopefully clear any cache but won't fix a stuck bit.
	     */
	    if ( ++safety > 5 )
	    {
		printf ("aimc%d: intr_status stuck = %x\n", sc->card_num
								, status );
		break;
	    }
	}

	/* Restore unit and page selection for interrupted code */
	PAGE_SEL ( sc, saved_unit, saved_page );

	/* Never expect an interrupt or it wouldn't be an interrupt */
	/* We now say we expected it if we processed at least on item, this
	 * stops the system recording the interrupt as spurious
	 */
	return safety ? INTR_EXPECTED : INTR_UNEXPECTED;
}

/*
 *	Watchdog polling loop
 */
void
aimcpoll ( sc )
	void *sc;
{
	PROTECT
	    aimcintr ( (struct aimc_softc *)sc );
	ALLOW

	timeout ( aimcpoll, sc, WATCHDOG_RATE );
}


/*
 *	Standard driver entry points
 *	============================
 *
 *	Open the device
 *
 *	Note: A lot of the control operations assume exclusive use of the
 *	appropriate devices. However I do not perform exclusion locking in
 *	the driver, this simplifies the driver, and besides anyone stupid
 *	enough to start multiple support demons gets what they deserve.
 */
int
aimcopen ( dev, flags, mode, p )
	dev_t dev;
	int flags;
	int mode;
	struct proc *p;
{
	struct aimc_softc *sc;
	int card = MCARD ( dev );
	int unit = MUNIT ( dev );
	int type = MTYPE ( dev );

	dbg ( DBG_ENTRY,("aimcopen: %x %x %x\n", dev, flags, mode ));

	/* Open actions depend on type */
	switch ( type )
	{
	case TYPE_ALLCARD:

	    /* Force use of correct minor number, we may have use of
	     * other bit combos in the future.
	     */
	    if ( unit != 0 || card != 0 )
		return ENOENT;

	    /* Initialise the info file data */
	    /* Find number of cards */
	    for ( card = 0 ; card < aimccd.cd_ndevs ; card++ )
	    {
		if ( aimccd.cd_devs[ card ] == NULL )
		    break;
	    }
	    aim_info_file[ IFILE_NCARDS_OFFSET ] = '0' + card;

	    break;

	case TYPE_CARD:

	    /* Force correct minor # */
	    if ( unit != 0 )
		return ENOENT;

	    /* Was card detected at system powerup ? */
	    if ( card >= aimccd.cd_ndevs
				|| ( sc = aimccd.cd_devs[ card ]) == NULL )
	    {
		return ENXIO;
	    }

	    /* Opening for reading indicates waiting for power up, for writing
	     * waits for power down.
	     */
	    if ( flags & FREAD )
		return wait_power_good ( sc );
	    else
		return wait_power_bad ( sc );
	    break;

	case TYPE_CONFIG:
	case TYPE_RAW:

	    /* Check unit limits */
	    if ( unit >= NUNITS )
		return ENOENT;

	    /* Was card detected at powerup ? */
	    if ( card >= aimccd.cd_ndevs
				|| ( sc = aimccd.cd_devs[ card ]) == NULL )
	    {
		return ENXIO;
	    }

	    /* Is the external unit present and powered ? */
	    if ( ! ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT ))
		return EIO;

	    if ( type == TYPE_CONFIG )
	    {
		/* Enable configuration interrupts from the unit */
		/* Also sets up and enables generation of interrupts by the
		 * unit. This side effect of configuration is relied on by
		 * the general data drivers which assume that the unit is
		 * already enabled and that they only need to enable the
		 * particular page.
		 */
		PROTECT
		    PAGE_SEL ( sc, unit, PGNUM_MASTER );
		    sc->card_mem->msp.cfflags |= CFFLAGS_CINTR | CFFLAGS_RINTR;
		    bit_set ( sc->card_mem->msp.itcmask, PGNUM_MASTER );
		    sc->card_mem->msp.itenab = ITENAB_10MS;
		ALLOW
	    }
	    break;
	}

	return ENOERROR;
}


/*
 *	Close the device
 */
int
aimcclose ( dev, flags, mode, p )
	dev_t dev;
	int flags;
	int mode;
	struct proc *p;
{
	/* Not a lot to do. But keep a place marker for later perhaps. */
	return ENOERROR;
}

/*
 *	Read external unit memory
 */
int
aimcread ( dev, uio, flags )
	dev_t dev;
	struct uio *uio;
	int flags;
{
	struct aimc_softc *sc = aimccd.cd_devs[ MCARD( dev )];
	int unit = MUNIT( dev );
	int error = ENOERROR;
	int off;
	int c;

	dbg ( DBG_ENTRY,("aimcread: %x %x\n", dev, flags ));

	/* Read action depends on type of device */
	switch ( MTYPE( dev ))
	{
	case TYPE_RAW:

	    /* Raw access to box memory */
	    /* Check the unit has not died at some time after the open */
	    if ( ! ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT ))
	    {
		error = EIO;
		break;
	    }

	    /* We do byte I/O so that any error will be detected at the earliest
	     * opportunity, a thousand AIM bus errors are a pain. Efficiency is
	     * not a problem since these accesses should only occur at system/
	     * external unit power up under normal circumstances.
	     */
	    while ( uio->uio_resid > 0 && error == ENOERROR )
	    {
		/* Check we've not hit EOF */
		if ( uio->uio_offset >= PGSIZE * NPAGES )
		    break;

		/* Get the next byte */
		PAGE_SEL ( sc, unit, uio->uio_offset / PGSIZE );
		c = sc->card_mem->hwp[ uio->uio_offset % PGSIZE ];

		/* Check we've not had an AIM bus error and pass data to user */
		if ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT )
		    error = ureadc ( c, uio );
		else
		    error = EIO;
	    }
	    break;

	case TYPE_CONFIG:

	    /* Configuration string access */
	    /* Insist that we read the whole string or nothing */
	    if ( uio->uio_resid < CFREQ_SIZE )
	    {
		error = EINVAL;
		break;
	    }

	    /* Check the unit has not died at some time after the open */
	    if ( ! ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT ))
	    {
		error = EIO;
		break;
	    }

	    /* Wait for the string */
	    PROTECT
		PAGE_SEL ( sc, unit, PGNUM_MASTER );
		while ( sc->card_mem->msp.cfrown != CFROWN_HOST )
		{
		    /* Check the unit has not died */
		    if ( ! ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT ))
		    {
			error = EIO;
			break;
		    }
		    sc->card_uinfo[ unit ].unit_flags |= UNIT_WRESPONSE;
		    if ( error = tsleep ( (caddr_t)&sc->card_uinfo[ unit ]
						, MYPRI | PCATCH,"aimrsp", 0 ))
			break;

		    /* Common gotcha. Context switch may have changed page */
		    PAGE_SEL ( sc, unit, PGNUM_MASTER );
		}
	    ALLOW

	    /* Read the string */
	    for ( off = 0 ; uio->uio_resid > 0 && error == ENOERROR ; ++off )
	    {
		/* Check we've not hit EOF */
		if ( off >= CFREQ_SIZE )
		    break;

		/* Get the next byte */
		c = sc->card_mem->msp.cfreq[ off ];

		/* Check we've not had an AIM bus error and pass data to user */
		if ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT )
		{
		    /* Terminate on NUL */
		    if ( c == '\0')
			break;

		    error = ureadc ( c, uio );
		}
		else
		    error = EIO;
	    }

	    /* Adding a new line will probably break fewer programs */
	    if ( error == ENOERROR && uio->uio_resid > 0 && c != '\n')
		error = ureadc ('\n', uio );

	    /* Release request/response region */
	    /* SPEC VIOLATION. We are supposed to process debug message requests
	     *      from the firmware for console display, also diagnostic
	     *      download requests. Neither are generated by the production
	     *      firmware so we should be OK.
	     */
	    sc->card_mem->msp.cfrown = CFROWN_BOX_OK;
	    break;

	case TYPE_ALLCARD:

	    /* Combined info access */
	    while ( uio->uio_resid > 0 && error == ENOERROR )
	    {
		/* Check we've not hit EOF */
		if ( uio->uio_offset >= IFILE_SIZE )
		    break;

		/* Get the next byte */
		error = ureadc ( aim_info_file[ uio->uio_offset ], uio );
	    }
	    break;

	default:
	    error = ENODEV;
	    break;
	}

	return error;
}

/*
 *	Write external unit memory
 */
int
aimcwrite ( dev, uio, flags )
	dev_t dev;
	struct uio *uio;
	int flags;
{
	struct aimc_softc *sc = aimccd.cd_devs[ MCARD( dev )];
	int unit = MUNIT( dev );
	int error = ENOERROR;
	int off;
	int c;

	dbg ( DBG_ENTRY,("aimcwrite: %x %x\n", dev, flags ));

	/* Check the unit has not died at some time after the open */
	if ( ! ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT ))
	    return EIO;

	/* Write action depends on type of device */
	switch ( MTYPE( dev ))
	{
	case TYPE_RAW:

	    off = uio->uio_offset;

	    /* Raw access to box memory */
	    while ( uio->uio_resid > 0 && error == ENOERROR )
	    {
		/* Check we've not hit EOF */
		if ( off >= PGSIZE * NPAGES )
		    break;

		/* Get the next byte */
		if (( c = uwritec ( uio )) == -1 )
		{
		    error = EFAULT;
		    break;
		}

		/* Stick it in external memory */
		PAGE_SEL ( sc, unit, off / PGSIZE );
		sc->card_mem->hwp[ off++ % PGSIZE ] = c;

		/* Check we've not had an AIM bus error */
		if ( ! ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT ))
		{
		    error = EIO;
		    break;
		}
	    }
	    break;

	case TYPE_CONFIG:

	    /* Configuration string access */
	    /* Wait for access to command area */
	    PROTECT
		PAGE_SEL ( sc, unit, PGNUM_MASTER );
		while ( sc->card_mem->msp.cfcown != CFCOWN_HOST )
		{
		    /* Check the unit has not died */
		    if ( ! ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT ))
		    {
			error = EIO;
			break;
		    }
		    sc->card_uinfo[ unit ].unit_flags |= UNIT_WCOMMAND;
		    if ( error = tsleep ( (caddr_t)&sc->card_uinfo[ unit ]
						, MYPRI | PCATCH,"aimcmd", 0 ))
			break;

		    /* Common gotcha. Context switch may have changed page */
		    PAGE_SEL ( sc, unit, PGNUM_MASTER );
		}
	    ALLOW

	    dbg ( DBG_MISC,("Wait for config write access complete\n"));

	    /* Write the string */
	    for ( off = 0 ; uio->uio_resid > 0 && error == ENOERROR ; ++off )
	    {
		/* Check we've not hit EOF */
		if ( off >= CFCMD_SIZE )
		    break;

		/* Get the next byte */
		if (( c = uwritec ( uio )) == -1 )
		{
		    error = EFAULT;
		    break;
		}

		/* Convert newline into nul termination */
		if ( c == '\n')
		    c = '\0';

		/* Write the byte */
		sc->card_mem->msp.cfcmd[ off ] = c;

		/* Check we've not had an AIM bus error */
		if ( ! ( sc->card_uinfo[ unit ].unit_flags & UNIT_PRESENT ))
		    error = EIO;

		if ( c == '\0')
		    break;
	    }

	    /* Add nul termination if not already done and we have space */
	    if ( off < CFCMD_SIZE && c != '\0' && error == ENOERROR )
		sc->card_mem->msp.cfcmd[ off ] = '\0';
		

	    /* Release request/response region */
	    sc->card_mem->msp.cfcown = CFCOWN_BOX;
	    break;

	default:
	    error = ENODEV;
	    break;
	}

	return error;
}

/*
 *	Ioctl handling
 */
					/* Typing short cut */
#define ainfo	(*(struct iopro_assign_info *)data)

int
#if BSDI_RELEASE >= 200
aimcioctl ( dev, cmd, data, flag, p )
#else
aimcioctl ( dev, cmd, data, flag )
#endif
	dev_t dev;
	int cmd;
	caddr_t data;
	int flag;
#if BSDI_RELEASE >= 200
	struct proc *p;
#endif
{
	struct aimc_softc *sc = aimccd.cd_devs[ MCARD( dev )];
	int unit = MUNIT( dev );
	int error;

	dbg ( DBG_ENTRY,("aimcioctl: %x %x\n", dev, cmd ));

	if ( sc == NULL )
	    return ENXIO;

	switch ( cmd )
	{
	case IOPRO_RESET:
	    sc->card_mem->proc_control[ unit ].hw_reg = PROC_RESET;
	    return ENOERROR;

	case IOPRO_RELEASE:
	    sc->card_mem->proc_control[ unit ].hw_reg = 0;
	    return ENOERROR;

	case IOPRO_ATTN:
	    sc->card_mem->proc_control[ unit ].hw_reg = PROC_INT;
	    return ENOERROR;

	case IOPRO_SWRESET:
	    sc->card_mem->proc_control[ unit ].hw_reg = PROC_NMI;
	    return ENOERROR;

	case IOPRO_ASSIGN:
	    /* Only support the async data driver at present */
	    if ( ainfo.ia_driver != IOPRO_ASYNC || ainfo.ia_page > 255 )
		return EINVAL;

	    error = aim_assign_minor ( ainfo.ia_minor, sc, unit, ainfo.ia_page);
	    if ( error != ENOERROR )
		return error;

	    PROTECT
		/* Enable interrupts from this channel */
		PAGE_SEL ( sc, unit, PGNUM_MASTER );
		bit_set ( sc->card_mem->msp.itcmask, ainfo.ia_page );
		/* Log highest interrupting channel */
		if ( ainfo.ia_page > sc->card_uinfo[ unit ].unit_maxpage )
		    sc->card_uinfo[ unit ].unit_maxpage = ainfo.ia_page;
	    ALLOW
	    return ENOERROR;

	case IOPRO_DEBUG:
#ifdef AIM_DEBUG
	    aim_debug = *(int *)data;
	    return ENOERROR;
#else
	    return ENXIO;
#endif

	default:
	    dbg ( DBG_MISC,("aimc: unknown ioctl %x\n", cmd ));
	    return EINVAL;
	}
}

#undef	ainfo

#endif


/*
 *	The device switch entry
 *	Placed here because I hate forward decls
 */
#if BSDI_RELEASE >= 200

struct devsw aimcsw = {
	&aimccd,
	aimcopen, aimcclose, aimcread, aimcwrite, aimcioctl, noselect, nommap,
	nostrat, nodump, nopsize, 0,
	nostop
};
#endif
