/*	BSDI aimdata.c,v 2.6 1998/06/03 19:14:53 karels Exp	*/

/*-
 *	Copyright (c) 1993, 1994 Robert J.Dunlop. All rights reserved.
 *
 *	Portions Copyright (c) 1991, 1992, 1993, 1994 Chase Research PLC.
 *	Portions Trade Secrets of Chase Research PLC.
 *	All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, is permitted provided the following conditions are met:
 *   1.	Redistributions in source form must retain the above copyright and trade
 *	secret notices, this list of conditions and the following disclaimers.
 *   2. Redistributions in binary form must reproduce all items from condition
 *	1 in the accompanying documentation and/or other materials.
 *   3. Neither the names of Chase Research PLC and it's subsidiary companies,
 *	nor that of Robert Dunlop may be used to endorse, advertise or promote
 *	products derived from this software without specific written permission
 *	from the party or parties concerned.
 *   4.	Adaptation of this software to operating systems other than BSD/OS
 *	requires the prior written permission of Chase Research PLC.
 *   5.	Adaptation of this software for operation with communications hardware
 *	other than the Chase IOPRO and IOLITE series requires the prior written
 *	permission of Chase Research PLC.
 */

/*
 *	Chase Research IOPRO data driver (aim)
 *
 *	This driver is used to perform the user control of and data transfer
 *	to and from a single channel page in the external AIM interface.
 *
 *	aimdata.c,v 1.21 1995/07/28 08:59:35 rjd Exp
 */


/* Only include code if we are configured in.
 * Also perform configuration sanity check.
 */
#include "aim.h"
#if !defined(NAIMC)
# include "aimc.h"
#endif
#if NAIM > 0

#if NAIMC <= 0
Configuration error: You must include the AIM control driver.
#endif

/* 1.0 and 1.1 system users may need to convert these <> back to "" */
#include <cdefs.h>
#include <types.h>
#include <errno.h>
#include <fcntl.h>
#include <device.h>
#include <uio.h>
#include "isavar.h"
#include "isa.h"
#include <ioctl.h>
#include <tty.h>
#include <param.h>
#include <systm.h>
#include <time.h>
#include <proc.h>
#include <conf.h>
#include <malloc.h>

#include "aimreg.h"
#include "aimvar.h"
#include "aimioctl.h"


/*
 *	The tty control structures.
 *
 *	These consist of two parts.
 *	    The tty structure itself used by the driver.
 *	    The tty attach structure used to gather stats etc for kinfo.
 */
struct super_tty {
	struct tty s_tty;		/* Driver info */
	struct ttydevice_tmp s_ttydev;	/* How to find the driver info */
};


/*
 *	Minor device number breakdown
 */
#define NUM_MINOR	NAIM		/* Number of minor device numbers
					 * available.
					 */

struct minor_info {
					/* Minor device number breakdown */
	struct aimc_softc *minor_sc;	/* Card information */
	Ubyte	minor_unit;		/* Unit number */
	Ubyte	minor_page;		/* Page number */
	Ubyte	minor_flags;		/* Special flags */

	struct super_tty *m_super_tty;	/* The "standard" tty information */

	tcflag_t	m_iflag;	/* Original values of flags */
	tcflag_t	m_oflag;
	tcflag_t	m_cflag;
};

					/* Minor flag values */
#define MINF_ASSIGNED	0x01		/* Card/unit/page are valid */
#define MINF_WASSIGN	0x02		/* Someone is waiting for port to be
					 * assigned a card/unit/page triple
					 */
#define MINF_TXWATER	0x04		/* Tx watermark is set */
#define MINF_OSTOP	0x08		/* Firmware output stopped by user */
#define MINF_IRUNNING	0x10		/* Driver input chain running */


					/* Parts handled by the external
					 * firmware so the line discipline
					 * needn't bother.
					 */
#define FW_IFLAG	( ISTRIP | INLCR | IGNCR | ICRNL | IXON | IXOFF | IXANY)
					/* ONOEOT's a new (or very old) one that
					 * I havn't come across before and hence
					 * it's not supported in the firmware.
					 * 27 Jul 95: It's very old, don't look
					 * like it's ever used these days.
					 */
#define FW_OFLAG	( ONLCR | OXTABS )
#define FW_CFLAG	( CCTS_OFLOW | CRTS_IFLOW | MDMBUF )

					/* The minor device data */
static struct minor_info minor_info[ NUM_MINOR ];


					/* Is the minor device # legal */
#define DEV_IS_LEGAL(D)		( minor(D) < NUM_MINOR )

					/* Get device flags. Assumes we have
					 * already checked device is legal.
					 */
#define DEV_FLAGS(D)		minor_info[ minor(D)].minor_flags

					/* Remaining device fields */
#define SC_FROM_DEV(D)		minor_info[ minor(D)].minor_sc
#define UNIT_FROM_DEV(D)	minor_info[ minor(D)].minor_unit
#define PAGE_FROM_DEV(D)	minor_info[ minor(D)].minor_page
#define SUPER_FROM_DEV(D)	minor_info[ minor(D)].m_super_tty

					/* Is the minor device # assigned. */
#define DEV_IS_ASSIGNED(D)	( DEV_FLAGS(D) & MINF_ASSIGNED )


					/* Pointer to tty structure given the
					 * device number.
					 */
#define TTY_FROM_DEV(D)		( &minor_info[ minor(D)].m_super_tty->s_tty )

					/* Pointer to ttydev structure given the
					 * device number.
					 */
#define TTYDEV_FROM_DEV(D)	( &minor_info[ minor(D)].m_super_tty->s_ttydev )


					/* Minor info given a tty pointer */
#define MINFO_FROM_TP(T)	( &minor_info[ minor((T)->t_dev ) ])


/*
 *	Minor device # control routines
 *	===============================
 *
 *	Assign a minor device to an external channel page.
 *	This entry is called from the control driver under prompting of the
 *	control daemon.
 */
int
aim_assign_minor ( minor_num, sc, unit, page )
	unsigned minor_num;
	struct aimc_softc *sc;
	unsigned unit;
	unsigned page;
{
	extern struct devsw aimcsw;

	/* Quick safety as control driver does not know our minor device range.
	 * We assume card, unit and page have already been checked.
	 */
	if ( DEV_IS_LEGAL ( minor_num ))
	{
	    /* If not already allocated get the super tty structure and set it
	     * up, linking it into the kerninfo list.
	     */
	    if ( SUPER_FROM_DEV( minor_num ) == NULL )
	    {
		/* Allocate and zero data structure */
		SUPER_FROM_DEV( minor_num )
		  = malloc ( sizeof ( struct super_tty ), M_DEVBUF, M_WAITOK );

		/* Give up if we can't get the memory */
		if ( SUPER_FROM_DEV( minor_num ) == NULL )
		    return ENOMEM;

		bzero ( SUPER_FROM_DEV( minor_num )
						, sizeof ( struct super_tty ));

		/* Initialise */
		/* The device name is "aim" but the most informative name to
		 * give the user is probably "iopro"
		 */
		strcpy ( TTYDEV_FROM_DEV( minor_num )->tty_name,"iopro");

		TTYDEV_FROM_DEV( minor_num )->tty_unit = unit;
		TTYDEV_FROM_DEV( minor_num )->tty_base = minor_num;
		TTYDEV_FROM_DEV( minor_num )->tty_count = 1;
		TTYDEV_FROM_DEV( minor_num )->tty_ttys
						= TTY_FROM_DEV( minor_num );

		/* Link */
		PROTECT
		    tty_attach ( TTYDEV_FROM_DEV( minor_num ), &aimcsw);
		ALLOW
	    }

	    /* Interrupt protect while we set values */
	    PROTECT
		SC_FROM_DEV ( minor_num ) = sc;
		UNIT_FROM_DEV ( minor_num ) = unit;
		PAGE_FROM_DEV ( minor_num ) = page;
		DEV_FLAGS ( minor_num ) |= MINF_ASSIGNED;

		/* Set external unit based back pointer */
		PAGE_SEL ( sc, unit, page );
		sc->card_mem->chp.osminor = minor_num;

		/* Wake up anyone waiting for the assignment */
		if ( DEV_FLAGS ( minor_num ) & MINF_WASSIGN )
		{
		    DEV_FLAGS ( minor_num ) &= ~MINF_WASSIGN;
		    wakeup ( (caddr_t)&DEV_FLAGS ( minor_num ) );
		}
	    ALLOW
	    return ENOERROR;
	}
	else
	    return ENXIO;
}

/*
 *	De-assign minor device numbers, based on card #.
 *	This entry is called from the control driver when an external units
 *	power signal goes bad. All units share a common signal so they are all
 *	de-assigned.
 */
void
aim_deassign_minor ( sc )
	struct aimc_softc *sc;
{
	int minor_num;

	for ( minor_num = 0 ; minor_num < NUM_MINOR ; ++minor_num )
	{
	    if ( minor_info[ minor_num ].minor_sc == sc )
		minor_info[ minor_num ].minor_flags &= ~MINF_ASSIGNED;
	}
}


/*
 *	Wait for a device to be assigned an external page.
 */
static int
wait_dev_assignment ( dev, flags )
	dev_t dev;
	int flags;
{
	int error = ENOERROR;
	
	if ( ! DEV_IS_LEGAL ( dev ))
	    return ENXIO;

	PROTECT
	    while ( ! ( DEV_FLAGS ( dev ) & MINF_ASSIGNED ))
	    {
		if ( flags & O_NONBLOCK )
		{
		    error = EWOULDBLOCK;
		    break;
		}
		else
		{

		    DEV_FLAGS ( dev ) |= MINF_WASSIGN;

		    if ( error = tsleep ((caddr_t)&DEV_FLAGS( dev )
						, MYPRI | PCATCH,"aimasg", 0 ))
		    {
			break;
		    }
		}
	    }
	ALLOW

	return error;
}



/*
 *	Driver configuration entry points
 *	=================================
 *
 *	These are dummys as the hardware belongs to the control driver.
 *
 *	Probe makes sure we only print one configuration message.
 */
static int
aimprobe ( parent, cf, aux )
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	static Bool done_probe = FALSE;

	if ( done_probe )
	    return 0;

	done_probe = TRUE;
	return 1;
}

void
aimattach ( parent, self, aux )
	struct device *parent;
	struct device *self;
	void *aux;
{
	/* Nothing for the attach routine to do.  */
}


/* The configuration control structure */
struct cfdriver aimcd =
{
#if BSDI_RELEASE >= 200
	NULL, "aim", aimprobe, aimattach, DV_TTY, 0, NULL, 0
#else
	NULL, "aim", aimprobe, aimattach, 0
#endif
};

/* The control drivers structure is the real one to watch */
extern struct cfdriver aimccd;



/*
 *	Misc support routines
 *	=====================
 *
 *	Select a channel page from the device number
 */
static volatile struct channel_page *
cp_from_dev ( dev )
	dev_t dev;
{
	register struct aimc_softc *sc;

	sc = SC_FROM_DEV ( dev );
	PAGE_SEL ( sc, UNIT_FROM_DEV ( dev ), PAGE_FROM_DEV ( dev ));
	return &sc->card_mem->chp;
}


/*
 *	Modem control routines
 *	======================
 *
 *	Raise or lower modem control lines ( DTR & RTS )
 */
static void
aim_modem_ctl ( dev, state, inclose )
	dev_t dev;
	Ubyte state;
	Bool  inclose;
{
	volatile struct channel_page *cp;
	Ubyte phoutln;

	/* Get output line control byte */
	cp = cp_from_dev ( dev );
	phoutln = cp->phoutln;

	/* Reprogram RTS unless it is under special control */
	if ( ! ( phoutln & PHOUTLN_RTS_FLOW ) || inclose )
	{
	    phoutln &= ~PHOUTLN_RTS_MASK;
	    phoutln |= state & PHOUTLN_RTS_MASK;
	}

	/* Ditto DTR */
	if ( ! ( phoutln & PHOUTLN_DTR_FLOW ) || inclose )
	{
	    phoutln &= ~PHOUTLN_DTR_MASK;
	    phoutln |= state & PHOUTLN_DTR_MASK;
	}

	/* Tell the unit and make the change immediate */
	cp->phoutln = phoutln;
	cp->mdflags = MDFLAGS_IMMED;
}

/*
 *	Convert ioctl (TIOC) modem control bits to firmware (PHOUTLN) format.
 *	Currently only support the two output bits DTR and RTS.
 *	XXX Should check what "line enable" does and add this ?
 */
static Ubyte
tioc2phoutln ( tioc )
	int tioc;
{
	Ubyte phout = PHOUTLN_DTR_OFF | PHOUTLN_RTS_OFF;

	if ( tioc & TIOCM_DTR )
	    phout |= PHOUTLN_DTR_ON;
	if ( tioc & TIOCM_RTS )
	    phout |= PHOUTLN_RTS_ON;

	return phout;
}

/*
 *	Sense modem status. Format returned is compatible with TIOC ioctls.
 *	XXX Again we are forcing "line enable" true.
 */
static int
aim_modem_status ( dev )
	dev_t dev;
{
	volatile struct channel_page *cp;
	Ubyte b;
	int status;

	status = TIOCM_LE;		/* Start with fixed parts */

	cp = cp_from_dev ( dev );

	/* Add in DTR and RTS outputs */
	/* Firmware 1.05 onward now reports state */
	b = cp->olstate;
	if ( b & OLSTATE_DTR )
	    status |= TIOCM_DTR;
	if ( b & OLSTATE_RTS )
	    status |= TIOCM_RTS;

	/* Now process the four lines input lines.
	 * Note three of these lines are deliberately inverted for easy special
	 * state detection when handled with input characters.
	 */
	b = cp->lnstate;
	if ( ! ( b & LNSTATE_NO_CTS ))
	    status |= TIOCM_CTS;
	if ( ! ( b & LNSTATE_NO_DSR ))
	    status |= TIOCM_DSR;
	if ( b & LNSTATE_RI )
	    status |= TIOCM_RI;
	if ( ! ( b & LNSTATE_NO_DCD ))
	    status |= TIOCM_CD;

	return status;
}

/*
 *	Enable/disable monitoring of modem status lines.
 */
static void
aim_monitor ( dev, lines )
	dev_t dev;
	Ubyte lines;
{
	volatile struct channel_page *cp;

	cp = cp_from_dev ( dev );
	cp->lnintr = lines;
	cp->lndelta = 0;
}


/*
 *	Configuration changing routines
 *	===============================
 *
 *	Change hardware parameters ( called by line discipline and others )
 */
static int
aimparam ( tp, t )
	struct tty *tp;
	struct termios *t;
{
	volatile struct channel_page *cp;
	Ubyte flags;


	/* Don't bother to proceed if hardware hasn't changed */
	if ( tp->t_ispeed == t->c_ispeed && tp->t_ospeed == t->c_ospeed
		&& tp->t_cflag == t->c_cflag && ( tp->t_state & TS_ISOPEN ))
	{
	    return ENOERROR;
	}

	/* If CLOCAL has just been added wakeup anyone who might be waiting for
	 * carrier in the open.
	 */
	if ( ! ( tp->t_cflag & CLOCAL ) && ( t->c_cflag & CLOCAL ))
	    wakeup ( (caddr_t)&tp->t_rawq );

	/* If old baudrate was 0 raise DTR & RTS ( unless rate is still 0 ) */
	if ( tp->t_ospeed == 0 && t->c_ospeed != 0 )
	    aim_modem_ctl ( tp->t_dev, PHOUTLN_DTR_ON | PHOUTLN_RTS_ON, FALSE );

	/* Copy to tty struct */
	tp->t_ispeed = t->c_ispeed;
	tp->t_ospeed = t->c_ospeed;
	tp->t_cflag = t->c_cflag;

	/* If baudrate is 0 this signals dropping DTR & RTS */
	if ( t->c_ospeed == 0 )
	{
	    aim_modem_ctl ( tp->t_dev, PHOUTLN_DTR_OFF | PHOUTLN_RTS_OFF
								, FALSE );
	    return ENOERROR;
	}

	/* Tell the unit about new setting */
	cp = cp_from_dev ( tp->t_dev );

	/* Baudrate ( let the firmware calc nearest divisor ) */
	/* V1.05 firmware onward supports split baudrate */
	cp->phbsel = PHBSEL_ACTUAL;
	cp->phibaud = t->c_ispeed * 256;
	cp->phbaud = t->c_ospeed * 256;

	/* Set parity unless already special */
	flags = cp->phfrmt & PHFRMT_CLR_PAR;
	if ( flags != PHFRMT_CLR_PAR && flags != PHFRMT_SET_PAR )
	{
	    flags = 0;
	    if ( t->c_cflag & PARENB )
	    {
		flags = ( t->c_cflag & PARODD ) ? PHFRMT_ODD_PAR
						: PHFRMT_EVEN_PAR;
	    }
	}
	/* Add in character size and stop bit information */
	switch ( t->c_cflag & CSIZE )
	{
	    case CS5:	flags |= PHFRMT_5BIT;	break;
	    case CS6:	flags |= PHFRMT_6BIT;	break;
	    case CS7:	flags |= PHFRMT_7BIT;	break;
	    case CS8:	flags |= PHFRMT_8BIT;	break;
	}
	if ( t->c_cflag & CSTOPB )
	    flags |= PHFRMT_2STOP;
	/* Now tell the unit */
	cp->phfrmt = flags;

	/* Hardware output flow control */
	flags = cp->phinact & ~( PHINACT_TX_NEEDS_CTS | PHINACT_TX_NEEDS_DCD );
	if ( t->c_cflag & CCTS_OFLOW )
	    flags |= PHINACT_TX_NEEDS_CTS;
	if ( t->c_cflag & MDMBUF )
	    flags |= PHINACT_TX_NEEDS_DCD;
	cp->phinact = flags;

	/* Hardware input flow control */
	flags = cp->phoutln & ~PHOUTLN_RTS_MASK;
	if ( t->c_cflag & CRTS_IFLOW )
	    flags |= PHOUTLN_RTS_FLOW;
	else
	    flags |= PHOUTLN_RTS_ON;
	cp->phoutln = flags;

	/* Backup the hardware flags and mask the bits we don't want the host
	 * to process
	 */
	MINFO_FROM_TP ( tp )->m_cflag = t->c_cflag;
	t->c_cflag &= ~FW_CFLAG;
	tp->t_cflag &= ~FW_CFLAG;

	/* Tell the unit to make the change */
	cp->mdflags = MDFLAGS_IMMED;

	return ENOERROR;
}

/*
 *	Record the software settings ( iflags & oflag ) and setup the external
 *	firmware accordingly.
 */
static void
aimparam2 ( tp )
	struct tty *tp;
{
	volatile struct channel_page *cp;
	Ubyte flags;

	/* Get access to firmware channel page */
	cp = cp_from_dev ( tp->t_dev );

	/* First setup the output translations */
	/* We preserve those translations not supported under BSD in case the
	 * user has set them indirectly via the configuration interface
	 */
	flags = cp->otflags & ~ ( OTFLAGS_LF_TO_CRLF | OTFLAGS_TABEXPAND );
	if ( tp->t_oflag & OPOST )
	{
	    if ( tp->t_oflag & ONLCR )
		flags |= OTFLAGS_LF_TO_CRLF;
	    if ( tp->t_oflag & OXTABS )
		flags |= OTFLAGS_TABEXPAND;
	}
	cp->otflags = flags;

	/* Now the input translation flags */
	flags = ITFLAGS_NOERRSTRIP;
	if ( tp->t_iflag & INLCR )
	    flags |= ITFLAGS_LF_TO_CR;
	if ( tp->t_iflag & IGNCR )
	    flags |= ITFLAGS_IGNORE_CR;
	if ( tp->t_iflag & ICRNL )
	    flags |= ITFLAGS_CR_TO_LF;
	cp->itflags = flags;
	cp->itstrip = ( tp->t_iflag & ISTRIP ) ? 0x7F : 0xFF;

	/* Finally the software flow control flags */
	/* Again we preserve any special conditions that may have been set
	 * indirectly ( the most useful being robust operation )
	 */
	if ( tp->t_iflag & IXOFF )
	{
	    cp->fcinflg |= FCINFLG_ENABLE;
	    cp->fctxon = tp->t_cc[ VSTART ];
	    cp->fctxoff = tp->t_cc[ VSTOP ];
	}
	else
	    cp->fcinflg &= ~FCINFLG_ENABLE;

	/* Only allow the setting of output flow control if it isn't set to
	 * something special
	 */
	flags = cp->fcotflg;
	if (( flags & FCOTFLG_MASK ) <= FCOTFLG_RESTART_ANY )
	{
	    flags &= ~FCOTFLG_MASK;
	    switch ( tp->t_iflag & ( IXON | IXANY ))
	    {
	    case 0:
	    case IXANY:
		break;

	    case IXON:
		flags |= FCOTFLG_STANDARD;
		break;

	    case IXON | IXANY:
		flags |= FCOTFLG_RESTART_ANY;
		break;
	    }
	    cp->fcrxon = tp->t_cc[ VSTART ];
	    cp->fcrxoff = tp->t_cc[ VSTOP ];
	}
	cp->fcotflg = flags;

	/* Backup the flags and mask the bits we don't want the host to process
	 */
	MINFO_FROM_TP ( tp )->m_oflag = tp->t_oflag;
	tp->t_oflag &= ~FW_OFLAG;
	if ( tp->t_oflag == OPOST )
	    tp->t_oflag = 0;

	MINFO_FROM_TP ( tp )->m_iflag = tp->t_iflag;
	tp->t_iflag &= ~FW_IFLAG;

	/* Tell the unit to make the change */
	cp->mdflags = MDFLAGS_IMMED;
}


/*
 *	Data transfer routines
 *	======================
 *
 *	Data flow control between the driver and firmware is handled as follows:
 *
 *	The driver hides all requests for flow control in the mode structures
 *	from the line discipline and passes them instead to the firmware.
 *	This reduces the work load on the line discipline and produces a quicker
 *	flow control response ( there is little point in the line discipline
 *	generating an XOFF if it is put on the end of a 256 character output
 *	buffer ).
 *
 *	On the transmit side data is put into the firmware buffer until it is
 *	full at which time a watermark is set requesting notification on a
 *	drain to 20% and further driver output is suspended. One problem with
 *	this is that there is possibly no way for the driver and line discipline
 *	to distinguish between suspension due to external flow control or simply
 *	because the buffer is full.
 *
 *	The receive side is harder. In a normal driver input received is always
 *	passed to the line discipline and if the line discipline buffers become
 *	full the data is dumped. Unfortunately the external firmware will only
 *	apply input flow control when it's internal buffers become full, if
 *	the firmware buffers are continually drained by the line discipline
 *	filling and dumping it's buffers input flow control will never be
 *	applied.
 *	One technique used to solve this problem is to use calls made by the
 *	line discipline to the driver when the input buffer become full to
 *	generate requests for firmware flow control. The BSD line discipline is
 *	not well adapted to this technique as it generates flow control requests
 *	directly rather than calling driver functions.
 *	The alternative technique used here is for the driver to stop copying
 *	data to the line discipline buffer when it approaches the flow control
 *	limit. This will cause the firmware buffer to fill and flow control to
 *	be applied.
 *	Two problems exist with this technique. Firstly if the line discipline
 *	input buffer should become drained for any reason the input interrupt
 *	chain must be restarted, unfortunately the line discipline has no
 *	equivalent of the t_start driver entry point to call in this case. Thus
 *	the driver must check the input buffer condition after every operation
 *	that may change the buffer state. That is after any read or ioctl.
 *	The second problem is that if flow control should fail for any reason
 *	the characteristics of firmware input buffer dumping would be different
 *	from those of line discipline buffering.
 */

/* Masks for buffer indexes. This assumes that the buffers are powers of 2 in
 * size, as do many of the algorithms that use them.
 */
#define TXBUF_MASK	( TXBUF_SIZE - 1 )
#define RXBUF_MASK	( RXBUF_SIZE - 1 )

/*
 *	Start or continue output of data
 *
 *	We assume we are interrupt protected and that the channel page and
 *	tty structures have already been loaded for us.
 */
static void
aim_do_tx ( tp, cp )
	struct tty *tp;
	volatile struct channel_page *cp;
{
	int	spaces;		/* Size of buffer */
	int	inp;		/* Copies of buffer indexes */
	int	outp;

	/* Don't bother if we are flagged as stopped */
	if ( tp->t_state & ( TS_TIMEOUT | TS_TTSTOP ))
	    return;

	/* Wake sleepers if we have passed the low watermark */
	if ( tp->t_outq.c_cc <= tp->t_lowat )
	{
	    if ( tp->t_state & TS_ASLEEP )
	    {
		tp->t_state &= ~TS_ASLEEP;
		wakeup ( (caddr_t)&tp->t_outq );
	    }

#if BSDI_RELEASE < 101
	    if ( tp->t_wsel )
	    {
		selwakeup ( tp->t_wsel, tp->t_state & TS_WCOLL );
		tp->t_wsel = 0;
		tp->t_state &= ~TS_WCOLL;
	    }
#else
	    selwakeup ( &tp->t_wsel );
#endif
	}

	/* Give up if nothing left to output */
	/* TS_BUSY and TS_FLUSH will be cleared in the interrupt service
	 * routine when the firmware buffer drains to zero.
	 */
	if ( tp->t_outq.c_cc == 0 )
	{
	    /* Clear the flags if firmware buffer is already drained as we won't
	     * be getting an interrupt.
	     */
	    if ( cp->txwater == 0 )
		tp->t_state &= ~ ( TS_BUSY | TS_FLUSH );
	    return;
	}

	/* Try to copy as many characters as possible to the cards output
	 * buffer. Buffer capacity is a maximum of TXBUF_SIZE - 1 characters
	 * or the pointers would wrap back to zero.
	 */
	tp->t_state |= TS_BUSY;
	inp = cp->txinp;
	outp = cp->txoutp;
	spaces = TXBUF_SIZE - 1 - (( inp - outp ) & TXBUF_MASK );
	dbg ( DBG_OUTPUT,("do_tx: C=%d S=%d\n", tp->t_outq.c_cc, spaces ));

	/* Two different algorithms as I am not sure about the portability of
	 * q_to_b ().
	 */
#ifdef USE_GETC
	while ( spaces-- > 0 && tp->t_outq.c_cc )
	    cp->txbuf[ inp++ & TXBUF_MASK ] = getc ( &tp->t_outq );
#else
	if ( inp >= outp )
	{
	    int	chars;

	    /* Copy data up to end of buffer and wrap if necessary */
	    chars = q_to_b ( &tp->t_outq, (char *)&cp->txbuf[ inp ]
					, min ( TXBUF_SIZE - inp, spaces ));

	    inp += chars;
	    if ( inp >= TXBUF_SIZE )
		inp = 0;
	    spaces -= chars;
	}

	/* Copy characters up to the output index */
	if ( spaces > 0 && tp->t_outq.c_cc > 0 )
	    inp += q_to_b ( &tp->t_outq, (char *)&cp->txbuf[ inp ], spaces );
#endif

	/* Write back the output Q input pointer */
	cp->txinp = inp & TXBUF_MASK;

	/* If we still have output to send set a watermark at 20% ( it's as
	 * good a guess as any ) otherwise set watermark to wait for output
	 * ( including CD180 ) drain.
	 */
	if ( tp->t_outq.c_cc > 0 )
	    cp->txwater = TXBUF_SIZE / 5;
	else
	    cp->txwater = TXWATER_DRAIN;

	MINFO_FROM_TP ( tp )->minor_flags |= MINF_TXWATER;
}

/*
 *	Start or continue input of data
 *
 *	We assume we are interrupt protected and that the channel page and
 *	tty structures have already been loaded for us.
 */
static void
aim_do_rx ( tp, cp )
	struct tty *tp;
	volatile struct channel_page *cp;
{
	int	chars;		/* Characters available from firmware */
	int	spaces;		/* Spaces available in driver */
	int	inp;		/* Copies of buffer indexes */
	int	outp;
	int	c;		/* Build input character and flags here */

	/* Don't bother if the tty is closed. This is necessary as this may be
	 * a last interrupt after closure and the clist buffers have gone away.
	 * I know that ttyinput() also checks this, but this way we don't set
	 * a watermark and the interrupt chain dies.
	 * XXX Even when the interrupt chain is turned off input will still be
	 *     accumulated in the firmware buffers. Perhaps we should do
	 *     something about this ?
	 *     Input is now flushed on the first open of a port. This is at
	 *     least a partial fix, although some devices may be puzzled by
	 *     being flow controlled by an otherwise closed port.
	 */
	if ( ! ( tp->t_state & TS_ISOPEN ))
	{
	    /* Flush the input buffer */
	    cp->rxoutp = cp->rxinp;

	    return;
	}

	dbg ( DBG_INPUT,("do_rx:"));

	/* Try to copy as many characters as possible to the line discipline
	 * input buffer. It is important not to overflow line discipline input
	 * capacity or it will cause a flush.
	 * XXX Curiously input overflow flushes both input and output buffers ?
	 */
	inp = cp->rxinp;
	outp = cp->rxoutp;
	chars = ( inp - outp ) & RXBUF_MASK;

	while ( chars > 0 )
	{
	    /* Calc how much room we have */
	    spaces = TTYHOG - 1 - tp->t_rawq.c_cc - tp->t_canq.c_cc;
	    dbg ( DBG_INPUT,(" C=%d S=%d", chars, spaces ));
	    spaces = min ( spaces, chars );

	    if ( spaces <= 0 )
		break;

	    /* Copy the characters */
	    while ( spaces-- > 0 )
	    {
		c = cp->rxbuf[ outp & RXBUF_MASK ] & TTY_CHARMASK;
		if ( cp->rxlbuf[ outp & RXBUF_MASK ])
		{
		    /* Add in exception flags */
		    if ( cp->rxlbuf[ outp & RXBUF_MASK ]
					& ( LNSTATE_FRAME | LNSTATE_BREAK ))
		    {
			c |= TTY_FE;
		    }
		    else if ( cp->rxlbuf[ outp & RXBUF_MASK ] & LNSTATE_PARITY )
		    {
			c |= TTY_PE;
		    }
		}
		++outp;

		(*linesw[ tp->t_line ].l_rint)( c, tp );
	    }

	    chars = ( inp - outp ) & RXBUF_MASK;
	}

	/* Write back the input Q output pointer */
	cp->rxoutp = outp & RXBUF_MASK;

	/* If we still have characters available then we ran out of space and
	 * needn't set a watermark. Otherwise ask for a report on the first
	 * character available.
	 * XXX Could possibly do some tuning with VMIN/VTIME here but the
	 *     current situation isn't as bad as it looks thanks to the cards
	 *     10mS interrupt holdoff which will tend to aggregate characters
	 *     when the lines are busy.
	 */
	if ( chars <= 0 )
	{
	    dbg ( DBG_INPUT,(" rxwater"));
	    cp->rxwater = 1;
	    MINFO_FROM_TP ( tp )->minor_flags |= MINF_IRUNNING;
	}
	dbg ( DBG_INPUT,("\n"));
}

/*
 *	Start output of data ( called from line discipline )
 */
static void
aimstart ( tp )
	struct tty *tp;
{
	volatile struct channel_page *cp;


#ifdef TS_XON_PEND
	/* Input flow control is handled by the external unit so just clear
	 * the pending flags to keep the line-discipline happy.
	 * XXX Is there any way we could interpret this as enabling robust
	 *     flow control ?
	 */
	tp->t_state &= ~ ( TS_XON_PEND | TS_XOFF_PEND );
#endif

	cp = cp_from_dev ( tp->t_dev );

	PROTECT
	    /* Restart firmware output if stopped */
	    if ( DEV_FLAGS ( tp->t_dev ) & MINF_OSTOP )
	    {
		cp->txflags |= TXFLAGS_RESUME;
		DEV_FLAGS ( tp->t_dev ) &= ~MINF_OSTOP;
	    }

	    /* Stuff chars in the buffer */
	    aim_do_tx ( tp, cp );
	ALLOW

	/* Pet peeve. Header files declare this func as returning int when
	 * usage always ignores the value, so it should return void.
	 * FIXED in 2.0 :-)
	 */
}

/*
 *	Stop output of data or flush it ( called from line discipline )
 */
static void
aimstop ( tp, rw_flag )
	struct tty *tp;
	int	rw_flag;
{
	volatile struct channel_page *cp;

	cp = cp_from_dev ( tp->t_dev );

	PROTECT
	    if ( tp->t_state & TS_BUSY )
	    {
		if (( tp->t_state & TS_TTSTOP ) == 0 )
		{
		    /* Tell firmware to flush output */
		    cp->txnewop = cp->txinp;
		    cp->txflags |= TXFLAGS_FLUSH;
		    /* Record flush in progress */
		    tp->t_state |= TS_FLUSH;
		}
	    }
	ALLOW
}

/*
 *	Standard driver entry points
 *	============================
 *
 *	Open a device
 */
int
aimopen ( dev, flags, mode, p )
	dev_t dev;
	int flags;
	int mode;
	struct proc *p;
{
	register struct tty *tp;
	volatile struct channel_page *cp;
	int error;


	dbg ( DBG_OPEN, ("aimopen:"));
	if ( error = wait_dev_assignment ( dev, flags ))
	    return error;
	dbg ( DBG_OPEN, (" ASSIGNED"));

	/* Device is legal and has been assigned an external port */

	tp = TTY_FROM_DEV ( dev );

	dbg ( DBG_OPEN, (" state=%x", tp->t_state ));
	/* If this is a first open set some defaults */
	if ( ! ( tp->t_state & TS_ISOPEN ))
	{
	    /* First perform a flush of the IOPRO's buffers  */
	    cp = cp_from_dev ( dev );
	    /* Flush the input buffer */
	    cp->rxoutp = cp->rxinp;
	    /* Tell firmware to flush output */
	    cp->txnewop = cp->txinp;
	    cp->txflags |= TXFLAGS_FLUSH;

	    /* Entry hooks for kernel functions */
	    tp->t_oproc = aimstart;
	    tp->t_stop = aimstop;
	    tp->t_param = aimparam;
	    tp->t_dev = dev;

	    /* Default the line mode unless it has already been set */
	    if (tp->t_ispeed == 0)
		tp->t_termios = deftermios;
	    else
		ttychars(tp);

	    aimparam2 ( tp );
	    aimparam ( tp, &tp->t_termios );
	    ttsetwater ( tp );
	}
	else
	{
	    /* Not first open check exclusive use */
	    /* XXX is there a race condition here with another process sitting
	     *     in the wait for DCD loop ?
	     */
	    if (( tp->t_state & TS_XCLUDE ) && ! issuser ( p ))
		return EBUSY;
	}

	PROTECT
	    /* Raise modem control lines and wait for DCD if required */
	    dbg ( DBG_OPEN, (" RAISE"));
	    aim_modem_ctl ( dev, PHOUTLN_DTR_ON | PHOUTLN_RTS_ON, FALSE );

	    /* Start the DCD monitor ( Will run until the line is closed ) */
	    aim_monitor ( dev, LNSTATE_NO_DCD );

	    /* Check current DCD state */
	    if ( aim_modem_status ( dev ) & TIOCM_CD )
		tp->t_state |= TS_CARR_ON;

	    dbg ( DBG_OPEN, (" state=%x", tp->t_state ));

	    /* Wait for DCD */
	    if ( ! ( flags & O_NONBLOCK ))
	    {
		while ( ! ( tp->t_cflag & CLOCAL )
					&& ! ( tp->t_state & TS_CARR_ON ))
		{
		    dbg ( DBG_OPEN, (" WAIT"));
		    tp->t_state |= TS_WOPEN;

		    error = ttysleep ( tp, (caddr_t)&tp->t_rawq, TTIPRI | PCATCH
								, ttopen, 0 );

		    /* If we've been interrupted drop modem control and quit */
		    if ( error )
		    {
			aim_modem_ctl ( dev, PHOUTLN_DTR_OFF | PHOUTLN_RTS_OFF
								, TRUE );
			break;
		    }
		}
	    }

	    if ( ! error )
		error = (*linesw[ tp->t_line ].l_open)( dev, tp );

	    /* Start input interrupt chain if not already running */
	    if ( ! ( DEV_FLAGS ( dev ) & MINF_IRUNNING ))
		aim_do_rx ( tp, cp_from_dev ( dev ));
	ALLOW


	dbg ( DBG_OPEN, (" DONE\n"));
	return error;
}

/*
 *	Close a device
 */
int
aimclose ( dev, flags, mode, p )
	dev_t dev;
	int flags;
	int mode;
	struct proc *p;
{
	register struct tty *tp;
	volatile struct channel_page *cp;


	dbg ( DBG_OPEN,("aimclose\n"));

	tp = TTY_FROM_DEV ( dev );

	PROTECT
	    (*linesw[ tp->t_line ].l_close)( tp, flags );

	    /* Clear any break condition left lying about */
	    cp = cp_from_dev ( dev );
	    cp->phfrmt &= ~PHFRMT_BREAK;
	    cp->mdflags = MDFLAGS_IMMED;

	    /* Drop modem control lines if required */
	    /* XXX I don't understand the checks (taken direct from com.c).
	     *     Isn't this the final close ? and why do we force dropping
	     *     the line if someone is waiting for open ?
	     */
	    if (( tp->t_cflag & HUPCL ) || ( tp->t_state & TS_WOPEN )
					|| ! ( tp->t_state & TS_ISOPEN ))
	    {
		aim_modem_ctl ( dev, PHOUTLN_DTR_OFF | PHOUTLN_RTS_OFF, TRUE );
	    }

	    /* Stop monitoring modem control lines */
	    aim_monitor ( dev, 0 );

	    ttyclose ( tp );
	ALLOW

	return ENOERROR;
}

/*
 *	Read data
 */
int
aimread ( dev, uio, flags )
	dev_t dev;
	struct uio *uio;
	int flags;
{
	register struct tty *tp;
	int error;

	/* Abort the read if the external hardware has been powered down */
	if ( ! DEV_IS_ASSIGNED ( dev ))
	    return EIO;

	tp = TTY_FROM_DEV ( dev );

	/* Start input interrupt chain if not already running */
	/* Note we do this in two places. Before the read processing in case
	 * we have no input ready and after in case we were blocked waiting for
	 * space.
	 */
	PROTECT
	    if ( ! ( DEV_FLAGS ( dev ) & MINF_IRUNNING ))
		aim_do_rx ( tp, cp_from_dev ( dev ));
	ALLOW

	/* Normal read processing */
	error = (*linesw[ tp->t_line ].l_read)( tp, uio, flags );

	/* Start input interrupt chain if not still running */
	PROTECT
	    if ( ! ( DEV_FLAGS ( dev ) & MINF_IRUNNING ))
		aim_do_rx ( tp, cp_from_dev ( dev ));
	ALLOW

	return error;
}

/*
 *	Write data
 */
int
aimwrite ( dev, uio, flags )
	dev_t dev;
	struct uio *uio;
	int flags;
{
	register struct tty *tp;

	/* Abort the write if the external hardware has been powered down */
	if ( ! DEV_IS_ASSIGNED ( dev ))
	    return EIO;

	/* Normal write processing */
	tp = TTY_FROM_DEV ( dev );

	return (*linesw[ tp->t_line ].l_write)( tp, uio, flags );
}

#if BSDI_RELEASE >= 101
/*
 *	1.1 introduced the select entry to the driver.
 *	XXX I'm flying blind here not being privy to the contents of tty.c
 *		thanks a lot USL :-(
 */
int
aimselect ( dev, flag, p )
	dev_t dev;
	int flag;
	struct proc *p;
{
	register struct tty *tp;

	/* Abort the select if the external hardware has been powered down */
	/* XXX I don't know if EIO is a good value to return, I don't even
	 *	know whether this should be an errno.
	 */
	if ( ! DEV_IS_ASSIGNED ( dev ))
	    return EIO;

	/* Normal select processing */
	tp = TTY_FROM_DEV ( dev );

	return ttyselect ( tp, flag, p );
}
#endif


/*
 *	Mode changing and special controls
 *
 *	Check if iflag or oflag have changed and call aimparam2() if necessary.
 *	aimparam() is automatically called by the line discipline if cflags
 *	changes although we may still need to apply the firmware mask if it
 *	dosn't.
 */
#define check_param2_flags()	{					\
	    if (( tp->t_iflag & FW_IFLAG ) != mp->m_iflag		\
	    		|| ( tp->t_oflag & FW_OFLAG ) != mp->m_oflag )	\
	    {								\
	    	aimparam2 ( tp );					\
	    }								\
	    else							\
	    {								\
	    	tp->t_iflag &= ~FW_IFLAG;				\
	    	tp->t_oflag &= ~FW_OFLAG;				\
	    }								\
	    tp->t_cflag &= ~FW_CFLAG;					\
	}

/* 2.0 introduced and additional parameter to pass through */
#if BSDI_RELEASE >= 200
# define PROC	, procp
#else
# define PROC
#endif

int
aimioctl ( dev, cmd, data, flag PROC )
	dev_t dev;
	int cmd;
	caddr_t data;
	int flag;
#if BSDI_RELEASE >= 200
	struct proc *procp;
#endif
{
	register struct tty *tp;
	register int error;
	volatile struct channel_page *cp;
	struct minor_info *mp;

	/* Abort if the external hardware has been powered down */
	if ( ! DEV_IS_ASSIGNED ( dev ))
	    return EIO;

	tp = TTY_FROM_DEV ( dev );
	mp = &minor_info[ minor( dev )];

	/* Restore "hidden" flags */
	tp->t_iflag = mp->m_iflag;
	tp->t_oflag = mp->m_oflag;
	tp->t_cflag = mp->m_cflag;

	/* If the ioctl is an input poll make sure the input buffer is as full
	 * as possible before we process it.
	 */
	if ( cmd == FIONREAD )
	{
	    PROTECT
		/* We don't condition this call with MINF_IRUNNING as in so
		 * many other cases because we want to include characters
		 * that may be delayed in the cards buffer due to the
		 * interrupt holdoff.
		 */
		aim_do_rx ( tp, cp_from_dev ( dev ));
	    ALLOW
	}

	/* Start with normal ioctl processing */
	error = (*linesw[ tp->t_line ].l_ioctl)( tp, cmd, data, flag PROC );
	if ( error >= 0 )
	{
	    check_param2_flags ();

	    return error;
	}

	error = ttioctl ( tp, cmd, data, flag PROC );
	check_param2_flags ();
	if ( error >= 0 )
	{

	    /* Several of the standard ioctls may drain or flush the input
	     * buffer requiring us to restart the input chain.
	     */
	    if ( cmd == TIOCSETD || cmd == TIOCSETAF || cmd == TIOCFLUSH )
	    {
		PROTECT
		    if ( ! ( DEV_FLAGS ( dev ) & MINF_IRUNNING ))
			aim_do_rx ( tp, cp_from_dev ( dev ));
		ALLOW
	    }
	    return error;
	}

	/* Now do the hardware ops and special checks */
	cp = cp_from_dev ( tp->t_dev );

	switch ( cmd )
	{
	case TIOCSBRK:
	    /* Start a break */
	    cp->phfrmt |= PHFRMT_BREAK;
	    cp->mdflags = MDFLAGS_IMMED;
	    break;

	case TIOCCBRK:
	    /* End a break */
	    cp->phfrmt &= ~PHFRMT_BREAK;
	    cp->mdflags = MDFLAGS_IMMED;
	    break;

	case TIOCSDTR:
	    /* Set DTR ( we hit RTS also ) */
	    aim_modem_ctl ( dev, PHOUTLN_DTR_ON | PHOUTLN_RTS_ON, FALSE );
	    break;

	case TIOCCDTR:
	    /* Clear DTR ( and RTS ) */
	    aim_modem_ctl ( dev, PHOUTLN_DTR_OFF | PHOUTLN_RTS_OFF, FALSE );
	    break;

	case TIOCMSET:
	    /* Set modem control bits directly */
	    aim_modem_ctl ( dev, tioc2phoutln ( *(int *)data ), FALSE );
	    break;

	case TIOCMBIS:
	    /* Or in modem control bits */
	    aim_modem_ctl ( dev, cp->phoutln | tioc2phoutln ( *(int *)data )
	    							, FALSE );
	    break;

	case TIOCMBIC:
	    /* And out modem control bits */
	    aim_modem_ctl ( dev, cp->phoutln & ~tioc2phoutln ( *(int *)data )
	    							, FALSE );
	    break;

	case TIOCMGET:
	    /* Get the current modem status */
	    *(int *)data = aim_modem_status ( dev );
	    break;

	default:
	    return ENOTTY;
	}

	return ENOERROR;
}

#undef check_param2_flags


/*
 *	The interrupt service routines
 *	==============================
 *
 *	Service a single channel (page)
 *	This routine is not a direct device driver entry point. It is called
 *	from the control driver interrupt service routine aimcintr().
 */
void
aimdata_service ( sc, unit, page )
	struct aimc_softc *sc;
	Ubyte unit;
	Ubyte page;
{
	volatile struct channel_page *cp;
	struct tty *tp;
	int minor_num;

	/* Bring channel page into view */
	PAGE_SEL ( sc, unit, page );
	cp = &sc->card_mem->chp;

	/* Grab back pointer and find OS data */
	minor_num = cp->osminor;

	/* Check minor number and get tty structure */
	if ( DEV_IS_LEGAL ( minor_num ) && DEV_IS_ASSIGNED ( minor_num ))
	    tp = TTY_FROM_DEV ( minor_num );
	else
	{
	    printf ("aim%d: Interrupt from unassigned page %d unit %d\n"
						, sc->card_num, page, unit );
	    return;
	}

	/* Look for sources of interrupt and process */

	/* Modem status change ? */
	if ( cp->lndelta != 0 )
	{
	    if ( cp->lndelta & LNSTATE_NO_DCD )
	    {
		/* Zero delta before we check state or we might lose one */
		cp->lndelta = 0;

		/* Report change to DCD */
	        if ( cp->lnstate & LNSTATE_NO_DCD )
	            (void)(*linesw[ tp->t_line ].l_modem)( tp, 0 );
	        else
	            (void)(*linesw[ tp->t_line ].l_modem )( tp, 1 );
	    }
	    else
	    	cp->lndelta = 0;
	}

	/* Output watermark ? */
	if (( DEV_FLAGS ( minor_num ) & MINF_TXWATER ) && cp->txwater == 0 )
	{
	    DEV_FLAGS ( minor_num ) &= ~MINF_TXWATER;

	    /* If we have got characters to output stuff the buffer */
	    if ( tp->t_outq.c_cc > 0 )
		aim_do_tx ( tp, cp );
	    else
	    {
		/* No output to send so last watermark set was a wait for drain
		 * so we can now safely clear the busy bits and recall the line
		 * discipline.
		 * XXX I think there may be a window here associated with output
		 *     flushing.
		 */
		tp->t_state &= ~( TS_BUSY | TS_FLUSH );
		(*linesw[tp->t_line].l_start)( tp );
	    }
	}

	/* Input watermark ? */
	if (( DEV_FLAGS ( minor_num ) & MINF_IRUNNING ) && cp->rxwater == 0 )
	{
	    DEV_FLAGS ( minor_num ) &= ~MINF_IRUNNING;
	    aim_do_rx ( tp, cp );
	}
}

/*
 *	The device switch entry
 *	Placed here because I hate forward decls
 */
#if BSDI_RELEASE >= 200

struct devsw aimsw = {
	&aimcd,
	aimopen, aimclose, aimread, aimwrite, aimioctl, aimselect, nommap,
	nostrat, nodump, nopsize, 0,
	aimstop
};
#endif

#endif
