/*
 * Copyright (c) 1997, 1998, 1999
 *	Bill Paul <wpaul@ctr.columbia.edu>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Bill Paul.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY Bill Paul AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL Bill Paul OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Modified for BSD/OS by Geert Jan de Groot 
 *
 *	Id: if_wi.c,v 1.7 1999/07/04 14:40:22 wpaul Exp 
 *	BSDI if_wi.c,v 2.13.4.3 2001/09/19 00:58:19 geertj Exp
 */

/*
 * Lucent WaveLAN/IEEE 802.11 PCMCIA driver for FreeBSD.
 *
 * Written by Bill Paul <wpaul@ctr.columbia.edu>
 * Electrical Engineering Department
 * Columbia University, New York City
 */

/*
 * The WaveLAN/IEEE adapter is the second generation of the WaveLAN
 * from Lucent. Unlike the older cards, the new ones are programmed
 * entirely via a firmware-driven controller called the Hermes.
 * Unfortunately, Lucent will not release the Hermes programming manual
 * without an NDA (if at all). What they do release is an API library
 * called the HCF (Hardware Control Functions) which is supposed to
 * do the device-specific operations of a device driver for you. 
 *
 * This driver does not use the HCF or HCF Light at all. Instead, it
 * programs the Hermes controller directly, using information gleaned
 * from the HCF Light code and corresponding documentation.
 *
 * This driver supports both the PCMCIA and ISA versions of the
 * WaveLAN/IEEE cards. Note however that the ISA card isn't really
 * anything of the sort: it's actually a PCMCIA bridge adapter
 * that fits into an ISA slot, into which a PCMCIA WaveLAN card is
 * inserted. Consequently, you need to use the pccard support for
 * both the ISA and PCMCIA adapters.
 *
 * BUGS: as of firmware 2.0, multicast filtering doesn't seem to
 * be working as all multicast packets get thru.
 */

#include "bpfilter.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/sockio.h>
#include <sys/mbuf.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/socket.h>
#include <sys/device.h>
#include <sys/proc.h>

#include <net/if.h>
#include <net/if_arp.h>
#include <net/if_dl.h>
#include <net/if_media.h>
#include <net/if_types.h>
#include <net/if_802_11.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#include <net/bpf.h>
#include <net/bpfdesc.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>
#include <machine/cpu.h>
#include "cs.h"
#if NCS > 0
#include <i386/pcmcia/pccs_meta.h>
#include <i386/pcmcia/cs_handle.h>
#include <i386/pcmcia/cs_irq.h>
#include <i386/pcmcia/cs_cisutil.h>
#include <i386/pcmcia/cs_errno.h>
#include <i386/pcmcia/cs_memrw.h>
#include <i386/pcmcia/cs_tplsvc.h>
#include <i386/pcmcia/cs_event.h>
#include <i386/pcmcia/cs_iop.h>
#include <i386/pcmcia/cs_rio.h>
#include <i386/pcmcia/cs_cli.h>
#include <i386/pcmcia/cs_conf.h>
#define PCCARD				/* PCMCIA support */
#else
#error if_wi needs pcmcia support 
#endif
#include <i386/isa/if_wireg.h>

struct wi_softc	{
	struct device		wi_dev;		/* base device (must be 1st) */
	struct isadev		wi_id;		/* ISA device */
	struct intrhand		wi_ih;		/* interrupt vectoring */
	short			wi_base;	/* ISA base address */
	short			wi_irq;		/* irq */
	struct arpcom		wi_ac;
#define wi_if	wi_ac.ac_if			/* network-visible interface */
#define wi_addr wi_ac.ac_enaddr			/* hardware address */

	struct	ifmedia 	wi_media;	/* media management */

	int			wi_tx_data_id;
	int			wi_tx_mgmt_id;
	int			wi_portnum;
	int			wi_max_data_len;
	int			wi_rts_thresh;
	int			wi_ap_density;
	int			wi_create_ibss;
	int			wi_chanmask;
	int			wi_channel;
	int			wi_pm_enabled;
	int			wi_max_sleep;
	char			wi_node_name[WI_MAXNAMELEN + 1];
	char			wi_net_name[WI_MAXNAMELEN + 1];
	char			wi_ibss_name[WI_MAXNAMELEN + 1];

	int			wi_haswep;
	int			wi_wepenable;
	int			wi_weptxkey;
	struct wi_key		wi_wepkey[IEEE_802_11_NWEPKEYS];

	u_int8_t 		wi_txbuf[ETHERMTU + 
				    IEEE_802_11_SNAPHDR_LEN + 2];
	struct wi_ltv_macaddr	wi_mac;

	struct wi_counters	wi_stats;

	int			wi_tx_timeout;
	int			wi_pm_timeout;
	int			wi_flags;

	struct wi_sigcache 	wi_sigcache[MAXWICACHE];
	int			wi_sigcacheslot;

	/* PCMCIA specific stuff */
	int			wi_ispccard;
	int			wi_socket;
	CliHandle_t 		wi_clihdl;
	int			wi_cfstat;
	cs_rio_t		wi_rio;
	cs_rirq_t 		wi_rirq;
	cs_cfg_t		wi_rcfg;
	struct tpcc 		wi_tpcc;
};

#define	WI_FL_PROMISC	0x0001		/* switched in promisc mode */


static int wi_probe 		__P((struct device *, struct cfdata *, void *));
static void wi_attach 		__P((struct device *, struct device *, void *));
struct mtdreq;
static int wi_cse_handler	__P((void *, int, int, int, struct mtdreq *,
				    char *, int));
static int wi_cc_probe		__P((struct wi_softc *, int));
static int wi_cc_attach		__P((struct wi_softc *, int));
static int wi_cc_detach		__P((struct wi_softc *, int));
static int wi_intr		__P((void *));
static void wi_reset		__P((struct wi_softc *));
static int wi_ioctl		__P((struct ifnet *, u_long, caddr_t));
static void wi_init		__P((void *));
static void wi_start		__P((struct ifnet *));
static void wi_stop		__P((struct wi_softc *));
static void wi_watchdog		__P((struct ifnet *));
static void wi_rxeof		__P((struct wi_softc *));
static void wi_txeof		__P((struct wi_softc *, int));
static void wi_update_stats	__P((struct wi_softc *));
static void wi_setmulti		__P((struct wi_softc *));

static int wi_cmd		__P((struct wi_softc *, int, int));
static int wi_read_record	__P((struct wi_softc *, struct wi_ltv *));
static int wi_read_int	__P((struct wi_softc *, u_int16_t rid, int *));
static int wi_read_str	__P((struct wi_softc *, u_int16_t rid, char *));
static int wi_write_record	__P((struct wi_softc *, struct wi_ltv *));
static int wi_write_int	__P((struct wi_softc *, u_int16_t rid, int));
static int wi_write_str	__P((struct wi_softc *, u_int16_t rid, char *));
static int wi_read_data		__P((struct wi_softc *, int,
					int, caddr_t, int));
static int wi_write_data	__P((struct wi_softc *, int,
					int, caddr_t, int));
static int wi_seek		__P((struct wi_softc *, int, int, int));
static int wi_alloc_nicmem	__P((struct wi_softc *, int, int *));
static int wi_setdef		__P((struct wi_softc *, struct wi_ltv *));
static int wi_mediachg		__P((struct ifnet *));
static void wi_mediastat	__P((struct ifnet *, 
				    struct ifmediareq *));
static int wi_sysctl		__P((struct ifnet *, int *, u_int, void *,
				    size_t *, void *, size_t));
static void wi_cache_store	__P((struct wi_softc *, u_int8_t *,
				    int signal, int noise));


struct cfdriver wicd = {
        NULL, "wi", wi_probe, wi_attach, DV_IFNET, sizeof(struct wi_softc)
};

/* PCMCIA debugging, based on printfs */
#define WI_CC_DEBUG
#ifdef WI_CC_DEBUG
int iwl_cc_debug = 0;
#define ccdprintf(x)	{ if (iwl_cc_debug) printf x; }
#else
#define ccdprintf(x)
#endif

/* Register space access macros */
#define CSR_WRITE_4(sc, reg, val)	outl((sc->wi_base) + (reg), (val))
#define CSR_WRITE_2(sc, reg, val)	outw((sc->wi_base) + (reg), (val))
#define CSR_WRITE_1(sc, reg, val)	outb((sc->wi_base) + (reg), (val))

#define CSR_READ_4(sc, reg)		inl((sc->wi_base) + (reg))
#define CSR_READ_2(sc, reg)		inw((sc->wi_base) + (reg))
#define CSR_READ_1(sc, reg)		inb((sc->wi_base) + (reg))

#define WI_DEFAULT_PORT	(WI_PORT0 << 8)	/* only 0 exists on stations */
#define WI_DEFAULT_NETNAME	""	/* none */
#define WI_DEFAULT_AP_DENSITY	1
#define WI_DEFAULT_RTS_THRESH	2347
#define WI_DEFAULT_DATALEN	2304
#define WI_DEFAULT_CREATE_IBSS	0
#define WI_DEFAULT_PM_ENABLED	0
#define WI_DEFAULT_MAX_SLEEP	100
#define WI_DEFAULT_NODENAME	"WaveLAN/IEEE node"
#define WI_DEFAULT_IBSS		"IBSS"
#define WI_DEFAULT_CHAN		3
#define WI_DEFAULT_WEPENABLE	0 /* XXX can't really set default keys */
#define WI_DEFAULT_WEPTXKEY	0

#define WI_HERMES_AUTOINC_WAR	/* Work around data write autoinc bug. */
#define WI_HERMES_STATS_WAR	/* Work around stats counter bug. */

/*
 * Probe routine
 */
static int
wi_probe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;

#ifdef PCCARD
	if (ia->ia_bustype == BUS_PCMCIA) {
		/*
		 * Always probe true here, the real work is done when the
		 * card is actually inserted
		 */
		ia->ia_irq = IRQNONE;
		ia->ia_aux = cf;
		return (1);
	}
#endif

	return (0);
}


/*
 * Interface exists: make available by filling in network interface
 * record.  System will initialize the interface when it is ready
 * to accept packets.  We get the ethernet address here.
 */
static void
wi_attach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct wi_softc *sc = (struct wi_softc *) self;
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	struct ifnet *ifp = &sc->wi_if;
	cs_rclient_t rcli;

#ifdef WI_CC_DEBUG
	if (getdevconf(self->dv_cfdata, NULL, -1) & 0x1)
		iwl_cc_debug++;
#endif

	if (ia->ia_bustype != BUS_PCMCIA) 
		panic("wi_attach");

	/*
	 * Initialize interface structure
	 */
	ifp->if_unit = sc->wi_dev.dv_unit;
	ifp->if_name = wicd.cd_name;
	ifp->if_mtu = ETHERMTU;
	ifp->if_flags = IFF_BROADCAST | IFF_MULTICAST | IFF_SIMPLEX;
	ifp->if_init = wi_init;
	ifp->if_start = wi_start;
	ifp->if_ioctl = wi_ioctl;
	ifp->if_watchdog = wi_watchdog;
	ifp->if_sysctl = wi_sysctl;
	ifp->if_softc = sc;
	ifp->if_snd.ifq_maxlen = IFQ_MAXLEN;
	ieee_802_11_attach(ifp);
	/* XXX how can we tell a normal from a turbo card? */
	ifp->if_baudrate = 0;

	bpfattach(&ifp->if_bpf, ifp, DLT_IEEE802_11, 
	    sizeof(struct ieee_802_11_header));

	bzero(sc->wi_addr, sizeof(sc->wi_addr));
	sc->wi_ispccard = 1;
	sc->wi_socket = -1;

	sc->wi_chanmask = -1;

	sc->wi_ih.ih_fun = wi_intr;
	sc->wi_ih.ih_arg = (void *)sc;
	sc->wi_ih.ih_flags = IH_APICLEVEL;

	/*
	 * Initialize as card client
	 */
	rcli.Attributes = CSCLI_IO;
	rcli.EventMask = ~CSEM_CD_CHANGE;
	rcli.CallBack = wi_cse_handler;
	rcli.CallBackArg = sc;

	cs_RegisterClient(&sc->wi_clihdl, &rcli);

	isa_establish(&sc->wi_id, &sc->wi_dev);

	printf(": PCCARD IEEE 802.11 WaveLAN\n");

	ifmedia_init(&sc->wi_media, IFM_802_11_IBSS,
	    wi_mediachg, wi_mediastat);

#ifdef XXX /* XXX should be revamped */
	ifmedia_add(&sc->wi_media, 
	    IFM_802_11 | IFM_802_11_1, WI_TXRATE_1MBPS, NULL);
	ifmedia_add(&sc->wi_media, 
	    IFM_802_11 | IFM_802_11_2, WI_TXRATE_2MBPS, NULL);
	ifmedia_add(&sc->wi_media, 
	    IFM_802_11 | IFM_802_11_5, WI_TXRATE_5_5MBPS, NULL);
	ifmedia_add(&sc->wi_media, 
	    IFM_802_11 | IFM_802_11_11, WI_TXRATE_11MBPS, NULL);
#endif
	ifmedia_add(&sc->wi_media, 
	    IFM_802_11 | IFM_AUTO, WI_TXRATE_AUTO , NULL);

	ifmedia_set(&sc->wi_media, 
	    IFM_802_11 | IFM_AUTO); 


	/*
	 * Set card defaults
	 */
	bzero(sc->wi_node_name, sizeof(sc->wi_node_name));
	bcopy(WI_DEFAULT_NODENAME, sc->wi_node_name,
	    sizeof(WI_DEFAULT_NODENAME) - 1);

	bzero(sc->wi_net_name, sizeof(sc->wi_net_name));
	bcopy(WI_DEFAULT_NETNAME, sc->wi_net_name,
	    sizeof(WI_DEFAULT_NETNAME) - 1);

	bzero(sc->wi_ibss_name, sizeof(sc->wi_ibss_name));
	bcopy(WI_DEFAULT_IBSS, sc->wi_ibss_name,
	    sizeof(WI_DEFAULT_IBSS) - 1);

	sc->wi_portnum = WI_DEFAULT_PORT;
	sc->wi_ap_density = WI_DEFAULT_AP_DENSITY;
	sc->wi_rts_thresh = WI_DEFAULT_RTS_THRESH;
	sc->wi_max_data_len = WI_DEFAULT_DATALEN;
	sc->wi_create_ibss = WI_DEFAULT_CREATE_IBSS;
	sc->wi_pm_enabled = WI_DEFAULT_PM_ENABLED;
	sc->wi_max_sleep = WI_DEFAULT_MAX_SLEEP;
	sc->wi_wepenable = WI_DEFAULT_WEPENABLE;
	sc->wi_weptxkey = WI_DEFAULT_WEPTXKEY;
	sc->wi_haswep = -1;

	return;
}

static void 
wi_rxeof(sc)
	struct wi_softc *sc;
{
	struct ifnet *ifp = &sc->wi_if;
	struct wi_header rx_header;
	struct mbuf *m;
	int signal, noise;
	int id;

	id = CSR_READ_2(sc, WI_RX_FID);

	/* First read in the frame header */
	if (wi_read_data(sc, id, 0, (caddr_t)&rx_header, 
	    sizeof(rx_header))) {
		ifp->if_ierrors++;
		return;
	}

	if (rx_header.wi_status & WI_STAT_ERRSTAT) {
		ifp->if_ierrors++;
		return;
	}

	MGETHDR(m, M_DONTWAIT, MT_DATA);
	if (m == NULL) {
		ifp->if_ierrors++;
		return;
	}
	MCLGET(m, M_DONTWAIT);
	if (!(m->m_flags & M_EXT)) {
		m_freem(m);
		ifp->if_ierrors++;
		return;
	}

	if((rx_header.wi_dat_len + sizeof(struct wi_header)) > MCLBYTES) {
		dvprintf(sc, "oversized packet received "
		    "(wi_dat_len=%d, wi_status=0x%x)\n", 
		    rx_header.wi_dat_len, rx_header.wi_status);
		m_freem(m);
		ifp->if_ierrors++;
		return;
	}

	bcopy(&rx_header.wi_802_11_header, 
	    mtod(m, struct ieee_802_11_header *),
	    sizeof(struct ieee_802_11_header));
	m->m_pkthdr.len = m->m_len = rx_header.wi_dat_len + 
	    sizeof(struct ieee_802_11_header);
	m->m_pkthdr.rcvif = ifp;

	if (wi_read_data(sc, id, sizeof(struct wi_header),
	    mtod(m, caddr_t) + sizeof(struct ieee_802_11_header), 
	     rx_header.wi_dat_len + 2)) {
		m_freem(m);
		ifp->if_ierrors++;
		return;
	}

	ifp->if_ipackets++;

	/* Update signal strength cache */
	signal = (rx_header.wi_q_info >> 8) & 0xff;
	noise = rx_header.wi_q_info & 0xff;
	wi_cache_store(sc, rx_header.wi_802_11_header.e11_addr2, 
	    signal, noise);

	/* Handle BPF listeners. */
	if (ifp->if_bpf) {
		bpf_mtap(ifp->if_bpf, m);

		if ((sc->wi_flags & WI_FL_PROMISC) &&
		    (bcmp(rx_header.wi_802_11_header.e11_addr1, 
		     sc->wi_addr, ETHER_ADDR_LEN)
		      && (rx_header.wi_802_11_header.e11_addr1[0] & 1) == 0)) {
			m_freem(m);
			return;
		}
	}

	/* Receive packet. */
	ieee_802_11_input(ifp, m);

	return;
}

static void 
wi_txeof(sc, status)
	struct wi_softc *sc;
	int status;
{
	struct ifnet *ifp = &sc->wi_if;

	sc->wi_tx_timeout = 0;
	ifp->if_flags &= ~IFF_OACTIVE;

	if (status & WI_EV_TX_EXC)
		ifp->if_oerrors++;
	else
		ifp->if_opackets++;

	return;
}

static void 
wi_update_stats(sc)
	struct wi_softc *sc;
{
	struct wi_ltv *ltvp;
	u_int16_t id;
	struct ifnet *ifp = &sc->wi_if;
	u_int32_t *ptr;
	int i;
	u_int16_t t;

	if ((ltvp = (struct wi_ltv *) malloc(sizeof(struct wi_ltv), 
	    M_TEMP, M_NOWAIT)) == NULL) { 
		dvprintf(sc, "update_stats: no memory\n");
		return;
	}

	id = CSR_READ_2(sc, WI_INFO_FID);

	wi_read_data(sc, id, 0, (char *)ltvp, 4);

	if (ltvp->wi_type != WI_INFO_COUNTERS)
		goto out;

	ptr = (u_int32_t *)&sc->wi_stats;

	for (i = 0; i < ltvp->wi_len - 1; i++) {
		t = CSR_READ_2(sc, WI_DATA1);
#ifdef WI_HERMES_STATS_WAR
		if (t > 0xF000)
			t = ~t & 0xFFFF;
#endif
		if (i >= (sizeof(sc->wi_stats) / 4) + 1)
			continue; /* drain */

		ptr[i] += t;
	}

	ifp->if_collisions = sc->wi_stats.wi_tx_single_retries +
	    sc->wi_stats.wi_tx_multi_retries +
	    sc->wi_stats.wi_tx_retry_limit;

out:
	free(ltvp, M_TEMP);
	return;
}

static int 
wi_intr(arg)
	void *arg;
{
	struct wi_softc *sc = arg;
	struct ifnet *ifp = &sc->wi_if;
	u_int16_t status;
	int rc = 0;

	if (sc->wi_ispccard && sc->wi_socket < 0) 
		return (0);

	if (!(ifp->if_flags & IFF_UP)) {
		CSR_WRITE_2(sc, WI_EVENT_ACK, 0xFFFF);
		CSR_WRITE_2(sc, WI_INT_EN, 0);
		return (1);
	}

	/* Disable interrupts. */
	CSR_WRITE_2(sc, WI_INT_EN, 0);

	status = CSR_READ_2(sc, WI_EVENT_STAT);
	CSR_WRITE_2(sc, WI_EVENT_ACK, ~WI_INTRS);

	if (status & WI_EV_RX) {
		wi_rxeof(sc);
		CSR_WRITE_2(sc, WI_EVENT_ACK, WI_EV_RX);
		rc = 1;
	}

	if (status & WI_EV_TX) {
		wi_txeof(sc, status);
		CSR_WRITE_2(sc, WI_EVENT_ACK, WI_EV_TX);
		rc = 1;
	}

	if (status & WI_EV_ALLOC) {
		int id;
		id = CSR_READ_2(sc, WI_ALLOC_FID);
		CSR_WRITE_2(sc, WI_EVENT_ACK, WI_EV_ALLOC);
		if (id == sc->wi_tx_data_id)
			wi_txeof(sc, status);
		rc = 1;
	}

	if (status & WI_EV_INFO) {
		wi_update_stats(sc);
		CSR_WRITE_2(sc, WI_EVENT_ACK, WI_EV_INFO);
		rc = 1;
	}

	if (status & WI_EV_TX_EXC) {
		wi_txeof(sc, status);
		CSR_WRITE_2(sc, WI_EVENT_ACK, WI_EV_TX_EXC);
		rc = 1;
	}

	if (status & WI_EV_INFO_DROP) {
		CSR_WRITE_2(sc, WI_EVENT_ACK, WI_EV_INFO_DROP);
		rc = 1;
	}

	/* Re-enable interrupts. */
	CSR_WRITE_2(sc, WI_INT_EN, WI_INTRS);

	if (ifp->if_snd.ifq_head != NULL) {
		wi_start(ifp);
		rc = 1;
	}

	return (rc);
}

static int 
wi_cmd(sc, cmd, val)
	struct wi_softc *sc;
	int cmd;
	int val;
{
	int i, s = 0;

	CSR_WRITE_2(sc, WI_PARAM0, val);
	CSR_WRITE_2(sc, WI_COMMAND, cmd);

	for (i = 0; i < WI_TIMEOUT; i++) {
		/*
		 * Wait for 'command complete' bit to be
		 * set in the event status register.
		 */
		s = CSR_READ_2(sc, WI_EVENT_STAT) & WI_EV_CMD;
		if (s) {
			/* Ack the event and read result code. */
			s = CSR_READ_2(sc, WI_STATUS);
			CSR_WRITE_2(sc, WI_EVENT_ACK, WI_EV_CMD);
#ifdef foo
			if ((s & WI_CMD_CODE_MASK) != (cmd & WI_CMD_CODE_MASK))
				return (EIO);
#endif
			if (s & WI_STAT_CMD_RESULT)
				return (EIO);
			break;
		}
	}

	if (i == WI_TIMEOUT)
		return (ETIMEDOUT);

	return (0);
}

static void 
wi_reset(sc)
	struct wi_softc *sc;
{
	if (wi_cmd(sc, WI_CMD_INI, 0))
		dvprintf(sc, "init failed\n");
	CSR_WRITE_2(sc, WI_INT_EN, 0);
	CSR_WRITE_2(sc, WI_EVENT_ACK, 0xFFFF);

	/* Calibrate timer. */
	wi_write_int(sc, WI_RID_TICK_TIME, 8);

	return;
}

/*
 * Read an LTV record from the NIC.
 */
static int 
wi_read_record(sc, ltvp)
	struct wi_softc *sc;
	struct wi_ltv *ltvp;
{
	u_int16_t *ptr;
	int i, len, code;

	/* Tell the NIC to enter record read mode. */
	if (wi_cmd(sc, WI_CMD_ACCESS|WI_ACCESS_READ, ltvp->wi_type))
		return (EIO);

	/* Seek to the record. */
	if (wi_seek(sc, ltvp->wi_type, 0, WI_BAP1))
		return (EIO);

	/*
	 * Read the length and record type and make sure they
	 * match what we expect (this verifies that we have enough
	 * room to hold all of the returned data).
	 */
	len = CSR_READ_2(sc, WI_DATA1);
	if (len > ltvp->wi_len)
		return (ENOSPC);
	code = CSR_READ_2(sc, WI_DATA1);
	if (code != ltvp->wi_type)
		return (EIO);

	ltvp->wi_len = len;
	ltvp->wi_type = code;

	/* Now read the data. */
	ptr = &ltvp->wi_val[0];
	for (i = 0; i < ltvp->wi_len - 1; i++)
		ptr[i] = CSR_READ_2(sc, WI_DATA1);

	return (0);
}

static int
wi_read_int(sc, rid, val)
	struct wi_softc *sc;
	u_int16_t rid;
	int *val;
{
	struct wi_ltv_int ltvi;
	int error = 0;

	ltvi.wi_type = rid;
	ltvi.wi_len = 2;
	if ((error = wi_read_record(sc, (struct wi_ltv *)&ltvi)) != 0)
		return (error);

	*val = ltvi.wi_val;

	return (0);
}

static int
wi_read_str(sc, rid, s)
	struct wi_softc *sc;
	u_int16_t rid;
	char *s;
{
	struct wi_ltv_str ltvs;
	int error = 0;

	ltvs.wi_len = (WI_MAXNAMELEN / 2) + 2;
	ltvs.wi_type = rid;
	if ((error = wi_read_record(sc, (struct wi_ltv *)&ltvs)) != 0)
		return (error);

	if (ltvs.wi_strlen > WI_MAXNAMELEN)
		panic("wi_read_str");

	bcopy(ltvs.wi_str, s, ltvs.wi_strlen);
	s[ltvs.wi_strlen] = 0;

	return (0);
}

/*
 * Write an LTV record to the NIC.
 */
static int 
wi_write_record(sc, ltvp)
	struct wi_softc *sc;
	struct wi_ltv *ltvp;
{
	u_int16_t *ptr;
	int i;

	if (wi_seek(sc, ltvp->wi_type, 0, WI_BAP1))
		return (EIO);

	CSR_WRITE_2(sc, WI_DATA1, ltvp->wi_len);
	CSR_WRITE_2(sc, WI_DATA1, ltvp->wi_type);

	ptr = &ltvp->wi_val[0];
	for (i = 0; i < ltvp->wi_len - 1; i++)
		CSR_WRITE_2(sc, WI_DATA1, ptr[i]);

	if (wi_cmd(sc, WI_CMD_ACCESS|WI_ACCESS_WRITE, ltvp->wi_type))
		return (EIO);

	return (0);
}

static int
wi_write_int(sc, rid, val)
	struct wi_softc *sc;
	u_int16_t rid;
	int val;
{
	struct wi_ltv_int ltvi;

	ltvi.wi_len = 2;
	ltvi.wi_type = rid;
	ltvi.wi_val = val;
	return (wi_write_record(sc, (struct wi_ltv *)&ltvi));
}


static int
wi_write_str(sc, rid, s)
	struct wi_softc *sc;
	u_int16_t rid;
	char *s;
{
	struct wi_ltv_str ltvs;
	int l;

	l = strlen(s);
	bzero(ltvs.wi_str, sizeof(ltvs.wi_str));

	ltvs.wi_len = ((l + 1) / 2) + 2;
	ltvs.wi_type = rid;
	ltvs.wi_strlen = l;
	bcopy(s, ltvs.wi_str, l);

	return (wi_write_record(sc, (struct wi_ltv *)&ltvs));
}


static int 
wi_seek(sc, id, off, chan)
	struct wi_softc *sc;
	int id, off, chan;
{
	int selreg, offreg;
	int i;

	switch (chan) {
	case WI_BAP0:
		selreg = WI_SEL0;
		offreg = WI_OFF0;
		break;
	case WI_BAP1:
		selreg = WI_SEL1;
		offreg = WI_OFF1;
		break;
	default:
		dvprintf(sc, "invalid data path: %x\n", chan);
		return (EIO);
	}

	CSR_WRITE_2(sc, selreg, id);
	CSR_WRITE_2(sc, offreg, off);

	for (i = 0; i < WI_TIMEOUT; i++) {
		if (!(CSR_READ_2(sc, offreg) & (WI_OFF_BUSY|WI_OFF_ERR)))
			break;
	}

	if (i == WI_TIMEOUT)
		return (ETIMEDOUT);

	return (0);
}

static int 
wi_read_data(sc, id, off, buf, len)
	struct wi_softc *sc;
	int id, off;
	caddr_t buf;
	int len;
{
	int i;
	u_int16_t *ptr;

	if (wi_seek(sc, id, off, WI_BAP1))
		return (EIO);

	ptr = (u_int16_t *)buf;
	for (i = 0; i < len / 2; i++)
		ptr[i] = CSR_READ_2(sc, WI_DATA1);

	return (0);
}

/*
 * According to the comments in the HCF Light code, there is a bug in
 * the Hermes (or possibly in certain Hermes firmware revisions) where
 * the chip's internal autoincrement counter gets thrown off during
 * data writes: the autoincrement is missed, causing one data word to
 * be overwritten and subsequent words to be written to the wrong memory
 * locations. The end result is that we could end up transmitting bogus
 * frames without realizing it. The workaround for this is to write a
 * couple of extra guard words after the end of the transfer, then
 * attempt to read then back. If we fail to locate the guard words where
 * we expect them, we preform the transfer over again.
 */
static int 
wi_write_data(sc, id, off, buf, len)
	struct wi_softc *sc;
	int id, off;
	caddr_t buf;
	int len;
{
	int i;
	u_int16_t *ptr;

#ifdef WI_HERMES_AUTOINC_WAR
again:
#endif

	if (wi_seek(sc, id, off, WI_BAP0))
		return (EIO);

	ptr = (u_int16_t *)buf;
	for (i = 0; i < (len / 2); i++)
		CSR_WRITE_2(sc, WI_DATA0, ptr[i]);

#ifdef WI_HERMES_AUTOINC_WAR
	CSR_WRITE_2(sc, WI_DATA0, 0x1234);
	CSR_WRITE_2(sc, WI_DATA0, 0x5678);

	if (wi_seek(sc, id, off + len, WI_BAP0))
		return (EIO);

	if (CSR_READ_2(sc, WI_DATA0) != 0x1234 ||
	    CSR_READ_2(sc, WI_DATA0) != 0x5678)
		goto again;
#endif

	return (0);
}

/*
 * Allocate a region of memory inside the NIC and zero
 * it out.
 */
static int 
wi_alloc_nicmem(sc, len, id)
	struct wi_softc *sc;
	int len;
	int *id;
{
	int i;

	if (wi_cmd(sc, WI_CMD_ALLOC_MEM, len)) {
		dvprintf(sc, "failed to allocate %d bytes on NIC\n", len);
		return (ENOMEM);
	}

	for (i = 0; i < WI_TIMEOUT; i++) {
		if (CSR_READ_2(sc, WI_EVENT_STAT) & WI_EV_ALLOC)
			break;
	}

	if (i == WI_TIMEOUT)
		return (ETIMEDOUT);

	CSR_WRITE_2(sc, WI_EVENT_ACK, WI_EV_ALLOC);
	*id = CSR_READ_2(sc, WI_ALLOC_FID);

	if (wi_seek(sc, *id, 0, WI_BAP0))
		return (EIO);

	for (i = 0; i < len / 2; i++)
		CSR_WRITE_2(sc, WI_DATA0, 0);

	return (0);
}

static void 
wi_setmulti(sc)
	struct wi_softc *sc;
{
	struct ifnet *ifp = &sc->wi_if;
	struct ether_multi *enm;
	struct ether_multistep estep;
	struct wi_ltv_mcast mcast;
	int promisc = 0;
	int i;

	if (sc->wi_ispccard && sc->wi_socket < 0) 
		return;

	bzero((char *)&mcast, sizeof(mcast));
	mcast.wi_type = WI_RID_MCAST_LIST;

	if ((ifp->if_flags & IFF_ALLMULTI) || 
	    (ifp->if_flags & IFF_PROMISC) ||
	    (sc->wi_ac.ac_multicnt > WI_NMCAST_FILTER))
		promisc++;

	if (!promisc) {
		ETHER_FIRST_MULTI(estep, &sc->wi_ac, enm);
		while (enm) {
			if (bcmp((caddr_t)&enm->enm_addrlo,
			    (caddr_t)&enm->enm_addrhi, 
			    sizeof(enm->enm_addrlo)) != 0) {
				promisc++;
				break;
			}
			ETHER_NEXT_MULTI(estep, enm);
		}
	}

	if (promisc) {
		wi_write_int(sc, WI_RID_PROMISC, 1);
		sc->wi_flags |= WI_FL_PROMISC;
	} else {
		wi_write_int(sc, WI_RID_PROMISC, 0);
		sc->wi_flags &= ~WI_FL_PROMISC;
		ETHER_FIRST_MULTI(estep, &sc->wi_ac, enm);
		for (i = 0; i < WI_NMCAST_FILTER && enm != NULL; i++) {
		    bcopy(((u_char *)(enm->enm_addrlo)),
		      (char *)&mcast.wi_mcast[i], ETHER_ADDR_LEN);
		    ETHER_NEXT_MULTI(estep, enm);
		}
		mcast.wi_len = (i * 3) + 1;
		wi_write_record(sc, (struct wi_ltv *)&mcast);
	}

	return;
}

/*
 * XXX this should eventually disappear - leave for now while we don't
 * have this in the 802.11 framework yet
 */
static int 
wi_setdef(sc, ltv)
	struct wi_softc *sc;
	struct wi_ltv *ltv;
{
	struct sockaddr_dl *sdl;
	struct ifaddr *ifa;
	struct ifnet *ifp = &sc->wi_if;
	int unit = ifp->if_unit;

	switch (ltv->wi_type) {
	case WI_RID_MAX_DATALEN:
		sc->wi_max_data_len = ltv->wi_val[0];
		break;
	case WI_RID_RTS_THRESH:
		sc->wi_rts_thresh = ltv->wi_val[0];
		break;
	case WI_RID_SYSTEM_SCALE:
		sc->wi_ap_density = ltv->wi_val[0];
		break;
	case WI_RID_CREATE_IBSS:
		sc->wi_create_ibss = ltv->wi_val[0];
		break;
	case WI_RID_PM_ENABLED:
		sc->wi_pm_enabled = ltv->wi_val[0];
		break;
	case WI_RID_MAX_SLEEP:
		sc->wi_max_sleep = ltv->wi_val[0];
		break;
	default:
		return (EIO);
	}

	/* Reinitialize WaveLAN. */
	wi_init(sc);

	return (0);
}

static void 
wi_init(xsc)
	void *xsc;
{
	struct wi_softc *sc = (struct wi_softc *)xsc;
	struct ifnet *ifp = &sc->wi_if;
	static struct wi_ltv_keys keys;
	int s;
	int id = 0;

	if (sc->wi_ispccard && sc->wi_socket < 0) 
		return;

	s = splimp();

	if (ifp->if_flags & IFF_RUNNING)
		wi_stop(sc);

	wi_reset(sc);

	if ((sc->wi_wepenable) && !(sc->wi_haswep)) {
		dvprintf(sc, "WEP requested but not supported by hardware, "
		    "interface shut down\n");
		ifp->if_flags &= ~IFF_UP;
		return;
	}
	/* Program max data length. */
	wi_write_int(sc, WI_RID_MAX_DATALEN, sc->wi_max_data_len);

	/* Enable/disable IBSS creation. */
	wi_write_int(sc, WI_RID_CREATE_IBSS, sc->wi_create_ibss);

	/* Set the port type. */
	if (sc->wi_media.ifm_media & IFM_802_11_IBSS)
		wi_write_int(sc, WI_RID_PORTTYPE, WI_PORTTYPE_ADHOC);
	else /* ESS mode */
		wi_write_int(sc, WI_RID_PORTTYPE, WI_PORTTYPE_BSS);

	/* Program the RTS/CTS threshold. */
	wi_write_int(sc, WI_RID_RTS_THRESH, sc->wi_rts_thresh);

	/* Program the TX rate */
#ifdef XXX
	wi_write_int(sc, WI_RID_TX_RATE, sc->wi_media.ifm_cur->ifm_data);
#else
	wi_write_int(sc, WI_RID_TX_RATE, 3);
#endif

	/* Access point density */
	wi_write_int(sc, WI_RID_SYSTEM_SCALE, sc->wi_ap_density);

	/* Power Management Enabled */
	if (sc->wi_pm_enabled >= 2) {
		wi_write_int(sc, WI_RID_PM_ENABLED, 0);
		sc->wi_pm_timeout = sc->wi_pm_enabled;
	} else 
		wi_write_int(sc, WI_RID_PM_ENABLED, sc->wi_pm_enabled);

	/* Power Managment Max Sleep */
	wi_write_int(sc, WI_RID_MAX_SLEEP, sc->wi_max_sleep);

	/* Specify the IBSS name */
	wi_write_str(sc, WI_RID_OWN_SSID, sc->wi_ibss_name);

	/* Specify the network name */
	wi_write_str(sc, WI_RID_DESIRED_SSID, sc->wi_net_name);

	/* Specify the frequency to use */
	wi_write_int(sc, WI_RID_OWN_CHNL, sc->wi_channel);

	/* Program the nodename. */
	wi_write_str(sc, WI_RID_NODENAME, sc->wi_node_name);

	/* Set our MAC address. */
	sc->wi_mac.wi_len = 4;
	sc->wi_mac.wi_type = WI_RID_MAC_NODE;
	bcopy((char *)&sc->wi_addr, &sc->wi_mac.wi_mac_addr, ETHER_ADDR_LEN);
	wi_write_record(sc, (struct wi_ltv *)&sc->wi_mac);

	/* Set multicast filter. */
	wi_setmulti(sc);

	/* Set WEP parameters */
	if (sc->wi_haswep == 1) {
		wi_write_int(sc, WI_RID_ENCRYPTION, sc->wi_wepenable);
		wi_write_int(sc, WI_RID_TX_CRYPT_KEY, sc->wi_weptxkey);
		keys.wi_len = (sizeof(struct wi_ltv_keys) / 2) + 1;
		keys.wi_type = WI_RID_DEFLT_CRYPT_KEYS;
		bcopy(&sc->wi_wepkey, &keys.wi_keys, 
		    sizeof(struct wi_key) * IEEE_802_11_NWEPKEYS);
		wi_write_record(sc, (struct wi_ltv *)&keys);
	}

	/* Enable desired port */
	wi_cmd(sc, WI_CMD_ENABLE|sc->wi_portnum, 0);

	if (wi_alloc_nicmem(sc, 1518 + sizeof(struct wi_header) + 8, &id))
		dvprintf(sc, "tx buffer allocation failed\n");
	sc->wi_tx_data_id = id;

	ifp->if_flags |= IFF_RUNNING;
	ifp->if_flags &= ~IFF_OACTIVE;

	/* enable interrupts */
	CSR_WRITE_2(sc, WI_INT_EN, WI_INTRS);

	splx(s);

	ifp->if_timer = 1;

	return;
}

static void 
wi_start(ifp)
	struct ifnet *ifp;
{
	struct wi_softc *sc = ifp->if_softc;
	struct mbuf *m0;
	struct wi_header tx_header;
	struct ieee_802_11_header *ih;
	int id;

	if (sc->wi_ispccard && sc->wi_socket < 0) 
		return;

	if (ifp->if_flags & IFF_OACTIVE)
		return;

	IF_DEQUEUE(&ifp->if_snd, m0);
	if (m0 == NULL)
		return;
	if (m0->m_len < sizeof(struct ieee_802_11_header) && 
	     (m0 = m_pullup(m0, sizeof(struct ieee_802_11_header))) == NULL)
		return;

	bzero((char *)&tx_header, sizeof(tx_header));
	id = sc->wi_tx_data_id;
	ih = mtod(m0, struct ieee_802_11_header *);

	bcopy((char *)&ih->e11_addr1,
	    (char *)&tx_header.wi_dst_addr, ETHER_ADDR_LEN);
	bcopy((char *)&ih->e11_addr2,
	    (char *)&tx_header.wi_src_addr, ETHER_ADDR_LEN);
	tx_header.wi_dat_len = m0->m_pkthdr.len - IEEE_802_11_HDR_LEN;
	tx_header.wi_len = htons(tx_header.wi_dat_len);

	/*
	 * Here we would call
	 * ieee_802_11_setbssid(ifp, ih, cur_bssid);
	 * but WaveLan firmware fills up the address[1234] fields itself.
	 */
	bzero((char *)&ih->e11_addr3, ETHER_ADDR_LEN);
	bzero((char *)&ih->e11_addr4, ETHER_ADDR_LEN);
	ih->e11_seq_ctl = 0;

	bcopy((char *)ih, &tx_header.wi_802_11_header,
	    sizeof(struct ieee_802_11_header));

	wi_write_data(sc, id, 0, (caddr_t)&tx_header,
	    sizeof(struct wi_header));

	m_copydata(m0, sizeof(struct ieee_802_11_header),
	    m0->m_pkthdr.len - sizeof(struct ieee_802_11_header),
	    sc->wi_txbuf);
	wi_write_data(sc, id, sizeof(struct wi_header), sc->wi_txbuf,
	    (m0->m_pkthdr.len - sizeof(struct ieee_802_11_header)) + 2);

	/*
	 * If there's a BPF listner, bounce a copy of
	 * this frame to him.
	 */
	if (ifp->if_bpf) 
		bpf_mtap(ifp->if_bpf, m0);

	m_freem(m0);

	if (wi_cmd(sc, WI_CMD_TX|WI_RECLAIM, id))
		dvprintf(sc, "xmit failed\n");

	ifp->if_flags |= IFF_OACTIVE;

	/*
	 * Set a timeout in case the chip goes out to lunch.
	 */
	sc->wi_tx_timeout = 5;

	/*
	 * Disable power management & start timer
	 */
	if (sc->wi_pm_enabled >= 2) {
		if (sc->wi_pm_timeout <= 0) {
			wi_write_int(sc, WI_RID_PM_ENABLED, 0);
			wi_cmd(sc, WI_CMD_DISABLE|sc->wi_portnum, 0);
			wi_cmd(sc, WI_CMD_ENABLE|sc->wi_portnum, 0);
		}
		sc->wi_pm_timeout = sc->wi_pm_enabled;
	}

	return;
}


static void 
wi_stop(sc)
	struct wi_softc *sc;
{
	struct ifnet *ifp = &sc->wi_if;

	if (sc->wi_ispccard && sc->wi_socket < 0) 
		return;

	ifp->if_timer = 0;
	sc->wi_tx_timeout = 0;
	sc->wi_pm_timeout = 0;

	CSR_WRITE_2(sc, WI_INT_EN, 0);
	wi_cmd(sc, WI_CMD_DISABLE|sc->wi_portnum, 0);

	ifp->if_flags &= ~(IFF_RUNNING|IFF_OACTIVE);

	return;
}


static void 
wi_watchdog(ifp)
	struct ifnet *ifp;
{
	struct wi_softc *sc = ifp->if_softc;

	if (sc->wi_ispccard && sc->wi_socket < 0) 
		return;

	if ((ifp->if_flags & IFF_OACTIVE) == 0)
		wi_cmd(sc, WI_CMD_INQUIRE, WI_INFO_COUNTERS);

	if (sc->wi_tx_timeout && (--sc->wi_tx_timeout == 0)) {
		dvprintf(sc, "device timeout\n");
		wi_init(sc);
		ifp->if_oerrors++;
	}

	/* Handle dynamic power mgmt */
	if (sc->wi_pm_timeout && (--sc->wi_pm_timeout == 0)) {
		wi_write_int(sc, WI_RID_PM_ENABLED, 1);
		wi_cmd(sc, WI_CMD_DISABLE|sc->wi_portnum, 0);
		wi_cmd(sc, WI_CMD_ENABLE|sc->wi_portnum, 0);
	}

	ifp->if_timer = 1;

	return;
}


/*
 * Callback when media options are changed
 */
static int
wi_mediachg(ifp)
	struct ifnet *ifp;
{
	if ((ifp->if_flags & IFF_RUNNING) == 0)
		return (0);

	wi_init(ifp->if_softc);
	return (0);
}


/*
 * Callback for media status
 */
static void
wi_mediastat(ifp, req)
	struct ifnet *ifp;
	struct ifmediareq *req;
{
	struct wi_softc *sc = ifp->if_softc;
	int stat;

	req->ifm_status = 0;

	/* No idea what's going on if the receiver is off */
	if ((sc->wi_if.if_flags & IFF_RUNNING) == 0)
		return;

	/* Read connection status */
	if (wi_read_int(sc, WI_RID_PORT_STAT, &stat) != 0)
		return;
	switch (stat) {
	/* 
	 * XXX These values were not documented and found by
	 * try and error. Ugh!
	 */
	case 2: /* ESS mode, no conn */
	case 5: /* ESS mode, lost conn */
		req->ifm_status = IFM_AVALID;
		break;

	case 4: /* ESS mode, conn up */
		req->ifm_status = IFM_ACTIVE | IFM_AVALID;
		break;

	case 1: /* interface down */
	case 3: /* IBSS mode, no status */
	default:
		break;
	}
}


static int 
wi_ioctl(ifp, command, data)
	struct ifnet *ifp;
	u_long command;
	caddr_t data;
{
	struct wi_softc *sc = ifp->if_softc;
	struct wi_ltv *ltvp = NULL;
	struct ifreq *ifr = (struct ifreq *)data;
	struct proc *p = PCPU(curproc);
	int s, error = 0;

	s = splimp();

	switch (command) {
	case SIOCSIFADDR:
	case SIOCGIFADDR:
	case SIOCSIFMTU:
		error = ieee_802_11_ioctl(ifp, command, data);
		break;

	case SIOCSIFFLAGS:
		if (ifp->if_flags & IFF_UP) 
			wi_init(sc);
		else 
			wi_stop(sc);

		error = 0;
		break;

        case SIOCADDMULTI:
                error = ether_addmulti((struct ifreq *)data, &sc->wi_ac);
                goto reset;
        case SIOCDELMULTI:
                error = ether_delmulti((struct ifreq *)data, &sc->wi_ac);
        reset:
                if (error == ENETRESET) {
			wi_setmulti(sc);
                        error = 0;
                }
                break;

	case SIOCSIFMEDIA:
	case SIOCGIFMEDIA:
		error = ifmedia_ioctl(ifp, (struct ifreq *)data,
		    &sc->wi_media, command);
		break;


	case SIOCGWAVELAN:
		if ((ltvp = (struct wi_ltv *) malloc(sizeof(struct wi_ltv), 
		    M_DEVBUF, M_NOWAIT)) == NULL)
			return (ENOMEM);

		if (error = copyin(ifr->ifr_data, ltvp, sizeof(struct wi_ltv)))
			break;

		if (ltvp->wi_type == WI_RID_IFACE_STATS) {
			bcopy((char *)&sc->wi_stats, (char *)&ltvp->wi_val,
			    sizeof(sc->wi_stats));
			ltvp->wi_len = (sizeof(sc->wi_stats) / 2) + 1;
		} else if (ltvp->wi_type == WI_RID_ZERO_CACHE) {
			bzero((char *)sc->wi_sigcache,
			sizeof(struct wi_sigcache) * MAXWICACHE);
			sc->wi_sigcacheslot = 0;
		} else if (ltvp->wi_type == WI_RID_READ_CACHE) {
			char *pt = (char *)&ltvp->wi_val;
			int nslots = MAXWICACHE;

			bcopy((char *) &nslots, (char *)pt, sizeof(int));
			pt += (sizeof (int));
			ltvp->wi_len = sizeof(int) / 2;

			bcopy((char *)&sc->wi_sigcache, (char *) pt,
			    sizeof(struct wi_sigcache) * MAXWICACHE);
			ltvp->wi_len += sizeof(struct wi_sigcache) * MAXWICACHE;
		} else {
			if (sc->wi_ispccard && sc->wi_socket < 0) {
				error = EIO;
				break;
			}
			if (wi_read_record(sc, ltvp)) {
				error = EINVAL;
				break;
			}
		}
		error = copyout(ltvp, ifr->ifr_data, sizeof(struct wi_ltv));
		break;

	case SIOCSWAVELAN:
		if ((error = suser(p->p_ucred, &p->p_acflag)) != 0)
                        break;

		if ((ltvp = (struct wi_ltv *) malloc(sizeof(struct wi_ltv), 
		    M_DEVBUF, M_NOWAIT)) == NULL) 
			return (ENOMEM);

		if ((error = copyin(ifr->ifr_data, ltvp, 
		    sizeof(struct wi_ltv))) != 0)
			break;
		error = wi_setdef(sc, ltvp);
		break;

	default:
		error = EINVAL;
		break;
	}

	if (ltvp != NULL)
		free(ltvp, M_DEVBUF);

	splx(s);

	return (error);
}

static int
wi_sysctl_field(oldp, oldlenp, newp, newlen, data, len, max)
	void *oldp;
	size_t *oldlenp;
	void *newp;
	size_t newlen;
	void *data;
	u_int16_t *len, max;
{
	int error = 0;

	if (oldp && *oldlenp < *len) {
		*oldlenp = *len;
		return (ENOMEM);
	}
	if (newp && newlen > max)
		return (EINVAL);
	if (oldp) {
		*oldlenp = *len;
		error = copyout(data, oldp, *len);
	}
	if (error == 0 && newp) {
		bzero(data, max);
		if (newlen)
			error = copyin(newp, data, newlen);
	}
	return (error);
}


int
wi_sysctl(ifp, name, namelen, oldp, oldlenp, newp, newlen)
	struct ifnet *ifp;
	int *name;
	u_int namelen;
	void *oldp;
	size_t *oldlenp;
	void *newp;
	size_t newlen;
{
        struct wi_softc *sc = ifp->if_softc;
	struct proc *p = PCPU(curproc);
	struct ifaddr *ifa;
	struct sockaddr_dl *sdl, dl;
	char buf[WI_MAXNAMELEN + 1];
	int error = 0, len, channel;

	if (newp && (error = suser(p->p_ucred, &p->p_acflag)) != 0)
		return (error);

	if (name[0] != ifp->if_index)
		goto default_sysctl;

	if (namelen < 2)
		return (EISDIR);

	if (name[1] != CTL_LINK_LINKTYPE)
		goto default_sysctl;

	if (namelen < 4)
		return (EISDIR);
	/*
	 * We know that name[2] == IFT_IEEE80211
	 */
	switch (name[3]) {
	case IEEE_802_11_SCTL_SSID:
		if (namelen != 5)
			return (ENOTDIR);
		switch (name[4]) {
		case 0:	/* Current SSID */
			if (newp)
				return (EPERM);
			if ((error = 
			    wi_read_str(sc, WI_RID_CURRENT_SSID, buf)) != 0)
				return (error);
			error = sysctl_rdstring(oldp, oldlenp, newp, buf);
			break;

		case 1:	/* Desired SSID */
			len = strlen(sc->wi_net_name);
			error = sysctl_string(oldp, oldlenp, newp, newlen, 
			    sc->wi_net_name, sizeof(sc->wi_net_name));
			break;

		default:
			return (EINVAL);
		}
		break;

	case IEEE_802_11_SCTL_WEPENABLE:
		if (namelen != 4)
			return (ENOTDIR);
		if (sc->wi_haswep == 0) 
			return (EOPNOTSUPP);
               	error = sysctl_int(oldp, oldlenp, newp, newlen, 
		    &sc->wi_wepenable);
		break;

	case IEEE_802_11_SCTL_WEPTXKEY:
		if (namelen != 4)
			return (ENOTDIR);
               	error = sysctl_int(oldp, oldlenp, newp, newlen, 
		    &sc->wi_weptxkey);
		break;

	case IEEE_802_11_SCTL_WEPKEY:
		if (namelen != 5)
			return (ENOTDIR);
		if ((name[4] < 0) || (name[4] > IEEE_802_11_NWEPKEYS))
			return (EOPNOTSUPP);
		if (oldp)
			return (EPERM);
		if (newp == NULL)
			return (0);
		if (newlen != 0 && newlen != 5 && newlen != 13)
			return (EINVAL);
		bzero(sc->wi_wepkey[name[4]].wi_keydat, WI_MAXMAXWEPKEYLEN);
		if (error = wi_sysctl_field(NULL, NULL, newp, newlen, 
		    sc->wi_wepkey[name[4]].wi_keydat, 
		    NULL, IEEE_802_11_MAXWEPKEYLEN))
			return (error);
		sc->wi_wepkey[name[4]].wi_keylen = newlen;
		break;

	case IEEE_802_11_SCTL_STATION:
		if (namelen != 4)
			return (ENOTDIR);
		len = strlen(sc->wi_node_name);
		error = sysctl_string(oldp, oldlenp, newp, newlen, 
		    sc->wi_node_name, sizeof(sc->wi_node_name));
		break;

	case IEEE_802_11_SCTL_CHANNEL:
		if (namelen != 4)
			return (ENOTDIR);

		if (oldp) {
			error = wi_read_int(sc, WI_RID_CURRENT_CHAN, &channel);
			if (error)
				return (error);
		}

               	error = sysctl_int(oldp, oldlenp, newp, newlen, &channel);
		if (error == 0 && newp) {
			/*
			 * It is possible to set an illegal channel
			 * by setting it before the card is inserted
			 * (and the legal channel mask is read in).
			 * This is unavoidable as we don't know the
			 * range of legal channels at that time;
			 * we expect the card to behave reasonable
			 * if an illegal channel is set 
			 * (as it turns out, wavelan does this)
			 */
			if ((channel < 1) || 
			    ((1 << (channel - 1)) & sc->wi_chanmask) == 0)
				return (EINVAL);
			sc->wi_channel = channel;
		}
		break;

	default:
		return (EOPNOTSUPP);
	}

	if ((error == 0) && newp && (ifp->if_flags & IFF_UP))
		wi_init(sc);
	return (error);


default_sysctl:
	return (ieee_802_11_sysctl(ifp, name, namelen, oldp, oldlenp, 
	    newp, newlen));
}

/*
 * wavelan signal strength cache code.
 * store signal/noise/quality on per MAC src basis in
 * a small fixed cache.  The cache wraps if > MAX slots
 * used.  The cache may be zeroed out to start over.
 *
 * The cache stores (MAC src(index), signal, quality, noise)
 *
 * Original comments:
 * -----------------
 * wi_cache_store, per rx packet store signal
 * strength in MAC (src) indexed cache.
 *
 * follows linux driver in how signal strength is computed.
 * In ad hoc mode, we use the rx_quality field.
 * signal and noise are trimmed to fit in the range from 47..138.
 * rx_quality field MSB is signal strength.
 * rx_quality field LSB is noise.
 * "quality" is (signal - noise) as is log value.
 * note: quality CAN be negative.
 *
 * In BSS mode, we use the RID for communication quality.
 * TBD:  BSS mode is currently untested.
 *
 * Bill's comments:
 * ---------------
 * Actually, we use the rx_quality field all the time for both "ad-hoc"
 * and BSS modes. Why? Because reading an RID is really, really expensive:
 * there's a bunch of PIO operations that have to be done to read a record
 * from the NIC, and reading the comms quality RID each time a packet is
 * received can really hurt performance. We don't have to do this anyway:
 * the comms quality field only reflects the values in the rx_quality field
 * anyway. The comms quality RID is only meaningful in infrastructure mode,
 * but the values it contains are updated based on the rx_quality from
 * frames received from the access point.
 *
 * Also, according to Lucent, the signal strength and noise level values
 * can be converted to dBms by subtracting 149, so I've modified the code
 * to do that instead of the scaling it did originally.
 */

static void
wi_cache_store (sc, ap, signal, noise)
	struct wi_softc *sc;
	u_int8_t *ap;
	int signal;
	int noise;
{
	int cache_slot = -1;
	int i;

	/* Do a linear search for a matching MAC address in the cache table */
	for (i = 0; i < MAXWICACHE; i++) {
		if (! bcmp(ap, sc->wi_sigcache[i].macsrc,  
		    ETHER_ADDR_LEN )) {
			/* Match!,
			 * so we already have this entry,
			 * update the data
			 */
			cache_slot = i;
			break;	
		}
	}

	/* If not found, have a new address entry, so add this new entry */
	if (i >= MAXWICACHE) {
		if (sc->wi_sigcacheslot == MAXWICACHE)
			sc->wi_sigcacheslot = 0;
		cache_slot = sc->wi_sigcacheslot;
		sc->wi_sigcacheslot++;
	}

	/* Invariant: cache_slot now points at some slot in cache */
	if (cache_slot < 0 || cache_slot >= MAXWICACHE)
		panic("wi_cache_store");

	/* Store items in cache: macsrc, signal, etc */
	bcopy(ap, sc->wi_sigcache[cache_slot].macsrc, ETHER_ADDR_LEN);
	sc->wi_sigcache[cache_slot].signal = signal - 149;
	sc->wi_sigcache[cache_slot].noise = noise - 149;

	return;
}


#ifdef PCCARD
/*
 * PCMCIA card service event handler (callback)
 */
static int
wi_cse_handler(clidata, func, sock, info, mtdreq, buf, misc)
	void *clidata;
	int func, sock, info;
	struct mtdreq *mtdreq;
	char *buf;
	int misc;
{
	struct wi_softc *sc = (struct wi_softc *)clidata;

	switch (func) {
	case CSE_CARD_INSERTION:
		if (cu_configured(&sc->wi_clihdl, sock) || sc->wi_socket >= 0)
			break;

		if (wi_cc_probe(sc, sock))
			wi_cc_attach(sc, sock);
		break;

	case CSE_CARD_REMOVAL:
		wi_cc_detach(sc, sock);
		break;

	case CSE_CLIENT_INFO:
		break;

	default:
		break;
	}
}

/*
 * Probe routine called upon the card insertion event.
 */
static int
wi_cc_probe(sc, sock)
	struct wi_softc *sc;
	int sock;
{
	struct ifnet *ifp = &sc->wi_if;
	cs_rtpl_t tplreq;
	u_char tpccbuf[32];
	struct ifaddr *ifa;
	struct sockaddr_dl *sdl;
	u_char *macaddr;
	int base = -1;
	int configured = 0;
	int ret = 0;
	int err;

	if (cs_spec_lookup(sock, "wi", NULL, 0)) {
		ccdprintf(("wi_cc_probe: no match\n"));
		return (0);
	}

	/*
	 * We would like to test if the card we are probing has been
	 * in the system before, but we can't do it before it is
	 * assigned its i/o ports.
	 */
	tplreq.Socket = sock;
	tplreq.Desired = CISTPL_CONF;
	tplreq.TupleData = tpccbuf;
	tplreq.TupleDataMax = sizeof(tpccbuf);

	if (err = cs_GetTuple(&tplreq)) {
		ccdprintf(("wi: can't find CISTPL_CONF, err 0x%x\n", err));
		return (0);
	}

	if (err = cu_parse_conf(tpccbuf, TupleLen(tplreq), &sc->wi_tpcc)) {
		ccdprintf(("wi: bad CISTPL_CONF\n"));
		return (0);
	}

	ccdprintf(("wi: tpcc_radr 0x%x\n", sc->wi_tpcc.tpcc_radr));
	ccdprintf(("wi: tpcc_rmsk 0x%x\n", sc->wi_tpcc.tpcc_rmsk[0]));

	sc->wi_rio.csrio_Socket = sock;
	sc->wi_rio.csrio_BasePort1 = 0;
	sc->wi_rio.csrio_NumPorts1 = WI_NPORT;
	sc->wi_rio.csrio_PortAttr1 = CSRIO_DATAPATH_16;
	sc->wi_rio.csrio_IOAddrLines = 6;
	sc->wi_rio.csrio_NumPorts2 = 0;

	ccdprintf(("wi: Looking for I/O ports using CS ..."));

	if (err = cs_RequestIO(&sc->wi_clihdl, &sc->wi_rio)) {
		ccdprintf(("wi: RequestIO err 0x%x\n", err));
		return (0);
	}
	ccdprintf(("done, got base 0x%x size 0%x\n",
		sc->wi_rio.csrio_BasePort1, sc->wi_rio.csrio_NumPorts1));
	/*
	 * From here on, we have to release the i/o window and/or
	 * configuration made above before returning.
	 */
	base = sc->wi_rio.csrio_BasePort1;

	sc->wi_rcfg.cscfg_Socket = sock;
	sc->wi_rcfg.cscfg_Attributes = 0;
	sc->wi_rcfg.cscfg_Vcc = 50;			/* 5V to Vcc */
	sc->wi_rcfg.cscfg_Vpp1 = sc->wi_rcfg.cscfg_Vpp2 = 0;
	sc->wi_rcfg.cscfg_IntType = CSCFG_IO;		/* I/O card */
	sc->wi_rcfg.cscfg_Present = CREGMAP_COR;
	sc->wi_rcfg.cscfg_ConfigBase = sc->wi_tpcc.tpcc_radr;
	sc->wi_rcfg.cscfg_ConfigIndex = 0x41;

	ccdprintf(("wi: Configuring using CS ..."));

	if (err = cs_RequestConfiguration(&sc->wi_clihdl, &sc->wi_rcfg)) {
		printf("wi: RequestConfiguration err 0x%x\n", err);
		goto out;
	}
	ccdprintf(("done\n"));

	configured = 1;
	sc->wi_base = base;

	/*
	 * Probe the device. If a value is returned, the
	 * device was found at the location.
	 */
	/* Reset the NIC. */
	wi_reset(sc);

	/* Read the station address. */
	sc->wi_mac.wi_type = WI_RID_MAC_NODE;
	sc->wi_mac.wi_len = 4;
	wi_read_record(sc, (struct wi_ltv *)&sc->wi_mac);
	macaddr = sc->wi_mac.wi_mac_addr;

	ccdprintf(("wi: MACaddr %s\n", ether_sprintf(macaddr)));

	if ((((u_short *)(sc->wi_addr))[0] == 0) &&
	    (((u_short *)(sc->wi_addr))[1] == 0) &&
	    (((u_short *)(sc->wi_addr))[2] == 0)) {
		bcopy(macaddr, sc->wi_addr, ETHER_ADDR_LEN);
		for (ifa = ifp->if_addrlist; ifa; ifa = ifa->ifa_next) {
			if ((sdl = (struct sockaddr_dl *)ifa->ifa_addr) &&
			    sdl->sdl_family == AF_LINK) {
				bcopy((caddr_t)((struct arpcom *)ifp)->ac_enaddr,
				      LLADDR(sdl), ifp->if_addrlen);
				break;
			}
		}
		ccdprintf(("wi: address %s, new card\n",
				ether_sprintf(sc->wi_addr)));
		/*
	 	* Read the default channel from the NIC. This may vary
	 	* depending on the country where the NIC was purchased, so
	 	* we can't hard-code a default and expect it to work for
	 	* everyone.
	 	*/
		wi_read_int(sc, WI_RID_OWN_CHNL, &sc->wi_channel);
		wi_read_int(sc, WI_RID_CHANNEL_LIST, &sc->wi_chanmask);

		wi_read_int(sc, WI_RID_WEP_AVAIL, &sc->wi_haswep);

		bzero((char *)&sc->wi_stats, sizeof(sc->wi_stats));

		ret = 1;
		goto out;
	}

	if (bcmp(macaddr, sc->wi_addr, ETHER_ADDR_LEN) == 0) {
		ccdprintf(("wi: old card\n"));
		ret = 1;
		goto out;
	}
out:
	/*
	 * Release the current configuration if configured.
	 */
	if (configured) {
		if (err = cs_ReleaseConfiguration(&sc->wi_clihdl, &sc->wi_rcfg))
			printf("wi: ReleaseConfiguration err %d\n", err);
			/* Nothing else significant we can do */
	}

	/*
	 * I/O window handling. Release the i/o window if returning error,
	 * otherwise record it for future use.
	 */
	if (ret == 0 && base >= 0) {
		if (err = cs_ReleaseIO(&sc->wi_clihdl, &sc->wi_rio))
			printf("wi: ReleaseIO err %d\n", err);
			/* Nothing else significant we can do */
	} else {
		/* Make sure interrupts are disabled. */
		CSR_WRITE_2(sc, WI_INT_EN, 0);
		CSR_WRITE_2(sc, WI_EVENT_ACK, 0xFFFF);
	}
	return (ret);
}

/*
 * Interface exists: make available by filling in network interface
 * record.  System will initialize the interface when it is ready
 * to accept packets.  
 */
static int
wi_cc_attach(sc, sock)
	struct wi_softc *sc;
	int sock;
{
	struct ifnet *ifp = &sc->wi_if;
	int i;
	int cnt;
	int err;
	int irq, ivec;

	ccdprintf(("wi: base %x\n", sc->wi_base));

	/*
	 * Allocate an interrupt vector, configure together with base
	 * address, setup interrupt steering, and see if the card interrupts.
	 */
	sc->wi_rirq.Socket = sock;
	sc->wi_rirq.Attributes = CSRIRQ_TYPE_EXCLUSIVE;
	sc->wi_rirq.IRQInfo1 = 0x20|0x10; /* 0x10 is forced here */
	sc->wi_rirq.IRQInfo2 = 0xffff;
	if (err = cs_RequestIRQ(&sc->wi_clihdl, &sc->wi_rirq)) {
		printf("wi_cc_attach: no irq\n");
		cs_ReleaseIO(&sc->wi_clihdl, &sc->wi_rio);
		sc->wi_base = 0;
		return (0);
	}

	if (sc->wi_rirq.AssignedIRQ == 0) {
		printf("wi_cc_attach: bad irq\n");
		cs_ReleaseIO(&sc->wi_clihdl, &sc->wi_rio);
		sc->wi_base = 0;
		return (0);
	}

	irq = 1 << sc->wi_rirq.AssignedIRQ;
	ivec = ffs(irq) - 1;

	sc->wi_rcfg.cscfg_Attributes = 2;	/* Enable IRQ */
	sc->wi_rcfg.cscfg_Vcc = 50;		/* 5V to Vcc */
	sc->wi_rcfg.cscfg_Vpp1 = 0;
	sc->wi_rcfg.cscfg_Vpp2 = 0;		/* Vpp's are not used */
	sc->wi_rcfg.cscfg_IntType = 2;		/* I/O card */
	sc->wi_rcfg.cscfg_ConfigBase = sc->wi_tpcc.tpcc_radr;
	sc->wi_rcfg.cscfg_Present = CREGMAP_COR;
	sc->wi_rcfg.cscfg_ConfigIndex = 0x41;
#if 0
	sc->wi_rcfg.cscfg_Irqno = ffs(irq) - 1;
#endif

	if (err = cs_RequestConfiguration(&sc->wi_clihdl, &sc->wi_rcfg)) {
		printf("wi: RequestConfiguration err 0x%x\n", err);
		cs_ReleaseIO(&sc->wi_clihdl, &sc->wi_rio);
		cs_ReleaseIRQ(&sc->wi_clihdl, &sc->wi_rirq);
		sc->wi_base = 0;
		return (0);
	}

	sc->wi_irq = ivec;

	ccdprintf(("wi_cc_attach: irq %x (ivec %d)\n", irq, ivec));

	dyna_intr_establish(irq, &sc->wi_ih, DV_NET);
	sc->wi_ih.ih_flags = IH_APICLEVEL;

	sc->wi_socket = sock;

	wi_init(sc);
	if (!(ifp->if_flags & IFF_UP)) 
		wi_stop(sc);

	return (1);
}

static int
wi_cc_detach(sc, sock)
	struct wi_softc *sc;
	int sock;
{
	struct ifnet *ifp = &sc->wi_if;

	if (sc->wi_socket != sock)
		return;

	if (sc->wi_socket < 0)	/* sanity */
		return;

	sc->wi_socket = -1;

	ifp->if_flags &= ~IFF_RUNNING;

	dyna_intr_release(1 << sc->wi_irq, &sc->wi_ih);
	cs_ReleaseConfiguration(&sc->wi_clihdl, &sc->wi_rcfg);
	cs_ReleaseIO(&sc->wi_clihdl, &sc->wi_rio);
	cs_ReleaseIRQ(&sc->wi_clihdl, &sc->wi_rirq);
}
#endif /*PCCARD*/



