/*
 * Copyright (c) 1997, 1998, 1999
 *	Bill Paul <wpaul@ctr.columbia.edu>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Bill Paul.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY Bill Paul AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL Bill Paul OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Modified for BSD/OS by Geert Jan de Groot
 *
 *	Id: if_wireg.h,v 1.5 1999/07/20 20:03:42 wpaul Exp 
 *	BSDI $Id: if_wireg.h,v 2.6.6.1 2001/07/13 17:45:10 geertj Exp $
 */

#ifndef _IF_WIREG_H
#define _IF_WIREG_H

#define	WI_NPORT	0x40

#define	WI_MAXNAMELEN	32		/* SSID / nodename */
#define	WI_MAXMAXWEPKEYLEN 14		/* WEP key length */

#define WI_TIMEOUT	65536

/*
 * The WaveLAN/IEEE cards contain an 802.11 MAC controller which Lucent
 * calls 'Hermes.' In typical fashion, getting documentation about this
 * controller is about as easy as squeezing blood from a stone. Here
 * is more or less what I know:
 *
 * - The Hermes controller is firmware driven, and the host interacts
 *   with the Hermes via a firmware interface, which can change.
 *
 * - The Hermes is described in a document called: "Hermes Firmware
 *   WaveLAN/IEEE Station Functions," document #010245, which of course
 *   Lucent will not release without an NDA.
 *
 * - Lucent has created a library called HCF (Hardware Control Functions)
 *   though which it wants developers to interact with the card. The HCF
 *   is needlessly complex, ill conceived and badly documented. Actually,
 *   the comments in the HCP code itself aren't bad, but the publically
 *   available manual that comes with it is awful, probably due largely to
 *   the fact that it has been emasculated in order to hide information
 *   that Lucent wants to keep proprietary. The purpose of the HCF seems
 *   to be to insulate the driver programmer from the Hermes itself so that
 *   Lucent has an excuse not to release programming in for it.
 *
 * - Lucent only makes available documentation and code for 'HCF Light'
 *   which is a stripped down version of HCF with certain features not
 *   implemented, most notably support for 802.11 frames.
 *
 * Rather than actually use the Lucent HCF library, I have stripped all
 * the useful information from it and used it to create a driver in the
 * usual BSD form. Note: I don't want to hear anybody whining about the
 * fact that the Lucent code is GPLed and mine isn't. I did not actually
 * put any of Lucent's code in this driver: I only used it as a reference
 * to obtain information about the underlying hardware. The Hermes
 * programming interface is not GPLed, so bite me.
 */


/*
 * Hermes register definitions and what little I know about them.
 */

/* Hermes command/status registers. */
#define WI_COMMAND		0x00
#define WI_PARAM0		0x02
#define WI_PARAM1		0x04
#define WI_PARAM2		0x06
#define WI_STATUS		0x08
#define WI_RESP0		0x0A
#define WI_RESP1		0x0C
#define WI_RESP2		0x0E

/* Command register values. */
#define WI_CMD_BUSY		0x8000 /* busy bit */
#define WI_CMD_INI		0x0000 /* initialize */
#define WI_CMD_ENABLE		0x0001 /* enable */
#define WI_CMD_DISABLE		0x0002 /* disable */
#define WI_CMD_DIAG		0x0003
#define WI_CMD_ALLOC_MEM	0x000A /* allocate NIC memory */
#define WI_CMD_TX		0x000B /* transmit */
#define WI_CMD_NOTIFY		0x0010
#define WI_CMD_INQUIRE		0x0011
#define WI_CMD_ACCESS		0x0021
#define WI_CMD_PROGRAM		0x0022

#define WI_CMD_CODE_MASK	0x003F

/*
 * Reclaim qualifier bit, applicable to the
 * TX and INQUIRE commands.
 */
#define WI_RECLAIM		0x0100 /* reclaim NIC memory */

/*
 * ACCESS command qualifier bits.
 */
#define WI_ACCESS_READ		0x0000
#define WI_ACCESS_WRITE		0x0100

/*
 * PROGRAM command qualifier bits.
 */
#define WI_PROGRAM_DISABLE	0x0000
#define WI_PROGRAM_ENABLE_RAM	0x0100
#define WI_PROGRAM_ENABLE_NVRAM	0x0200
#define WI_PROGRAM_NVRAM	0x0300

/* Status register values */
#define WI_STAT_CMD_CODE	0x003F
#define WI_STAT_DIAG_ERR	0x0100
#define WI_STAT_INQ_ERR		0x0500
#define WI_STAT_CMD_RESULT	0x7F00

/* memory handle management registers */
#define WI_INFO_FID		0x10
#define WI_RX_FID		0x20
#define WI_ALLOC_FID		0x22
#define WI_TX_CMP_FID		0x24

/*
 * Buffer Access Path (BAP) registers.
 * These are I/O channels. I believe you can use each one for
 * any desired purpose independently of the other. In general
 * though, we use BAP1 for reading and writing LTV records and
 * reading received data frames, and BAP0 for writing transmit
 * frames. This is a convention though, not a rule.
 */
#define WI_SEL0			0x18
#define WI_SEL1			0x1A
#define WI_OFF0			0x1C
#define WI_OFF1			0x1E
#define WI_DATA0		0x36
#define WI_DATA1		0x38
#define WI_BAP0			WI_DATA0
#define WI_BAP1			WI_DATA1

#define WI_OFF_BUSY		0x8000
#define WI_OFF_ERR		0x4000
#define WI_OFF_DATAOFF		0x0FFF

/* Event registers */
#define WI_EVENT_STAT		0x30	/* Event status */
#define WI_INT_EN		0x32	/* Interrupt enable/disable */
#define WI_EVENT_ACK		0x34	/* Ack event */

/* Events */
#define WI_EV_TICK		0x8000	/* aux timer tick */
#define WI_EV_RES		0x4000	/* controller h/w error (time out) */
#define WI_EV_INFO_DROP		0x2000	/* no RAM to build unsolicited frame */
#define WI_EV_NO_CARD		0x0800	/* card removed (hunh?) */
#define WI_EV_DUIF_RX		0x0400	/* wavelan management packet received */
#define WI_EV_INFO		0x0080	/* async info frame */
#define WI_EV_CMD		0x0010	/* command completed */
#define WI_EV_ALLOC		0x0008	/* async alloc/reclaim completed */
#define WI_EV_TX_EXC		0x0004	/* async xmit completed with failure */
#define WI_EV_TX		0x0002	/* async xmit completed succesfully */
#define WI_EV_RX		0x0001	/* async rx completed */

#define WI_INTRS	\
	(WI_EV_RX|WI_EV_TX|WI_EV_TX_EXC|WI_EV_ALLOC|WI_EV_INFO|WI_EV_INFO_DROP)

/* Host software registers */
#define WI_SW0			0x28
#define WI_SW1			0x2A
#define WI_SW2			0x2C
#define WI_SW3			0x2E

#define WI_CNTL			0x14

#define WI_CNTL_AUX_ENA		0xC000
#define WI_CNTL_AUX_ENA_STAT	0xC000
#define WI_CNTL_AUX_DIS_STAT	0x0000
#define WI_CNTL_AUX_ENA_CNTL	0x8000
#define WI_CNTL_AUX_DIS_CNTL	0x4000

#define WI_AUX_PAGE		0x3A
#define WI_AUX_OFFSET		0x3C
#define WI_AUX_DATA		0x3E

/*
 * These are all the LTV record types that we can read or write
 * from the WaveLAN. Not all of them are temendously useful, but I
 * list as many as I know about here for completeness.
 */

#define WI_RID_DNLD_BUF		0xFD01
#define WI_RID_MEMSZ		0xFD02
#define WI_RID_DOMAINS		0xFD11
#define WI_RID_CIS		0xFD13
#define WI_RID_COMMQUAL		0xFD43
#define WI_RID_SCALETHRESH	0xFD46
#define WI_RID_PCF		0xFD87

/*
 * Network parameters, static configuration entities.
 */
#define WI_RID_PORTTYPE		0xFC00 /* Connection control characteristics */
#define WI_RID_MAC_NODE		0xFC01 /* MAC address of this station */
#define WI_RID_DESIRED_SSID	0xFC02 /* Service Set ID for connection */
#define WI_RID_OWN_CHNL		0xFC03 /* Comm channel for BSS creation */
#define WI_RID_OWN_SSID		0xFC04 /* IBSS creation ID */
#define WI_RID_OWN_ATIM_WIN	0xFC05 /* ATIM window time for IBSS creation */
#define WI_RID_SYSTEM_SCALE	0xFC06 /* scale that specifies AP density */
#define WI_RID_MAX_DATALEN	0xFC07 /* Max len of MAC frame body data */
#define WI_RID_MAC_WDS		0xFC08 /* MAC addr of corresponding WDS node */
#define WI_RID_PM_ENABLED	0xFC09 /* ESS power management enable */
#define WI_RID_PM_EPS		0xFC0A /* PM EPS/PS mode */
#define WI_RID_MCAST_RX		0xFC0B /* ESS PM mcast reception */
#define WI_RID_MAX_SLEEP	0xFC0C /* max sleep time for ESS PM */
#define WI_RID_HOLDOVER		0xFC0D /* holdover time for ESS PM */
#define WI_RID_NODENAME		0xFC0E /* ID name of this node for diag */
#define WI_RID_DTIM_PERIOD	0xFC10 /* beacon interval between DTIMs */
#define WI_RID_WDS_ADDR1	0xFC11 /* port 1 MAC of WDS link node */
#define WI_RID_WDS_ADDR2	0xFC12 /* port 1 MAC of WDS link node */
#define WI_RID_WDS_ADDR3	0xFC13 /* port 1 MAC of WDS link node */
#define WI_RID_WDS_ADDR4	0xFC14 /* port 1 MAC of WDS link node */
#define WI_RID_WDS_ADDR5	0xFC15 /* port 1 MAC of WDS link node */
#define WI_RID_WDS_ADDR6	0xFC16 /* port 1 MAC of WDS link node */
#define WI_RID_MCAST_PM_BUF	0xFC17 /* PM buffering of mcast */
#define WI_RID_ENCRYPTION	0xFC20 /* enable/disable WEP */
#define WI_RID_AUTHTYPE		0xFC21 /* specify authentication type */

/*
 * Network parameters, dynamic configuration entities
 */
#define WI_RID_MCAST_LIST	0xFC80 /* list of multicast addrs */
#define WI_RID_CREATE_IBSS	0xFC81 /* create IBSS */
#define WI_RID_FRAG_THRESH	0xFC82 /* frag len, unicast msg xmit */
#define WI_RID_RTS_THRESH	0xFC83 /* frame len for RTS/CTS handshake */
#define WI_RID_TX_RATE		0xFC84 /* data rate for message xmit */
#define WI_RID_PROMISC		0xFC85 /* enable promisc mode */
#define WI_RID_FRAG_THRESH0	0xFC90
#define WI_RID_FRAG_THRESH1	0xFC91
#define WI_RID_FRAG_THRESH2	0xFC92
#define WI_RID_FRAG_THRESH3	0xFC93
#define WI_RID_FRAG_THRESH4	0xFC94
#define WI_RID_FRAG_THRESH5	0xFC95
#define WI_RID_FRAG_THRESH6	0xFC96
#define WI_RID_RTS_THRESH0	0xFC97
#define WI_RID_RTS_THRESH1	0xFC98
#define WI_RID_RTS_THRESH2	0xFC99
#define WI_RID_RTS_THRESH3	0xFC9A
#define WI_RID_RTS_THRESH4	0xFC9B
#define WI_RID_RTS_THRESH5	0xFC9C
#define WI_RID_RTS_THRESH6	0xFC9D
#define WI_RID_TX_RATE0		0xFC9E
#define WI_RID_TX_RATE1		0xFC9F
#define WI_RID_TX_RATE2		0xFCA0
#define WI_RID_TX_RATE3		0xFCA1
#define WI_RID_TX_RATE4		0xFCA2
#define WI_RID_TX_RATE5		0xFCA3
#define WI_RID_TX_RATE6		0xFCA4
#define WI_RID_DEFLT_CRYPT_KEYS	0xFCB0
#define WI_RID_TX_CRYPT_KEY	0xFCB1
#define WI_RID_TICK_TIME	0xFCE0

/*
 * NIC information
 */
#define WI_RID_FIRM_ID		0xFD02 /* Primary func firmware ID. */
#define WI_RID_PRI_SUP_RANGE	0xFD03 /* primary supplier compatibility */
#define WI_RID_CIF_ACT_RANGE	0xFD04 /* controller sup. compatibility */
#define WI_RID_SERIALNO		0xFD0A /* card serial number */
#define WI_RID_CARD_ID		0xFD0B /* card identification */
#define WI_RID_MFI_SUP_RANGE	0xFD0C /* modem supplier compatibility */
#define WI_RID_CFI_SUP_RANGE	0xFD0D /* controller sup. compatibility */
#define WI_RID_CHANNEL_LIST	0xFD10 /* allowd comm. frequencies. */
#define WI_RID_REG_DOMAINS	0xFD11 /* list of intended regulatory doms */
#define WI_RID_TEMP_TYPE	0xFD12 /* hw temp range code */
#define WI_RID_CIS		0xFD13 /* PC card info struct */
#define WI_RID_STA_IDENEITY	0xFD20 /* station funcs firmware ident */
#define WI_RID_STA_SUP_RANGE	0xFD21 /* station supplier compat */
#define WI_RID_MFI_ACT_RANGE	0xFD22
#define WI_RID_CFI_ACT_RANGE	0xFD33

/*
 * MAC information
 */
#define WI_RID_PORT_STAT	0xFD40 /* actual MAC port con control stat */
#define WI_RID_CURRENT_SSID	0xFD41 /* ID of actually connected SS */
#define WI_RID_CURRENT_BSSID	0xFD42 /* ID of actually connected BSS */
#define WI_RID_COMMS_QUALITY	0xFD43 /* quality of BSS connection */
#define WI_RID_CUR_TX_RATE	0xFD44 /* current TX rate */
#define WI_RID_OWN_BEACON_INT	0xFD45 /* beacon xmit time for BSS creation */
#define WI_RID_CUR_SCALE_THRESH	0xFD46 /* actual system scane thresh setting */
#define WI_RID_PROT_RESP_TIME	0xFD47 /* time to wait for resp to req msg */
#define WI_RID_SHORT_RTR_LIM	0xFD48 /* max tx attempts for short frames */
#define WI_RID_LONG_RTS_LIM	0xFD49 /* max tx attempts for long frames */
#define WI_RID_MAX_TX_LIFE	0xFD4A /* max tx frame handling duration */
#define WI_RID_MAX_RX_LIFE	0xFD4B /* max rx frame handling duration */
#define WI_RID_CF_POLL		0xFD4C /* contention free pollable ind */
#define WI_RID_AUTH_ALGS	0xFD4D /* auth algorithms available */
#define WI_RID_AUTH_TYPE	0xFD4E /* availanle auth types */
#define WI_RID_WEP_AVAIL	0xFD4F /* WEP privacy option available */
#define WI_RID_CUR_TX_RATE1	0xFD80
#define WI_RID_CUR_TX_RATE2	0xFD81
#define WI_RID_CUR_TX_RATE3	0xFD82
#define WI_RID_CUR_TX_RATE4	0xFD83
#define WI_RID_CUR_TX_RATE5	0xFD84
#define WI_RID_CUR_TX_RATE6	0xFD85
#define WI_RID_OWN_MAC		0xFD86 /* unique local MAC addr */
#define WI_RID_PCF		0xFD87 /* point coordination func cap */

/*
 * Modem information
 */
#define WI_RID_PHY_TYPE		0xFDC0 /* phys layer type indication */
#define WI_RID_CURRENT_CHAN	0xFDC1 /* current frequency */
#define WI_RID_PWR_STATE	0xFDC2 /* pwr consumption status */
#define WI_RID_CCA_MODE		0xFDC3 /* clear chan assess mode indication */
#define WI_RID_CCA_TIME		0xFDC4 /* clear chan assess time */
#define WI_RID_MAC_PROC_DELAY	0xFDC5 /* MAC processing delay time */
#define WI_RID_DATA_RATES	0xFDC6 /* supported data rates */

/*
 * One form of communication with the Hermes is with what Lucent calls
 * LTV records, where LTV stands for Length, Type and Value. The length
 * and type are 16 bits and are in native byte order. The value is in
 * multiples of 16 bits and is in little endian byte order.
 */
#define WI_MAX_DATALEN	512

struct wi_ltv {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_val[WI_MAX_DATALEN];
};

struct wi_ltv_int {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_val;
};

struct wi_ltv_str {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_strlen;
	u_char			wi_str[WI_MAXNAMELEN];
};

/* Download buffer location and length (WI_RID_DNLD_BUF) */
struct wi_ltv_dnld_buf {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_buf_pg; /* page addr of intermediate dl buf*/
	u_int16_t		wi_buf_off; /* offset of idb */
	u_int16_t		wi_buf_len; /* len of idb */
};

/* Mem sizes (WI_RID_MEMSZ) */
struct wi_ltv_memsz {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_mem_ram;
	u_int16_t		wi_mem_nvram;
};

/* List of intended regulatory domains (WI_RID_DOMAINS) */
struct wi_ltv_domains {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_domains[6];
};

/* CIS struct (WI_RID_CIS) */
struct wi_ltv_cis {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_cis[240];
};

/* Communications quality (WI_RID_COMMQUAL) */
struct wi_ltv_commqual {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_coms_qual;
	u_int16_t		wi_sig_lvl;
	u_int16_t		wi_noise_lvl;
};

/* Actual system scale thresholds (WI_RID_SYSTEM_SCALE/WI_RID_SCALETHRESH) */
struct wi_ltv_scalethresh {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_energy_detect;
	u_int16_t		wi_carrier_detect;
	u_int16_t		wi_defer;
	u_int16_t		wi_cell_search;
	u_int16_t		wi_out_of_range;
	u_int16_t		wi_delta_snr;
};

/* PCF info struct (WI_RID_PCF) */
struct wi_ltv_pcf {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int16_t		wi_energy_detect;
	u_int16_t		wi_carrier_detect;
	u_int16_t		wi_defer;
	u_int16_t		wi_cell_search;
	u_int16_t		wi_range;
};

/* Mac addresses (WI_RID_MAC_NODE/WI_RID_MAC_WDS) */
struct wi_ltv_macaddr {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_int8_t		wi_mac_addr[ETHER_ADDR_LEN];
};

/* WEP keys (WI_RID_DEFLT_CRYPT_KEYS) */
struct wi_key {
	u_int16_t		wi_keylen;
	u_int8_t		wi_keydat[WI_MAXMAXWEPKEYLEN];
};

struct wi_ltv_keys {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	struct wi_key		wi_keys[IEEE_802_11_NWEPKEYS];
};

/*
 * Multicast addresses to be put in filter (WI_RID_MCAST_LIST)
 * We're allowed up to 16 addresses in the filter.
 */
#define	WI_NMCAST_FILTER	16
struct wi_ltv_mcast {
	u_int16_t		wi_len;
	u_int16_t		wi_type;
	u_char			wi_mcast[WI_NMCAST_FILTER][ETHER_ADDR_LEN];
};

/* Connection control characteristics (WI_RID_PORTTYPE) */
#define WI_PORTTYPE_BSS		1	/* Basic Service Set (BSS) */
#define WI_PORTTYPE_WDS		2	/* Wireless Distribudion System (WDS) */
#define WI_PORTTYPE_ADHOC	3	/* Pseudo IBSS */

/* TX rate control (WI_RID_TX_RATE) */
#define WI_TXRATE_1MBPS		1
#define WI_TXRATE_2MBPS		2
#define WI_TXRATE_AUTO		3
#define WI_TXRATE_5_5MBPS	4
#define WI_TXRATE_11MBPS	5

/*
 * Information frame types.
 */
#define WI_INFO_NOTIFY		0xF000	/* Handover address */
#define WI_INFO_COUNTERS	0xF100	/* Statistics counters */
#define WI_INFO_SCAN_RESULTS	0xF101	/* Scan results */
#define WI_INFO_LINK_STAT	0xF200	/* Link status */
#define WI_INFO_ASSOC_STAT	0xF201	/* Association status */

/*
 * Statistics counters (WI_INFO_COUNTERS)
 */
struct wi_counters {
	u_int32_t		wi_tx_unicast_frames;
	u_int32_t		wi_tx_multicast_frames;
	u_int32_t		wi_tx_fragments;
	u_int32_t		wi_tx_unicast_octets;
	u_int32_t		wi_tx_multicast_octets;
	u_int32_t		wi_tx_deferred_xmits;
	u_int32_t		wi_tx_single_retries;
	u_int32_t		wi_tx_multi_retries;
	u_int32_t		wi_tx_retry_limit;
	u_int32_t		wi_tx_discards;
	u_int32_t		wi_rx_unicast_frames;
	u_int32_t		wi_rx_multicast_frames;
	u_int32_t		wi_rx_fragments;
	u_int32_t		wi_rx_unicast_octets;
	u_int32_t		wi_rx_multicast_octets;
	u_int32_t		wi_rx_fcs_errors;
	u_int32_t		wi_rx_discards_nobuf;
	u_int32_t		wi_tx_discards_wrong_sa;
	u_int32_t		wi_rx_WEP_cant_decrypt;
	u_int32_t		wi_rx_msg_in_msg_frags;
	u_int32_t		wi_rx_msg_in_bad_msg_frags;
	u_int32_t		wi_rx_discards_WEPICVerror;
	u_int32_t		wi_rx_discards_WEPexcluded;
};

/*
 * Hermes transmit/receive header structure
 */
struct wi_header {
	u_int16_t		wi_status;	/* 0x00 */
	u_int16_t		wi_rsvd0;	/* 0x02 */
	u_int16_t		wi_rsvd1;	/* 0x04 */
	u_int16_t		wi_q_info;	/* 0x06 */
	u_int16_t		wi_rsvd2;	/* 0x08 */
	u_int16_t		wi_rsvd3;	/* 0x0A */
	u_int16_t		wi_tx_ctl;	/* 0x0C */
	struct ieee_802_11_header wi_802_11_header; /* 0x0E */
	u_int16_t		wi_dat_len;	/* 0x2C */
	u_int8_t		wi_dst_addr[6];	/* 0x2E */
	u_int8_t		wi_src_addr[6];	/* 0x34 */
	u_int16_t		wi_len;		/* 0x3A */
};

#define WI_STAT_BADCRC		0x0001
#define WI_STAT_UNDECRYPTABLE	0x0002
#define WI_STAT_ERRSTAT		0x0003
#define WI_STAT_MAC_PORT	0x0700
#define WI_STAT_1042		0x2000	/* RFC1042 encoded */
#define WI_STAT_TUNNEL		0x4000	/* Bridge-tunnel encoded */
#define WI_STAT_WMP_MSG		0x6000	/* WaveLAN-II management protocol */
#define WI_RXSTAT_MSG_TYPE	0xE000

#define WI_ENC_TX_802_3		0x00
#define WI_ENC_TX_802_11	0x11
#define WI_ENC_TX_E_II		0x0E

#define WI_ENC_TX_1042		0x00
#define WI_ENC_TX_TUNNEL	0xF8

#define WI_TXCNTL_MACPORT	0x00FF
#define WI_TXCNTL_STRUCTTYPE	0xFF00

#define WI_PORT0		0
#define WI_PORT1		1
#define WI_PORT2		2
#define WI_PORT3		3
#define WI_PORT4		4
#define WI_PORT5		5

/*
 * XXX these should be phased out when the 802.11 framework will
 * have the same functionality
 */
#define	SIOCSWAVELAN	_IOW('i', 113, struct ifreq)
#define	SIOCGWAVELAN	_IOWR('i', 114, struct ifreq)

/*
 * Private LTV records (interpreted only by the driver). This is
 * a minor kludge to allow reading the interface statistics from
 * the driver.
 */
#define WI_RID_IFACE_STATS	0x0100
#define WI_RID_ZERO_CACHE	0x0300
#define WI_RID_READ_CACHE	0x0301

/*
 * Lucent/wavelan IEEE signal strength cache
 *
 * driver keeps cache of last
 * MAXWICACHE packets to arrive including signal strength info.
 * daemons may read this via ioctl
 *
 * Each entry in the wi_sigcache has a unique macsrc.
 */
#define	MAXWICACHE	10

struct wi_sigcache {
	char	macsrc[ETHER_ADDR_LEN];	/* unique MAC address for entry */
	short	signal;		/* signal strength of the packet */
	short	noise;		/* noise value */
};


#endif
