/*-
 * Copyright (c) 2000 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *      BSDI if_802_11.c,v 1.9 2000/08/30 14:15:34 geertj Exp
 */
/*
 * This file has been slightly modified by NRL for use with IPv6+IPsec.
 * Search for INET6 and/or IPSEC to see the blocks where this happened.
 * See the NRL Copyright notice for conditions on the modifications.
 */

/*
 * Copyright (c) 1982, 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)if_ethersubr.c	8.2 (Berkeley) 4/4/96
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>

#include <machine/cpu.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/if_dl.h>
#include <net/if_types.h>
#include <net/if_802_11.h>

#if defined(INET) || defined(INET6)
#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/if_ether.h>
#endif

struct protoq ieee_802_11_protoq;
extern u_char etherbroadcastaddr[ETHER_ADDR_LEN];
void ieee_802_11_rawintr __P((void *));

#define senderr(e) { error = (e); goto bad;}

/*
 * IEEE 802.11 output routine.
 * Encapsulate a packet of type family for the local net.
 * Assumes that ifp is actually pointer to arpcom structure.
 */
int
ieee_802_11_output(ifp, m0, dst, rt0)
	struct ifnet *ifp;
	struct mbuf *m0;
	struct sockaddr *dst;
	struct rtentry *rt0;
{
	u_short type;
	int s, error = 0;
 	u_int16_t edst[ETHER_ADDR_LEN/2];
	struct mbuf *m = m0;
	struct rtentry *rt;
	struct mbuf *mcopy = (struct mbuf *)0;
	struct ieee_802_11_header *ih;
	struct ether_header *oeh;
	int len = m->m_pkthdr.len;
	struct arpcom *ac = (struct arpcom *)ifp;
	char *cp;
	extern struct ifnet *loifp;
	int snaplen;
	u_char *snap;

	if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
		senderr(ENETDOWN);
	if (rt = rt0) {
		if ((rt->rt_flags & RTF_UP) == 0) {
			if (rt0 = rt = rtalloc1(dst, 1)) {
				rt->rt_refcnt--;
				if (rt->rt_ifp != ifp)
					return (*rt->rt_ifp->if_output)
							(ifp, m0, dst, rt);
			} else 
				senderr(EHOSTUNREACH);
		}
		if ((rt->rt_flags & RTF_GATEWAY) && dst->sa_family != AF_NS) {
			if (rt->rt_gwroute == 0)
				goto lookup;
			if (((rt = rt->rt_gwroute)->rt_flags & RTF_UP) == 0) {
				rtfree(rt); rt = rt0;
			lookup: rt->rt_gwroute = rtalloc1(rt->rt_gateway, 1);
				if ((rt = rt->rt_gwroute) == 0)
					senderr(EHOSTUNREACH);
				/* the "G" test below also prevents rt == rt0 */
				if ((rt->rt_flags & RTF_GATEWAY) ||
				    (rt->rt_ifp != ifp)) {
					rt->rt_refcnt--;
					rt0->rt_gwroute = 0;
					senderr(EHOSTUNREACH);
				}
			}
		}
		if (rt->rt_flags & RTF_REJECT)
			senderr(rt == rt0 ? EHOSTDOWN : EHOSTUNREACH);
	}
	snaplen = IEEE_802_11_SNAPHDR_LEN;	/* normally use a SNAP header */
	switch (dst->sa_family) {

#ifdef INET
	case AF_INET:
		if (!arpresolve(ac, rt, m, dst, (u_char *)edst))
			return (0);	/* if not yet resolved */
		/* If broadcasting on a simplex interface, loopback a copy */
		if ((m->m_flags & M_BCAST) && (ifp->if_flags & IFF_SIMPLEX) &&
		    loifp)
			mcopy = m_copy(m, 0, (int)M_COPYALL);
		type = htons(ETHERTYPE_IP);
		break;
#endif
#ifdef INET6
	case AF_INET6:
#ifndef OLDIP6OUTPUT
		if (!nd6_storelladdr(ifp, rt, m, dst, (u_char *)edst))
			return(0);	/* it must be impossible, but... */
#else
		if (!nd6_resolve(ifp, rt, m, dst, (u_char *)edst))
			return(0);	/* if not yet resolves */
#endif
		type = htons(ETHERTYPE_IPV6);
		break;
#endif /* INET6 */

	case AF_UNSPEC:
		oeh = (struct ether_header *)dst->sa_data;
 		ETHER_COPY(oeh->ether_dhost, edst);
		type = oeh->ether_type;
		if (ntohs(type) < 1536)
			snaplen = 2;
		break;

	case AF_LINK:
		ih = mtod(m, struct ieee_802_11_header *);
		ETHER_COPY(ih->e11_addr2, ac->ac_enaddr);
		if (ETHER_IS_MULTICAST(ih->e11_addr1)) {
			if (bcmp((caddr_t)etherbroadcastaddr,
			    (caddr_t)ih->e11_addr1,
			    sizeof(etherbroadcastaddr)) == 0)
				m->m_flags |= M_BCAST;
			else
				m->m_flags |= M_MCAST;  
		}
		type = htons(ETHERTYPE_IP);
		goto queue_it;
		break;

	default:
		printf("%s%d: can't handle af%d\n", ifp->if_name, ifp->if_unit,
			dst->sa_family);
		senderr(EAFNOSUPPORT);
	}


	if (mcopy)
		(void) (*loifp->if_output)(ifp, mcopy, dst, rt);
	/*
	 * Add local net header.  If no space in first mbuf,
	 * allocate another.
	 */
	M_PREPEND(m, IEEE_802_11_HDR_LEN + snaplen, M_DONTWAIT);
	if (m == 0)
		senderr(ENOBUFS);
	ih = mtod(m, struct ieee_802_11_header *);

	ETHER_COPY(edst, ih->e11_addr1);   /* ieee_802_11_setbssid may move */
	ETHER_COPY(ac->ac_enaddr, ih->e11_addr2);
queue_it:
	/* addr3 always set by ieee_802_11_setbssid */
	ETHER_ZERO(ih->e11_addr4);		/* Only used for repeaters */
	ih->e11_frame_ctl0 = IEEE_802_11_TYPE_DATA;
	ih->e11_frame_ctl1 = 0;		/* ad-hoc mode by default */
	ih->e11_duration = 0;
	ih->e11_seq_ctl = 0;

	snap = mtod(m, u_char *) + IEEE_802_11_HDR_LEN;

	if (snaplen == IEEE_802_11_SNAPHDR_LEN) {
		snap[0] = IEEE_802_11_SNAP_K1;
		snap[1] = IEEE_802_11_SNAP_K1;
		snap[2] = IEEE_802_11_SNAP_CONTROL;
		snap[3] = IEEE_802_11_SNAP_K2;
		snap[4] = 0;
		snap[5] = 0;
		snap += 6;
	}
	*(u_int16_t *)snap = type;

	s = splimp();
	/*
	 * Queue message on interface, and start output if interface
	 * not yet active.
	 */
	if (IF_QFULL(&ifp->if_snd)) {
		IF_DROP(&ifp->if_snd);
		splx(s);
		senderr(ENOBUFS);
	}
	IF_ENQUEUE(&ifp->if_snd, m);
	if ((ifp->if_flags & IFF_OACTIVE) == 0)
		(*ifp->if_start)(ifp);
	splx(s);
	ifp->if_obytes += len + IEEE_802_11_HDR_LEN + snaplen;
	if (m->m_flags & M_MCAST)
		ifp->if_omcasts++;
	return (error);

bad:
	if (m)
		m_freem(m);
	return (error);
}

/*
 * IEEE 802.11 address processing
 * Set the bssid, the second byte of the frame control and
 * assure the addresses are in the correct places.
 * Assumes that ifp is actually pointer to arpcom structure.
 */
int
ieee_802_11_setbssid(ifp, ih, bssid)
	struct ifnet *ifp;
	struct ieee_802_11_header *ih;
	u_char *bssid;
{
	struct arpcom *ac = (struct arpcom *)ifp;

	switch (ac->ac_flags & AC_802_11_MODE) {
	case AC_802_11_IBSS:
		ETHER_COPY(bssid, ih->e11_addr3);
		break;

	case AC_802_11_ESS:
		ih->e11_frame_ctl1 = IEEE_802_11_FCTL_TODS;
		ETHER_COPY(ih->e11_addr1, ih->e11_addr3);
		ETHER_COPY(bssid, ih->e11_addr1);
		break;

	case AC_802_11_AP:
		ih->e11_frame_ctl1 = IEEE_802_11_FCTL_FROMDS;
		ETHER_COPY(ih->e11_addr1, ih->e11_addr3);
		ETHER_COPY(bssid, ih->e11_addr1);
		break;

	case AC_802_11_REP:
		ih->e11_frame_ctl1 =
		    IEEE_802_11_FCTL_TODS | IEEE_802_11_FCTL_FROMDS;
		panic("802.11 repeater mode not supported!");
	}
}

/*
 * Process a received IEEE 802.11 packet
 * the packet is in the mbuf chain m without
 * The 802.11 header SNAP header must be in the first mbuf
 */
void
ieee_802_11_input(ifp, m)
	struct ifnet *ifp;
	struct mbuf *m;
{
	struct ifqueue *inq;
	struct ieee_802_11_header *ih;
	struct arpcom *ac = (struct arpcom *)ifp;
	u_char *dhost, *shost, *bssid;
	u_char *snap;
	int s;
	int ifam = ifp->if_addrmask;
	u_short ctl, type;

	if ((ifp->if_flags & IFF_UP) == 0) {
		m_freem(m);
		return;
	}
	ifp->if_ibytes += m->m_pkthdr.len;

	ih = mtod(m, struct ieee_802_11_header *);

	/*
	 * Trim off the 802.11 header
	 */
	m_adj(m, IEEE_802_11_HDR_LEN);

	/*
	 * We now look for ether the SNAP header or the packet length
	 */
	snap = mtod(m, u_char *);
	if (snap[0] == IEEE_802_11_SNAP_K1 && snap[1] == IEEE_802_11_SNAP_K1 &&
	    snap[2] == IEEE_802_11_SNAP_CONTROL && snap[3] == 0) {
		type = (snap[6] << 8) | snap[7];
		m_adj(m, 8);	/* Trim SNAP header and ether type */
	} else {
		type = (snap[0] << 8) | snap[1];
		m_adj(m, 2);	/* Trim length */
	}

	switch ((ih->e11_frame_ctl1 << 8) & AC_802_11_MODE) {
	case AC_802_11_IBSS:
		dhost = ih->e11_addr1;
		shost = ih->e11_addr2;
		bssid = ih->e11_addr3;
		break;

	case AC_802_11_ESS:
		dhost = ih->e11_addr1;
		bssid = ih->e11_addr2;
		shost = ih->e11_addr3;
		break;

	case AC_802_11_AP:
		bssid = ih->e11_addr1;
		shost = ih->e11_addr2;
		dhost = ih->e11_addr3;
		break;

	case AC_802_11_REP:
		bssid = NULL;
		dhost = ih->e11_addr3;
		shost = ih->e11_addr4;
		break;
	}

	if (ETHER_IS_MULTICAST(dhost)) {
		if (bcmp((caddr_t)etherbroadcastaddr, (caddr_t)dhost,
		    sizeof(etherbroadcastaddr)) == 0)
			m->m_flags |= M_BCAST;
		else
			m->m_flags |= M_MCAST;
	}

	if (m->m_flags & (M_BCAST|M_MCAST))
		ifp->if_imcasts++;

	switch (type) {

#ifdef INET6
	/*
	 * Schedule IPv6 software interrupt for incoming IPv6 packet.
	 */
	case ETHERTYPE_IPV6:
		schednetisr(NETISR_IPV6);
		inq = &ip6intrq;
		break;
#endif /* INET6 */
#ifdef INET
	case ETHERTYPE_IP:
		schednetisr(NETISR_IP);
		inq = &ipintrq;
		break;

	case ETHERTYPE_ARP:
		schednetisr(NETISR_ARP);
		inq = &arpintrq;
		break;
#endif
	default:
dropanyway:
	    /*
	     * Add the 802.11 header back on
	     */
	    m->m_len += m->m_data - (char *)ih;
	    m->m_pkthdr.len += m->m_data - (char *)ih;
	    m->m_data = (char *)ih;

	    s = splimp();
	    if (IF_QFULL(&ieee_802_11_protoq.pq_queue)) {
		    IF_DROP(&ieee_802_11_protoq.pq_queue);
		    m_freem(m);
	    } else
		    PQ_ENQUEUE(&ieee_802_11_protoq, m);
	    splx(s);
	    return;
	}

	s = splimp();
	if (IF_QFULL(inq)) {
		IF_DROP(inq);
		m_freem(m);
	} else
		IF_ENQUEUE(inq, m);
	splx(s);
}

/*
 * Perform common duties while attaching to interface list
 */
void
ieee_802_11_attach(ifp)
	struct ifnet *ifp;
{
#ifdef DEBUG
	if (ifp->if_softc == NULL)
		panic("ieee_802_11_attach");
#endif
	ifp->if_output = ether_output;
	ifp->if_output = ieee_802_11_output;
	ifp->if_type = IFT_IEEE80211;
	ifp->if_baudrate = 1*1000*1000;	/* Default, drivers may update */
	ifp->if_addrlen = ETHER_ADDR_LEN;
	ifp->if_hdrlen = IEEE_802_11_HDR_LEN;
	ifp->if_mtu = ETHERMTU;
	if (ifp->if_sysctl == NULL)
		ifp->if_sysctl = ieee_802_11_sysctl;
	if (ETHER_IS_MULTICAST((caddr_t)((struct arpcom *)ifp)->ac_enaddr))
		printf("WARNING: MAC address has multicast bit set\n");
	if_attach(ifp);
	if_newaddr(ifp, ifp->if_type,
	    (caddr_t)((struct arpcom *)ifp)->ac_enaddr);
	PQ_INIT(&ieee_802_11_protoq, ieee_802_11_rawintr);
}

/*
 * Common ioctl functions.
 * Note that XNS is omitted as no definition exists to run XNS on 802.11
 */
int
ieee_802_11_ioctl(ifp, command, data)
	struct ifnet *ifp;
	int command;
	caddr_t data;
{
	struct ifaddr *ifa = (struct ifaddr *) data;
	struct ifreq *ifr = (struct ifreq *) data;
	struct arpcom *ac = (struct arpcom *)(ifp);
	struct sockaddr *sa;
	int error = 0;

	switch (command) {
	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;

		switch (ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET:
			ifp->if_init(ifp->if_softc);	/* before arpwhohas */
			arp_ifinit((struct arpcom *)(ifp), ifa);
			break;
#endif
		default:
			ifp->if_init(ifp->if_softc);
			break;
		}
		break;

	case SIOCGIFADDR:
		sa = (struct sockaddr *) & ifr->ifr_data;
		bcopy(((struct arpcom *)(ifp))->ac_enaddr,
		     (caddr_t) sa->sa_data, sizeof(ac->ac_enaddr));
		break;

	case SIOCSIFMTU:
		/*
		 * Set the interface MTU.
		 */
		if (ifr->ifr_mtu > ETHERMTU || ifr->ifr_mtu < 64) 
			error = EINVAL;
		else
			ifp->if_mtu = ifr->ifr_mtu;
		break;
	}
	return (error);
}


void
ieee_802_11_rawintr(arg)
	void *arg;
{
	struct mbuf *m;
	struct ieee_802_11_header *ih;
	struct sockproto proto;
	struct sockaddr_dl ether_src;
	struct sockaddr_dl ether_dst;

	bzero(&ether_src, sizeof(ether_src));
	bzero(&ether_dst, sizeof(ether_dst));
	proto.sp_family = PF_LINK;
	ether_src.sdl_family = AF_LINK;
	ether_src.sdl_alen = ETHER_ADDR_LEN;
	ether_dst.sdl_family = AF_LINK;
	ether_dst.sdl_alen = ETHER_ADDR_LEN;
	ieee_802_11_protoq.pq_wayout.pending = 0;
	for (;;) {
		IF_DEQUEUE(&ieee_802_11_protoq.pq_queue, m);
		if (m == 0)
			return;
		ih = mtod(m, struct ieee_802_11_header *);
		proto.sp_protocol = m->m_pkthdr.rcvif->if_index;
		switch ((ih->e11_frame_ctl1 << 8) & AC_802_11_MODE) {
		case AC_802_11_IBSS:
			ETHER_COPY(ih->e11_addr1, ether_dst.sdl_data);
			ETHER_COPY(ih->e11_addr2, ether_src.sdl_data);
			break;

		case AC_802_11_ESS:
			ETHER_COPY(ih->e11_addr1, ether_dst.sdl_data);
			ETHER_COPY(ih->e11_addr3, ether_src.sdl_data);
			break;

		case AC_802_11_AP:
			ETHER_COPY(ih->e11_addr3, ether_dst.sdl_data);
			ETHER_COPY(ih->e11_addr2, ether_src.sdl_data);
			break;

		case AC_802_11_REP:
			ETHER_COPY(ih->e11_addr3, ether_dst.sdl_data);
			ETHER_COPY(ih->e11_addr4, ether_src.sdl_data);
			break;
		}
		m_adj(m, IEEE_802_11_HDR_LEN);
		raw_input(m, 0, &proto, &ether_src, &ether_dst);
	}
}

int
ieee_802_11_sysctl(ifp, name, namelen, oldp, oldlenp, newp, newlen)
	struct	ifnet *ifp;
	int	*name;
	u_int	namelen;
	void	*oldp;
	size_t	*oldlenp;
	void	*newp;
	size_t	newlen;
{
	struct arpcom *ac;
	struct ether_multi *ep, *op, *enm;
	struct ifaddr *ifa;
	struct sockaddr_dl *sdl;
	void *eoldp;
	int error, len;

	if (namelen < 2)
		return (ENOTDIR);

	/* We currently do not implment any generic operations */
	if (name[0] == 0)
		return (EOPNOTSUPP);

	eoldp = (char *)oldp + *oldlenp;
	ac = (struct arpcom *)ifp;

	switch (name[1]) {
	case CTL_LINK_GENERIC:
		if (namelen != 3)
			return (ENOTDIR);
		switch (name[2]) {
		case LINK_GENERIC_ADDR:
			/*
			 * XXX - Should we be assuming this that
			 * the first address on the list is an sdl?
			 */
			ifa = ifnet_addrs[ifp->if_index];
			sdl = (struct sockaddr_dl *)ifa->ifa_addr;
			if (oldp) {
				len = *oldlenp < sdl->sdl_len ?
				    *oldlenp : sdl->sdl_len;
				error = copyout(sdl, oldp, len);
				if (error) {
					*oldlenp = sdl->sdl_len;
					return (error);
				}
			} else
				len = sdl->sdl_len;
			*oldlenp = sdl->sdl_len;
			if (newlen) {
				struct sockaddr_dl dl;

				if (newlen > sizeof(dl))
					return (EINVAL);
				error = copyin(newp, &dl, newlen);
				if (error)
					return (error);
				if (newlen < dl.sdl_len ||
				    dl.sdl_family != sdl->sdl_family ||
				    dl.sdl_alen != sdl->sdl_alen)
					return (EINVAL);
				bcopy(LLADDR(&dl), ac->ac_enaddr, dl.sdl_alen);
				if_newaddr(ifp, IFT_IEEE80211, ac->ac_enaddr);
				ac->ac_flags |= AC_ADDRSET;
				/*
				 * XXX - We expect the if_init routine to
				 * do the right thing.  This includes reverting
				 * back to the old mac addr if the new one
				 * could not be set.
				 */
				(*ifp->if_init)(ifp->if_softc);
				arp_ifinit(ac, ifa);
			}
			error = 0;
			break;

		default:
			error = EOPNOTSUPP;
			break;
		}
		break;

	case CTL_LINK_LINKTYPE:
		if (namelen < 4)
			return (ENOTDIR);
		switch (name[3]) {
		case IEEE_802_11_SCTL_MULTIADDRS:
			op = (struct ether_multi *)oldp;
			if (oldp == NULL) {
				for (ep = ac->ac_multiaddrs; ep != NULL;
				     ep = ep->enm_next)
					op++;
				*oldlenp =
				    11 * ((caddr_t)op - (caddr_t)oldp) / 10;
				return (0);
			}
			enm = malloc(sizeof(*op), M_TEMP, M_WAITOK);
			if (enm == NULL)
				return (ENOMEM);
			error = 0;
			for (ep = ac->ac_multiaddrs; ep != NULL;
			    ep = ep->enm_next) {
				*enm = *ep;
				enm->enm_ac = NULL;
				if (enm->enm_next != NULL)
					enm->enm_next = op + 1;
				if ((void *)(op + 1) > eoldp) {
					error = ENOMEM;
					break;
				}
				error = copyout(enm, op, sizeof(*op));
				if (error)
					break;
				op++;
			}
			if (error == 0)
				*oldlenp = (caddr_t)op - (caddr_t)oldp;
			free(enm, M_TEMP);
			break;

		default:
			error = EOPNOTSUPP;
			break;
		}	
		break;

	default:
		error = EOPNOTSUPP;
		break;
	}

	return (error);
}
