/*	BSDI if_ethersubr.c,v 2.37.2.1 2001/08/02 21:39:27 dab Exp	*/
/*
 * This file has been slightly modified by NRL for use with IPv6+IPsec.
 * Search for INET6 and/or IPSEC to see the blocks where this happened.
 * See the NRL Copyright notice for conditions on the modifications.
 */

/*
 * Copyright (c) 1982, 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)if_ethersubr.c	8.2 (Berkeley) 4/4/96
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>

#include <machine/cpu.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/if_llc.h>
#include <net/if_dl.h>
#include <net/if_types.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_var.h>
#endif
#include <netinet/if_ether.h>

#ifdef INET6
#ifndef INET
#include <netinet/in.h>
#endif
#include <netinet6/in6_var.h>
#endif /* INET6 */

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#ifdef ISO
#include <netiso/argo_debug.h>
#include <netiso/iso.h>
#include <netiso/iso_var.h>
#include <netiso/iso_snpac.h>
#endif

#ifdef LLC
#include <netccitt/dll.h>
#include <netccitt/llc_var.h>
#endif

#if defined(LLC) && defined(CCITT)
extern struct ifqueue pkintrq;
#endif

struct protoq ether_protoq;

u_char	etherbroadcastaddr[ETHER_ADDR_LEN] =
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

u_char	etherflowcontroladdr[ETHER_ADDR_LEN] =
	{ 0x01, 0x80, 0xc2, 0x00, 0x00, 0x01 };

#ifdef NS
int ns_typeset = 0;
u_short	ns_nettype = 0xffff;	/* Encapsulation type for IPX (NS) packets */
#endif

#include "vlan.h"
#if NVLAN > 0
#include <net/if_vlan.h>
#include <net/if_vlan_var.h>
#endif /* NVLAN > 0 */

#define senderr(e) { error = (e); goto bad;}

/*
 * Ethernet output routine.
 * Encapsulate a packet of type family for the local net.
 * Assumes that ifp is actually pointer to arpcom structure.
 */
int
ether_output(ifp, m0, dst, rt0)
	register struct ifnet *ifp;
	struct mbuf *m0;
	struct sockaddr *dst;
	struct rtentry *rt0;
{
	u_short type;
	int s, error = 0;
 	u_int16_t edst[ETHER_ADDR_LEN/2];
	register struct mbuf *m = m0;
	register struct rtentry *rt;
	struct mbuf *mcopy = (struct mbuf *)0;
	register struct ether_header *eh;
	int len = m->m_pkthdr.len;
	struct arpcom *ac = (struct arpcom *)ifp;
	char *cp;
	extern struct ifnet *loifp;
	struct ifnet *oifp;
#if	NVLAN > 0
	struct ifv_softc *ifvsc;
	struct ether_8021q_header *ehq;
	u_int16_t tci;

	if (ifp->if_type == IFT_L2VLAN) {
		ifvsc = (struct ifv_softc *)ifp;

		/* Parent must be up and running */
		if ((oifp = ifvsc->ifv_p) == NULL ||
		    (oifp->if_flags & (IFF_UP|IFF_RUNNING))
		    != (IFF_UP|IFF_RUNNING))
			senderr(ENETDOWN);
	} else {
		ifvsc = NULL;
		oifp = ifp;
	}
#else
	oifp = ifp;
#endif


	if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
		senderr(ENETDOWN);
	if (rt = rt0) {
		if ((rt->rt_flags & RTF_UP) == 0) {
			if (rt0 = rt = rtalloc1(dst, 1)) {
				rt->rt_refcnt--;
				if (rt->rt_ifp != ifp)
					return (*rt->rt_ifp->if_output)
							(ifp, m0, dst, rt);
			} else 
				senderr(EHOSTUNREACH);
		}
		if ((rt->rt_flags & RTF_GATEWAY) && dst->sa_family != AF_NS) {
			if (rt->rt_gwroute == 0)
				goto lookup;
			if (((rt = rt->rt_gwroute)->rt_flags & RTF_UP) == 0) {
				rtfree(rt); rt = rt0;
			lookup: rt->rt_gwroute = rtalloc1(rt->rt_gateway, 1);
				if ((rt = rt->rt_gwroute) == 0)
					senderr(EHOSTUNREACH);
				/* the "G" test below also prevents rt == rt0 */
				if ((rt->rt_flags & RTF_GATEWAY) ||
				    (rt->rt_ifp != ifp)) {
					rt->rt_refcnt--;
					rt0->rt_gwroute = 0;
					senderr(EHOSTUNREACH);
				}
			}
		}
		if (rt->rt_flags & RTF_REJECT)
			senderr(rt == rt0 ? EHOSTDOWN : EHOSTUNREACH);
	}
	switch (dst->sa_family) {

#ifdef INET
	case AF_INET:
		if (!arpresolve(ac, rt, m, dst, (u_char *)edst))
			return (0);	/* if not yet resolved */
		/* If broadcasting on a simplex interface, loopback a copy */
		if ((m->m_flags & M_BCAST) && (ifp->if_flags & IFF_SIMPLEX) &&
		    loifp)
			mcopy = m_copy(m, 0, (int)M_COPYALL);
		type = htons(ETHERTYPE_IP);
		break;
#endif
#ifdef INET6
	case AF_INET6:
#ifndef OLDIP6OUTPUT
		if (!nd6_storelladdr(ifp, rt, m, dst, (u_char *)edst))
			return(0); /* it must be impossible, but... */
#else
		if (!nd6_resolve(ifp, rt, m, dst, (u_char *)edst))
			return(0);	/* if not yet resolves */
#endif
		type = htons(ETHERTYPE_IPV6);
		break;
#endif /* INET6 */
#ifdef NS
	case AF_NS:
		type = htons(ns_nettype);
		switch (ns_nettype) {
		case 0xffff:
			type = htons(m->m_pkthdr.len);
			break;
		case 0xe0e0:
			M_PREPEND(m, 3, M_WAIT);
			type = htons(m->m_pkthdr.len);
			cp = mtod(m, u_char *);
			cp[0] = 0xe0;		/* LLC/IPX */
			cp[1] = 0xe0;
			cp[2] = 0x03;
			break;
		}
 		ETHER_COPY(&(((struct sockaddr_ns *)dst)->sns_addr.x_host),
			    edst);
		if (!bcmp((caddr_t)edst, (caddr_t)&ns_thishost, sizeof(edst))) {
			if (loifp == NULL) {
				m_freem(m);
				return (ENODEV);
			}
			return ((*loifp->if_output)(ifp, m, dst, rt));
		}
		/* If broadcasting on a simplex interface, loopback a copy */
		if ((m->m_flags & M_BCAST) && (ifp->if_flags & IFF_SIMPLEX) &&
		    loifp)
			mcopy = m_copy(m, 0, (int)M_COPYALL);
		break;
#endif
#ifdef	ISO
	case AF_ISO: {
		int	snpalen;
		struct	llc *l;
		register struct sockaddr_dl *sdl;

		if (rt && (sdl = (struct sockaddr_dl *)rt->rt_gateway) &&
		    sdl->sdl_family == AF_LINK && sdl->sdl_alen > 0) {
			ETHER_COPY(LLADDR(sdl), edst);
		} else if (error = iso_snparesolve(ifp,
		    (struct sockaddr_iso *)dst, (char *)edst, &snpalen))
			goto bad; /* Not Resolved */
		/* If multicasting on a simplex interface, loopback a copy */
		if (ETHER_IS_MULTICAST((u_char *)edst))
			m->m_flags |= (M_BCAST|M_MCAST);
		if ((m->m_flags & M_BCAST) && (ifp->if_flags & IFF_SIMPLEX) &&
		    loifp && (mcopy = m_copy(m, 0, (int)M_COPYALL))) {
			M_PREPEND(mcopy, sizeof (*eh), M_DONTWAIT);
			if (mcopy) {
				eh = mtod(mcopy, struct ether_header *);
				ETHER_COPY(edst, eh->ether_dhost);
				ETHER_COPY(ac->ac_enaddr, eh->ether_shost);
			}
		}
		M_PREPEND(m, 3, M_DONTWAIT);
		if (m == NULL)
			return (0);
		type = htons(m->m_pkthdr.len);
		l = mtod(m, struct llc *);
		l->llc_dsap = l->llc_ssap = LLC_ISO_LSAP;
		l->llc_control = LLC_UI;
		len += 3;
		IFDEBUG(D_ETHER)
			int i;
			printf("unoutput: sending pkt to: ");
			for (i=0; i<ETHER_ADDR_LEN; i++)
				printf("%x ", ((u_char *)edst)[i] & 0xff);
			printf("\n");
		ENDDEBUG
		} break;
#endif /* ISO */
#ifdef	LLC
/*	case AF_NSAP: */
	case AF_CCITT: {
		register struct sockaddr_dl *sdl = 
			(struct sockaddr_dl *) rt -> rt_gateway;

		if (sdl && sdl->sdl_family == AF_LINK
		    && sdl->sdl_alen > 0) {
			ETHER_COPY(LLADDR(sdl), edst);
		} else
			goto bad; /* Not a link interface ? Funny ... */
		if (ETHER_IS_MULTICAST((u_char *)edst) &&
		    (ifp->if_flags & IFF_SIMPLEX) && loifp &&
		    (mcopy = m_copy(m, 0, (int)M_COPYALL))) {
			M_PREPEND(mcopy, sizeof (*eh), M_DONTWAIT);
			if (mcopy) {
				eh = mtod(mcopy, struct ether_header *);
				ETHER_COPY(edst, eh->ether_dhost);
				ETHER_COPY(ac->ac_enaddr, eh->ether_shost);
			}
		}
		type = htons((u_short) m->m_pkthdr.len);
#ifdef LLC_DEBUG
		{
			int i;
			register struct llc *l = mtod(m, struct llc *);

			printf("ether_output: sending LLC2 pkt to: ");
			for (i=0; i<ETHER_ADDR_LEN; i++)
				printf("%x ", ((u_char *)edst)[i] & 0xff);
			printf(" len 0x%x dsap 0x%x ssap 0x%x control 0x%x\n", 
			       m->m_pkthdr.len, l->llc_dsap & 0xff,
			       l->llc_ssap &0xff, l->llc_control & 0xff);

		}
#endif /* LLC_DEBUG */
		} break;
#endif /* LLC */	

	case AF_UNSPEC:
		eh = (struct ether_header *)dst->sa_data;
 		ETHER_COPY(eh->ether_dhost, edst);
		type = eh->ether_type;
		break;

	case AF_LINK:
#if     NVLAN > 0
		if (ifp->if_type == IFT_L2VLAN) {
			/*
			 * When sending a raw packet on a VLAN, if it
			 * isn't an 802.1Q header, turn it into one.  In
			 * all cases, force the tag to the right value.
			 */
			ehq = mtod(m, struct ether_8021q_header *);
			type = ehq->evlq_encap_proto;
			if (type != htons(vlan_8021q_proto)) {
				int offset = sizeof(*ehq) - sizeof(*eh);
				M_PREPEND(m, offset, M_DONTWAIT);
				if (m == NULL ||
				    (m->m_len < sizeof(*ehq) &&
				    (m = m_pullup(m, sizeof(*ehq))) == NULL))
					senderr(ENOBUFS);
				ehq = mtod(m, struct ether_8021q_header *);
				eh = (struct ether_header *)
				    ((u_char *)ehq + offset);
				ETHER_COPY(eh->ether_dhost, ehq->evlq_dhost);
				ehq->evlq_encap_proto = htons(vlan_8021q_proto);
				ehq->evlq_proto = type;
			}
			ehq->evlq_tci = htons(EVLQ_TCI(0, 0, ifvsc->ifv_vid));
		}
#endif
		eh = mtod(m, struct ether_header *);
		ETHER_COPY(ac->ac_enaddr, eh->ether_shost);
		if (ETHER_IS_MULTICAST(eh->ether_dhost)) {
			if (bcmp((caddr_t)etherbroadcastaddr, (caddr_t)eh->ether_dhost,
			    sizeof(etherbroadcastaddr)) == 0)
				m->m_flags |= M_BCAST;
			else
				m->m_flags |= M_MCAST;
		}
		goto queue_it;

	default:
		printf("%s%d: can't handle af%d\n", ifp->if_name, ifp->if_unit,
			dst->sa_family);
		senderr(EAFNOSUPPORT);
	}


	if (mcopy)
		(void) (*loifp->if_output)(ifp, mcopy, dst, rt);
	/*
	 * Add local net header.  If no space in first mbuf,
	 * allocate another.
	 */
	M_PREPEND(m, ifp->if_hdrlen, M_DONTWAIT);
	if (m == 0)
		senderr(ENOBUFS);
	eh = mtod(m, struct ether_header *);
#if defined(LLC) || defined(ISO)
	/* If it is not aligned, use bcopy() */
	if (((char *)eh - (char *)0) & 1) {
#if	NVLAN > 0
		if (ifp->if_type == IFT_L2VLAN) {
			tci = EVLQ_TCI(0, 0, ifvsc->ifv_vid);
			ehq = (struct ether_8021q_header *)eh;
			bcopy(&type, &ehq->evlq_proto, sizeof(ehq->evlq_proto));
			bcopy(&tci, &ehq->evlq_tci, sizeof(ehq->evlq_tci));
			type = htons(vlan_8021q_proto);
		}
#endif
		bcopy((caddr_t)&type, (caddr_t)&eh->ether_type,
		    sizeof(eh->ether_type));
		bcopy((caddr_t)edst, (caddr_t)eh->ether_dhost, sizeof (edst));
		bcopy((caddr_t)ac->ac_enaddr, (caddr_t)eh->ether_shost,
		    sizeof(eh->ether_shost));
	} else
#endif
	{
#if	NVLAN > 0
		if (ifp->if_type == IFT_L2VLAN) {
			ehq = (struct ether_8021q_header *)eh;
			ehq->evlq_proto = type;
			ehq->evlq_tci = htons(EVLQ_TCI(0, 0, ifvsc->ifv_vid));
			type = htons(vlan_8021q_proto);
		}
#endif
		ETHER_FILL_HDR(edst, ac->ac_enaddr, type, eh);
	}
queue_it:
	s = splimp();
	/*
	 * Queue message on interface, and start output if interface
	 * not yet active.
	 */
	if (IF_QFULL(&oifp->if_snd)) {
		IF_DROP(&oifp->if_snd);
		splx(s);
		senderr(ENOBUFS);
	}
#if	NVLAN > 0
	if (ifvsc != NULL) {
		ifp->if_obytes += len + ifp->if_hdrlen;
		if (ifp->if_bpf != NULL)
			bpf_mtap(ifp->if_bpf, m);
	}
#endif
	if (ifp->if_pif && ifp->if_pif->if_bpf != NULL)
		bpf_mtap(ifp->if_pif->if_bpf, m);
	if (m->m_flags & M_MCAST)
		ifp->if_omcasts++;
	IF_ENQUEUE(&oifp->if_snd, m);
	if ((oifp->if_flags & IFF_OACTIVE) == 0)
		(*oifp->if_start)(oifp);
	splx(s);
	oifp->if_obytes += len + oifp->if_hdrlen;
	return (error);

bad:
	if (m)
		m_freem(m);
	return (error);
}

/*
 * Process a received Ethernet packet;
 * the packet, with header, is in the mbuf chain m
 */
void
ether_input(ifp, m)
	struct ifnet *ifp;
	struct mbuf *m;
{
	struct ifqueue *inq;
	int isr;
#if	defined(ISO) || defined(LLC)
	struct llc *l;
#endif
	struct ether_header *eh;
	struct arpcom *ac = (struct arpcom *)ifp;
	u_int adjlen = 0, tadjlen;
	int s;
	int ifam = ifp->if_addrmask;
	u_short csum;
	u_int16_t ether_type;

	eh = mtod(m, struct ether_header *);

	if ((ifp->if_flags & IFF_UP) == 0) {
		m_freem(m);
		return;
	}
	ifp->if_ibytes += m->m_pkthdr.len;
	if (ETHER_IS_MULTICAST(eh->ether_dhost)) {
		if (bcmp((caddr_t)etherbroadcastaddr, (caddr_t)eh->ether_dhost,
		    sizeof(etherbroadcastaddr)) == 0)
			m->m_flags |= M_BCAST;
		else
			m->m_flags |= M_MCAST;
	}
	if (m->m_flags & (M_BCAST|M_MCAST))
		ifp->if_imcasts++;

	ether_type = ntohs(eh->ether_type);
#if NVLAN > 0
	if (ether_type == vlan_8021q_proto) {
		u_int16_t vid;
		u_int16_t tci;
		struct ether_8021q_header *ehq;

		ehq = mtod(m, struct ether_8021q_header *);

		/* 
		 * Extract TCI and proto fields from the remainder of
		 * the header
		 */
		tci = ntohs(ehq->evlq_tci);
		ether_type = ntohs(ehq->evlq_proto);

		/*
		 * If CFI is set we need to figure out how long the routing
		 * field is so we can skip over it.
		 */
		if (EVLQ_CFIOFTCI(tci)) {
			/* XXX
			 * For now we just ignore frames with CFI set as the
			 * E-RIF is variable length and we don't
			 * understand how to handle it (yet?).
			 */
			ifp->if_noproto++;
			goto dropanyway;
		} else
			tadjlen = sizeof(*ehq);

		vid = EVLQ_VIDOFTCI(tci);
		/* Ignore frames with the reserved VID */
		if (vid == EVLQ_VID_RESV) {
			ifp->if_noproto++;
			goto dropanyway;
		}

		if (vid != EVLQ_VID_NULL) {
			struct ifnet *vifp;

			if ((vifp = VLAN_8021Q_LOOKUP(ifp, vid)) == NULL) {
				ifp->if_noproto++;
				goto dropanyway;
			}
			ifp = vifp;
			if ((ifp->if_flags & IFF_UP) == 0) {
				m_freem(m);
				return;
			}

			/*
			 * Having found a valid vlan interface corresponding to the
			 * given source interface and vlan id, update statistics and
			 * do any requested BPF tracing.
			 */
			ifp->if_ipackets++;
			ifp->if_ibytes += m->m_pkthdr.len;
			if (m->m_flags & (M_BCAST|M_MCAST))
				ifp->if_imcasts++;
			m->m_pkthdr.rcvif = ifp;
 
			if (ifp->if_bpf)
				bpf_mtap(ifp->if_bpf, m);
		}
	} else
#endif /* NVLAN > 0 */
		tadjlen = ifp->if_hdrlen;

	adjlen = tadjlen;
	if (m->m_len < tadjlen &&
	    (m = m_pullup(m, tadjlen)) == NULL)
		return;
	m->m_data += adjlen;
	m->m_len -= adjlen;
	m->m_pkthdr.len -= adjlen;

	switch (ether_type) {

#ifdef INET6
	/*
	 * Schedule IPv6 software interrupt for incoming IPv6 packet.
	 */
	case ETHERTYPE_IPV6:
#ifdef NOTYET
		if ((ifam & 1 << NETISR_IPV6) == 0)
			goto dropanyway;
#endif
		isr = NETISR_IPV6;
		inq = &ip6intrq;
		break;
#endif /* INET6 */
#ifdef INET
	case ETHERTYPE_IP:
#ifdef NOTYET
		if ((ifam & 1 << NETISR_IP) == 0)
			goto dropanyway;
#endif
		isr = NETISR_IP;
		inq = &ipintrq;
		break;

	case ETHERTYPE_ARP:
#ifdef NOTYET
		if ((ifam & 1 << NETISR_IP) == 0)
			goto dropanyway;
#endif
		isr = NETISR_ARP;
		inq = &arpintrq;
		break;
#endif
#ifdef NS
	case 0x8137:		/* Ethernet II */
	case ETHERTYPE_NS:
		if ((ifam & 1 << NETISR_NS) == 0)
			goto dropanyway;
		if (ns_typeset == 0) {
			ns_nettype = ether_type;
			ns_typeset = 1;
		}
		isr = NETISR_NS;
		inq = &nsintrq;
		break;

#endif
	default:
#ifdef NS
		csum = *mtod(m, u_short *);
		if (ether_type <= ETHERMTU && 
		    (csum == 0xe0e0 || csum == 0xffff) &&
		    (ifam & 1 << NETISR_NS)) {
			if (ns_typeset == 0) {
				ns_nettype = csum;
				ns_typeset = 1;
			}
			if (csum == 0xe0e0) {
				/*
				 * Trim LLC header, next bytes should be 
				 * 0xffff (the IPX checksum)
				 */
				m->m_len -= 3;
				m->m_data += 3;
				if (m->m_flags & M_PKTHDR)
					m->m_pkthdr.len -= 3;
			}
			isr = NETISR_NS;
			inq = &nsintrq;
			break;
		}
#endif /* NS */
#if defined (ISO) || defined (LLC)
		if (ether_type > ETHERMTU)
			goto dropanyway;
		l = mtod(m, struct llc *);
		switch (l->llc_dsap) {
#ifdef	ISO
		case LLC_ISO_LSAP: 
#ifdef NOTYET
			if ((ifam & 1 << NETISR_ISO) == 0)
				goto dropanyway;
#endif
			switch (l->llc_control) {
			case LLC_UI:
				/* LLC_UI_P forbidden in class 1 service */
				if ((l->llc_dsap == LLC_ISO_LSAP) &&
				    (l->llc_ssap == LLC_ISO_LSAP)) {
					/* LSAP for ISO */
					if (m->m_pkthdr.len > ether_type)
						m_adj(m, ether_type - m->m_pkthdr.len);
					m->m_data += 3;		/* XXX */
					m->m_len -= 3;		/* XXX */
					m->m_pkthdr.len -= 3;	/* XXX */
					adjlen += 3;		/* XXX */
					M_PREPEND(m, sizeof (*eh), M_DONTWAIT);
					if (m == 0)
						return;
					ovbcopy(eh, m->m_data, sizeof(*eh));
					IFDEBUG(D_ETHER)
						printf("clnp packet");
					ENDDEBUG
					isr = NETISR_ISO;
					inq = &clnlintrq;
					break;
				}
				ifp->if_noproto++;
				goto dropanyway;
				
			case LLC_XID:
			case LLC_XID_P:
				if(m->m_len < 6)
					goto dropanyway;
				l->llc_window = 0;
				l->llc_fid = 9;
				l->llc_class = 1;
				l->llc_dsap = l->llc_ssap = 0;
				/* Fall through to */
			case LLC_TEST:
			case LLC_TEST_P:
			{
				struct sockaddr sa;
				register struct ether_header *eh2;
				int i;
				u_char c = l->llc_dsap;

				l->llc_dsap = l->llc_ssap;
				l->llc_ssap = c;
				if (m->m_flags & (M_BCAST | M_MCAST))
					ETHER_COPY(ac->ac_enaddr,
					      eh->ether_dhost);
				sa.sa_family = AF_UNSPEC;
				sa.sa_len = sizeof(sa);
				eh2 = (struct ether_header *)sa.sa_data;
				for (i = 0; i < ETHER_ADDR_LEN; i++) {
					eh2->ether_shost[i] = c = eh->ether_dhost[i];
					eh2->ether_dhost[i] = 
						eh->ether_dhost[i] = eh->ether_shost[i];
					eh->ether_shost[i] = c;
				}
				ifp->if_output(ifp, m, &sa, NULL);
				return;
			}
			default:
				m_freem(m);
				return;
			}
			break;
#endif /* ISO */
#ifdef LLC
		case LLC_X25_LSAP:
		{
#ifdef NOTYET
			if ((ifam & 1 << NETISR_CCITT) == 0)
				goto dropanyway;
#endif
			if (m->m_pkthdr.len > ether_type)
				m_adj(m, ether_type - m->m_pkthdr.len);
			M_PREPEND(m, sizeof(struct sdl_hdr) , M_DONTWAIT);
			if (m == 0)
				return;
			if ( !sdl_sethdrif(ifp, eh->ether_shost, LLC_X25_LSAP,
					    eh->ether_dhost, LLC_X25_LSAP, 6, 
					    mtod(m, struct sdl_hdr *)))
				panic("ETHER cons addr failure");
			mtod(m, struct sdl_hdr *)->sdlhdr_len = ether_type;
#ifdef LLC_DEBUG
				printf("llc packet\n");
#endif /* LLC_DEBUG */
			isr = NETISR_CCITT;
			inq = &llcintrq;
			break;
		}
#endif /* LLC */
		default:
			ifp->if_noproto++;
		dropanyway:
			if (adjlen != 0) {
				m->m_data -= adjlen;
				m->m_len += adjlen;
				m->m_pkthdr.len += adjlen;
			}
			s = splimp();
			if (IF_QFULL(&ether_protoq.pq_queue)) {
				IF_DROP(&ether_protoq.pq_queue);
				m_freem(m);
			} else
				PQ_ENQUEUE(&ether_protoq, m);
			splx(s);
			return;
		}
#else /* ISO || LLC */
dropanyway:
	    if (adjlen != 0) {
		    m->m_data -= adjlen;
		    m->m_len += adjlen;
		    m->m_pkthdr.len += adjlen;
	    }
	    s = splimp();
	    if (IF_QFULL(&ether_protoq.pq_queue)) {
		    IF_DROP(&ether_protoq.pq_queue);
		    m_freem(m);
	    } else
		    PQ_ENQUEUE(&ether_protoq, m);
	    splx(s);
	    return;
#endif /* ISO || LLC */
	}

	if (ifp->if_pif && ifp->if_pif->if_input) {
		/* Put the header back on... */
		if (adjlen) {
			m->m_data -= adjlen;
			m->m_len += adjlen;
			m->m_pkthdr.len += adjlen;
		}
		(*ifp->if_pif->if_input)(ifp->if_pif, m, adjlen, isr);
		return;
	}

	s = splimp();
	if (IF_QFULL(inq)) {
		IF_DROP(inq);
		m_freem(m);
	} else {
		IF_ENQUEUE(inq, m);
		schednetisr(isr);
	}
	splx(s);
}


void
ether_rawintr(arg)
	void *arg;
{
	register struct mbuf *m;
	struct ether_header *eh;
	struct sockproto proto;
	struct sockaddr_dl ether_src;
	struct sockaddr_dl ether_dst;

	bzero(&ether_src, sizeof(ether_src));
	bzero(&ether_dst, sizeof(ether_dst));
	proto.sp_family = PF_LINK;
	ether_src.sdl_len = sizeof(ether_src);
	ether_src.sdl_family = AF_LINK;
	ether_src.sdl_alen = ETHER_ADDR_LEN;
	ether_dst.sdl_len = sizeof(ether_dst);
	ether_dst.sdl_family = AF_LINK;
	ether_dst.sdl_alen = ETHER_ADDR_LEN;
	ether_protoq.pq_wayout.pending = 0;
	for (;;) {
		IF_DEQUEUE(&ether_protoq.pq_queue, m);
		if (m == 0)
			return;
		eh = mtod(m, struct ether_header *);
		proto.sp_protocol = m->m_pkthdr.rcvif->if_index;
		bcopy(eh->ether_shost, ether_src.sdl_data, ETHER_ADDR_LEN);
		bcopy(eh->ether_dhost, ether_dst.sdl_data, ETHER_ADDR_LEN);
		raw_input(m, 0, &proto, &ether_src, &ether_dst);
	}
}

/*
 * Convert Ethernet address to printable (loggable) representation.
 */
static char digits[] = "0123456789abcdef";
char *
ether_sprintf(ap)
	register u_char *ap;
{
	register i;
	static char etherbuf[18];
	register char *cp = etherbuf;

	for (i = 0; i < ETHER_ADDR_LEN; i++) {
		*cp++ = digits[*ap >> 4];
		*cp++ = digits[*ap++ & 0xf];
		*cp++ = ':';
	}
	*--cp = 0;
	return (etherbuf);
}

/*
 * Perform common duties while attaching to interface list
 */
void
ether_attach(ifp)
	register struct ifnet *ifp;
{
#ifdef DEBUG
	if (ifp->if_softc == NULL)
		panic("ether_attach");
#endif
	ifp->if_output = ether_output;
	ifp->if_type = IFT_ETHER;
	ifp->if_baudrate = 10*1000*1000;	/* Default, drivers may update */
	ifp->if_addrlen = ETHER_ADDR_LEN;
	ifp->if_hdrlen = sizeof(struct ether_header);
	ifp->if_mtu = ETHERMTU;
	if (ifp->if_sysctl == NULL)
		ifp->if_sysctl = ether_sysctl;
	if (ETHER_IS_MULTICAST((caddr_t)((struct arpcom *)ifp)->ac_enaddr))
		printf("WARNING: MAC address has multicast bit set\n");

	if_attach(ifp);
	if_newaddr(ifp, IFT_ETHER, (caddr_t)((struct arpcom *)ifp)->ac_enaddr);
	PQ_INIT(&ether_protoq, ether_rawintr);
}

int
ether_ioctl(ifp, command, data)
	struct ifnet *ifp;
	int command;
	caddr_t data;
{
	struct ifaddr *ifa = (struct ifaddr *) data;
	struct ifreq *ifr = (struct ifreq *) data;
	struct arpcom *ac = (struct arpcom *)(ifp);
	struct ns_addr *ina;
	struct sockaddr *sa;
	int error = 0;

	switch (command) {
	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;

		switch (ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET:
			ifp->if_init(ifp->if_softc);	/* before arpwhohas */
			arp_ifinit((struct arpcom *)(ifp), ifa);
			break;
#endif
#ifdef NS
		case AF_NS:
			ina = &(IA_SNS(ifa)->sns_addr);
			if (ns_nullhost(*ina))
				ina->x_host =
				    *(union ns_host *) (ac->ac_enaddr);
			else {
				ifp->if_flags &= ~IFF_RUNNING;
				bcopy((caddr_t) ina->x_host.c_host,
				      (caddr_t) ac->ac_enaddr,
				      sizeof(ac->ac_enaddr));
			}

			/*
			 * Set new address
			 */
			ifp->if_init(ifp->if_softc);
			break;
#endif
		default:
			ifp->if_init(ifp->if_softc);
			break;
		}
		break;

	case SIOCGIFADDR:
		sa = (struct sockaddr *) & ifr->ifr_data;
		bcopy(((struct arpcom *)(ifp))->ac_enaddr,
		     (caddr_t) sa->sa_data, sizeof(ac->ac_enaddr));
		break;

	case SIOCSIFMTU:
		/*
		 * Set the interface MTU.
		 */
		if (ifr->ifr_mtu > ETHERMTU || ifr->ifr_mtu < 64) 
			error = EINVAL;
		else
			ifp->if_mtu = ifr->ifr_mtu;
		break;
	}
	return (error);
}

int
ether_sysctl(ifp, name, namelen, oldp, oldlenp, newp, newlen)
	struct	ifnet *ifp;
	int	*name;
	u_int	namelen;
	void	*oldp;
	size_t	*oldlenp;
	void	*newp;
	size_t	newlen;
{
	struct arpcom *ac;
	struct ether_multi *ep, *op, *enm;
	struct ifaddr *ifa;
	struct sockaddr_dl *sdl;
	void *eoldp;
	int error, len;

	if (namelen < 2)
		return (ENOTDIR);

	/* We currently do not implment any generic operations */
	if (name[0] == 0)
		return (EOPNOTSUPP);

	eoldp = (char *)oldp + *oldlenp;
	ac = (struct arpcom *)ifp;

	switch (name[1]) {
	case CTL_LINK_GENERIC:
		if (namelen != 3)
			return (ENOTDIR);
		switch (name[2]) {
		case LINK_GENERIC_ADDR:
			/*
			 * XXX - Should we be assuming this that
			 * the first address on the list is an sdl?
			 */
			ifa = ifnet_addrs[ifp->if_index];
			sdl = (struct sockaddr_dl *)ifa->ifa_addr;
			if (oldp) {
				len = *oldlenp < sdl->sdl_len ?
				    *oldlenp : sdl->sdl_len;
				error = copyout(sdl, oldp, len);
				if (error) {
					*oldlenp = sdl->sdl_len;
					return (error);
				}
			} else
				len = sdl->sdl_len;
			*oldlenp = sdl->sdl_len;
			if (newlen) {
				struct sockaddr_dl dl;

				if (newlen > sizeof(dl))
					return (EINVAL);
				error = copyin(newp, &dl, newlen);
				if (error)
					return (error);
				if (newlen < dl.sdl_len ||
				    dl.sdl_family != sdl->sdl_family ||
				    dl.sdl_alen != sdl->sdl_alen)
					return (EINVAL);
				bcopy(LLADDR(&dl), ac->ac_enaddr, dl.sdl_alen);
				if_newaddr(ifp, IFT_ETHER, ac->ac_enaddr);
				ac->ac_flags |= AC_ADDRSET;
				/*
				 * XXX - We expect the if_init routine to
				 * do the right thing.  This includes reverting
				 * back to the old mac addr if the new one
				 * could not be set.
				 */
				(*ifp->if_init)(ifp->if_softc);
				arp_ifinit(ac, ifa);
			}
			error = 0;
			break;

		default:
			error = EOPNOTSUPP;
			break;
		}
		break;

	case CTL_LINK_LINKTYPE:
		if (namelen < 4)
			return (ENOTDIR);
		switch (name[3]) {
		case ETHERCTL_MULTIADDRS:
			op = (struct ether_multi *)oldp;
			if (oldp == NULL) {
				for (ep = ac->ac_multiaddrs; ep != NULL;
				     ep = ep->enm_next)
					op++;
				*oldlenp =
				    11 * ((caddr_t)op - (caddr_t)oldp) / 10;
				return (0);
			}
			enm = malloc(sizeof(*op), M_TEMP, M_WAITOK);
			if (enm == NULL)
				return (ENOMEM);
			error = 0;
			for (ep = ac->ac_multiaddrs; ep != NULL;
			    ep = ep->enm_next) {
				*enm = *ep;
				enm->enm_ac = NULL;
				if (enm->enm_next != NULL)
					enm->enm_next = op + 1;
				if ((void *)(op + 1) > eoldp) {
					error = ENOMEM;
					break;
				}
				error = copyout(enm, op, sizeof(*op));
				if (error)
					break;
				op++;
			}
			if (error == 0)
				*oldlenp = (caddr_t)op - (caddr_t)oldp;
			free(enm, M_TEMP);
			break;

		default:
			error = EOPNOTSUPP;
			break;
		}	
		break;

	default:
		error = EOPNOTSUPP;
		break;
	}

	return (error);
}

u_char	ether_ipmulticast_min[ETHER_ADDR_LEN] =
	{ 0x01, 0x00, 0x5e, 0x00, 0x00, 0x00 };
u_char	ether_ipmulticast_max[ETHER_ADDR_LEN] =
	{ 0x01, 0x00, 0x5e, 0x7f, 0xff, 0xff };
#ifdef INET6
u_char	ether_ip6multicast_min[ETHER_ADDR_LEN] =
	{ 0x33, 0x33, 0x00, 0x00, 0x00, 0x00 };
u_char	ether_ip6multicast_max[ETHER_ADDR_LEN] =
	{ 0x33, 0x33, 0xff, 0xff, 0xff, 0xff };
#endif /* INET6 */

/*
 * Add an Ethernet multicast address or range of addresses to the list for a
 * given interface.
 */
int
ether_addmulti(ifr, ac)
	struct ifreq *ifr;
	register struct arpcom *ac;
{
	register struct ether_multi *enm;
	u_char addrlo[ETHER_ADDR_LEN];
	u_char addrhi[ETHER_ADDR_LEN];
	int rc;
	int s = splimp();

	if ((rc = ether_mapmcaddr(ifr, addrlo, addrhi)) != 0) {
		splx(s);
		return (rc);
	}

	/*
	 * See if the address range is already in the list.
	 */
	ETHER_LOOKUP_MULTI(addrlo, addrhi, ac, enm);
	if (enm != NULL) {
		/*
		 * Found it; just increment the reference count.
		 */
		++enm->enm_refcount;
		splx(s);
		return (0);
	}
	/*
	 * New address or range; malloc a new multicast record
	 * and link it into the interface's multicast list.
	 */
	enm = (struct ether_multi *)malloc(sizeof(*enm), M_IFMADDR, M_NOWAIT);
	if (enm == NULL) {
		splx(s);
		return (ENOBUFS);
	}
	ETHER_COPY(addrlo, enm->enm_addrlo);
	ETHER_COPY(addrhi, enm->enm_addrhi);
	enm->enm_ac = ac;
	enm->enm_refcount = 1;
	enm->enm_next = ac->ac_multiaddrs;
	ac->ac_multiaddrs = enm;
	ac->ac_multicnt++;
	splx(s);
	/*
	 * Return ENETRESET to inform the driver that the list has changed
	 * and its reception filter should be adjusted accordingly.
	 */
	return (ENETRESET);
}

/*
 * Delete a multicast address record.
 */
int
ether_delmulti(ifr, ac)
	struct ifreq *ifr;
	register struct arpcom *ac;
{
	register struct ether_multi *enm;
	register struct ether_multi **p;
	u_char addrlo[ETHER_ADDR_LEN];
	u_char addrhi[ETHER_ADDR_LEN];
	int rc;
	int s = splimp();

	if ((rc = ether_mapmcaddr(ifr, addrlo, addrhi)) != 0) {
		splx(s);
		return (rc);
	}

	/*
	 * Look up the address in our list.
	 */
	ETHER_LOOKUP_MULTI(addrlo, addrhi, ac, enm);
	if (enm == NULL) {
		splx(s);
		return (ENXIO);
	}
	if (--enm->enm_refcount != 0) {
		/*
		 * Still some claims to this record.
		 */
		splx(s);
		return (0);
	}
	/*
	 * No remaining claims to this record; unlink and free it.
	 */
	for (p = &enm->enm_ac->ac_multiaddrs;
	     *p != enm;
	     p = &(*p)->enm_next)
		continue;
	*p = (*p)->enm_next;
	free(enm, M_IFMADDR);
	ac->ac_multicnt--;
	splx(s);
	/*
	 * Return ENETRESET to inform the driver that the list has changed
	 * and its reception filter should be adjusted accordingly.
	 */
	return (ENETRESET);
}

/*
 * Translate a multicast address to ethernet addresses
 */
int
ether_mapmcaddr(ifr, addrlo, addrhi)
	struct ifreq *ifr;
	u_char *addrlo;
	u_char *addrhi;
{
	struct sockaddr_in *sin;
#ifdef INET6
	struct sockaddr_in6 *sin6;
#endif /* INET6 */

	switch (ifr->ifr_addr.sa_family) {

	case AF_UNSPEC:
		ETHER_COPY(ifr->ifr_addr.sa_data, addrlo);
		ETHER_COPY(addrlo, addrhi);
		break;

#ifdef INET
	case AF_INET:
		sin = (struct sockaddr_in *)&(ifr->ifr_addr);
		if (sin->sin_addr.s_addr == INADDR_ANY) {
			/*
			 * An IP address of INADDR_ANY means stop listening
			 * to the range of Ethernet multicast addresses used
			 * for IP.
			 */
			ETHER_COPY(ether_ipmulticast_min, addrlo);
			ETHER_COPY(ether_ipmulticast_max, addrhi);
		} else {
			ETHER_MAP_IP_MULTICAST(&sin->sin_addr, addrlo);
			ETHER_COPY(addrlo, addrhi);
		}
		break;
#endif
#ifdef INET6
	case AF_INET6:
		sin6 = (struct sockaddr_in6 *)&(ifr->ifr_addr);
		if (IN6_IS_ADDR_UNSPECIFIED(&sin6->sin6_addr)) {
			/*
			 * An unspecified IPv6 address means stop listening to
			 * all IPv6 multicast addresses on this Ethernet.'
			 *
			 * (This might not be healthy, given IPv6's reliance on
			 *  multicast for things like neighbor discovery.
			 *  Perhaps initializing all-nodes, solicited nodes, and
			 *  possibly all-routers for this interface afterwards
			 *  is not a bad idea.)
			 */
			bcopy(ether_ip6multicast_min, addrlo, ETHER_ADDR_LEN);
			bcopy(ether_ip6multicast_max, addrhi, ETHER_ADDR_LEN); 
		} else {
			ETHER_MAP_IPV6_MULTICAST(&sin6->sin6_addr, addrlo);
			bcopy(addrlo, addrhi, ETHER_ADDR_LEN);
		}
		break;
#endif /* INET6 */

	default:
		return (EAFNOSUPPORT);
	}

	/*
	 * Verify that we have valid Ethernet multicast addresses.
	 */
	if (!ETHER_IS_MULTICAST(addrlo) || !ETHER_IS_MULTICAST(addrhi)) 
		return (EINVAL);

	return (0);
}

