/*	BSDI if_fddisubr.c,v 2.23 2000/08/30 14:15:35 geertj Exp	*/

/*
 * This file has been slightly modified by NRL for use with IPv6+IPsec.
 * Search for INET6 and/or IPSEC to see the blocks where this happened.
 * See the NRL Copyright notice for conditions on the modifications.
 */
/*
 * Copyright (c) 1995
 *	Matt Thomas.  All rights reserved.
 * Copyright (c) 1982, 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)if_fddisubr.c	8.1 (Berkeley) 6/10/93
 *
 * Id: if_fddisubr.c,v 1.7 1995/04/20 20:17:33 thomas Exp
 * Log: if_fddisubr.c,v
 * Revision 1.7  1995/04/20  20:17:33  thomas
 * fix typo
 *
 * Revision 1.6  1995/04/20  19:21:58  thomas
 * *** empty log message ***
 *
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>

#include <machine/cpu.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/if_llc.h>
#include <net/if_dl.h>
#include <net/if_types.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_var.h>
#endif
#include <netinet/if_ether.h>
#include <net/if_fddi.h>

#ifdef INET6
#ifndef INET
#include <netinet/in.h>
#endif
#include <netinet6/in6_var.h>
#endif /* INET6 */

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#ifdef DECNET
#include <netdnet/dn.h>
#endif

#ifdef ISO
#include <netiso/argo_debug.h>
#include <netiso/iso.h>
#include <netiso/iso_var.h>
#include <netiso/iso_snpac.h>
#endif

#include "bpfilter.h"

#ifdef LLC
#include <netccitt/dll.h>
#include <netccitt/llc_var.h>
#endif

#if defined(LLC) && defined(CCITT)
extern struct ifqueue pkintrq;
#endif

#define senderr(e) { error = (e); goto bad;}

/*
 * FDDI output routine.
 * Encapsulate a packet of type family for the local net.
 * Use trailer local net encapsulation if enough data in first
 * packet leaves a multiple of 512 bytes of data in remainder.
 * Assumes that ifp is actually pointer to arpcom structure.
 */
int
fddi_output(ifp, m0, dst, rt0)
	register struct ifnet *ifp;
	struct mbuf *m0;
	struct sockaddr *dst;
	struct rtentry *rt0;
{
	short type;
	int s, error = 0;
 	u_char edst[6];
	register struct mbuf *m = m0;
	register struct rtentry *rt;
	struct mbuf *mcopy = (struct mbuf *)0;
	register struct fddi_header *fh;
	struct arpcom *ac = (struct arpcom *)ifp;
	extern struct ifnet *loifp;

	if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
		senderr(ENETDOWN);
	if (rt = rt0) {
		if ((rt->rt_flags & RTF_UP) == 0) {
			if (rt0 = rt = rtalloc1(dst, 1)) {
				rt->rt_refcnt--;
				if (rt->rt_ifp != ifp)
					return (*rt->rt_ifp->if_output)
							(ifp, m0, dst, rt);
			} else 
				senderr(EHOSTUNREACH);
		}
		if ((rt->rt_flags & RTF_GATEWAY) && dst->sa_family != AF_NS) {
			if (rt->rt_gwroute == 0)
				goto lookup;
			if (((rt = rt->rt_gwroute)->rt_flags & RTF_UP) == 0) {
				rtfree(rt); rt = rt0;
			lookup: rt->rt_gwroute = rtalloc1(rt->rt_gateway, 1);
				if ((rt = rt->rt_gwroute) == 0)
					senderr(EHOSTUNREACH);
				/* the "G" test below also prevents rt == rt0 */
				if ((rt->rt_flags & RTF_GATEWAY) ||
				    (rt->rt_ifp != ifp)) {
					rt->rt_refcnt--;
					rt0->rt_gwroute = 0;
					senderr(EHOSTUNREACH);
				}
			}
		}
		if (rt->rt_flags & RTF_REJECT)
			senderr(rt == rt0 ? EHOSTDOWN : EHOSTUNREACH);
	}
	switch (dst->sa_family) {

#ifdef INET
	case AF_INET:
		if (!arpresolve(ac, rt, m, dst, edst))
			return (0);	/* if not yet resolved */
		/* If broadcasting on a simplex interface, loopback a copy */
		if ((m->m_flags & M_BCAST) && (ifp->if_flags & IFF_SIMPLEX) &&
		    loifp)
			mcopy = m_copy(m, 0, (int)M_COPYALL);
		type = htons(ETHERTYPE_IP);
		break;
#endif
#ifdef INET6
	case AF_INET6:
#ifndef OLDIP6OUTPUT
		if (!nd6_storelladdr(ifp, rt, m, dst, (u_char *)edst))
			return(0); /* it must be impossible, but... */
#else
		if (!nd6_resolve(ifp, rt, m, dst, (u_char *)edst))
			return(0);	/* if not yet resolves */
#endif
		type = htons(ETHERTYPE_IPV6);
		break;
#endif /* INET6 */
#ifdef NS
	case AF_NS:
		type = htons(ETHERTYPE_NS);
 		bcopy((caddr_t)&(((struct sockaddr_ns *)dst)->sns_addr.x_host),
		    (caddr_t)edst, sizeof (edst));
		if (!bcmp((caddr_t)edst, (caddr_t)&ns_thishost, sizeof(edst))) {
			if (loifp == NULL) {
				m_freem(m);
				return (ENODEV);
			}
			return ((*loifp->if_output)(ifp, m, dst, rt));
		}
		/* If broadcasting on a simplex interface, loopback a copy */
		if ((m->m_flags & M_BCAST) && (ifp->if_flags & IFF_SIMPLEX) &&
		    loifp)
			mcopy = m_copy(m, 0, (int)M_COPYALL);
		break;
#endif
#ifdef	ISO
	case AF_ISO: {
		int	snpalen;
		struct	llc *l;
		register struct sockaddr_dl *sdl;

		if (rt && (sdl = (struct sockaddr_dl *)rt->rt_gateway) &&
		    sdl->sdl_family == AF_LINK && sdl->sdl_alen > 0) {
			bcopy(LLADDR(sdl), (caddr_t)edst, sizeof(edst));
		} else if (error =
			    iso_snparesolve(ifp, (struct sockaddr_iso *)dst,
					    (char *)edst, &snpalen))
			goto bad; /* Not Resolved */
		/* If broadcasting on a simplex interface, loopback a copy */
		if (*edst & 1)
			m->m_flags |= (M_BCAST|M_MCAST);
		if ((m->m_flags & M_BCAST) && (ifp->if_flags & IFF_SIMPLEX) &&
		    loifp && (mcopy = m_copy(m, 0, (int)M_COPYALL))) {
			M_PREPEND(mcopy, sizeof (*fh), M_DONTWAIT);
			if (mcopy) {
				fh = mtod(mcopy, struct fddi_header *);
				bcopy((caddr_t)edst,
				      (caddr_t)fh->fddi_dhost, sizeof (edst));
				bcopy((caddr_t)ac->ac_enaddr,
				      (caddr_t)fh->fddi_shost, sizeof (edst));
			}
		}
		M_PREPEND(m, 3, M_DONTWAIT);
		if (m == NULL)
			return (0);
		type = 0;
		l = mtod(m, struct llc *);
		l->llc_dsap = l->llc_ssap = LLC_ISO_LSAP;
		l->llc_control = LLC_UI;
		IFDEBUG(D_ETHER)
			int i;
			printf("unoutput: sending pkt to: ");
			for (i=0; i<6; i++)
				printf("%x ", edst[i] & 0xff);
			printf("\n");
		ENDDEBUG
		} break;
#endif /* ISO */
#ifdef	LLC
/*	case AF_NSAP: */
	case AF_CCITT: {
		register struct sockaddr_dl *sdl = 
			(struct sockaddr_dl *) rt -> rt_gateway;

		if (sdl && sdl->sdl_family == AF_LINK
		    && sdl->sdl_alen > 0) {
			bcopy(LLADDR(sdl), (char *)edst,
				sizeof(edst));
		} else goto bad; /* Not a link interface ? Funny ... */
		if ((ifp->if_flags & IFF_SIMPLEX) && (*edst & 1) && loifp &&
		    (mcopy = m_copy(m, 0, (int)M_COPYALL))) {
			M_PREPEND(mcopy, sizeof (*fh), M_DONTWAIT);
			if (mcopy) {
				fh = mtod(mcopy, struct fddi_header *);
				bcopy((caddr_t)edst,
				      (caddr_t)fh->fddi_dhost, sizeof (edst));
				bcopy((caddr_t)ac->ac_enaddr,
				      (caddr_t)fh->fddi_shost, sizeof (edst));
				fh->fddi_fc = FDDIFC_LLC_ASYNC|FDDIFC_LLC_PRIO4;
			}
		}
		type = 0;
#ifdef LLC_DEBUG
		{
			int i;
			register struct llc *l = mtod(m, struct llc *);

			printf("fddi_output: sending LLC2 pkt to: ");
			for (i=0; i<6; i++)
				printf("%x ", edst[i] & 0xff);
			printf(" len 0x%x dsap 0x%x ssap 0x%x control 0x%x\n", 
			       type & 0xff, l->llc_dsap & 0xff, l->llc_ssap &0xff,
			       l->llc_control & 0xff);

		}
#endif /* LLC_DEBUG */
		} break;
#endif /* LLC */	

	case AF_UNSPEC:
	{
		struct ether_header *eh;
		eh = (struct ether_header *)dst->sa_data;
 		bcopy((caddr_t)eh->ether_dhost, (caddr_t)edst, sizeof (edst));
		if (*edst & 1)
			m->m_flags |= (M_BCAST|M_MCAST);
		type = eh->ether_type;
		break;
	}

#if NBPFILTER > 0
	case AF_LINK:
	{
		fh = mtod(m, struct fddi_header *);
		error = EPROTONOSUPPORT;
		switch (fh->fddi_fc & (FDDIFC_C|FDDIFC_L|FDDIFC_F)) {
			case FDDIFC_LLC_ASYNC: {
				/* legal priorities are 0 through 7 */
				if ((fh->fddi_fc & FDDIFC_Z) > 7)
			        	goto bad;
				break;
			}
			case FDDIFC_LLC_SYNC: {
				/* FDDIFC_Z bits reserved, must be zero */
				if (fh->fddi_fc & FDDIFC_Z)
					goto bad;
				break;
			}
			case FDDIFC_SMT: {
				/* FDDIFC_Z bits must be non zero */
				if ((fh->fddi_fc & FDDIFC_Z) == 0)
					goto bad;
				break;
			}
			default: {
				/* anything else is too dangerous */
               	 		goto bad;
			}
		}
		error = 0;
		if (fh->fddi_dhost[0] & 1)
			m->m_flags |= (M_BCAST|M_MCAST);
		goto queue_it;
	}
#endif
	default:
		printf("%s%d: can't handle af%d\n", ifp->if_name, ifp->if_unit,
			dst->sa_family);
		senderr(EAFNOSUPPORT);
	}


	if (mcopy)
		(void) (*loifp->if_output)(ifp, mcopy, dst, rt);
	if (type != 0) {
		register struct llc *l;
		M_PREPEND(m, sizeof (struct llc), M_DONTWAIT);
		if (m == 0)
			senderr(ENOBUFS);
		l = mtod(m, struct llc *);
		l->llc_control = LLC_UI;
		l->llc_dsap = l->llc_ssap = LLC_SNAP_LSAP;
		l->llc_org_code[0] = l->llc_org_code[1] = 
			l->llc_org_code[2] = 0;
		bcopy((caddr_t) &type, (caddr_t) &l->llc_ether_type,
			sizeof(u_short));
	}
	/*
	 * Add local net header.  If no space in first mbuf,
	 * allocate another.
	 */
	M_PREPEND(m, sizeof (struct fddi_header), M_DONTWAIT);
	if (m == 0)
		senderr(ENOBUFS);
	fh = mtod(m, struct fddi_header *);
	fh->fddi_fc = FDDIFC_LLC_ASYNC|FDDIFC_LLC_PRIO4;
 	bcopy((caddr_t)edst, (caddr_t)fh->fddi_dhost, sizeof (edst));
  queue_it:
 	bcopy((caddr_t)ac->ac_enaddr, (caddr_t)fh->fddi_shost,
	    sizeof(fh->fddi_shost));
	s = splimp();
	/*
	 * Queue message on interface, and start output if interface
	 * not yet active.
	 */
	if (IF_QFULL(&ifp->if_snd)) {
		IF_DROP(&ifp->if_snd);
		splx(s);
		senderr(ENOBUFS);
	}
	ifp->if_obytes += m->m_pkthdr.len;
	IF_ENQUEUE(&ifp->if_snd, m);
	if ((ifp->if_flags & IFF_OACTIVE) == 0)
		(*ifp->if_start)(ifp);
	splx(s);
	if (m->m_flags & M_MCAST)
		ifp->if_omcasts++;
	return (error);

bad:
	if (m)
		m_freem(m);
	return (error);
}

/*
 * Process a received FDDI packet;
 * the packet is in the mbuf chain m without
 * the fddi header, which is provided separately.
 */
void
fddi_input(ifp, fh, m)
	struct ifnet *ifp;
	register struct fddi_header *fh;
	struct mbuf *m;
{
	register struct ifqueue *inq;
	register struct llc *l;
	int s;
	int ifam = ifp->if_addrmask;

	if ((ifp->if_flags & IFF_UP) == 0) {
		m_freem(m);
		return;
	}
	ifp->if_ibytes += m->m_pkthdr.len + sizeof (*fh);
	if (fh->fddi_dhost[0] & 1) {
		if (bcmp((caddr_t)fddibroadcastaddr, (caddr_t)fh->fddi_dhost,
		    sizeof(fddibroadcastaddr)) == 0)
			m->m_flags |= M_BCAST;
		else
			m->m_flags |= M_MCAST;
		ifp->if_imcasts++;
	}

	l = mtod(m, struct llc *);
	switch (l->llc_dsap) {
#if defined(INET) || defined(NS) || defined(DECNET) || defined(INET6)
	case LLC_SNAP_LSAP:
	{
		unsigned fddi_type;
		if (l->llc_control != LLC_UI)
			goto notui;

		if (l->llc_ssap != LLC_SNAP_LSAP) {
			ifp->if_noproto++;
			goto dropanyway;
		}
		if (l->llc_org_code[0] != 0 || l->llc_org_code[1] != 0 ||
		    l->llc_org_code[2] != 0)
			goto dropanyway;
		fddi_type = ntohs(l->llc_ether_type);
		m_adj(m, 8);
		switch (fddi_type) {
#ifdef INET
		case ETHERTYPE_IP:
#ifdef NOTYET
			if ((ifam & 1 << NETISR_IP) == 0)
				goto dropanyway;
#endif
			schednetisr(NETISR_IP);
			inq = &ipintrq;
			break;

		case ETHERTYPE_ARP:
#ifdef NOTYET
			if ((ifam & 1 << NETISR_IP) == 0)
				goto dropanyway;
#endif
			schednetisr(NETISR_ARP);
			inq = &arpintrq;
			break;
#endif
#ifdef INET6
		/*
		 * Schedule IPv6 software interrupt for incoming IPv6 packet.
		 */
		case ETHERTYPE_IPV6:
#ifdef NOTYET
			if ((ifam & 1 << NETISR_IPV6) == 0)
				goto dropanyway;
#endif
			schednetisr(NETISR_IPV6);
			inq = &ip6intrq;
			break;
#endif /* INET6 */
#ifdef NS
		case ETHERTYPE_NS:
			if ((ifam & 1 << NETISR_NS) == 0)
				goto dropanyway;
			schednetisr(NETISR_NS);
			inq = &nsintrq;
			break;
#endif
#ifdef DECNET
		case ETHERTYPE_DECNET:
			schednetisr(NETISR_DECNET);
			inq = &decnetintrq;
			break;
#endif
		default:
			/* printf("fddi_input: unknown protocol 0x%x\n", fddi_type); */
			ifp->if_noproto++;
			goto dropanyway;
		}
		break;
	}
#endif /* INET || NS */
#ifdef	ISO
	case LLC_ISO_LSAP: 
		/* LLC_UI_P forbidden in class 1 service */
		if (l->llc_control != LLC_UI)
			goto notui;

		if ((l->llc_dsap == LLC_ISO_LSAP) &&
		    (l->llc_ssap == LLC_ISO_LSAP)
#ifdef NOTYET
		    && (ifam & 1 << NETISR_ISO)
#endif
		    ) {
			/* LSAP for ISO */
			m->m_data += 3;		/* XXX */
			m->m_len -= 3;		/* XXX */
			m->m_pkthdr.len -= 3;	/* XXX */
			M_PREPEND(m, sizeof *fh, M_DONTWAIT);
			if (m == 0)
				return;
			*mtod(m, struct fddi_header *) = *fh;
			IFDEBUG(D_ETHER)
				printf("clnp packet");
			ENDDEBUG
			schednetisr(NETISR_ISO);
			inq = &clnlintrq;
			break;
		}
		ifp->if_noproto++;
		goto dropanyway;
#endif /* ISO */

	case LLC_GLOBAL_LSAP: 
	case LLC_NULL_LSAP: 
notui:
		switch (l->llc_control) {
			
		case LLC_XID:
		case LLC_XID_P:
			if(m->m_len < 6)
				goto dropanyway;
			l->llc_window = 0;
			l->llc_fid = 9;
			l->llc_class = 1;
			l->llc_dsap = l->llc_ssap = 0;
			/* Fall through to */
		case LLC_TEST:
		case LLC_TEST_P:
		{
			struct sockaddr sa;
			register struct ether_header *eh2;
			int i;
			u_char c = l->llc_dsap;
			struct arpcom *ac = (struct arpcom *)ifp;

			l->llc_dsap = l->llc_ssap;
			l->llc_ssap = c;
			if (m->m_flags & (M_BCAST | M_MCAST))
				bcopy((caddr_t)ac->ac_enaddr,
				      (caddr_t)fh->fddi_dhost, 6);
			sa.sa_family = AF_UNSPEC;
			sa.sa_len = sizeof(sa);
			eh2 = (struct ether_header *)sa.sa_data;
			for (i = 0; i < 6; i++) {
				eh2->ether_shost[i] = c = fh->fddi_dhost[i];
				eh2->ether_dhost[i] = 
					fh->fddi_dhost[i] = fh->fddi_shost[i];
				fh->fddi_shost[i] = c;
			}
			eh2->ether_type = 0;
			ifp->if_output(ifp, m, &sa, NULL);
			return;
		}
		default:
			m_freem(m);
			return;
		}
		break;
#ifdef LLC
	case LLC_X25_LSAP:
	{
#ifdef NOTYET
		if ((ifam & 1 << NETISR_CCITT) == 0)
			goto dropanyway;
#endif
		M_PREPEND(m, sizeof(struct sdl_hdr) , M_DONTWAIT);
		if (m == 0)
			return;
		if ( !sdl_sethdrif(ifp, fh->fddi_shost, LLC_X25_LSAP,
				    fh->fddi_dhost, LLC_X25_LSAP, 6, 
				    mtod(m, struct sdl_hdr *)))
			panic("ETHER cons addr failure");
		mtod(m, struct sdl_hdr *)->sdlhdr_len = m->m_pkthdr.len - sizeof(struct sdl_hdr);
#ifdef LLC_DEBUG
		printf("llc packet\n");
#endif /* LLC_DEBUG */
		schednetisr(NETISR_CCITT);
		inq = &llcintrq;
		break;
	}
#endif /* LLC */
		
	default:
		/* printf("fddi_input: unknown dsap 0x%x\n", l->llc_dsap); */
		ifp->if_noproto++;
	dropanyway:
		m_freem(m);
		return;
	}

	s = splimp();
	if (IF_QFULL(inq)) {
		IF_DROP(inq);
		m_freem(m);
	} else
		IF_ENQUEUE(inq, m);
	splx(s);
}
/*
 * Perform common duties while attaching to interface list
 */
void
fddi_ifattach(ifp)
	register struct ifnet *ifp;
{
	struct ifaddr *ifa;
	struct sockaddr_dl *sdl;

#ifdef DEBUG
	if (ifp->if_softc == NULL)
		panic("fddi_attach");
#endif
	ifp->if_type = IFT_FDDI;
	ifp->if_baudrate = 100*1000*1000;
	ifp->if_addrlen = 6;
	ifp->if_hdrlen = 21;
	ifp->if_mtu = FDDIMTU;
	if (ifp->if_sysctl == NULL)
		ifp->if_sysctl = fddi_sysctl;
	if (ETHER_IS_MULTICAST((caddr_t)((struct arpcom *)ifp)->ac_enaddr))
		printf("WARNING: MAC address has multicast bit set\n");

	if_attach(ifp);
	if_newaddr(ifp, IFT_FDDI, (caddr_t)((struct arpcom *)ifp)->ac_enaddr);
}

int
fddi_ioctl(ifp, cmd, data)
        struct ifnet *ifp;
        int cmd;
        caddr_t data;
{
	struct ifreq *ifr = (struct ifreq *)data;
	struct ifaddr *ifa = (struct ifaddr *)data;
	struct arpcom *ac = (struct arpcom *)(ifp);
	struct ns_addr *ina;
	struct sockaddr *sa;
	int error = 0;
 
	switch (cmd) {
	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;

		switch (ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET:
			ifp->if_init(ifp->if_softc);	/* before arpwhohas */
			arp_ifinit((struct arpcom *)(ifp), ifa);
			break;
#endif
#ifdef NS
		case AF_NS:
			ina = &(IA_SNS(ifa)->sns_addr);
			if (ns_nullhost(*ina))
				ina->x_host =
				    *(union ns_host *) (ac->ac_enaddr);
			else {
				bcopy((caddr_t) ina->x_host.c_host,
				      (caddr_t) ac->ac_enaddr,
				      sizeof(ac->ac_enaddr));
			}

			/*
			 * Set new address
			 */
			ifp->if_init(ifp->if_softc);
			break;
#endif
		default:
			ifp->if_init(ifp->if_softc);
			break;
		}
		break;

	case SIOCGIFADDR:
		sa = (struct sockaddr *) & ifr->ifr_data;
		bcopy(((struct arpcom *)(ifp))->ac_enaddr,
		    (caddr_t) sa->sa_data, sizeof(ac->ac_enaddr));
		break;

	case SIOCSIFMTU:
		/*
		 * Set the interface MTU.
		 */
		if (ifr->ifr_mtu > FDDIMTU || ifr->ifr_mtu < 64) {
			error = EINVAL;
			break;
		}
		ifp->if_mtu = ifr->ifr_mtu;
		break;
	}
	return (error);
}

int
fddi_sysctl(ifp, name, namelen, oldp, oldlenp, newp, newlen)
	struct	ifnet *ifp;
	int	*name;
	u_int	namelen;
	void	*oldp;
	size_t	*oldlenp;
	void	*newp;
	size_t	newlen;
{
	struct arpcom *ac;
	struct ether_multi *ep, *op, *enm;
	void *eoldp;
	int error;

	if (namelen < 2)
		return (ENOTDIR);

	/* We currently do not implement any generic operations */
	if (name[0] == 0)
		return (EOPNOTSUPP);

	/* We only handle link-type operations here */
	if (name[1] != CTL_LINK_LINKTYPE)
		return (EOPNOTSUPP);

	if (namelen < 4)
		return (ENOTDIR);

	eoldp = (char *)oldp + *oldlenp;

	switch (name[3]) {
	case FDDICTL_MULTIADDRS:
		op = (struct ether_multi *)oldp;
		ac = (struct arpcom *)ifp;
		if (oldp == NULL) {
			for (ep = ac->ac_multiaddrs; ep != NULL;
			     ep = ep->enm_next)
				op++;
			*oldlenp = 11 * ((caddr_t)op - (caddr_t)oldp) / 10;
			return (0);
		}
		if ((enm = malloc(sizeof(*op), M_TEMP, M_WAITOK)) == NULL)
			return (ENOMEM);
		error = 0;
		for (ep = ac->ac_multiaddrs; ep != NULL; ep = ep->enm_next) {
			*enm = *ep;
			enm->enm_ac = NULL;
			if (enm->enm_next != NULL)
				enm->enm_next = op + 1;
			if ((void *)(op + 1) > eoldp) {
				error = ENOMEM;
				break;
			}
			if ((error = copyout(enm, op, sizeof(*op))) != 0)
				break;
			op++;
		}
		if (error == 0)
			*oldlenp = (caddr_t)op - (caddr_t)oldp;
		free(enm, M_TEMP);
		return (error);

	default:
		error = EOPNOTSUPP;
		break;
	}	

	return (error);
}
