/*	BSDI if_loop.c,v 2.11.4.1 2001/09/04 15:29:23 dab Exp	*/

/*
 * This file has been slightly modified by NRL for use with IPv6+IPsec.
 * Search for INET6 and/or IPSEC to see the blocks where this happened.
 * See the NRL Copyright notice for conditions on the modifications.
 */
/*
 * Copyright (c) 1982, 1986, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)if_loop.c	8.2 (Berkeley) 1/9/95
 */

/*
 * Loopback interface driver for protocol testing and timing.
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <machine/cpu.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/bpf.h>

#ifdef	INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#ifdef ISO
#include <netiso/iso.h>
#include <netiso/iso_var.h>
#endif

#include "bpfilter.h"

#define	LOMTU	(4096+256)

struct	ifnet *loifp;
int	nloif;		/* The number of loopback interfaces */

static int losysctl  __P((struct ifnet *, int *, u_int, void *, size_t *, void *,
    size_t));


static
void
loinit(ifp, unit)
	struct ifnet *ifp;
	int unit;
{

	bzero(ifp, sizeof(*ifp));
	ifp->if_name = "lo";
	ifp->if_unit = unit;
	ifp->if_mtu = LOMTU;
	ifp->if_flags = IFF_LOOPBACK | IFF_MULTICAST;
	ifp->if_ioctl = loioctl;
	ifp->if_output = looutput;
	ifp->if_type = IFT_LOOP;
	ifp->if_hdrlen = 0;
	ifp->if_addrlen = 0;
	ifp->if_sysctl = losysctl;
	if_attach(ifp);
	bpfattach(&ifp->if_bpf, ifp, DLT_NULL, sizeof(u_int));
}

void
loopattach(n)
	int n;
{
	register struct ifnet *ifp;
	int i;

	/*
	 * If "nloif" has been patched, use it instead of the
	 * value from the config file.
	 */
	if (nloif)
		n = nloif;
	nloif = n;

	loifp = malloc(n * sizeof(struct ifnet), M_DEVBUF, M_WAIT);
	bzero(loifp, n * sizeof(struct ifnet));
	for (i = 0, ifp = loifp; i < nloif; i++, ifp++)
		loinit(ifp, i);
}

void
lomore(n)
	int n;
{
	struct ifnet *ifp;

	ifp = malloc((n - nloif) * sizeof(*ifp), M_DEVBUF, M_WAIT);
	bzero(ifp, (n - nloif) * sizeof(*ifp));
	for (; nloif < n; ifp++, nloif++)
		loinit(ifp, nloif);
}

int
looutput(ifp, m, dst, rt)
	struct ifnet *ifp;
	register struct mbuf *m;
	struct sockaddr *dst;
	register struct rtentry *rt;
{
	int s, isr;
	register struct ifqueue *ifq = 0;

	if ((m->m_flags & M_PKTHDR) == 0)
		panic("looutput no HDR");
	/*
	 * We can get called with a non-loopback interface
	 * pointer.  This happens in the ether/fddi/token
	 * code when they need to loop back a copy of a
	 * broadcast packet.
	 *
	 * Ideally, those calls should pass in loifp as
	 * the ifp, but that causes m_pkthdr.rcvif to point
	 * to the loopback driver, and that can cause problems
	 * in things like the src route check and IP filter
	 * code.  So, set the rcvif, and then reset ifp to
	 * loifp if we weren't passed in a loopback interface,
	 * so that things like bpf and statistics work
	 * correctly.
	 */
	m->m_pkthdr.rcvif = ifp;
	if ((ifp->if_flags & IFF_LOOPBACK) == 0)
		ifp = loifp;

	if (ifp->if_bpf) {
		/*
		 * We need to prepend the address family as
		 * a four byte field.  Cons up a dummy header
		 * to pacify bpf.  This is safe because bpf
		 * will only read from the mbuf (i.e., it won't
		 * try to free it or keep a pointer a to it).
		 */
		struct mbuf m0;
		u_int af = dst->sa_family;

		m0.m_next = m;
		m0.m_len = 4;
		m0.m_data = (char *)&af;
		
		bpf_mtap(ifp->if_bpf, &m0);
	}

	if (rt && rt->rt_flags & (RTF_REJECT|RTF_BLACKHOLE)) {
		m_freem(m);
		return (rt->rt_flags & RTF_BLACKHOLE ? 0 :
		        rt->rt_flags & RTF_HOST ? EHOSTUNREACH : ENETUNREACH);
	}

#ifndef PULLDOWN_TEST
	/*
	 * KAME requires that the packet to be contiguous on the
	 * mbuf.  We need to make that sure.
	 * this kind of code should be avoided.
	 * XXX other conditions to avoid running this part?
	 */
	if (m && m->m_next != NULL) {
		struct mbuf *n;

		MGETHDR(n, M_DONTWAIT, MT_HEADER);
		if (n) {
			MCLGET(n, M_DONTWAIT);
			if ((n->m_flags & M_EXT) == 0) {
				m_free(n);
				n = NULL;
			}
		}
		if (!n) {
			printf("looutput: mbuf allocation failed\n");
			m_freem(m);
			return ENOBUFS;
		}

		n->m_pkthdr.rcvif = m->m_pkthdr.rcvif;
		n->m_pkthdr.len = m->m_pkthdr.len;
		if (m->m_pkthdr.len <= MCLBYTES) {
			m_copydata(m, 0, m->m_pkthdr.len, mtod(n, caddr_t));
			n->m_len = m->m_pkthdr.len;
			m_freem(m);
		} else {
			m_copydata(m, 0, MCLBYTES, mtod(n, caddr_t));
			m_adj(m, MCLBYTES);
			n->m_len = MCLBYTES;
			n->m_next = m;
			m->m_flags &= ~M_PKTHDR;
		}
		m = n;
	}
#if 0
	if (m && m->m_next != NULL) {
		printf("loop: not contiguous...\n");
		m_freem(m);
		return ENOBUFS;
	}
#endif
#endif

	ifp->if_opackets++;
	ifp->if_obytes += m->m_pkthdr.len;
	switch (dst->sa_family) {

#ifdef INET
	case AF_INET:
#if	BYTE_ORDER != BIG_ENDIAN
		/* 
		 * XXX - Make sure the data in the IP header is not
		 * shared as ip_input() byte swaps at least the length
		 * field.
		 */
		if ((m->m_flags & M_EXT) &&
		    mclrefcnt[mtocl(m->m_ext.ext_buf)] > 1 &&
		    (m = m_pullup(m, sizeof(struct ip))) == NULL)
			return (ENOBUFS);
#endif
		ifq = &ipintrq;
		isr = NETISR_IP;
		break;
#endif
#ifdef INET6
	case AF_INET6:
		m->m_flags |= M_LOOP;
		ifq = &ip6intrq;
		isr = NETISR_IPV6;
		break;
#endif /* INET6 */
#ifdef NS
	case AF_NS:
		ifq = &nsintrq;
		isr = NETISR_NS;
		break;
#endif
#ifdef ISO
	case AF_ISO:
		ifq = &clnlintrq;
		isr = NETISR_ISO;
		break;
#endif
	default:
		printf("lo%d: can't handle af%d\n", ifp->if_unit,
			dst->sa_family);
		m_freem(m);
		return (EAFNOSUPPORT);
	}
	s = splimp();
	if (IF_QFULL(ifq)) {
		IF_DROP(ifq);
		m_freem(m);
		splx(s);
		return (ENOBUFS);
	}
	IF_ENQUEUE(ifq, m);
	schednetisr(isr);
	ifp->if_ipackets++;
	ifp->if_ibytes += m->m_pkthdr.len;
	splx(s);
	return (0);
}

/* ARGSUSED */
void
lortrequest(cmd, rt, info)
	int cmd;
	struct rtentry *rt;
	struct rt_addrinfo *info;
{

	if (rt)
		rt->rt_rmx.rmx_mtu = LOMTU;
}

/*
 * Process an ioctl request.
 */
/* ARGSUSED */
int
loioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	u_long cmd;
	caddr_t data;
{
	register struct ifaddr *ifa;
	register struct ifreq *ifr;
	register int error = 0;

	switch (cmd) {

	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;
		ifa = (struct ifaddr *)data;
		if (ifa != 0 && ifa->ifa_addr->sa_family == AF_ISO)
			ifa->ifa_rtrequest = lortrequest;
		/*
		 * Everything else is done at a higher level.
		 */
		break;

	case SIOCSIFMTU:
		ifr = (struct ifreq *)data;
		if (ifr->ifr_mtu < 64 || ifr->ifr_mtu > 65*1024) {
			error = EINVAL;
			break;
		}
		ifp->if_mtu = ifr->ifr_mtu;
		break;

	case SIOCADDMULTI:
	case SIOCDELMULTI:
		ifr = (struct ifreq *)data;
		if (ifr == 0) {
			error = EAFNOSUPPORT;		/* XXX */
			break;
		}
		switch (ifr->ifr_addr.sa_family) {

#ifdef INET6
		case AF_INET6:
			break;
#endif /* INET6 */
#ifdef INET
		case AF_INET:
			break;
#endif

		default:
			error = EAFNOSUPPORT;
			break;
		}
		break;

	default:
		error = EINVAL;
	}
	return (error);
}


/*
 * Sysctl for lo.  At this level we currently only handle type
 * specific calls.
 */
/* ARGSUSED */
static int
losysctl(ifp, name, namelen, oldp, oldlenp, newp, newlen)
	struct	ifnet *ifp;
	int	*name;
	u_int	namelen;
	void	*oldp;
	size_t	*oldlenp;
	void	*newp;
	size_t	newlen;
{
	int error;
	int val;

	/* We currently only support generic sysctls */
	if (name[0] != 0)
		return (EOPNOTSUPP);

	if (namelen != 3)
		return (ENOTDIR);
	switch (name[2]) {
	case CTL_LINK_IFTYPE_NUMIF:
		val = nloif;
		error = sysctl_int(oldp, oldlenp, newp, newlen, &val);
		if (error == 0 && newp) {
			if (val < nloif)
				return (EINVAL);
			if (val > nloif)
				lomore(val);
		}
		break;

	default:
		error = EOPNOTSUPP;
		break;
	}
	return (error);
}
