/*-
 * Copyright (c) 1993, 1994, 1995, 1996 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI if_p2psubr.c,v 2.17 1998/10/19 15:02:50 jch Exp
 */

/*
 * Common code for point-to-point synchronous serial links
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/proc.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/syslog.h>

#include <machine/pcpu.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/if_dl.h>
#include <net/if_types.h>

#include <net/if_p2p.h>

extern int np2pproto;

/*
 * Point-to-point interface attachment.
 * Checks the interface type and calls the
 * corresponding link-layer protocol attach routine.
 */
int
p2p_attach(pp)
	struct p2pcom *pp;
{
	register struct ifnet *ifp = &pp->p2p_if;
	short type;
	int s;
	int error = 0;

	s = splimp();

#ifdef	DEBUG
	if (pp->p2p_private)
		panic("p2p_attach: protocol did not clean up p2p_private");
#endif

	for (type = 0; type < np2pproto; ++type)
		if (ifp->if_type == p2pprotosw[type].pp_type)
			break;

	if (type == np2pproto) {
		error = EPFNOSUPPORT;
		goto out;
	}

	if (pp->p2p_proto != &p2pprotosw[type]) {
		ifp->if_output = p2pprotosw[type].pp_output;
		ifp->if_ctloutput = p2p_ctloutput;
		pp->p2p_input = p2pprotosw[type].pp_input;
		pp->p2p_modem = p2pprotosw[type].pp_modem;
		pp->p2p_hdrinput = p2pprotosw[type].pp_hdrinput;
		pp->p2p_proto = &p2pprotosw[type];
		pp->p2p_getmdm = 0;	/* force the attach to assign this */
	}
	if (pp->p2p_proto->pp_attach)
		error = ((*pp->p2p_proto->pp_attach)(pp));
	if (error == 0)
		bpfattach(&ifp->if_bpf, ifp, p2pprotosw[type].pp_bpftype,
		    p2pprotosw[type].pp_hdrlen);

out:
	splx(s);
	return (error);
}

/*
 * Point-to-point protocol detach routine
 */
int
p2p_detach(pp)
	struct p2pcom *pp;
{
	struct ifaddr *ifa;
	int s;

	/* All addresses except the AF_LINK should have been removed. */
	for (ifa = pp->p2p_if.if_addrlist; ifa != NULL; ifa = ifa->ifa_next)
		if (ifa->ifa_addr == NULL || 
		    ifa->ifa_addr->sa_family != AF_LINK)
			return (EBUSY);

	s = splimp();

	if (pp->p2p_proto && pp->p2p_proto->pp_detach)
		(*pp->p2p_proto->pp_detach)(pp);
#ifdef	DEBUG
	if (pp->p2p_private)
		panic("p2p_detach: protocol did not clean up p2p_private");
#endif
	/*
	 * By default we keep the POINTOPOINT flag on.
	 * Individual protocols might optionally clear this flag.
	 */
	pp->p2p_if.if_flags |= IFF_POINTOPOINT;
	splx(s);

	return (0);
}

/*
 * Point-to-point interface initialization.
 */
int
p2p_init(pp)
	struct p2pcom *pp;
{
	int error = 0;

	if (pp->p2p_proto && pp->p2p_proto->pp_init)
		error = (*pp->p2p_proto->pp_init)(pp);
	return (error);
}

/*
 * Point-to-point interface shutdown.
 */
void
p2p_shutdown(pp)
	struct p2pcom *pp;
{
	if (pp->p2p_proto && pp->p2p_proto->pp_shutdown)
		(*pp->p2p_proto->pp_shutdown)(pp);
}

/*
 * Handle protocol-specific ioctl
 */
int
p2p_ioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	u_long cmd;
	caddr_t data;
{
	struct proc *p = PCPU(curproc);	/* XXX should be parameter */
	struct p2pcom *pp = (struct p2pcom *) ifp;
	struct ifreq *ifr;
	struct ifaddr *ifa;
	struct sockaddr_dl *sdl;
	short type, otype;
	int error;
	int s;

	/*
	 * Call the protocol-specific ioctl handler
	 */
	s = splimp();
	if (pp->p2p_proto && pp->p2p_proto->pp_ioctl) {
		error = (*pp->p2p_proto->pp_ioctl)(pp, cmd, data);
		if (error != EINVAL)
			goto out;
	}

	otype = type = pp->p2p_if.if_type;

	/*
	 * Process the common ioctls
	 */
	error = 0;
	switch (cmd) {
	case SIOCSIFADDR:
		if (p && (error = suser(p->p_ucred, &p->p_acflag)))
			goto out;
		break;
	}
	switch (cmd) {

	default:
		error = EINVAL;
		goto out;

	case SIOCSIFADDR:
		if (type == IFT_NONE)
			error = ENETDOWN;
		else
			ifp->if_flags |= IFF_UP;
		break;

	case _SIOCSLINK:
		/*
		 * Internal-use version of SIOCSIFADDR for AF_LINK only;
		 * used to prevent type from changing accidentally
		 * if driver has default case for setting protocol addr.
		 */
		if (p && (error = suser(p->p_ucred, &p->p_acflag)))
			goto out;

		ifa = (struct ifaddr *)data;
		sdl = (struct sockaddr_dl *)(ifa->ifa_addr);
		for (type = 0; type < np2pproto; ++type)
			if (sdl->sdl_type == p2pprotosw[type].pp_type)
				break;
		if (type == np2pproto) {
			error = EAFNOSUPPORT;
			goto out;
		}
		type = sdl->sdl_type;
		break;

	case SIOCSIFDSTADDR:
	case SIOCSIFFLAGS:
		break;

	/*
	 * These should probably be done at a lower level,
	 * but this should be ok for now.
	 */
	case SIOCADDMULTI:
	case SIOCDELMULTI:
		ifr = (struct ifreq *)data;
		switch (ifr->ifr_addr.sa_family) {
#ifdef INET
		case AF_INET:
			break;
#endif
		default:
			error = EAFNOSUPPORT;
			break;
		}
		break;
	}

	/*
	 * An interface always goes down when changing types.
	 */
	if (type != otype)
		ifp->if_flags &= ~IFF_UP;

	/*
	 * Notify raw link layer sockets of IFF_UP changes
	 */
	if ((pp->p2p_oldflags & IFF_UP) != (ifp->if_flags & IFF_UP))
		if_statechg(ifp);

	/*
	 * On a change from UP to DOWN call shutdown.
	 */
	if ((pp->p2p_oldflags & IFF_UP) && !(ifp->if_flags & IFF_UP))
		p2p_shutdown(pp);

	/*
	 * If changing types detach the old protocol and attach the new one.
	 */
	if (type != otype) {
		if ((error = p2p_detach(pp)))
		    goto out;

		pp->p2p_if.if_type = type;
		if (error = p2p_attach(pp)) {
			/*
			 * Hmm, we failed.  Try to go back to the
			 * old type.  If we can't do that, default
			 * to the IFT_NONE case as a last resort.
			 */
			pp->p2p_if.if_type = otype;
			if (p2p_attach(pp)) {
				pp->p2p_if.if_type = IFT_NONE;
				p2p_attach(pp);
			}
			goto out;
		}
	}

	/*
	 * On a change from DOWN to UP call init.
	 */
	if (!(pp->p2p_oldflags & IFF_UP) && (ifp->if_flags & IFF_UP))
		error = p2p_init(pp);

	pp->p2p_oldflags = ifp->if_flags;
out:
	splx(s);
	return (error);
}

#define dprintf(x)	if (pp->p2p_if.if_flags & IFF_DEBUG) printf x

/*
 * Put nbytes of data starting at cp into the input mbuf.
 * May be called multiple times as more data in the packet
 * arrives.
 */
int
p2p_hdr(pp, cp, nbytes)
	struct p2pcom *pp;
	u_char *cp;
	int nbytes;
{
	struct mbuf *m = pp->p2p_ibuf;
	struct mbuf *m1;
	int r;

	if (nbytes == -1) {
		if (m) {
			if (m->m_next)
				m_freem(m->m_next);
			m->m_next = 0;
			m->m_len = 0;
		}
		return (0);
	}

	if (m == 0) {
		MGETHDR(m, M_DONTWAIT, MT_DATA);
		if (m == 0)
			return (-1);
		pp->p2p_ibuf = m;
		m->m_len = 0;
		m->m_next = 0;
	}

	/*
	 * Since we know nothing about the headers,
	 * the first mbuf will always be empty.
	 * If we do not have a second mbuf yet then
	 * this must be the first byte going in
	 * and it is safe to reset everything.
	 */
	if (m->m_next == 0) {
		m->m_pkthdr.rcvif = &pp->p2p_if;
		m->m_pkthdr.len = 0;
		m->m_data = m->m_pktdat;
		m->m_len = 0;
	} else
		while (m->m_next)
			m = m->m_next;

	while (nbytes-- > 0) {
	 	r = ((m->m_flags & M_EXT) ? MCLBYTES :
		     (m->m_flags & M_PKTHDR) ? MHLEN : MLEN) - m->m_len;

		if (m == pp->p2p_ibuf || r <= 0) {
			if (m == pp->p2p_ibuf) {
				MGETHDR(m1, M_DONTWAIT, MT_DATA);
				if (m1)
					m1->m_pkthdr.rcvif = &pp->p2p_if;
				r = MHLEN;
			} else {
				MGET(m1, M_DONTWAIT, MT_DATA);
				r = MLEN;
			}

			if (m1 == 0) {	/* No space available */
				dprintf(("%s%d: no mbufs left\n",
				    pp->p2p_if.if_name, pp->p2p_if.if_unit));

				printf("MGET -- no space\n");
				m = pp->p2p_ibuf;
				if (m->m_next)
					m_freem(m->m_next);
				m->m_len = 0;
				m->m_next = 0;
				return (-1);
			}

			if (nbytes >= r || m != pp->p2p_ibuf) {
				MCLGET(m1, M_DONTWAIT);
				if (m1->m_flags & M_EXT)
					r = MCLBYTES;
			}
			m1->m_len = 0;
			m->m_next = m1;
			m = m1;
		}
		if (nbytes > 0) {
			int n = nbytes < r ? nbytes + 1 : r;
			bcopy(cp, mtod(m, u_char *) + m->m_len, n);

			m->m_len += n;
			nbytes -= (n + 1);
			cp += n;
			pp->p2p_ibuf->m_pkthdr.len += n;
		} else {
			m->m_data[m->m_len++] = *cp++;
			pp->p2p_ibuf->m_pkthdr.len++;
		}
	}
	return (1);
}

int
p2p_ctloutput(op, ifp, level, optname, m)
	int op;
	struct ifnet *ifp;
	int level, optname;
	struct mbuf **m;
{
	struct proc *p = PCPU(curproc);	/* XXX should be parameter */
	struct p2pcom *pp = (struct p2pcom *) ifp;
	int error;

	if (level != IFT_NONE)
		return (pp->p2p_proto->pp_ctloutput ?
		    (*pp->p2p_proto->pp_ctloutput)(op, ifp, level, optname, m) :
		    EINVAL);

	error = 0;

	switch (optname) {
	case P2P_DTR:
		if (p && (error = suser(p->p_ucred, &p->p_acflag)))
			return(error);
		if (op != PRCO_SETOPT)
			return (EINVAL);
		if ((*m)->m_len != sizeof(int))
			return (EINVAL);
                if (pp->p2p_mdmctl)
                        return((*pp->p2p_mdmctl)(pp, *(int *)(*m)->m_data));
		return (ENODEV);

	case P2P_DCD:
		if (op == PRCO_SETOPT)
			return (EINVAL);
		*m = m_get(M_WAIT, MT_SOOPTS);
		(*m)->m_len = sizeof(int);

                if (pp->p2p_getmdm)
                        error = (*pp->p2p_getmdm)(pp, (*m)->m_data);
                else
                        *(int *)(*m)->m_data = TIOCM_CAR;
		return (error);
	default:
		return (EINVAL);
	}
}
