/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI if_tokensubr.c,v 2.28 2000/08/30 14:15:35 geertj Exp
 */

/*
 * Based on code from Roger Florkowski, which in turn
 * was based on net/if_ethersubr.c.
 */

/*
 * Copyright (c) 1982, 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>

#include <machine/cpu.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/if_llc.h>
#include <net/if_dl.h>
#include <net/if_types.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>		/* grr */
#endif
#include <netinet/if_ether.h>
#include <net/if_token.h>

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#include <netns/idp.h>
#endif

#ifdef ISO
#include <netiso/argo_debug.h>
#include <netiso/iso.h>
#include <netiso/iso_var.h>
#include <netiso/iso_snpac.h>
#endif

#ifdef LLC
#include <netccitt/dll.h>
#include <netccitt/llc_var.h>
#endif

#if defined(LLC) && defined(CCITT)
extern struct ifqueue pkintrq;
#endif

#ifdef INET
struct	llc llc_snap_ip;
struct	llc llc_snap_arp;
u_char	token_ip_lbcast[sizeof(struct token_header) + LLC_SNAPLEN];
u_char	token_ip_mbcast[sizeof(struct token_header) + 2 + LLC_SNAPLEN];
u_char	token_ip_lmcast[sizeof(struct token_header) + LLC_SNAPLEN];
u_char	token_ip_mmcast[sizeof(struct token_header) + 2 + LLC_SNAPLEN];
#endif

#ifdef NS
struct	llc llc_ipx;
u_char	token_ipx_lbcast[] = {
	ACF_PRIORITY3,					/* ACF */
	FCF_LLC_FRAME,					/* FCF */
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff,		/* Dhost */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,		/* Shost */
	0xe0, 0xe0, 0x03				/* LLC/IPX */
};
u_char	token_ipx_mbcast[] = {
	ACF_PRIORITY3,					/* ACF */
	FCF_LLC_FRAME,					/* FCF */
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff,		/* Dhost */
	RI_PRESENT, 0x00, 0x00, 0x00, 0x00, 0x00,	/* Shost */
	0,						/* RCF0 */
	RCF1_FRAME_MASK,				/* RCF1 */
	0xe0, 0xe0, 0x03				/* LLC/IPX */
};
#define SDL(s)	((struct sockaddr_dl *)s)

struct sockaddr_ns sipx = {sizeof(sipx), AF_NS};
void token_rresolve __P((struct rtentry *rt, struct ifnet *ifp,
    struct sockaddr_ns *dst));
int token_nshdr __P((struct token_max_hdr *hp, char *src, char *dst));
void token_update_srt __P((struct ifnet *ifp, struct token_max_hdr *mac,
    struct idp *idpp, int srtlen, int hdrlen));
#endif
extern	struct ifnet *loifp;
#define senderr(e) { error = (e); goto bad;}
#define ADDR_EQ(a, b) \
	((*(u_long *)&(a)[2] == *(u_long *)&(b)[2]) &&	\
	(*(u_short *)&(a)[0] == *(u_short *)&(b)[0]))

/*
 * Perform common duties while attaching to interface list
 */
void
token_attach(ifp)
	struct ifnet *ifp;
{
	struct token_max_hdr *tp;
	int i;

#ifdef DEBUG
	if (ifp->if_softc == NULL)
		panic("token_attach");
#endif
	ifp->if_type = IFT_ISO88025;
	ifp->if_addrlen = ISO88025_ADDR_LEN;
	ifp->if_hdrlen = sizeof(struct token_header);	/* sometimes */
	ifp->if_mtu = ISO88025_MTU;
	ifp->if_output = token_output;
	if (ifp->if_sysctl == NULL)
		ifp->if_sysctl = token_sysctl;
	if (max_linkhdr < ALIGN(sizeof(struct token_header) + LLC_SNAPLEN))
		max_linkhdr = ALIGN(sizeof(struct token_header) + LLC_SNAPLEN);
	if (ETHER_IS_MULTICAST((caddr_t)((struct arpcom *)ifp)->ac_enaddr))
		printf("WARNING: MAC address has multicast bit set\n");

	if_attach(ifp);
	if_newaddr(ifp, IFT_ISO88025, (caddr_t)((struct arpcom *)ifp)->ac_enaddr);

#ifdef INET
	/* construct prototype llc/snap header for IP */
	llc_snap_ip.llc_dsap = LLC_SNAP_LSAP;
	llc_snap_ip.llc_ssap = LLC_SNAP_LSAP;
	llc_snap_ip.llc_control = LLC_UI;
#if 0	/* this is in bss */
	llc_snap_ip.llc_org_code[0] = 0;
	llc_snap_ip.llc_org_code[1] = 0;
	llc_snap_ip.llc_org_code[2] = 0;
#endif
	llc_snap_ip.llc_ether_type = htons(ETHERTYPE_IP);

	/* construct prototype llc/snap header for ARP */
	llc_snap_arp = llc_snap_ip;
	llc_snap_arp.llc_ether_type = htons(ETHERTYPE_ARP);

	/* Construct prototype broadcast headers */
	tp = (struct token_max_hdr *)token_ip_lbcast;
	for (i = 0; i < ISO88025_ADDR_LEN; i++)
		tp->hdr.token_dhost[i] = 0xff;
	tp->hdr.token_acf = ACF_PRIORITY3;
	tp->hdr.token_fcf = FCF_LLC_FRAME;
	bcopy(&llc_snap_ip, &tp->rif, LLC_SNAPLEN);

	tp = (struct token_max_hdr *)token_ip_mbcast;
	bcopy(token_ip_lbcast, token_ip_mbcast, sizeof(struct token_header));
	/* rcf0 is filled in at runtime */
	tp->rif.rcf1 = RCF1_FRAME_MASK;
	bcopy(&llc_snap_ip, (u_char *)tp->rif.rseg, LLC_SNAPLEN);

	/* rfc1469 multicast prototype header */
	tp = (struct token_max_hdr *)token_ip_lmcast;
	bcopy(token_ip_lbcast, token_ip_lmcast, sizeof(struct token_header));
	tp->hdr.token_dhost[0] = 0xc0;
	tp->hdr.token_dhost[1] = 0x00;
	tp->hdr.token_dhost[2] = 0x00;
	tp->hdr.token_dhost[3] = 0x04;
	tp->hdr.token_dhost[4] = 0x00;
	tp->hdr.token_dhost[5] = 0x00;
	tp->hdr.token_acf = ACF_PRIORITY3;
	tp->hdr.token_fcf = FCF_LLC_FRAME;
	bcopy(&llc_snap_ip, &tp->rif, LLC_SNAPLEN);

	tp = (struct token_max_hdr *)token_ip_mmcast;
	bcopy(token_ip_lbcast, token_ip_mmcast, sizeof(struct token_header));
	tp->hdr.token_dhost[0] = 0xc0;
	tp->hdr.token_dhost[1] = 0x00;
	tp->hdr.token_dhost[2] = 0x00;
	tp->hdr.token_dhost[3] = 0x04;
	tp->hdr.token_dhost[4] = 0x00;
	tp->hdr.token_dhost[5] = 0x00;
	tp->hdr.token_acf = ACF_PRIORITY3;
	tp->hdr.token_fcf = FCF_LLC_FRAME;
	/* rcf0 is filled in at runtime */
	tp->rif.rcf1 = RCF1_FRAME_MASK;
	bcopy(&llc_snap_ip, (u_char *)tp->rif.rseg, LLC_SNAPLEN);
#endif /* INET */
}

/*
 * generic token-ring output routine.
 * Encapsulate a packet of type family for the local net.
 * Assumes that ifp is actually pointer to arpcom structure.
 */
int
token_output(ifp, m0, dst, rt0)
	struct ifnet *ifp;
	struct mbuf *m0;
	struct sockaddr *dst;
	struct rtentry *rt0;
{
	int s;
	int error = 0;
	struct token_max_hdr *thdr;
	u_char ttmp[8 + sizeof(struct token_max_hdr)];
	struct mbuf *m = m0;
	struct rtentry *rt;
	struct mbuf *mcopy = (struct mbuf *)0;
	u_char *phdr;
	int hdr_len;
	struct arpcom *ac = (struct arpcom *)ifp;
	int ismulti = 0;
#ifdef NS
	struct sockaddr_dl *sdl;
	struct sockaddr_dl_8025 sdlt;
#endif

	if ((ifp->if_flags & (IFF_UP | IFF_RUNNING)) != (IFF_UP | IFF_RUNNING)) 
		senderr(ENETDOWN);
	if (rt = rt0) {
		if ((rt->rt_flags & RTF_UP) == 0)
			if (rt0 = rt = rtalloc1(dst, 1)) {
				rt->rt_refcnt--;
				if (rt->rt_ifp != ifp)
					return (*rt->rt_ifp->if_output)
							(ifp, m0, dst, rt);
			} else 
				senderr(EHOSTUNREACH);
		if (rt->rt_flags & RTF_GATEWAY) {
			if (rt->rt_gwroute == 0)
				goto lookup;
			if (((rt = rt->rt_gwroute)->rt_flags & RTF_UP) == 0) {
				rtfree(rt); rt = rt0;
			lookup: rt->rt_gwroute = rtalloc1(rt->rt_gateway, 1);
				if ((rt = rt->rt_gwroute) == 0)
					senderr(EHOSTUNREACH);
				/* the "G" test below also prevents rt == rt0 */
				if ((rt->rt_flags & RTF_GATEWAY) ||
				    (rt->rt_ifp != ifp)) {
					rt->rt_refcnt--;
					rt0->rt_gwroute = 0;
					senderr(EHOSTUNREACH);
				}
			}
		}
		if (rt->rt_flags & RTF_REJECT)
			senderr(rt == rt0 ? EHOSTDOWN : EHOSTUNREACH);
	}

	switch (dst->sa_family) {

	case AF_LINK:
		break;

#ifdef INET
	case AF_INET:
		if (m->m_flags & (M_BCAST | M_MCAST)) {
			/*
			 * If broadcasting on a simplex interface, 
			 * loopback a copy.
			 */
			if ((m->m_flags & M_BCAST) &&
			    (ifp->if_flags & IFF_SIMPLEX))
				mcopy = m_copy(m, 0, (int)M_COPYALL);

			/*
			 * Patch up a canned header
			 */
			if (ac->ac_flags & AC_SOURCERT) {
				if ((m->m_flags & M_MCAST) && 
				    (ac->ac_flags & AC_MFA))
					thdr = (struct token_max_hdr *)
					    token_ip_mmcast;
				else
					thdr = (struct token_max_hdr *)
					    token_ip_mbcast;
				if (ac->ac_flags & AC_ALLBC)
					thdr->rif.rcf0 =
					    2 | RCF0_ALL_BROADCAST;
				else
					thdr->rif.rcf0 = 
					    2 | RCF0_SINGLE_BROADCAST;
				bcopy(ac->ac_enaddr, thdr->hdr.token_shost,
				    ISO88025_ADDR_LEN); 
				thdr->hdr.token_shost[0] |= RI_PRESENT;
				hdr_len = sizeof(struct token_header) + 2 +
				    LLC_SNAPLEN;
			} else {
				if ((m->m_flags & M_MCAST) && 
				    (ac->ac_flags & AC_MFA))
					thdr = (struct token_max_hdr *)
					    token_ip_lmcast;
				else
					thdr = (struct token_max_hdr *)
					    token_ip_lbcast;
				bcopy(ac->ac_enaddr, thdr->hdr.token_shost,
				    ISO88025_ADDR_LEN); 
				hdr_len = sizeof(struct token_header) +
				    LLC_SNAPLEN;
			}
		} else {
			/* XXX It would be nice to avoid arpresolve's copy. */
			if ((hdr_len = arpresolve(ac, rt, m, dst, ttmp)) == 0)
				return (0);	/* if not yet resolved */
			thdr = (struct token_max_hdr *)ttmp;
			if (ETHER_IS_MULTICAST(&ttmp[2]))
				ismulti++;
		}

		M_PREPEND(m, hdr_len, M_DONTWAIT);
		if (m == 0) {
			error = ENOBUFS;
			goto bad;
		}
		phdr = mtod(m, u_char *);
		bcopy(thdr, phdr, hdr_len);
		break;
#endif
#ifdef NS
	case AF_NS:
		if (m->m_flags & M_BCAST) {
			/*
			 * Patch up a canned header
			 */
			if (ac->ac_flags & AC_SOURCERT) {
				thdr = (struct token_max_hdr *)token_ipx_mbcast;
				if (ac->ac_flags & AC_ALLBC)
					thdr->rif.rcf0 = 2 | RCF0_ALL_BROADCAST;
				else
					thdr->rif.rcf0 =
					    2 | RCF0_SINGLE_BROADCAST;
				ETHER_COPY(ac->ac_enaddr,
				    thdr->hdr.token_shost);
				thdr->hdr.token_shost[0] |= RI_PRESENT;
				hdr_len = sizeof(struct token_header) + 2 +
				    LLC_MINLEN;
			} else {
				thdr = (struct token_max_hdr *)token_ipx_lbcast;
				ETHER_COPY(ac->ac_enaddr,
				    thdr->hdr.token_shost);
				hdr_len = sizeof(struct token_header) +
				    LLC_MINLEN;
			}
		} else {
			thdr = (struct token_max_hdr *)ttmp;
			if (rt == NULL) {
				/*
				 * SO_DONTROUTE gets us here, just send using
				 * the destination address as the MAC address
				 * (there's nowhere to cache it in this case).
				 */
				thdr = (struct token_max_hdr *)sdlt.sdl_data;
				hdr_len = token_nshdr(thdr, ac->ac_enaddr,
				    (char *)&(((struct sockaddr_ns *)dst)->
				    sns_addr.x_host));
			} else {
				sdl = SDL(rt->rt_gateway);
				if (sdl->sdl_family == AF_NS) {
					token_rresolve(rt, ifp,
					    (struct sockaddr_ns *)dst);
					sdl = SDL(rt->rt_gateway);
				}
#ifdef DEBUG
				if (sdl->sdl_family != AF_LINK)
					panic("ns: route still NS");

#endif
				thdr = (struct token_max_hdr *)sdl->sdl_data;
				hdr_len = sdl->sdl_alen;
			}
			if (ETHER_IS_MULTICAST(&thdr->hdr.token_dhost[0]))
				ismulti++;
		}
		M_PREPEND(m, hdr_len, M_DONTWAIT);
		if (m == 0) {
			error = ENOBUFS;
			goto bad;
		}
		phdr = mtod(m, u_char *);
		bcopy(thdr, phdr, hdr_len);
		/* Loop back packets to ourselves */
		if (ADDR_EQ(thdr->hdr.token_dhost, &ns_thishost)) {
			if (loifp)
				return((*loifp->if_output)(ifp, m, dst, rt));
			m_freem(m);
			return(ENODEV);
		}
		/* If broadcasting on a simplex interface loopback a copy. */
		if ((m->m_flags & M_BCAST) && (ifp->if_flags & IFF_SIMPLEX))
			mcopy = m_copy(m, 0, (int)M_COPYALL);

		break;
#endif

	/* 
	 * we expect to be passed a *completed* header here (mac + llc) in
	 * a sockaddr_dl (actually sockaddr_dl_8025, which has enough room).
	 */
	case AF_UNSPEC:
	    {
		caddr_t hdrp, hdrp2;

		hdrp = LLADDR((struct sockaddr_dl *) dst);
		hdr_len = ((struct sockaddr_dl *) dst)->sdl_alen;

		M_PREPEND(m, hdr_len, M_DONTWAIT);
		if (m == 0) {
		    error = ENOBUFS;
		    goto bad;
		}

		/* now fill the header */
		hdrp2 = mtod(m, caddr_t);
		bcopy(hdrp, hdrp2, hdr_len);
		break;
	    }

	default:
		printf("%s%d: can't handle af%d\n", ifp->if_name, ifp->if_unit,
			dst->sa_family);
		error = EAFNOSUPPORT;
		goto bad;
	}

	/*
	 * Queue message on interface, and start output if interface
	 * not yet active.
	 */
	s = splimp();
	if (IF_QFULL(&ifp->if_snd)) {
		IF_DROP(&ifp->if_snd);
		splx(s);
		error = ENOBUFS;
		goto bad;
	}
	IF_ENQUEUE(&ifp->if_snd, m);
	if ((ifp->if_flags & IFF_OACTIVE) == 0)
		(*ifp->if_start)(ifp);
	splx(s);
	if (mcopy) {
		if (loifp)
			(void) (*loifp->if_output)(ifp, mcopy, dst, rt);
		else
			m_freem(mcopy);
	}
	ifp->if_omcasts += ismulti;
	return (error);

bad:
	if (mcopy)
		m_freem(mcopy);
	if (m)
		m_freem(m);
	return (error);
}

#ifdef NS
/*
 * Fill in basic NS/IPX header
 */
int
token_nshdr(hp, src, dst)
	struct token_max_hdr *hp;
	char *src;
	char *dst;
{
	hp->hdr.token_acf = ACF_PRIORITY3;
	hp->hdr.token_fcf = FCF_LLC_FRAME;
	ETHER_COPY(dst, hp->hdr.token_dhost);
	ETHER_COPY(src, hp->hdr.token_shost);
	hp->hdr.token_shost[ISO88025_ADDR_LEN+0] = LLC_IPX_LSAP;
	hp->hdr.token_shost[ISO88025_ADDR_LEN+1] = LLC_IPX_LSAP;
	hp->hdr.token_shost[ISO88025_ADDR_LEN+2] = LLC_UI;
	return (sizeof(struct token_header) + LLC_MINLEN);
}

/*
 * Resolve an NS host route into a LINK route (prebuild a header).
 */
void
token_rresolve(rt, ifp, dst)
	struct rtentry *rt;
	struct ifnet *ifp;
	struct sockaddr_ns *dst;
{
	struct rtentry *grt;
	int hdr_len;
	struct arpcom *ac = (struct arpcom *)ifp;
	struct sockaddr_dl_8025 sdlt;

#ifdef DEBUG
	if ((rt->rt_flags & RTF_HOST) == 0)
		panic("ns: not a host route");
#endif
	/*
	 * See if there is a link route for this host route (this will happen
	 * when routing to a gateway across a source-routed bridge).
	 */
	grt = rtalloc1(rt->rt_gateway, 0);
	if (grt != NULL) {
		if (grt->rt_flags & RTF_LLINFO) {
#ifdef DEBUG
			if (grt->rt_gateway->sa_family != AF_LINK)
				panic("token ns: bad gateway route");
#endif
			rt_setgate(rt, rt_key(rt), grt->rt_gateway);
			RTFREE(grt);
			return;
		}
		RTFREE(grt);
	}

	/*
	 * No prebuilt route available, build a simple (not source routed)
	 * header.
	 */
	hdr_len = token_nshdr((struct token_max_hdr *)sdlt.sdl_data,
	    ac->ac_enaddr, (char *)(&dst->sns_addr.x_host));
	sdlt.sdl_len = sizeof(sdlt);
	sdlt.sdl_family = AF_LINK;
	sdlt.sdl_index = ifp->if_index;
	sdlt.sdl_type = ifp->if_type;
	sdlt.sdl_nlen = 0;
	sdlt.sdl_alen = hdr_len;
	sdlt.sdl_slen = 0;
	rt_setgate(rt, rt_key(rt), (struct sockaddr *)&sdlt);
	rt->rt_flags |= RTF_LLINFO;
}

/*
 * Received a source routed packet, update the source route if appropriate
 */
void
token_update_srt(ifp, mac, idpp, srtlen, hdrlen)
	struct ifnet *ifp;
	struct token_max_hdr *mac;
	struct idp *idpp;
	int srtlen;
	int hdrlen;
{
	u_long nsnet;
	struct rtentry *rt;
	struct ifaddr *ifa;
	struct rt_addrinfo info;
	struct token_max_hdr *mmac;
	struct sockaddr_dl_8025 sdl;
	struct arpcom *ac = (struct arpcom *)ifp;

	ETHER_COPY(idpp->idp_sna.x_host.c_host, &sipx.sns_addr.x_host);
	srtlen += LLC_MINLEN;		/* Include LLC header */

	/*
	 * If host is using an all 0's network address we
	 * assume its the last IPX network on this interface.
	 */
	nsnet = *(u_long *)&idpp->idp_sna.x_net;
	if (nsnet == 0) {
		ifa = ifaof_ifpforaddr((struct sockaddr *)&sipx,
		    ifp);
		nsnet = *(u_long *)&((struct sockaddr_ns *)
		    ifa->ifa_addr)->sns_addr.x_net;
	}
	*(u_long *)&sipx.sns_addr.x_net = nsnet;
	rt = rtalloc1((struct sockaddr *)&sipx, 0);
	if (rt != NULL &&
	    rt->rt_flags & RTF_LLINFO &&
	    rt->rt_gateway->sa_family == AF_LINK) {
		/* Update an existing entry */
		mmac = (struct token_max_hdr *)SDL(rt->rt_gateway)->sdl_data;
		mmac->hdr.token_shost[0] |= RI_PRESENT;
		bcopy(&mac->rif, &mmac->rif, srtlen);
		mmac->rif.rcf1 ^= RCF1_DIRECTION;
		mmac->rif.rcf0 &= ~RCF0_BCAST_MASK;
		SDL(rt->rt_gateway)->sdl_alen = hdrlen;
	} else {
		/*
		 * New entry - prebuild the header that will
		 *	get us to this host.
		 */
		mmac = (struct token_max_hdr *)sdl.sdl_data;
		ac = (struct arpcom *)ifp;
		mmac->hdr.token_acf = ACF_PRIORITY3;
		mmac->hdr.token_fcf = FCF_LLC_FRAME;
		ETHER_COPY(ac->ac_enaddr, mmac->hdr.token_shost);
		mmac->hdr.token_shost[0] |= RI_PRESENT;
		ETHER_COPY(mac->hdr.token_shost, mmac->hdr.token_dhost);
		bcopy(&mac->rif, &mmac->rif, srtlen);
		mmac->rif.rcf1 ^= RCF1_DIRECTION;
		mmac->rif.rcf0 &= ~RCF0_BCAST_MASK;
		sdl.sdl_len = sizeof(sdl);
		sdl.sdl_family = AF_LINK;
		sdl.sdl_index = ifp->if_index;
		sdl.sdl_type = ifp->if_type;
		sdl.sdl_nlen = 0;
		sdl.sdl_alen = hdrlen;
		sdl.sdl_slen = 0;
		bzero(&info, sizeof(info));
		info.rti_info[RTAX_DST] = (struct sockaddr *)&sipx;
		info.rti_info[RTAX_GATEWAY] = (struct sockaddr *)&sdl;
		info.rti_ifp = ifp;
		info.rti_flags = RTF_HOST | RTF_CLONED | RTF_LLINFO;
		rtrequest1(RTM_ADD, &info, NULL);
	}
	if (rt != NULL)
		RTFREE(rt);
}
#endif

/*
 * Process a received Token-ring packet;
 * the packet is in the mbuf chain m without
 * the ether header, which is provided separately.
 *
 * TODO: deal with test & XID requests, etc.
 */
void
token_input(ifp, m)
	struct ifnet *ifp;
	struct mbuf *m;
{
	struct token_header *mac;
	struct ifqueue *inq;
	struct llc *llc;
	struct ip *ip;
	int hdrlen;
	int srtlen;
	int s;
	int ifam = ifp->if_addrmask;
#ifdef NS
	struct idp *idpp;
#endif

	if ((ifp->if_flags & IFF_UP) == 0) {
		m_freem(m);
		return;
	}
	mac = mtod(m, struct token_header *);
	hdrlen = sizeof(*mac);
	if (HAS_ROUTE(mac)) {
		srtlen = ROUTE_BYTES(mac);
		hdrlen += srtlen;
	}
	llc = (struct llc *)((caddr_t)mac + hdrlen);
	ip = (struct ip *)((caddr_t)mac + hdrlen + LLC_SNAPLEN);
	ifp->if_ibytes += m->m_pkthdr.len;
	if (ADDR_EQ(etherbroadcastaddr, mac->token_dhost)) {
		/* Broadcast or pre-1469 multicast. Check IP dst addr */
		if ((llc->llc_dsap == LLC_SNAP_LSAP) &&
		    (ntohs(llc->llc_ether_type) == ETHERTYPE_IP) &&
		    (IN_MULTICAST(ntohl(ip->ip_dst))))
			m->m_flags |= M_MCAST;
		else
			m->m_flags |= M_BCAST;
		ifp->if_imcasts++;
	} else if (mac->token_dhost[0] & 1) {
		m->m_flags |= M_MCAST;
		ifp->if_imcasts++;
	}

	switch (llc->llc_dsap) {

	case LLC_SNAP_LSAP:
#ifdef NOTYET
		if ((ifam & 1 << NETISR_IP) == 0)
			goto dropanyway;
#endif
		switch (ntohs(llc->llc_ether_type)) {
#ifdef INET
		case ETHERTYPE_IP:
			hdrlen += LLC_SNAPLEN;
			m->m_data += hdrlen;
			m->m_len -= hdrlen;
			m->m_pkthdr.len -= hdrlen;
			schednetisr(NETISR_IP);
			inq = &ipintrq;
			break;

		case ETHERTYPE_ARP:
			schednetisr(NETISR_ARP);
			inq = &arpintrq;
			break;
#endif

		default:
			goto dropanyway;
		}
		break;
#ifdef NS
	case LLC_IPX_LSAP:
		if ((ifam & 1 << NETISR_NS) == 0)
			goto dropanyway;
		hdrlen += LLC_MINLEN;
		m->m_data += hdrlen;
		m->m_len -= hdrlen;
		m->m_pkthdr.len -= hdrlen;
		while (HAS_ROUTE(mac) && srtlen > 2) {
			mac->token_shost[0] &= ~RI_PRESENT;
			idpp = mtod(m, struct idp *);
#if 0
			/*
			 * Ignore packets that have been routed from another
			 * net (the source route is to the router). This is
			 * only because we need to know what net the packet
			 * was from to enter it into the routine table.
			 */
			if (!ADDR_EQ(mac->token_shost,
			    idpp->idp_sna.x_host.c_host))
				break;
#endif
			/*
			 * Only pay attention to broadcasts and NCP requests
			 *
			 * XXX This is IPX specific
			 */
			if (!(m->m_flags & M_BCAST) &&
			    (*(u_short *)(idpp+1) != 0x1111 ||
			    idpp->idp_dna.x_port != htons(0x451)))
				break;
			
			/* Enter/Update the source route */
			token_update_srt(ifp, (struct token_max_hdr *)mac,
			    idpp, srtlen, hdrlen);
			break;
		}
		schednetisr(NETISR_NS);
		inq = &nsintrq;
		break;
#endif
	
	default:
dropanyway:;
		ifp->if_noproto++;
		m_freem(m);
		return;
	}

	s = splimp();
	if (IF_QFULL(inq)) {
		IF_DROP(inq);
		m_freem(m);
	} else 
		IF_ENQUEUE(inq, m);
	splx(s);
}

int
token_ioctl(ifp, cmd, data)
	struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	struct ifaddr *ifa = (struct ifaddr *) data;
	struct ifreq *ifr = (struct ifreq *) data;
	struct arpcom *ac = (struct arpcom *)(ifp);
	struct ns_addr *ina;
	struct sockaddr *sa;
	int error = 0;

	switch (cmd) {
	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;

		switch (ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET:
			ifp->if_init(ifp->if_softc);	/* before arpwhohas */
			arp_ifinit((struct arpcom *)(ifp), ifa);
			break;
#endif
#ifdef NS
		case AF_NS:
			ina = &(IA_SNS(ifa)->sns_addr);
			if (ns_nullhost(*ina))
				ina->x_host =
				    *(union ns_host *) (ac->ac_enaddr);
			else {
				ifp->if_flags &= ~IFF_RUNNING;
				bcopy((caddr_t) ina->x_host.c_host,
				      (caddr_t) ac->ac_enaddr,
				      sizeof(ac->ac_enaddr));
			}

			/*
			 * Set new address
			 */
			ifp->if_init(ifp->if_softc);
			break;
#endif
		default:
			ifp->if_init(ifp->if_softc);
			break;
		}
		break;

	case SIOCGIFADDR:
		sa = (struct sockaddr *) & ifr->ifr_data;
		bcopy(((struct arpcom *)(ifp))->ac_enaddr,
		     (caddr_t) sa->sa_data, sizeof(ac->ac_enaddr));
		break;

	case SIOCSIFMTU:
                /*
                 * Set the interface MTU.
		 * Note that ISO88025_MTU_16MB isn't correct on 4M rings,
		 * and that some adapters are known not to be able to
		 * support MTU's this large, however the possibility
		 * to fiddle with this is kept in the hope that 
		 * people who fiddle with this know what they are doing.
                 */
                if (ifr->ifr_mtu > ISO88025_MTU_16MB || ifr->ifr_mtu < 64) {
                        error = EINVAL;
                        break;
                }
                ifp->if_mtu = ifr->ifr_mtu;
                break;
	}
	return (error);
}

int
token_sysctl(ifp, name, namelen, oldp, oldlenp, newp, newlen)
	struct	ifnet *ifp;
	int	*name;
	u_int	namelen;
	void	*oldp;
	size_t	*oldlenp;
	void	*newp;
	size_t	newlen;
{
	struct arpcom *ac;
	struct ether_multi *ep, *op, *enm;
	void *eoldp;
	int error;

	if (namelen < 2)
		return (ENOTDIR);

	/* We currently do not implement any generic operations */
	if (name[0] == 0)
		return (EOPNOTSUPP);

	if (name[1] != CTL_LINK_LINKTYPE)
		return (EOPNOTSUPP);

	if (namelen < 4)
		return (ENOTDIR);

	eoldp = (char *)oldp + *oldlenp;

	switch (name[3]) {
	case TOKENCTL_MULTIADDRS:
		op = (struct ether_multi *)oldp;
		ac = (struct arpcom *)ifp;
		if (oldp == NULL) {
			for (ep = ac->ac_multiaddrs; ep != NULL;
			     ep = ep->enm_next)
				op++;
			*oldlenp = 11 * ((caddr_t)op - (caddr_t)oldp) / 10;
			return (0);
		}
		if ((enm = malloc(sizeof(*op), M_TEMP, M_WAITOK)) == NULL)
			return (ENOMEM);
		error = 0;
		for (ep = ac->ac_multiaddrs; ep != NULL; ep = ep->enm_next) {
			*enm = *ep;
			enm->enm_ac = NULL;
			if (enm->enm_next != NULL)
				enm->enm_next = op + 1;
			if ((void *)(op + 1) > eoldp) {
				error = ENOMEM;
				break;
			}
			if ((error = copyout(enm, op, sizeof(*op))) != 0)
				break;
			op++;
		}
		if (error == 0)
			*oldlenp = (caddr_t)op - (caddr_t)oldp;
		free(enm, M_TEMP);
		return (error);

	default:
		error = EOPNOTSUPP;
		break;
	}	

	return (error);
}

/*
 * Print a token ring mac destination address (leave out the source part)
 */
char *
token_sprintf(buf, addr)
	char *buf;
	struct token_max_hdr *addr;
{
	char *p;
	int nseg;
	int i;

	bcopy(ether_sprintf(addr->hdr.token_dhost), buf, 18);
	if (!HAS_ROUTE(addr))
		return (buf);
	p = &buf[17];
	p[0] = '/';
	p[1] = (addr->rif.rcf1 & RCF1_DIRECTION) ? 'r' : 'f';
	p += 2;
	nseg = addr->rif.rcf0 & RCF0_LEN_MASK;
	if (nseg > 2) {
		nseg = (nseg >> 1) - 1;
		for (i = 0; i < nseg; i++) {
			sprintf(p, ":%x", ntohs(addr->rif.rseg[i]));
			while (*++p != NULL) ;
		}
	} else
		*p = NULL;
	return (buf);
}
