/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI if_ether.c,v 2.41 2000/08/31 06:30:57 dab Exp
 */

/*
 * Copyright (c) 1982, 1986, 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)if_ether.c	8.2 (Berkeley) 9/26/94
 */

/*
 * Ethernet address resolution protocol.
 * TODO:
 *	add "inuse/lock" bit (or ref. count) along with valid bit
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/kernel.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/syslog.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/if_llc.h>
#include <net/if_media.h>
#include <net/route.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>

#include "token.h"
#if NTOKEN > 0
#include <net/if_token.h>

extern struct llc llc_snap_ip;
#endif

#define SIN(s) ((struct sockaddr_in *)s)
#define SDL(s) ((struct sockaddr_dl *)s)
#define SRP(s) ((struct sockaddr_inarp *)s)

/*
 * ARP trailer negotiation.  Trailer protocol is not IP specific,
 * but ARP request/response use IP addresses.
 */
#define ETHERTYPE_IPTRAILERS ETHERTYPE_TRAIL


/* timer values */
int	arpt_probe = (2*60);	/* expire unresolved entries after 2 minutes */
int	arpt_keep = (10*60);	/* once resolved, good for 10 more minutes, */
int	arpt_refresh = (10*60);	/*       plus another 10 minutes to refresh */
int	arpt_down = 20;		/* once declared down, don't send for 20 secs */

#define	ARPT_MIN_PROBE	   10	/* wait at least 20 seconds for reply */
#define	ARPT_MIN_KEEP	   30	/* keep entries at least 30 seconds */
#define	ARPT_MIN_MAXTRIES   2	/* at least 2 tries before reject */
#define	ARPT_MAX_MAXTRIES  30	/* at most 30 tries before reject */
#define	ARPT_MIN_DOWN	    5	/* reject for at least 5 seconds */


static	void arprequest __P((struct arpcom *, u_int32_t *, u_int32_t *,
	    u_char *, u_char *));
static	void arptfree __P((struct rtentry *));
static	struct rtentry *arplookup __P((u_int32_t, int, int));
static	void in_arpinput __P((struct mbuf *, caddr_t));
char 	*in_sprintf __P((u_int32_t));

extern	struct ifnet *loifp;
extern	struct timeval time;
struct	ifqueue arpintrq = {0, 0, 0, 50};
int	arp_inuse, arp_allocated;
int	arp_maxtries = 5;
int	useloopback = 1;	/* use loopback interface for local traffic */

struct llinfo_arp {
	struct	mbuf *la_hold;		/* last packet until resolved/timeout */
	short	la_asked;		/* # times we QUERIED for this addr */
	short	la_flags;
};

#define	LA_INCOMPLETE	0x01		/* entry is not yet complete */
#define	LA_REFRESH	0x02		/* need refresh if idle */
#define	LA_IDLE		0x04		/* entry is idle */

static struct llinfo_arp llinfo_arp;		/* head of the llinfo queue */

#if NTOKEN > 0
int smtu_tab[] = {
	RCF1_FRAME0_MAX,
	RCF1_FRAME1_MAX,
	RCF1_FRAME2_MAX,
	RCF1_FRAME3_MAX,
	RCF1_FRAME4_MAX,
	RCF1_FRAME5_MAX,
	RCF1_FRAME6_MAX,
	0
};
#endif

/*
 * Timeout routine.  Age arp_tab entries periodically.
 */

void
arp_expire(rt, rtt)
	register struct rtentry *rt;
	struct rttimer *rtt;
{
	register struct llinfo_arp *la = (struct llinfo_arp *)rt->rt_llinfo;

	if (la == NULL) {
		if (rt->rt_flags & RTF_UP)
			log(LOG_DEBUG, "arp_expire: %x: no llinfo for %s\n",
			    rt, in_sprintf(SIN(rt_key(rt))->sin_addr.s_addr));
		Free(rtt);
		return;
	}

	/*
	 * If address was resolved, and we aren't in the refresh
	 * state yet, transition to refresh state and reset the
	 * timer.
	 */
	if (arpt_refresh && (la->la_flags & (LA_INCOMPLETE|LA_REFRESH)) == 0) {
		la->la_flags |= LA_REFRESH|LA_IDLE;
		rt_timer_add(rt, time.tv_sec + arpt_refresh, arp_expire, rtt);
		return;
	}

	Free(rtt);
	arptfree(rt);
}


/*
 * Parallel to llc_rtrequest.
 */
void
arp_rtrequest(req, rt, info)
	int req;
	register struct rtentry *rt;
	struct rt_addrinfo *info;
{
	register struct sockaddr *gate = rt->rt_gateway;
	register struct llinfo_arp *la = (struct llinfo_arp *)rt->rt_llinfo;
	static struct sockaddr_dl null_sdl = {sizeof(null_sdl), AF_LINK};
#if NTOKEN > 0
	static struct sockaddr_dl_8025
	    null_sdl_8025 = {sizeof(null_sdl_8025), AF_LINK};
#endif
	struct sockaddr_dl *null_sdlp;
	int max_len;
	int min_len;

	if (rt->rt_flags & RTF_GATEWAY)
		return;
	switch (rt->rt_ifp->if_type) {
	case IFT_ETHER:	
	case IFT_FDDI:
	case IFT_LOOP:
	case IFT_IEEE80211:	
	case IFT_L2VLAN:
	case IFT_PROPMUX:	/* PIF interface */
	case IFT_AIF:		/* AIF interface */
		null_sdlp = &null_sdl; 
		max_len = min_len = sizeof(null_sdl);
		break;
#if NTOKEN > 0
	case IFT_ISO88025: 		/* token ring */
		null_sdlp = (struct sockaddr_dl *) &null_sdl_8025; 
		max_len = sizeof(null_sdl_8025);
		min_len = max_len - sizeof(null_sdl_8025.sdl_data) +
			ISO88025_ADDR_LEN;
		break;
#endif
	default:
		log(LOG_DEBUG, "arp_rtrequest: bad if_type %d\n",
			rt->rt_ifp->if_type);
		return;
	}
	switch (req) {

	case RTM_ADD:
		/*
		 * XXX: If this is a manually added route to interface
		 * such as older version of routed or gated might provide,
		 * restore cloning bit.
		 */
		if ((rt->rt_flags & RTF_HOST) == 0 &&
		    SIN(rt_mask(rt))->sin_addr.s_addr != 0xffffffff)
			rt->rt_flags |= RTF_CLONING;
		if (rt->rt_flags & RTF_CLONING) {
			/*
			 * Case 1: This route should come from a route to iface.
			 */
			rt_setgate(rt, rt_key(rt),
					(struct sockaddr *)null_sdlp);
			gate = rt->rt_gateway;
			SDL(gate)->sdl_type = rt->rt_ifp->if_type;
			SDL(gate)->sdl_index = rt->rt_ifp->if_index;
			break;
		}
		/* Announce a new entry if requested. */
		if (rt->rt_flags & RTF_ANNOUNCE)
			arprequest((struct arpcom *)rt->rt_ifp,
			    &SIN(rt_key(rt))->sin_addr.s_addr,
			    &SIN(rt_key(rt))->sin_addr.s_addr,
			    (u_char *)LLADDR(SDL(gate)), NULL);
		/*FALLTHROUGH*/
	case RTM_RESOLVE:
		if (gate->sa_family != AF_LINK ||
		    gate->sa_len < min_len || gate->sa_len > max_len) {
			log(LOG_DEBUG, "arp_rtrequest: bad gateway value\n");
			break;
		}
		SDL(gate)->sdl_type = rt->rt_ifp->if_type;
		SDL(gate)->sdl_index = rt->rt_ifp->if_index;
		if (la != 0)
			break; /* This happens on a route change */
		/*
		 * Case 2:  This route may come from cloning, or a manual route
		 * add with a LL address.
		 */
		R_Malloc(la, struct llinfo_arp *, sizeof(*la));
		rt->rt_llinfo = (caddr_t)la;
		if (la == 0) {
			log(LOG_DEBUG, "arp_rtrequest: malloc failed\n");
			break;
		}
		arp_inuse++, arp_allocated++;
		Bzero(la, sizeof(*la));
		rt->rt_flags |= RTF_LLINFO;
		if (SIN(rt_key(rt))->sin_addr.s_addr ==
		    (IA_SIN(rt->rt_ifa))->sin_addr.s_addr) {
			/*
			 * This test used to be
			 *	if (loifp->if_flags & IFF_UP)
			 * It allowed local traffic to be forced through
			 * the hardware by configuring the loopback down.
			 * However, it causes problems during network
			 * configuration for boards that can't receive
			 * packets they send.  It is now necessary to clear
			 * "useloopback" and remove the route to force
			 * traffic out to the hardware.
			 */
			Bcopy(((struct arpcom *)rt->rt_ifp)->ac_enaddr,
				LLADDR(SDL(gate)), SDL(gate)->sdl_alen = 6);
			if (useloopback && loifp)
				rt->rt_ifp = loifp;
		} else if (SDL(gate)->sdl_alen == 0) {
			la->la_flags |= LA_INCOMPLETE;
			rt_timer_add(rt, time.tv_sec + arpt_probe,
			    arp_expire, NULL);
		}
		break;

	case RTM_DELETE:
		if (la == 0)
			break;
		arp_inuse--;
		rt->rt_llinfo = 0;
		rt->rt_flags &= ~RTF_LLINFO;
		if (la->la_hold)
			m_freem(la->la_hold);
		Free((caddr_t)la);
	}
}

extern struct llc llc_snap_arp;

/*
 * Broadcast an ARP request. Caller specifies:
 *	- arp header source ip address
 *	- arp header target ip address
 *	- arp header source ethernet address
 *	- arp header target ethernet address (only for refresh requests)
 */
static void
arprequest(ac, sip, tip, sea, tea)
	register struct arpcom *ac;
	register u_int32_t *sip, *tip;
	register u_char *sea;
	register u_char *tea;
{
	register struct mbuf *m;
	register struct ether_header *eh;
	register struct ether_arp *ea;
	struct sockaddr sa, *sap;
#if NTOKEN > 0
	struct sockaddr_dl_8025 sa8025;
#endif

	if ((m = m_gethdr(M_DONTWAIT, MT_DATA)) == NULL)
		return;
	m->m_len = sizeof(*ea);
	m->m_pkthdr.len = sizeof(*ea);
	MH_ALIGN(m, sizeof(*ea));
	ea = mtod(m, struct ether_arp *);
	bzero((caddr_t)ea, sizeof (*ea));
	ea->arp_pro = htons(ETHERTYPE_IP);
	ea->arp_hln = sizeof(ea->arp_sha);	/* hardware address length */
	ea->arp_pln = sizeof(ea->arp_spa);	/* protocol address length */
	ea->arp_op = htons(ARPOP_REQUEST);
	ETHER_COPY(sea, ea->arp_sha);
	bcopy((caddr_t)sip, (caddr_t)ea->arp_spa, sizeof(ea->arp_spa));
	bcopy((caddr_t)tip, (caddr_t)ea->arp_tpa, sizeof(ea->arp_tpa));

	if (tea == NULL)
		tea = etherbroadcastaddr;

	switch (ac->ac_if.if_type) {
	case IFT_ETHER:	
	case IFT_FDDI:
	case IFT_IEEE80211:	
	case IFT_L2VLAN:
		sap = &sa; 
		ea->arp_hrd = htons(ARPHRD_ETHER);
		eh = (struct ether_header *)sa.sa_data;
		ETHER_COPY(tea, eh->ether_dhost);
		eh->ether_type = htons(ETHERTYPE_ARP);
		break;

#if NTOKEN > 0
	case IFT_ISO88025: 		/* token ring */
	    {
		register struct token_max_hdr *mac;

		/* XXX change this to use AF_LINK */
		sap = (struct sockaddr *) &sa8025;
		ea->arp_hrd = htons(ARPHRD_802);
		bzero((caddr_t)sap, sizeof (*sap));
		mac = (struct token_max_hdr *)sa8025.sdl_data;
		mac->hdr.token_acf = ACF_PRIORITY3;
		mac->hdr.token_fcf = FCF_LLC_FRAME;
		bcopy(tea, mac->hdr.token_dhost,
		    ISO88025_ADDR_LEN);
		bcopy(ac->ac_enaddr, mac->hdr.token_shost, ISO88025_ADDR_LEN);
		if (ac->ac_flags & AC_SOURCERT) {
			mac->hdr.token_shost[0] |= RI_PRESENT;

			if (ac->ac_flags & AC_ALLBC)
				mac->rif.rcf0 = 2 | RCF0_ALL_BROADCAST;
			else
				mac->rif.rcf0 = 2 | RCF0_SINGLE_BROADCAST;
			mac->rif.rcf1 = RCF1_FRAME_MASK;
		}
		bcopy(&llc_snap_arp, (caddr_t)mac + TOKEN_MAC_SIZE(mac),
		    LLC_SNAPLEN);
		sa8025.sdl_alen = TOKEN_MAC_SIZE(mac) + LLC_SNAPLEN;

		break;
	    }
#endif
	default:
		log(LOG_DEBUG, "arprequest: bad if_type %d\n",
			ac->ac_if.if_type);
		return;
	}

	sap->sa_family = AF_UNSPEC;
	sap->sa_len = sizeof(*sap);
	(*ac->ac_if.if_output)(&ac->ac_if, m, sap, (struct rtentry *)0);
}

/*
 * Resolve an IP address into an ethernet address.  If success,
 * desten is filled in.  If there is no entry in arptab,
 * set one up and broadcast a request for the IP address.
 * Hold onto this mbuf and resend it once the address
 * is finally resolved.  A non-zero return value indicates
 * that desten has been filled in (the return value is the length
 * of the mac address copied back) and the packet should be sent
 * normally; a 0 return indicates that the packet has been
 * taken over here, either now or for later transmission.
 */
int
arpresolve(ac, rt, m, dst, desten)
	register struct arpcom *ac;
	register struct rtentry *rt;
	struct mbuf *m;
	register struct sockaddr *dst;
	register u_char *desten;
{
	register struct llinfo_arp *la;
	struct sockaddr_dl *sdl;
	time_t when;

	if (m->m_flags & M_MCAST) {	/* multicast */
		ETHER_MAP_IP_MULTICAST(&SIN(dst)->sin_addr, desten);
		return (sizeof(etherbroadcastaddr));
	}
	if ((rt == NULL &&
	    (rt = arplookup(SIN(dst)->sin_addr.s_addr, 
	        ~m->m_flags & M_BCAST, 0)) == NULL) ||
	    (la = (struct llinfo_arp *)rt->rt_llinfo) == NULL) {
		if (m->m_flags & M_BCAST) {
			/* broadcast w/IP_ROUTETOIF */
			bcopy((caddr_t)etherbroadcastaddr, (caddr_t)desten,
			    sizeof(etherbroadcastaddr));
			return (sizeof(etherbroadcastaddr));
		}
		log(LOG_DEBUG, "arpresolve: no %s for %s\n",
		    rt == NULL ? "route" : "llinfo",
		    in_sprintf(SIN(dst)->sin_addr.s_addr));
		m_freem(m);
		return (0);
	}
	sdl = SDL(rt->rt_gateway);
	/*
	 * Check the address family and length is valid, the address
	 * is resolved; otherwise, try to resolve.
	 */
	if (sdl->sdl_family == AF_LINK && sdl->sdl_alen != 0) {
		/* If idle, refresh the cache with a directed ARP request */
		if (la->la_flags & LA_IDLE) {
			la->la_flags &= ~LA_IDLE;
			arprequest(ac,
			    &(IA_SIN(rt->rt_ifa)->sin_addr.s_addr),
			    &(SIN(dst)->sin_addr.s_addr),
			    ac->ac_enaddr, (u_char *)LLADDR(sdl));
		}
		bcopy(LLADDR(sdl), desten, sdl->sdl_alen);
		return (sdl->sdl_alen);
	}
	/*
	 * There is an arptab entry, but no ethernet address
	 * response yet.  Replace the held mbuf with this
	 * latest one.
	 */
	if (la->la_hold)
		m_freem(la->la_hold);
	la->la_hold = m;

	rt->rt_flags &= ~RTF_REJECT;
	/*
	 * If we've already sent an ARP request, make sure
	 * we wait at least 1 second before trying again.
	 */
	if (la->la_asked) {
		RTT_FIND(rt, arp_expire, when);
		if (when == time.tv_sec + arpt_probe)
			return (0);
	}
	if (m->m_flags & M_BCAST) {
		switch (rt->rt_ifp->if_type) {
		case IFT_ETHER:	
		case IFT_FDDI:
		case IFT_IEEE80211:	
		case IFT_L2VLAN:
			sdl->sdl_alen = sizeof(etherbroadcastaddr);
			bcopy((caddr_t)etherbroadcastaddr,
			      LLADDR(sdl),
			      sdl->sdl_alen);
			break;

#if NTOKEN > 0
		case IFT_ISO88025: 		/* token ring */
			/* XXX - Token ring does not currently call */
			/* XXX - arpresolve for a broadcast address. */
			/* Fall through */
#endif
			
		default:
			log(LOG_DEBUG, "arpresolve: bad if_type %d\n",
			    rt->rt_ifp->if_type);
			break;
		}
		RTT_FIND(rt, arp_expire, when);
		if (when)
			rt_timer_add(rt, 0, arp_expire, NULL);
	} else if (la->la_asked++ < arp_maxtries) {
		rt_timer_add(rt, time.tv_sec + arpt_probe, arp_expire, NULL);
		arprequest(ac, &(IA_SIN(rt->rt_ifa)->sin_addr.s_addr),
		    &(SIN(dst)->sin_addr.s_addr), ac->ac_enaddr, NULL);
	} else {
		rt->rt_flags |= RTF_REJECT;
		rt_timer_add(rt, time.tv_sec + arpt_down, arp_expire, NULL);
		la->la_asked = 0;
	}
	return (0);
}

/*
 * Common length and type checks are done here,
 * then the protocol-specific routine is called.
 */
void
arpintr()
{
	register struct mbuf *m;
	register struct arphdr *ar;
	int s;
	caddr_t hdr;

	while (arpintrq.ifq_head) {
		s = splimp();
		IF_DEQUEUE(&arpintrq, m);
		splx(s);
		if (m == 0 || (m->m_flags & M_PKTHDR) == 0)
			panic("arpintr");

		if (m->m_len < sizeof(struct arphdr) &&
		    (m = m_pullup(m, sizeof (struct arphdr))) == 0)
			goto free_it;

		switch (m->m_pkthdr.rcvif->if_type) {
		case IFT_ETHER:	
		case IFT_FDDI:
		case IFT_IEEE80211:	
		case IFT_L2VLAN:
			hdr = (caddr_t) 0;
			ar = mtod(m, struct arphdr *);
			if (ntohs(ar->ar_hrd) != ARPHRD_ETHER)
				goto free_it;
			break;
#if NTOKEN > 0
		case IFT_ISO88025: 		/* token ring */
		    {
			struct token_max_hdr *mac;
			int hdrlen;

			mac = mtod(m, struct token_max_hdr *);
			hdrlen = TOKEN_MAC_SIZE(mac) + LLC_SNAPLEN;

			hdr = mtod(m, caddr_t);	/* XXX: mac hdr is first !! */
			m->m_len -= hdrlen;
			m->m_pkthdr.len -= hdrlen;
			m->m_data += hdrlen;
			if (m->m_len < sizeof(struct ether_arp) &&
			    (m = m_pullup(m, sizeof(struct ether_arp))) == 0)
				goto free_it;
			ar = mtod(m, struct arphdr *);
			if (ntohs(ar->ar_hrd) != ARPHRD_802)
				goto free_it;

			/* switch direction, and turn off broadcast bits */
			if (HAS_ROUTE(mac)) {
				mac->rif.rcf1 ^= RCF1_DIRECTION;
				mac->rif.rcf0 &= ~RCF0_BCAST_MASK;
			}
			break;
		    }
#endif
		default:
			log(LOG_DEBUG, "arpintr: bad if_type %d\n",
				m->m_pkthdr.rcvif->if_type);
			return;
		}

		if (m->m_len >=
		      sizeof(struct arphdr) + 2 * ar->ar_hln + 2 * ar->ar_pln){

			switch (ntohs(ar->ar_pro)) {

			case ETHERTYPE_IP:
			case ETHERTYPE_IPTRAILERS:
				in_arpinput(m, hdr);
				continue;
			}
		}
free_it:;
		m_freem(m);
	}
}

/*
 * ARP for Internet protocols on 10 Mb/s Ethernet.
 * Algorithm is that given in RFC 826.
 * In addition, a sanity check is performed on the sender
 * protocol address, to catch impersonators.
 * We no longer handle negotiations for use of trailer protocol:
 * Formerly, ARP replied for protocol type ETHERTYPE_TRAIL sent
 * along with IP replies if we wanted trailers sent to us,
 * and also sent them in response to IP replies.
 * This allowed either end to announce the desire to receive
 * trailer packets.
 * We no longer reply to requests for ETHERTYPE_TRAIL protocol either,
 * but formerly didn't normally send requests.
 */
static void
in_arpinput(m, hdr)
	struct mbuf *m;
	caddr_t hdr;
{
	register struct ether_arp *ea;
	register struct arpcom *ac = (struct arpcom *)m->m_pkthdr.rcvif;
	struct ether_header *eh;
	register struct llinfo_arp *la = 0;
	register struct rtentry *rt;
	struct in_ifaddr *ia;
	struct sockaddr_dl *sdl;
	struct sockaddr sa, *sap;
	struct inhash *ih;
#if NTOKEN > 0
	struct sockaddr_dl_8025 sa8025;
#endif
	struct in_addr isaddr, itaddr;
	time_t when;
	int iamtarget, op, notify = 0;
	struct ifnet *ifp;

	ea = mtod(m, struct ether_arp *);

	op = ntohs(ea->arp_op);

	/* Validate hardware addresses */
	if (!bcmp((caddr_t)ea->arp_sha, (caddr_t)ac->ac_enaddr,
	    sizeof (ea->arp_sha)))
		goto out;	/* it's from me, ignore it. */
	bcopy((caddr_t)ea->arp_spa, (caddr_t)&isaddr, sizeof (isaddr));
	bcopy((caddr_t)ea->arp_tpa, (caddr_t)&itaddr, sizeof (itaddr));
	if (!bcmp((caddr_t)ea->arp_sha, (caddr_t)etherbroadcastaddr,
	    sizeof (ea->arp_sha))) {
		log(LOG_ERR,
		    "arp: ether address is broadcast for IP address %s!\n",
		    in_sprintf(isaddr.s_addr));
		goto out;
	}

	/*
	 * Lookup the source address to be sure it is not one of our
	 * addresses.
	 */
	ifp = &ac->ac_if;
	ih = inh_lookup(isaddr, ifp);
	if (ifp->if_pif && (!ih || !(ia = ih->inh_ifa) || ia->ia_ifp != ifp)) {
		ifp = ifp->if_pif;
		ih = inh_lookup(isaddr, ifp);
	}
	if (ih != NULL) {
		if ((ia = ih->inh_ifa) == NULL || ia->ia_ifp != ifp ||
		    (ih->inh_flags & INH_BCAST))
			goto out;
		log(LOG_ERR,
		    "duplicate IP address %s!! sent from address: %s\n",
		    in_sprintf(isaddr.s_addr), 
		    ether_sprintf(ea->arp_sha));
		itaddr = ia->ia_addr.sin_addr;
		iamtarget = 1;
		goto reply;
	} 

	/* 
	 * Lookup the target address.  If it is one of our addresses
	 * on this interface and the source is on the same subnet, use
	 * that address. 
	 */
	ifp = &ac->ac_if;
	ih = inh_lookup(itaddr, ifp);
	if (ifp->if_pif && (!ih || !(ia = ih->inh_ifa) || ia->ia_ifp != ifp)) {
		ifp = ifp->if_pif;
		ih = inh_lookup(itaddr, ifp);
	}
		
	if (ih == NULL || (ia = ih->inh_ifa) == NULL) {
		/* Not one of our addresses, look up as a proxy */
		iamtarget = 0;
		ia = NULL;
	} else if (ia->ia_ifp != ifp || (ih->inh_flags & INH_BCAST) ||
	    !IA_ONNET(ia, isaddr)) {
		/* 
		 * Not one of our addresses on this interface, or not
		 * on the same subnet.  Just look up as a proxy.
		 */
		iamtarget = 0;
		ia = NULL;
	} else {
		/* This is one of our addresses on this interface */
		iamtarget = 1;
	}

	/*
	 * If we have not already located the interface address, look
	 * for an an interface on a subnet shared with the target.
	 */
	if (ia == NULL) {
		struct sockaddr_in *sinp;

		sinp = (struct sockaddr_in *)&sa;
		bzero(sinp, sizeof(*sinp));
		sinp->sin_len = sizeof(*sinp);
		sinp->sin_family = AF_INET;
		sinp->sin_addr = itaddr;
		ia = (struct in_ifaddr *)ifaof_ifpforaddr(&sa, &ac->ac_if);
		if (ia == NULL || !IA_ONNET(ia, itaddr))
			goto out;
	}
	/* Ignore requests from hosts we are unable to respond to */
	if (!IA_ONNET(ia, isaddr))
		goto out;

	rt = arplookup(isaddr.s_addr, iamtarget, 0);
	if (rt && (la = (struct llinfo_arp *)rt->rt_llinfo) &&
	    (sdl = SDL(rt->rt_gateway))) {

		switch (ac->ac_if.if_type) {
#if NTOKEN > 0
		case IFT_ISO88025:
			/*
			 * For token ring the ARP entry consists of a ready to
			 * send header (up through the end of the SNAP header).
			 * Out of paranoia we take the source address in the 
			 * ARP area and merge it with the source route info 
			 * from the mac header; this implies that proxy-arp 
			 * will work as long as the proxy is on the same 
			 * ring with the node being arp'ed for.
			 *
			 * This also (safely) implies that IP is always 
			 * sent with SNAP encapsulation.
			 */
		    {
			struct token_max_hdr *macsrc;
			int rt_len;
			struct token_max_hdr *arpent =
			    (struct token_max_hdr *)LLADDR(sdl);
			char oldhdr[sizeof(struct token_max_hdr) + LLC_SNAPLEN];
			int oldlen;
			char abuf1[64], abuf2[64];
			int smtu;

			oldlen = sdl->sdl_alen;
			bcopy(arpent, oldhdr, oldlen);
			macsrc = (struct token_max_hdr *)hdr;
			arpent->hdr.token_acf = ACF_PRIORITY3;
			arpent->hdr.token_fcf = FCF_LLC_FRAME;
			bcopy(ac->ac_enaddr, arpent->hdr.token_shost,
			    ISO88025_ADDR_LEN);
			bcopy(ea->arp_sha, arpent->hdr.token_dhost,
			    ISO88025_ADDR_LEN);
			sdl->sdl_alen = sizeof(struct token_header);
			if (HAS_ROUTE(macsrc) &&
			    (rt_len = ROUTE_BYTES(macsrc)) > 2 &&
			    rt_len <= 18 && !(rt_len & 0x1)) {
				bcopy(&macsrc->rif, &arpent->rif, rt_len);
				arpent->hdr.token_shost[0] |= RI_PRESENT;
				sdl->sdl_alen += rt_len;
				smtu = smtu_tab[(macsrc->rif.rcf1 >> 4) & 0x07];
				if (smtu && 
				    (smtu < ac->ac_if.if_mtu) &&
				    (smtu < rt->rt_rmx.rmx_mtu) &&
				    ((rt->rt_rmx.rmx_locks & RTV_MTU) == 0))
					rt->rt_rmx.rmx_mtu = smtu;
			}
			bcopy(&llc_snap_ip, (caddr_t)arpent + sdl->sdl_alen,
			    LLC_SNAPLEN);
			sdl->sdl_alen += LLC_SNAPLEN;
			if (oldlen) {
				if ((sdl->sdl_alen != oldlen ||
				    bcmp(oldhdr, arpent, oldlen) != 0))
					log(LOG_INFO, 
					    "arp info overwritten for %s: %s => %s\n",
					    in_sprintf(isaddr.s_addr),
					    token_sprintf(abuf1, oldhdr),
					    token_sprintf(abuf2, arpent));
				else
					break;
			}
			notify = 1;
			break;
		    }
#endif
		default:
			/* For everyone else just store the 6 byte MAC addr */
			if (sdl->sdl_alen != 0) {
				if (bcmp(ea->arp_sha, LLADDR(sdl), sizeof(ea->arp_sha)))
					log(LOG_INFO, 
					    "arp info overwritten for %s by %s\n",
					    in_sprintf(isaddr.s_addr),
					    ether_sprintf(ea->arp_sha));
				else
					break;
			}
			notify = 1;
			ETHER_COPY(ea->arp_sha, LLADDR(sdl));
			sdl->sdl_alen = sizeof(ea->arp_sha);
			break;
		}

		/*
		 * If the entry was incomplete, or if the arp_expire()
		 * timer is already running, reset the timer.
		 */
		if (la->la_flags & LA_INCOMPLETE) {
			rt_timer_add(rt, time.tv_sec + arpt_keep,
			    arp_expire, NULL);
		} else {
			RTT_FIND(rt, arp_expire, when);
			if (when) {
				rt_timer_add(rt, time.tv_sec + arpt_keep,
				    arp_expire, NULL);
			}
		}

		rt->rt_flags &= ~RTF_REJECT;
		la->la_asked = 0;
		la->la_flags &= ~(LA_REFRESH|LA_INCOMPLETE);

		if (la->la_hold) {
			(*ac->ac_if.if_output)(&ac->ac_if, la->la_hold,
				rt_key(rt), rt);
			la->la_hold = 0;
		}

		/* Report the change to routing socket listener */
		if (notify) {
			struct rt_addrinfo info;

			bzero((caddr_t)&info, sizeof(info));
			info.rti_info[RTAX_DST] = rt_key(rt);
			info.rti_info[RTAX_NETMASK] = rt_mask(rt);
			info.rti_info[RTAX_GATEWAY] = rt->rt_gateway;
			if (rt->rt_ifp != NULL) {
				info.rti_info[RTAX_IFP] =
				    rt->rt_ifp->if_addrlist->ifa_addr;
				info.rti_info[RTAX_IFA] = rt->rt_ifa->ifa_addr;
			}
			rt_missmsg(RTM_CHANGE, &info, rt->rt_flags, 0);
		}

	}
reply:
	if (op != ARPOP_REQUEST) {
	out:
		m_freem(m);
		return;
	}
	if (iamtarget) {
		/* I am the target */
		ETHER_COPY(ea->arp_sha, ea->arp_tha);
		ETHER_COPY(ac->ac_enaddr, ea->arp_sha);
	} else {
		rt = arplookup(itaddr.s_addr, 0, SIN_PROXY);
		if (rt == NULL)
			goto out;
		ETHER_COPY(ea->arp_sha, ea->arp_tha);
		sdl = SDL(rt->rt_gateway);
		ETHER_COPY(LLADDR(sdl), ea->arp_sha);
	}

	bcopy((caddr_t)ea->arp_spa, (caddr_t)ea->arp_tpa, sizeof(ea->arp_spa));
	bcopy((caddr_t)&itaddr, (caddr_t)ea->arp_spa, sizeof(ea->arp_spa));
	ea->arp_op = htons(ARPOP_REPLY);
	ea->arp_pro = htons(ETHERTYPE_IP); /* let's be sure! */

	switch (ac->ac_if.if_type) {
	case IFT_ETHER:	
	case IFT_FDDI:
	case IFT_IEEE80211:	
	case IFT_L2VLAN:
		sap = &sa; 
		eh = (struct ether_header *)sa.sa_data;
		ETHER_COPY(ea->arp_tha, eh->ether_dhost);
		eh->ether_type = htons(ETHERTYPE_ARP);
		break;

#if NTOKEN > 0
	case IFT_ISO88025: 		/* token ring */
	    {
		register struct token_header *mac;
		struct token_max_hdr *macsrc = (struct token_max_hdr *) hdr;

		/* XXX change this to use AF_LINK */
		sap = (struct sockaddr *) &sa8025;
		bzero((caddr_t)sap, sizeof (*sap));
		mac = (struct token_header *)sa8025.sdl_data;
		mac->token_acf = ACF_PRIORITY3;
		mac->token_fcf = FCF_LLC_FRAME;
		bcopy((caddr_t)ea->arp_tha, (caddr_t)mac->token_dhost, 
		    ISO88025_ADDR_LEN);
		bcopy((caddr_t)ac->ac_enaddr, (caddr_t)mac->token_shost, 
		    ISO88025_ADDR_LEN);

		/* fill in routing info, if needed. */
		if (HAS_ROUTE(macsrc)) {
			mac->token_shost[0] |= RI_PRESENT;
			bcopy(&macsrc->rif, &((struct token_max_hdr *)mac)->rif,
			    ROUTE_BYTES(macsrc));
		}
		bcopy(&llc_snap_arp, (caddr_t)mac + TOKEN_MAC_SIZE(mac),
		    LLC_SNAPLEN);
		sa8025.sdl_alen = TOKEN_MAC_SIZE(mac) + LLC_SNAPLEN;
		break;
	    }
#endif

	default:
		panic("in_arpinput: if_type (%s%d)",
		    ac->ac_if.if_name, ac->ac_if.if_unit);
	}
	
	sap->sa_family = AF_UNSPEC;
	sap->sa_len = sizeof(*sap);
	(*ac->ac_if.if_output)(&ac->ac_if, m, sap, (struct rtentry *)0);
}

/*
 * Free an arp entry.
 */
static void
arptfree(rt)
	register struct rtentry *rt;
{
	register struct llinfo_arp *la = (struct llinfo_arp *)rt->rt_llinfo;
	register struct sockaddr_dl *sdl;

	if (rt->rt_refcnt > 0 && (sdl = SDL(rt->rt_gateway)) &&
	    sdl->sdl_family == AF_LINK) {
		sdl->sdl_alen = 0;
		la->la_flags = LA_INCOMPLETE;
		la->la_asked = 0;
		rt->rt_flags &= ~RTF_REJECT;
		return;
	}
	(void)rtdeletemsg(rt);
}

/*
 * Lookup or enter a new address in arptab.
 */
static struct rtentry *
arplookup(addr, create, proxy)
	u_int32_t addr;
	int create, proxy;
{
	register struct rtentry *rt;
	static struct sockaddr_inarp sin = {sizeof(sin), AF_INET };

	sin.sin_addr.s_addr = addr;
	sin.sin_other = proxy ? SIN_PROXY : 0;
	rt = rtalloc1((struct sockaddr *)&sin, create);
	if (rt == 0)
		return (0);
	rt->rt_refcnt--;
	if ((rt->rt_flags & RTF_GATEWAY) || (rt->rt_flags & RTF_LLINFO) == 0 ||
	    rt->rt_gateway->sa_family != AF_LINK) {
		if (create) {
			log(LOG_DEBUG,
			    "arp: unable to create entry for %s",
			    in_sprintf(addr));
			addlog(" from route %s",
			    in_sprintf(SIN(rt_key(rt))->sin_addr.s_addr));
			if (rt_mask(rt) != NULL)
				addlog("/%s",
				    in_sprintf(SIN(rt_mask(rt))->sin_addr.s_addr));
			if (rt->rt_ifa != NULL) {
				addlog(", interface %s",
				    in_sprintf(SIN(rt->rt_ifa->ifa_addr)->sin_addr.s_addr));
				if (rt->rt_ifp != NULL)
					addlog("(%s%u)",
					    rt->rt_ifp->if_name,
					    rt->rt_ifp->if_unit);
			}
			addlog("\n");
		}
		return (0);
	}
	return (rt);
}

int
arpioctl(cmd, data)
	int cmd;
	caddr_t data;
{

	return (EOPNOTSUPP);
}

/*
 * Arp common initialization
 */
void
arp_ifinit(ac, ifa)
	struct arpcom *ac;
	struct ifaddr *ifa;
{
	arprequest(ac, &(IA_SIN(ifa)->sin_addr.s_addr),
	    &(IA_SIN(ifa)->sin_addr.s_addr), ac->ac_enaddr, NULL);
	ifa->ifa_rtrequest = arp_rtrequest;
	ifa->ifa_flags |= RTF_CLONING;
}

/*
 * Generate a printable representation of an IP address.
 * The value is stored in a static buffer, so can be called
 * only at splnet().
 * This does not really belong here, but is currently used only here.
 */
char *
in_sprintf(addr)
	u_int32_t addr;
{
	static char addrbuf[16];
	register u_char *ap = (u_char *)&addr;

	sprintf(addrbuf, "%d.%d.%d.%d", ap[0], ap[1], ap[2], ap[3]); 
	return (addrbuf);
}

int *arp_sysvars[] = ARPCTL_VARS;

/*
 * arp_sysctlfn is used by route_sysctl to tell whether arp is present.
 * XXX should be type "sysctlfn", but the proc pointer isn't passed.
 */
int arp_sysctl __P((int *, u_int, void *, size_t *, void *, size_t));
int (*arp_sysctlfn) __P((int *, u_int, void *, size_t *, void *, size_t)) =
    arp_sysctl;

/*
 * Sysctl for ARP variables.
 */
arp_sysctl(name, namelen, oldp, oldlenp, newp, newlen)
	int *name;
	u_int namelen;
	void *oldp;
	size_t *oldlenp;
	void *newp;
	size_t newlen;
{
	int error, t;

	switch (name[0]) {
	case ARPCTL_KEEP:
		t = arpt_keep;
		if ((error = sysctl_int(oldp, oldlenp, newp, newlen, &t)) != 0)
			break;
		if (newp) {
			if (t < ARPT_MIN_KEEP) {
				error = EINVAL;
				break;
			}
			arpt_keep = t;
		}
		break;

	case ARPCTL_REFRESH:
		t = arpt_refresh;
		if ((error = sysctl_int(oldp, oldlenp, newp, newlen, &t)) != 0)
			break;
		if (newp)  {
			if (t < 0) {
				error = EINVAL;
				break;
			}
			arpt_refresh = t;
		}
		break;

	case ARPCTL_PROBE:
		t = arpt_probe;
		if ((error = sysctl_int(oldp, oldlenp, newp, newlen, &t)) != 0)
			break;
		if (newp) {
			if (t < ARPT_MIN_PROBE) {
				error = EINVAL;
				break;
			}
			arpt_probe = t;
		}
		break;

	case ARPCTL_DOWN:
		t = arpt_down;
		if ((error = sysctl_int(oldp, oldlenp, newp, newlen, &t)) != 0)
			break;
		if (newp) {
			if (t < ARPT_MIN_DOWN) {
				error = EINVAL;
				break;
			}
			arpt_down = t;
		}
		break;

	case ARPCTL_MAXTRIES:
		t = arp_maxtries;
		if ((error = sysctl_int(oldp, oldlenp, newp, newlen, &t)) != 0)
			break;
		if (newp) {
			if (t < ARPT_MIN_MAXTRIES || t > ARPT_MAX_MAXTRIES) {
				error = EINVAL;
				break;
			}
			arp_maxtries = t;
		}
		break;

	case ARPCTL_USELOOPBACK:
		error = sysctl_int(oldp, oldlenp, newp, newlen, &useloopback);
		break;

	default:
		error = EOPNOTSUPP;
		break;
	}

	return (error);
}

#if NTOKEN > 0
/*
 * Update ARP flags from a media options word (called by drivers) 
 */
void
arp_setopt(ac, mword)
	struct arpcom *ac;
	int mword;
{

	ac->ac_flags &= ~(AC_SOURCERT | AC_ALLBC | AC_MFA);
	if (mword & IFM_TOK_SRCRT)
		ac->ac_flags |= AC_SOURCERT;
	if (mword & IFM_TOK_ALLR)
		ac->ac_flags |= AC_ALLBC;
	if (mword & IFM_TOK_MFA)
		ac->ac_flags |= AC_MFA;
}
#endif
