/*	BSDI if_ether.h,v 2.19 2001/02/26 14:26:31 geertj Exp	*/

/*
 * Copyright (c) 1982, 1986, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)if_ether.h	8.3 (Berkeley) 5/2/95
 */

#ifndef _IF_ETHER_H_
#define _IF_ETHER_H_

#include <net/if_arp.h>
#include <net/if_ethernet.h>

#ifdef KERNEL
/*
 * Macro to map an IP multicast address to an Ethernet multicast address.
 * The high-order 25 bits of the Ethernet address are statically assigned,
 * and the low-order 23 bits are taken from the low end of the IP address.
 */
#define ETHER_MAP_IP_MULTICAST(ipaddr, enaddr) \
	/* struct in_addr *ipaddr; */ \
	/* u_char enaddr[ETHER_ADDR_LEN];	   */ \
do { \
	(enaddr)[0] = 0x01; \
	(enaddr)[1] = 0x00; \
	(enaddr)[2] = 0x5e; \
	(enaddr)[3] = ((u_char *)ipaddr)[1] & 0x7f; \
	(enaddr)[4] = ((u_char *)ipaddr)[2]; \
	(enaddr)[5] = ((u_char *)ipaddr)[3]; \
} while (0)

/*
 * Macro to map an IPv6 multicast address to an Ethernet multicast address.
 * The high-order 16 bits of the Ethernet address are statically assigned,
 * and the low-order 32 bits are taken from the low end of the IPv6 address.
 */
#define ETHER_MAP_IPV6_MULTICAST(ip6addr, enaddr)			\
	/* struct in6_addr *ip6addr; */					\
	/* u_int8_t enaddr[ETHER_ADDR_LEN]; */				\
do {									\
	(enaddr)[0] = 0x33;						\
	(enaddr)[1] = 0x33;						\
	(enaddr)[2] = ((u_int8_t *)ip6addr)[12];			\
	(enaddr)[3] = ((u_int8_t *)ip6addr)[13];			\
	(enaddr)[4] = ((u_int8_t *)ip6addr)[14];			\
	(enaddr)[5] = ((u_int8_t *)ip6addr)[15];			\
} while (0)
#endif

/*
 * Ethernet Address Resolution Protocol.
 *
 * See RFC 826 for protocol description.  Structure below is adapted
 * to resolving internet addresses.  Field names used correspond to 
 * RFC 826.
 */
struct	ether_arp {
	struct	arphdr ea_hdr;			/* fixed-size header */
	u_char	arp_sha[ETHER_ADDR_LEN];	/* sender hardware address */
	u_char	arp_spa[4];			/* sender protocol address */
	u_char	arp_tha[ETHER_ADDR_LEN];	/* target hardware address */
	u_char	arp_tpa[4];			/* target protocol address */
};
#define	arp_hrd	ea_hdr.ar_hrd
#define	arp_pro	ea_hdr.ar_pro
#define	arp_hln	ea_hdr.ar_hln
#define	arp_pln	ea_hdr.ar_pln
#define	arp_op	ea_hdr.ar_op


/* Used for 802.11 */
#define	AC_802_11_MODE	0x0300		/* 802.11 mode mask */
#define	AC_802_11_ADHOC	0x0000		/* 802.11 ad-hoc */
#define	AC_802_11_ISS	0x0100		/* 802.11 infrastructure mode */
#define	AC_802_11_AP	0x0200		/* 802.11 access point mode */
#define	AC_802_11_REP	0x0300		/* 802.11 repeater mode */

#define AC_SNAP_K1	0xaa   		/* assigned global SAP for SNAP */
#define AC_SNAP_K2	0x00
#define AC_SNAP_CONTROL	0x03		/* unnumbered information format */
#define AC_SNAPHDR_LEN	8		/* 6 byte header + 2 byte type */

/*
 * 802.11 Frame Control bits
 */
/* Byte 0 */
#define IEEE_802_11_FTYPE_MGMT		0x00
#define IEEE_802_11_FTYPE_CTL		0x04
#define IEEE_802_11_FTYPE_DATA		0x08

/* Byte 1 */
#define IEEE_802_11_FCTL_VERS		0x02
#define IEEE_802_11_FCTL_FTYPE		0x0C
#define IEEE_802_11_FCTL_STYPE		0xF0
#define IEEE_802_11_FCTL_TODS		0x01
#define IEEE_802_11_FCTL_FROMDS		0x02
#define IEEE_802_11_FCTL_MOREFRAGS	0x04
#define IEEE_802_11_FCTL_RETRY		0x08
#define IEEE_802_11_FCTL_PM		0x10
#define IEEE_802_11_FCTL_MOREDATA	0x20
#define IEEE_802_11_FCTL_WEP		0x40
#define IEEE_802_11_FCTL_ORDER		0x80

struct sockaddr_inarp {
	u_char	sin_len;
	u_char	sin_family;
	u_short sin_port;
	struct	in_addr sin_addr;
	struct	in_addr sin_srcaddr;
	u_short	sin_tos;
	u_short	sin_other;
#define SIN_PROXY 1
};
/*
 * IP and ethernet specific routing flags
 */
#ifdef notdef	/* not used anymore, RTF_PROTO1 is needed for RTF_PROBEMTU */
#define	RTF_USETRAILERS	RTF_PROTO1	/* use trailers */
#endif
#define RTF_ANNOUNCE	RTF_PROTO2	/* announce new arp entry */

#ifdef	KERNEL
u_char	ether_ipmulticast_min[ETHER_ADDR_LEN];
u_char	ether_ipmulticast_max[ETHER_ADDR_LEN];
struct	ifqueue arpintrq;

void	arp_rtrequest __P((int, struct rtentry *, struct rt_addrinfo *));
void	arpintr __P((void));
int	arpresolve __P((struct arpcom *,
	    struct rtentry *, struct mbuf *, struct sockaddr *, u_char *));
void	arp_ifinit __P((struct arpcom *, struct ifaddr *));
void	arp_setopt __P((struct arpcom *, int));

int	ether_addmulti __P((struct ifreq *, struct arpcom *));
int	ether_delmulti __P((struct ifreq *, struct arpcom *));
int	ether_mapmcaddr __P((struct ifreq *, u_char *, u_char *));
int	arp_sysctl __P((int *, u_int, void *, size_t *, void *, size_t));

#endif /* KERNEL */

/*
 * Ethernet multicast address structure.  There is one of these for each
 * multicast address or range of multicast addresses that we are supposed
 * to listen to on a particular interface.  They are kept in a linked list,
 * rooted in the interface's arpcom structure.  (This really has nothing to
 * do with ARP, or with the Internet address family, but this appears to be
 * the minimally-disrupting place to put it.)
 */
struct ether_multi {
	u_char	enm_addrlo[ETHER_ADDR_LEN]; /* low  or only address of range */
	u_char	enm_addrhi[ETHER_ADDR_LEN]; /* high or only address of range */
	struct	arpcom *enm_ac;		/* back pointer to arpcom */
	u_int	enm_refcount;		/* no. claims to this addr/range */
	struct	ether_multi *enm_next;	/* ptr to next ether_multi */
};

/*
 * Structure used by macros below to remember position when stepping through
 * all of the ether_multi records.
 */
struct ether_multistep {
	struct ether_multi  *e_enm;
};

/*
 * Macro for looking up the ether_multi record for a given range of Ethernet
 * multicast addresses connected to a given arpcom structure.  If no matching
 * record is found, "enm" returns NULL.
 */
#define ETHER_LOOKUP_MULTI(addrlo, addrhi, ac, enm) \
	/* u_char addrlo[ETHER_ADDR_LEN]; */ \
	/* u_char addrhi[ETHER_ADDR_LEN]; */ \
	/* struct arpcom *ac; */ \
	/* struct ether_multi *enm; */ \
{ \
	for ((enm) = (ac)->ac_multiaddrs; \
	    (enm) != NULL && \
	    (bcmp((enm)->enm_addrlo, (addrlo), ETHER_ADDR_LEN) != 0 || \
	     bcmp((enm)->enm_addrhi, (addrhi), ETHER_ADDR_LEN) != 0); \
		(enm) = (enm)->enm_next); \
}

/*
 * Macro to step through all of the ether_multi records, one at a time.
 * The current position is remembered in "step", which the caller must
 * provide.  ETHER_FIRST_MULTI(), below, must be called to initialize "step"
 * and get the first record.  Both macros return a NULL "enm" when there
 * are no remaining records.
 */
#define ETHER_NEXT_MULTI(step, enm) \
	/* struct ether_multistep step; */  \
	/* struct ether_multi *enm; */  \
{ \
	if (((enm) = (step).e_enm) != NULL) \
		(step).e_enm = (enm)->enm_next; \
}

#define ETHER_FIRST_MULTI(step, ac, enm) \
	/* struct ether_multistep step; */ \
	/* struct arpcom *ac; */ \
	/* struct ether_multi *enm; */ \
{ \
	(step).e_enm = (ac)->ac_multiaddrs; \
	ETHER_NEXT_MULTI((step), (enm)); \
}

#define	ARPCTL_PROBE		1
#define	ARPCTL_MAXTRIES		2
#define	ARPCTL_DOWN		3
#define	ARPCTL_KEEP		4
#define	ARPCTL_REFRESH		5
#define	ARPCTL_USELOOPBACK	6
#define	ARPCTL_MAXID		7

#define	ARPCTL_NAMES {			\
	{ 0, 0 },			\
	{ "probe", CTLTYPE_INT },	\
	{ "maxtries", CTLTYPE_INT },	\
	{ "down", CTLTYPE_INT },	\
	{ "keep", CTLTYPE_INT },	\
	{ "refresh", CTLTYPE_INT },	\
	{ "useloopback", CTLTYPE_INT },	\
}

#define	ARPCTL_VARS {	\
	0,		\
	&arpt_probe,	\
	&arp_maxtries,	\
	&arpt_down,	\
	&arpt_keep,	\
	&arpt_refresh,	\
	&useloopback,	\
}

#define	ETHERCTL_MULTIADDRS	1	/* Return list of multicast addresses */
#define	ETHERCTL_MAXID		2

#define	ETHERCTL_NAMES {		\
	{ 0, 0 },			\
	{ "multiaddrs", CTLTYPE_STRUCT }, \
}

#endif /* !_IF_ETHER_H_ */
