/*-
 * Copyright (c) 1997 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI ipfw_circuit.h,v 2.3.8.1 2001/09/12 15:46:02 prb Exp
 */

/*
 * This structure is passed in/out of the kernel to describe a cirucit cache.
 * The length field is ignored when read in by the kernel.
 */
typedef struct {
	u_int	size;			/* number of buckets in circuit */
	u_int	length;			/* number of entries in the cache */
	u_long	datamask;		/* Which bits of first 4 data bytes */
	u_long	hitcode;		/* IPFW return code to send on hit */
	u_long	misscode;		/* IPFW return code to send on miss */
	int	index;			/* interface to filter */
	u_char	protocol;		/* What protocol we are filtering */
	u_char	cfsize;			/* size of cf arena */
	u_char	cfarena;		/* which of cf arena */
	u_char	which:3;		/* which address to filter on */
	u_char	flip16:1;		/* Data is 16bit order dependent */
	u_char	route:1;		/* return attached route to entry */
	u_char	chain:1;		/* call chained filter for entry */ 
	u_char	follow:1;		/* follow tcp FIN/REJ requests */
	u_char	translate:1;		/* reserverd */
	u_int	timeout;		/* Maximum seconds idle for a circuit */
	u_int	slices;			/* Number of slices of idle time */
	u_int	maxcircuits;		/* Maximum number of circuits */
} ipfw_cf_hdr_t;

#define	CIRCUIT_SRCONLY	1
#define	CIRCUIT_DSTONLY	2

/*
 * We allow a BPF filter to install a filter by setting the
 * CIRCUIT_INSTALL bit in the direction field.
 *
 * This allows a BPF filter to maintain a circuit cache of entries
 * for established circuits without sending the packet to user level.
 */
#define	CIRCUIT_INSTALL	IPFWF_FILTER1

/*
 * This structure is passed in/out of the kernel to describe a circuit.
 * Only the address and port fields are valid when passed into the kernel.
 * The next/prev values are never valid when passed in/out of the kernel.
 * (The kernel uses them internally and they are left available for what
 * ever use the user code would like to make of them)
 */
typedef struct {
#ifdef	KERNEL
	struct	ipfw_cf *cl_next;	/* Next entry in bucket */
	struct	ipfw_cf *cl_prev;	/* Previous entry in bucket */
#else
	void	*cl_private1;
	void	*cl_private2;
#endif
	union {
	    struct {
		u_quad_t
		    gotfin1:1,		/* We have got a fin in dir1 */
		    gotfin2:1,		/* We have got a fin in dir2 */
		    seenfin1:1,		/* We have seen a fin in dir1 */
		    seenfin2:1,		/* We have seen a fin in dir2 */
		    when:60;		/* time last used */
	    } cl_s0;
	    u_quad_t	bitfield;
	    int		size;		/* length of entry for sysctl's */
	} cl_u0;
	struct	in_addr a1;		/* Source address for match */
	struct	in_addr a2;		/* Destination address for match */
	u_int32_t	ports;		/* First 4 data bytes of packet */
} ipfw_cf_core_list_t;

typedef union {
	int		chain;	/* Filter to chain to */
	struct in_addr	route;	/* IP address to route circuit to */
} ipfw_cf_extra_list_t;

typedef struct {
	ipfw_cf_core_list_t cl_core;
	ipfw_cf_extra_list_t cl_extra;
} ipfw_cf_list_t;

#define	cl_route	cl_extra.route
#define	cl_chain	cl_extra.chain
#define	cl_a1		cl_core.a1
#define	cl_a2		cl_core.a2
#define	cl_ports	cl_core.ports
#define	cl_gotfin1	cl_core.cl_u0.cl_s0.gotfin1
#define	cl_gotfin2	cl_core.cl_u0.cl_s0.gotfin2
#define	cl_seenfin1	cl_core.cl_u0.cl_s0.seenfin1
#define	cl_seenfin2	cl_core.cl_u0.cl_s0.seenfin2
#define	cl_when		cl_core.cl_u0.cl_s0.when
#define	cl_bitfield	cl_core.cl_u0.bitfield
#define	cl_size		cl_core.cl_u0.size

#define	IPFW_CF_SET_PORTS(p1,p2)	((HTONS(p1) << 16) | HTONS(p2))
#define	IPFW_CF_GET_PORT1(x)	(NTOHS((x) >> 16))
#define	IPFW_CF_GET_PORT2(x)	(NTOHS((x) & 0xffff))

#define	CIRCUITCTL_CLOCK	1
#define	CIRCUITCTL_ADD		2
#define	CIRCUITCTL_DELETE	3
#define	CIRCUITCTL_BUCKETS	4
