/*-
 * Copyright (c) 1999 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI ipfw_nat.h,v 1.5 1999/11/18 01:03:40 prb Exp
 */

/*
 * This structure is passed in/out of the kernel to describe a cirucit cache.
 * The length field is ignored when read in by the kernel.
 */
typedef struct {
	u_int	size;			/* number of buckets in cache */
	u_int	prefill;		/* sessions to pre-allocate */
	u_int	sessions;		/* sessions in the cache */
	u_int	maxsessions;		/* maximum sessions allowed */
	int	glue;			/* NAT we are glued to */
	int	index;			/* interface we are assigned to */
	u_int	timeouts[256];		/* timeouts on a per-protocol basis */
} ip_nat_hdr_t;

/*
 * Structure defining a NAT
 *
 * We are inserted into 2 doubly linked lists
 *
 * nat_i*	hash indexed list based on incoming packets
 * nat_o*	hash indexed list based on outgoing packets
 *
 * Packets coming in are translated as follows:
 *
 *	Src Addr:	remote		->	remote
 *	Dst Addr:	external	->	internal
 *	Ports:		rpep		->	rpip
 *
 * Packets going out are translated as follows:
 *
 *	Src Addr:	internal	->	external
 *	Dst Addr:	remote		->	remote
 *	Ports:		iprp		->	eprp
 */

typedef struct ip_nat ip_nat_t;
typedef struct ip_nat_head ip_nat_head_t;

typedef union {
	u_int32_t	ports;
	struct {
	    u_int16_t	src;
	    u_int16_t	dst;
	} p;
} nat_port_t;

struct ip_nat {
#ifdef	KERNEL
	ip_nat_t	*nat_inext;	/* Next entry in bucket */
	ip_nat_t	*nat_iprev;	/* Previous entry in bucket */
	ip_nat_head_t	*nat_ihead;	/* Head of the in list */

	ip_nat_t	*nat_onext;	/* Next entry in bucket */
	ip_nat_t	*nat_oprev;	/* Previous entry in bucket */
	ip_nat_head_t	*nat_ohead;	/* Head of the out list */
#else
	void	*nat_private[6];	/* Place holders for above */
#endif
	struct	in_addr	nat_remote;	/* Address of remote side */
	struct	in_addr	nat_external;	/* Address external world sees */
	struct	in_addr	nat_internal;	/* Address internal world sees */

	nat_port_t	nat_rpep;	/* Ports coming in from outside */
	nat_port_t	nat_rpip;	/* Incoming ports mapped to this */
	nat_port_t	nat_eprp;	/* Ports going out to outside */
	nat_port_t	nat_iprp;	/* Outgoing ports mapped from this */

	u_long		nat_ihash;	/* Hash for incoming packets */
	u_long		nat_ohash;	/* Hash for outgoing packets */

	u_long		nat_delta;	/* How to adjust cksum */
	clock_t		nat_when;	/* Last time we saw this nat */

	u_char		nat_protocol;	/* Protocol in this nat */
	u_char		nat_direction;	/* How we got here */

	/* Here down is just to do FIN and RST chasing for TCP -- yuck */

	u_char		nat_gotfin1:1,	/* We have got a fin in dir1 */
                    	nat_gotfin2:1,	/* We have got a fin in dir2 */
                    	nat_seenfin1:1,	/* We have seen a fin in dir1 */
                    	nat_seenfin2:1;	/* We have seen a fin in dir2 */ 
	u_long	nat_ack1;		/* highest ack seen in dir1 */
	u_long	nat_ack2;		/* highest ack seen in dir2 */
	u_long	nat_fin1;		/* seq number of fin in dir1 */
	u_long	nat_fin2;		/* seq number of fin in dir2 */
};

#define	IPFWNAT_BUCKETS		2
#define	IPFWNAT_SETUP		3
#define	IPFWNAT_REMOVESERVICE	4
#define	IPFWNAT_REMOVEMAP	5
#define	IPFWNAT_INTERFACE	6
#define	IPFWNAT_ADDNAT		7
#define	IPFWNAT_REMOVENAT	8
#define	IPFWNAT_LOOKUP		9

#define NAT_HASH_SIZE		997
#define	NAT_TIMEOUT		(5 * 60)	/* 5 minutes */
#define	NAT_TCPTIMEOUT		(8 * 60 * 60)	/* 8 hours */

#define CCPG    (NBPG / sizeof(ip_nat_t))

/*
 * Structure of service offering, used to map incoming connection
 * requests to the appropriate internal address.
 */
typedef struct ip_natservice ip_natservice_t;

struct ip_natservice {
	ip_natservice_t	*ns_next;
	u_int	ns_serial;		/* serial number of service entry */
	time_t	ns_expire;		/* what time this service expires */
	struct	in_addr ns_external;	/* external address we are watching */
	struct	in_addr ns_internal;	/* internal address we map to */
	u_short	ns_protocol;		/* protocol of service */
	u_short	ns_eport;		/* external port we look for */
	u_short	ns_iport;		/* internal port we map to */
};

/*
 * Each entry describe a range of ports and addresses available to
 * mapping nat sessions.
 */
typedef	struct ip_natent ip_natent_t;

struct ip_natent {
	ip_natent_t	*ne_next;	/* next entry in list */
	struct	in_addr	ne_firstaddr;	/* First address available to nat */
	struct	in_addr	ne_lastaddr;	/* Last address available to nat */
	struct	in_addr	ne_luaddr;	/* Last address available to nat */
	u_short		ne_firstport;	/* First port available to nat */
	u_short		ne_lastport;	/* Last port available to nat */
	u_short		ne_luport;	/* Last port we used in this list */
};

/*
 * Each map describes a subnet that is subject to nat.
 */
typedef struct ip_natmap ip_natmap_t;

struct ip_natmap {
	u_int		nm_serial;	/* serial number of map */
	time_t		nm_expire;	/* what time this map expires */
	ip_natmap_t	*nm_next;	/* next entry in list */
	union {
	    ip_natent_t	*_nm_entries;	/* Possible entries we can use */
	    int		_nm_nentries;	/* How many entries follow for sysctl */
	} u;
	struct	in_addr	nm_internal;	/* internal address(s) we are mapping */
	struct	in_addr	nm_mask;	/* mask of internal addresses */
	u_short		nm_protocol;	/* protocl this map is for */
};

#define	nm_entries	u._nm_entries
#define	nm_nentries	u._nm_nentries

/*
 * A NAT Def structure preceeds a list of services and maps to describe
 * how many of each there are.
 */
typedef struct ip_natdef {
	int	nd_nservices;
	int	nd_nmaps;
} ip_natdef_t;
