/*
 * Copyright (c) 1996, 1997 Berkeley Software Design, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that this notice is retained,
 * the conditions in the following notices are met, and terms applying
 * to contributors in the following notices also apply to Berkeley
 * Software Design, Inc.
 *
 *	BSDI tcp_var.h,v 2.22 2000/04/21 07:44:16 dab Exp
 */

/*
 * tcp_var.h  --  Definitions for this implementation of TCP.
 *
 * Modifications to support IPv6 are copyright 1995 
 * by Randall Atkinson, Dan McDonald, and Bao Phan, All Rights Reserved.
 * All Rights under this copyright have been assigned to NRL.  
 *
 * See the file netinet6/COPYRIGHT.NRL for the licensing terms
 * on these modifications.
 */

/*
 * Copyright (c) 1982, 1986, 1993, 1994, 1995
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)tcp_var.h	8.4 (Berkeley) 5/24/95
 */

#ifndef _NETINET_TCP_VAR_H_
#define _NETINET_TCP_VAR_H_

/*
 * Kernel variables for tcp.
 */
struct tcp_segq {
	LIST_ENTRY(tcp_segq) seg_next;	/* ordered list of segments */
	struct	tcphdr *seg_th;		/* TCP header for this segment */
	struct	mbuf *seg_m;		/* data for same (possibly cluster) */
};

/*
 * Tcp control block, one per tcp; fields:
 */
struct tcpcb {
	LIST_HEAD(, tcp_segq) t_segq;	/* reassembly queue, OO packets only */
	int	t_xxx;			/* (padding -- temporary) */
	short	t_state;		/* state of this connection */
	short	t_timer[TCPT_NTIMERS];	/* tcp timers */
	short	t_rxtshift;		/* log(2) of rexmt exp. backoff */
	short	t_rxtcur;		/* current retransmit value */
	short	t_dupacks;		/* consecutive dup acks recd */
	u_short	t_maxseg;		/* maximum segment size */
	char	t_force;		/* 1 if forcing out a byte */
	u_char	pf;			/* Protocol family of Internet-layer */
					/* datagrams (if known), 0 if an */
					/* undecided socket (PF_INET6, but */
					/* unbound to an address.). */
	u_short	t_flags;
#define	TF_ACKNOW	0x0001		/* ack peer immediately */
/* #define TF_DELACK	0x0002		 * ack, but try to delay it */
#define	TF_NODELAY	0x0004		/* don't delay packets to coalesce */
#define	TF_NOOPT	0x0008		/* don't use tcp options */
#define	TF_SENTFIN	0x0010		/* have sent FIN */
#define	TF_USE_SCALE	0x0020		/* request/use window scaling */
#define	TF_SEND_TSTMP	0x0040		/* request/send timestamps */
#define	TF_NO_PMTU	0x0080		/* Don't use path MTU discovery */
#define	TF_SACK_PERMIT	0x0200		/* other side said I could SACK */
#define	TF_STDURG	0x0400		/* URG ptr is last byte of urg data */
#define	TF_WASIDLE	0x0800		/* tcp_output() was idle on last call */
#define	TF_RAW		0x1000		/* raw TCP socket */

	void	*t_unused;		/* placeholder... */
	struct	inpcb *t_inpcb;		/* back pointer to internet pcb */
/*
 * The following fields are used as in the protocol specification.
 * See RFC783, Dec. 1981, page 21.
 */
/* send sequence variables */
	tcp_seq	snd_una;		/* send unacknowledged */
	tcp_seq	snd_nxt;		/* send next */
	tcp_seq	snd_up;			/* send urgent pointer */
	tcp_seq	snd_wl1;		/* window update seg seq number */
	tcp_seq	snd_wl2;		/* window update seg ack number */
	tcp_seq	iss;			/* initial send sequence number */
	u_int32_t snd_wnd;		/* send window */
/* receive sequence variables */
	u_int32_t rcv_wnd;		/* receive window */
	tcp_seq	rcv_nxt;		/* receive next */
	tcp_seq	rcv_up;			/* receive urgent pointer */
	tcp_seq	irs;			/* initial receive sequence number */
/*
 * Additional variables for this implementation.
 */
/* receive variables */
	tcp_seq	rcv_adv;		/* advertised window */
/* retransmit variables */
	tcp_seq	snd_max;		/* highest sequence number sent;
					 * used to recognize retransmits
					 */
/* congestion control (for slow start, source quench, retransmit after loss) */
	u_int32_t snd_cwnd;		/* congestion-controlled window */
	u_int32_t snd_ssthresh;		/* snd_cwnd size threshhold for
					 * for slow start exponential to
					 * linear switch
					 */
/*
 * transmit timing stuff.  See below for scale of srtt and rttvar.
 * "Variance" is actually smoothed difference.
 */
	short	t_rtt;			/* round trip time */
	u_short	t_rttmin;		/* minimum rtt allowed */
	tcp_seq	t_rtseq;		/* sequence number being timed */
	short	t_srtt;			/* smoothed round-trip time */
	short	t_rttvar;		/* variance in round-trip time */
	u_int32_t t_idle;			/* inactivity time */
	u_int32_t max_sndwnd;		/* largest window peer has offered */

/* out-of-band data */
	char	t_oobflags;		/* have some */
	char	t_iobc;			/* input character */
#define	TCPOOB_HAVEDATA	0x01
#define	TCPOOB_HADDATA	0x02
	short	t_softerror;		/* possible error not yet reported */

/* RFC 1323 variables */
	u_char	snd_scale;		/* window scaling for send window */
	u_char	rcv_scale;		/* window scaling for recv window */
	u_char	request_r_scale;	/* pending window scaling */
	u_char	requested_s_scale;
	u_int32_t ts_recent;		/* timestamp echo data */
	u_int32_t ts_recent_age;	/* when last updated */
	tcp_seq	last_ack_sent;

/* TUBA stuff */
	caddr_t	t_tuba_pcb;		/* next level down pcb for TCP over z */

/* should be moved up */
	LIST_ENTRY(tcpcb) t_delacks;	/* list of connections needing delack */
	u_short	t_peermaxseg;		/* MSS offered by peer */
	u_short	t_fullmaxseg;		/* Unrounded MSS value */
	u_int32_t t_tcpiphdr[15];	/* IPV4/TCP hdr, 40 bytes */
					/* IPV6/TCP hdr, 60 bytes */
};

/* This structure should not exceed 32 bytes. */
struct syn_cache {
	struct	syn_cache *sc_next;
	u_int32_t sc_tstmp:1,
		sc_v6:1,
		sc_hash:30;
	tcp_seq	sc_irs;
	tcp_seq	sc_iss;
	u_short	sc_sport;
	u_short	sc_dport;
	u_short	sc_peermaxseg;
	u_char	sc_timer;
	u_char	sc_request_r_scale:4,
		sc_requested_s_scale:4;
	union {
		struct scu_4 {
			struct	in_addr scu4_src;
			struct	in_addr scu4_dst;
		} scu_4;
		struct scu_6 {
			struct	in6_addr scu6_src;
			struct	in6_addr scu6_dst;
		} scu_6;
	} sc_u;
#define	sc_src4 sc_u.scu_4.scu4_src
#define	sc_dst4 sc_u.scu_4.scu4_dst
#define	sc_src6 sc_u.scu_6.scu6_src
#define	sc_dst6 sc_u.scu_6.scu6_dst
};

#define sizeof_SYN_CACHE4 \
	(sizeof(struct syn_cache) - sizeof(struct scu_6) + sizeof(struct scu_4))
#define sizeof_SYN_CACHE6 sizeof(struct syn_cache)

struct syn_cache_head {
	struct	syn_cache *sch_first;		/* First entry in the bucket */
	struct	syn_cache *sch_last;		/* Last entry in the bucket */
	struct	syn_cache_head *sch_headq;	/* The next non-empty bucket */
	short	sch_timer_sum;			/* Total time in this bucket */
	u_short	sch_length;			/* @ # elements in bucket */
};

#define	intotcpcb(ip)	((struct tcpcb *)(ip)->inp_ppcb)
#define	sototcpcb(so)	(intotcpcb(sotoinpcb(so)))

/*
 * The following results in generation of delayed acks
 * in the opposite order in which they were requested...
 */
#define	tcp_delack(tp) { \
	if ((tp)->t_delacks.le_prev == 0) \
		LIST_INSERT_HEAD(&tcp_delacks, (tp), t_delacks); \
}

#define	tcp_delack_done(tp) { \
	if ((tp)->t_delacks.le_prev) { \
		LIST_REMOVE((tp), t_delacks); \
		(tp)->t_delacks.le_prev = 0; \
	} \
}

/*
 * Cancel 2msl timer (to restart, or to delete connection prematurely):
 * if this is the newest 2msl connection, reduce the total time for
 * the queue, otherwise transfer the remaining time to the next newest
 * 2msl connection.
 */
#define	tcp_cancel2msl(inp, tp) { \
	if (inp->inp_next == &tcb) \
		tcp_msltime -= tp->t_timer[TCPT_2MSL]; \
	else \
		intotcpcb(inp->inp_next)->t_timer[TCPT_2MSL] += \
		    tp->t_timer[TCPT_2MSL]; \
}

/*
 * The smoothed round-trip time and estimated variance
 * are stored as fixed point numbers scaled by the values below.
 * For convenience, these scales are also used in smoothing the average
 * (smoothed = (1/scale)sample + ((scale-1)/scale)smoothed).
 * With these scales, srtt has 3 bits to the right of the binary point,
 * and thus an "ALPHA" of 0.875.  rttvar has 2 bits to the right of the
 * binary point, and is smoothed with an ALPHA of 0.75.
 */
#define	TCP_RTT_SCALE		8	/* multiplier for srtt; 3 bits frac. */
#define	TCP_RTT_SHIFT		3	/* shift for srtt; 3 bits frac. */
#define	TCP_RTTVAR_SCALE	4	/* multiplier for rttvar; 2 bits */
#define	TCP_RTTVAR_SHIFT	2	/* multiplier for rttvar; 2 bits */

/*
 * The initial retransmission should happen at rtt + 4 * rttvar.
 * Because of the way we do the smoothing, srtt and rttvar
 * will each average +1/2 tick of bias.  When we compute
 * the retransmit timer, we want 1/2 tick of rounding and
 * 1 extra tick because of +-1/2 tick uncertainty in the
 * firing of the timer.  The bias will give us exactly the
 * 1.5 tick we need.  But, because the bias is
 * statistical, we have to test that we don't drop below
 * the minimum feasible timer (which is 2 ticks).
 * This macro assumes that the value of TCP_RTTVAR_SCALE
 * is the same as the multiplier for rttvar.
 */
#define	TCP_REXMTVAL(tp) \
	(((tp)->t_srtt >> TCP_RTT_SHIFT) + (tp)->t_rttvar)


/*
 * TCP statistics.
 * Many of these should be kept per connection,
 * but that's inconvenient at the moment.
 */
struct	tcpstat {
	u_quad_t tcps_connattempt;	/* connections initiated */
	u_quad_t tcps_accepts;		/* connections accepted */
	u_quad_t tcps_connects;		/* connections established */
	u_quad_t tcps_drops;		/* connections dropped */
	u_quad_t tcps_conndrops;	/* embryonic connections dropped */
	u_quad_t tcps_closed;		/* conn. closed (includes drops) */
	u_quad_t tcps_segstimed;	/* segs where we tried to get rtt */
	u_quad_t tcps_rttupdated;	/* times we succeeded */
	u_quad_t tcps_delack;		/* delayed acks sent */
	u_quad_t tcps_timeoutdrop;	/* conn. dropped in rxmt timeout */
	u_quad_t tcps_rexmttimeo;	/* retransmit timeouts */
	u_quad_t tcps_persisttimeo;	/* persist timeouts */
	u_quad_t tcps_keeptimeo;	/* keepalive timeouts */
	u_quad_t tcps_keepprobe;	/* keepalive probes sent */
	u_quad_t tcps_keepdrops;	/* connections dropped in keepalive */

	u_quad_t tcps_sndtotal;		/* total packets sent */
	u_quad_t tcps_sndpack;		/* data packets sent */
	u_quad_t tcps_sndbyte;		/* data bytes sent */
	u_quad_t tcps_sndrexmitpack;	/* data packets retransmitted */
	u_quad_t tcps_sndrexmitbyte;	/* data bytes retransmitted */
	u_quad_t tcps_sndrexmitfast;	/* Fast retransmits */
	u_quad_t tcps_sndacks;		/* ack-only packets sent */
	u_quad_t tcps_sndprobe;		/* window probes sent */
	u_quad_t tcps_sndurg;		/* packets sent with URG only */
	u_quad_t tcps_sndwinup;		/* window update-only packets sent */
	u_quad_t tcps_sndctrl;		/* control (SYN|FIN|RST) packets sent */

	u_quad_t tcps_rcvtotal;		/* total packets received */
	u_quad_t tcps_rcvpack;		/* packets received in sequence */
	u_quad_t tcps_rcvbyte;		/* bytes received in sequence */
	u_quad_t tcps_rcvbadsum;	/* packets received with ccksum errs */
	u_quad_t tcps_rcvbadoff;	/* packets received with bad offset */
	u_quad_t tcps_rcvshort;		/* packets received too short */
	u_quad_t tcps_rcvduppack;	/* duplicate-only packets received */
	u_quad_t tcps_rcvdupbyte;	/* duplicate-only bytes received */
	u_quad_t tcps_rcvpartduppack;	/* packets with some duplicate data */
	u_quad_t tcps_rcvpartdupbyte;	/* dup. bytes in part-dup. packets */
	u_quad_t tcps_rcvoopack;	/* out-of-order packets received */
	u_quad_t tcps_rcvoobyte;	/* out-of-order bytes received */
	u_quad_t tcps_rcvpackafterwin;	/* packets with data after window */
	u_quad_t tcps_rcvbyteafterwin;	/* bytes rcvd after window */
	u_quad_t tcps_rcvafterclose;	/* packets rcvd after "close" */
	u_quad_t tcps_rcvwinprobe;	/* rcvd window probe packets */
	u_quad_t tcps_rcvdupack;	/* rcvd duplicate acks */
	u_quad_t tcps_rcvacktoomuch;	/* rcvd acks for unsent data */
	u_quad_t tcps_rcvackpack;	/* rcvd ack packets */
	u_quad_t tcps_rcvackbyte;	/* bytes acked by rcvd acks */
	u_quad_t tcps_rcvwinupd;	/* rcvd window update packets */
	u_quad_t tcps_pawsdrop;		/* segments dropped due to PAWS */
	u_quad_t tcps_predack;		/* times hdr predict ok for acks */
	u_quad_t tcps_preddat;		/* times hdr predict ok for data pkts */
	u_quad_t tcps_pcbcachemiss;
	u_quad_t tcps_persistdrop;	/* timeout in persist state */
	u_quad_t tcps_badsyn;		/* bogus SYN, e.g. premature ACK */
	u_quad_t tcps_droppedsyn;	/* dropped SYN's because sonewconn() failed */

	/* These statistics deal with the SYN cache. */
	u_quad_t tcps_sc_added;		/* # of entries added */
	u_quad_t tcps_sc_completed;	/* # of connections completed */
	u_quad_t tcps_sc_timed_out;	/* # of entries timed out */
	u_quad_t tcps_sc_overflowed;	/* # dropped due to overflow */
	u_quad_t tcps_sc_reset;		/* # dropped due to RST */
	u_quad_t tcps_sc_unreach;	/* # dropped due to ICMP unreach */
	u_quad_t tcps_sc_bucketoverflow;/* # dropped due to bucket overflow */
	u_quad_t tcps_sc_aborted;	/* # of entries aborted (no mem) */
	u_quad_t tcps_sc_dupesyn;	/* # of duplicate SYNs received */
	u_quad_t tcps_sc_dropped;	/* # of SYNs dropped (no route/mem) */

	u_quad_t tcps_rcvmemdrop;	/* packets dropped for lack of memory */
};

#ifdef KERNEL
struct	inpcb tcb;		/* head of queue of active tcpcb's */
struct	tcpstat tcpstat;	/* tcp statistics */
u_int32_t tcp_now;		/* for RFC 1323 timestamps */
int	tcp_msltime;		/* total of 2MSL timers already in queue */

int	tcp_listen_hash_size;
int	tcp_conn_hash_size;
extern LIST_HEAD(tcp_hash_list, inpcb) tcp_listen_hash[], tcp_conn_hash[];
#ifndef NORAWTCP
struct tcp_hash_list tcp_raw_hash;
#endif /* NORAWTCP */
LIST_HEAD(tcp_delacks, tcpcb) tcp_delacks;

int	tcp_syn_cache_size;
int	tcp_syn_cache_timeo;
extern struct syn_cache_head tcp_syn_cache[];
struct	syn_cache_head *tcp_syn_cache_first;
u_int32_t syn_cache_count;

struct	tcp_opt_info;
struct	tcpiphdr;
struct	in6_hdrindex;

int	 tcp_attach __P((struct socket *));
void	 tcp_canceltimers __P((struct tcpcb *));
struct tcpcb *
	 tcp_close __P((struct tcpcb *));
#if defined(INET6) && !defined(TCP6)
void	 tcp6_ctlinput __P((int, struct sockaddr *, void *));
#endif
void	 tcp_ctlinput __P((int, struct sockaddr *, void *));
int	 tcp_ctloutput __P((int, struct socket *, int, int, struct mbuf **));
struct tcpcb *
	 tcp_disconnect __P((struct tcpcb *));
struct tcpcb *
	 tcp_drop __P((struct tcpcb *, int));
void	 tcp_dooptions __P((struct tcpcb *,
	    u_char *, int, struct tcphdr *, struct tcp_opt_info *));
void	 tcp_drain __P((void));
void	 tcp_fasttimo __P((void));
void	 tcp_init __P((void));
#if defined(INET6) && !defined(TCP6)
int	 tcp6_input __P((struct mbuf **, int *, int));
#endif
void	 tcp_input __P((struct mbuf *, int));
struct rtentry *
	 tcp_rtlookup __P((register struct inpcb *));
void	 tcp_maxseg __P((struct tcpcb *, u_int));
void	 tcp_maxseg_init __P((struct tcpcb *));
void	 tcp_peer_mss __P((struct tcpcb *, u_int));
u_int32_t tcp_send_mss __P((struct inpcb *));
void	 tcp_changemss __P((register struct tcpcb *, u_int));
struct tcpcb *
	 tcp_newtcpcb __P((struct inpcb *));
void	 tcp_notify __P((struct inpcb *, int));
int	 tcp_output __P((struct tcpcb *));
void	 tcp_pulloutofband __P((struct socket *,
	    struct tcphdr *, struct mbuf *, int));
void	 tcp_quench __P((struct inpcb *, int));
void	 tcp_mtudisc __P((struct inpcb *, int));
int	 tcp_reass __P((struct tcpcb *, struct tcphdr *, struct mbuf *, int));
int	 tcp_respond __P((struct tcpcb *,
	    struct tcpiphdr *, struct mbuf *, tcp_seq, tcp_seq, int));
void	 tcp_setpersist __P((struct tcpcb *));
void	 tcp_slowtimo __P((void));
int	 tcp_sysctl __P((int *, u_int, void *, size_t *, void *, size_t));
void	 tcp_template __P((struct tcpcb *));
struct tcpcb *
	 tcp_timers __P((struct tcpcb *, int));
void	 tcp_trace __P((int, int, struct tcpcb *, struct tcpiphdr *, int, int));
struct tcpcb *
	 tcp_usrclosed __P((struct tcpcb *));
#if defined(INET6) && !defined(TCP6)
int	 tcp6_usrreq __P((struct socket *,
	    int, struct mbuf *, struct mbuf *, struct mbuf *, struct proc *));
#endif
int	 tcp_usrreq __P((struct socket *,
	    int, struct mbuf *, struct mbuf *, struct mbuf *));
void	 tcp_xmit_timer __P((struct tcpcb *, int));

int	 syn_cache_add __P((struct socket *, struct mbuf *, struct ip *,
	    struct tcphdr *, u_char *, int, struct tcp_opt_info *));
void	 syn_cache_unreach __P((struct ip *, struct tcphdr *));
struct socket *
	 syn_cache_get __P((struct socket *so, struct mbuf *,
	    struct ip *, struct tcphdr *, int));
void	 syn_cache_insert __P((struct syn_cache *, struct syn_cache ***,
	    struct syn_cache_head **));
struct syn_cache *
	 syn_cache_lookup __P((struct ip *, struct tcphdr *,
	    struct syn_cache ***, struct syn_cache_head **));
void	 syn_cache_reset __P((struct ip *, struct tcphdr *));
int	 syn_cache_respond __P((struct syn_cache *, struct mbuf *,
	    struct ip *, struct tcphdr *, int32_t, u_int32_t));
void	 syn_cache_timer __P((int));

#endif

#endif /* !_NETINET_TCP_VAR_H_ */
