/*-
 * Copyright (c) 1998 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *      BSDI nfs_lock.h,v 2.2 1998/04/28 19:38:41 don Exp
 */

/*
 * lockd has two system calls it uses to get unique kernel services it needs.
 * Each takes two arguments, a version number, and a pointer to a structure.
 * This prevents libc from needing to change if the information passed 
 * between lockd and the kernel needs to change.
 *
 * If a structure changes, you must bump the version number.
 */

#include <nfs/nfsproto.h>


#define LOCKD_REQ_VERSION	1

struct lockd_req {
	int 		op;		/* F_GETLK | F_SETLK | F_UNLCK */
	int		owner;		/* owner of lock, -1 to allocate one */
	int		owner_rel_ok;	/* release owner if no locks left ? */
	int		*owner_ret;	/* owner alloc/free result target */
	void 		*fh;		/* NFS file handle */
	size_t 		fh_len;		/* NFS file handle length */
	u_quad_t 	offset;		/* offset of where to start lock */
	u_quad_t 	len;		/* length of range to lock */
	int 		type;		/* F_RDLCK | F_WRLCK | F_UNLCK */
	struct ucred	cred;		/* user credentials to use for lock */
	struct sockaddr saddr;		/* XXX how about non AF_INET ?? */
	int		pid;		/* pid of lock requester */
};

/* 
 * The fifo where the kernel writes requests for locks on remote NFS files,
 * and where lockd reads these requests.
 *
 */
#define	_PATH_LCKFIFO	"/var/run/lock"

/*
 * This structure is used to uniquely identify the process which originated
 * a particular message to lockd.  A sequence number is used to differentiate
 * multiple messages from the same process.  A process start time is used to
 * detect the unlikely, but possible, event of the recycling of a pid.
 */
struct lockd_msg_ident {
	pid_t		pid;            /* The process ID. */
	struct timeval	pid_start;	/* Start time of process id */
	int		msg_seq;	/* Sequence number of message */
};

#define LOCKD_MSG_VERSION	1

/*
 * The structure that the kernel hands us for each lock request.
 */
typedef struct __lock_msg {
	int			version;	/* which version is this */
	struct lockd_msg_ident	msg_ident;	/* originator of the message */
	struct flock		fl;             /* The lock request. */
	int			wait;           /* The F_WAIT flag. */
	int			getlk;		/* is this a F_GETLK request */
	struct sockaddr 	addr;		/* The address. */
	int			nfsv3;		/* If NFS version 3. */
	size_t			fh_len;		/* The file handle length. */
	struct ucred		cred;		/* user cred for lock req */
	u_int8_t		fh[NFS_SMALLFH];/* The file handle. */
} LOCKD_MSG;

#define LOCKD_ANS_VERSION	1

struct lockd_ans {
	struct lockd_msg_ident	msg_ident;	/* originator of the message */
	int		errno;
	int		set_getlk_pid;		/* use returned pid */
	int		getlk_pid;		/* returned pid for F_GETLK */
};

#ifndef KERNEL
extern int nfslockdreq(int version, struct lockd_req *reqp);
extern int nfslockdans(int version, struct lockd_ans *ansp);
#endif
