/*
 * Copyright (c) 1994 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI stand.h,v 2.16.2.2 2001/08/16 15:53:16 prb Exp
 */
/*-
 * Copyright (c) 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)stand.h	8.1 (Berkeley) 6/11/93
 */

#ifndef _STAND_H_
#define _STAND_H_

struct proc;
struct tty;
struct timeval;

#ifdef SMALL
#define	lseek	big_lseek	/* subvert the declaration in <sys/types.h> */
#endif

#include <sys/param.h>
#include <sys/cdefs.h>
#include <sys/reboot.h>
#include <sys/stat.h>

#include <stand/saerrno.h>

#ifdef SMALL
#undef	lseek			/* revert to private lseek */
#define off_t	int		/* and override typedef */
off_t	lseek __P((int, off_t, int)); /* now prototype our private lseek */
#endif

#ifndef NULL
#define	NULL	0
#endif

int errno;

struct open_file;

/*
 * This structure is used to define file system operations in a file system
 * independent way.
 */
struct fs_ops {
	char	*fsname;
	int	fstype_required;
	int	(*open) __P((char *path, struct open_file *f));
	off_t	(*seek) __P((struct open_file *f, off_t offset, int where));
	int	(*read) __P((struct open_file *f, char *buf,
			u_int size, u_int *resid));
#ifndef SMALL
	int	(*write) __P((struct open_file *f, char *buf,
			u_int size, u_int *resid));
	int	(*stat) __P((struct open_file *f, struct stat *sb));
	int	(*close) __P((struct open_file *f));
	int	(*ls) __P((struct open_file *f));
#endif
};

extern struct fs_ops file_system[];

/* where values for lseek(2) */
#define	SEEK_SET	0	/* set file offset to offset */
#define	SEEK_CUR	1	/* set file offset to current plus offset */
#define	SEEK_END	2	/* set file offset to EOF plus offset */

/*
 * Device switch.  The final argument to the strategy function is written
 * with the actual number of bytes transferred, *not* a residual count as
 * previously documented (but not implemented that way).
 */
struct devsw {
	char	*dv_name;
	int	(*dv_open) __P((void *));
	int	(*dv_close) __P((void *));
	int	(*dv_strategy) __P((void *IO, int rw,
			daddr_t bn, u_int cc, char *ma, u_int *result));
#if 0
	int	(*dv_ioctl) __P((struct open_file *f, int cmd, void *data));
#endif
};

extern struct devsw *devsw[];	/* device array */
extern int ndevs;		/* number of elements in devsw[] */

struct open_file {
	int		f_flags;	/* see F_* below */
	struct devsw	*f_dev;		/* pointer to device operations */
	void		*f_devdata;	/* device specific data */
	struct fs_ops	*f_ops;		/* pointer to file system operations */
	void		*f_fsdata;	/* file system specific data */
	off_t		f_seek;		/* seek pointer for raw devices */
};

#define	SOPEN_MAX	4
extern struct open_file files[SOPEN_MAX];

/* f_flags values */
#define	F_READ		0x0001	/* file opened for reading */
#define	F_WRITE		0x0002	/* file opened for writing */
#define	F_RAW		0x0004	/* raw device open - no file system */

/*
 * Function prototypes.
 * We include declarations for functions that
 * the machine-dependent code must define for us.
 */

/*
 * The machine dependent code must provide one of two routines for
 * turning a boot string into a device and (optional) filename.
 *
 * for SMALL (tiny boot-blocks):
 *	MD code must open the device, setting the device switch pointer
 *	(f_dev) and MD-data (f_devdata).  There is only one device around
 *	so there is no name string involved, and the open must not fail
 *	(return type void).
 *
 * for non-SMALL (`boot' program etc.):
 *	MD code must translate a string (in some machine-dependent
 *	format, though things like sd(3,1,0) are typical) to a device
 *	switch pointer.  It should then open the device, setting f_dev
 *	and f_devdata.  It must also give back a pointer to the filename
 *	portion; this should be the empty string to say `not a file after
 *	all'.  It can return an errno (implying `nothing opened').
 */
#ifdef SMALL
void	devopen __P((struct open_file *f));
#else
int	devopen __P((struct open_file *f, char *fname, char **file));
#endif

void	*alloc __P((u_int size));
void	free __P((void *ptr, u_int size));

struct	disklabel;
char	*getdisklabel __P((const char *buf, struct disklabel *lp));

int	nullop __P((void *));	/* for drivers */

void	putchar __P((int));
int	getchar __P((void));
int	cngetc __P((void));

char	*index __P((const char *, int));
size_t	strcspn __P((const char *, const char *));
long	strtol __P((const char *, char **, int));

#ifdef __powerpc__
/* XXX Blech!  But the DCBZ instruction doesn't work with VM turned off...  */
#define	bzero(v, n)	memset(v, 0x100, n)
#endif

#if defined(__sparc__) || defined(__sparc_v9__) /* XXX but saves space */
void	*memcpy __P((void *, const void *, size_t));
#define bcopy(src, dst, len) memcpy(dst, src, len)
#endif

/*
 * Standalone versions of various C library interfaces.
 * XXX these should have the same types as the real ones!
 */
int	close __P((int));
__dead void
	exit __P((int)) __attribute__((__volatile));
int	fstat __P((int, struct stat *));
char	*ges __P((int));			/* renamed strerror() */
int	getchar __P((void));
int	getfile __P((char *, int));
char	*gets __P((char *));
int	ls __P((int));
int	open __P((char *, int));
__dead void
	panic __P((const char *, ...)) __attribute__((__volatile));
void	printf __P((const char *, ...));
int	read __P((int, void *, unsigned int));
int	stat __P((const char *, struct stat *));
int	write __P((int, void *, unsigned int));

/*
 * The remainder of this file is really only for third and possibly
 * fourth stage boots which use parser.c. It is always include
 * because machine depended versions of stand.h sometimes
 * include mi_boot_options in side a machine dependent version
 * of mi_boot_options. 
 */

#define DEFAULTBOOT_MAXPATHLEN	100

typedef enum 	{LD_INIT, LD_DATA, LD_DONE} ld_cmd_t;

struct mi_boot_options {
	int	 o_echoon;
	int	 o_howto;
	ld_cmd_t o_cmd;		/* used by load_{prog,ramdisk} */
	int	 o_loaded;
	int	 o_ramdisksize;
	int	 o_pausedone;
	int	 o_stoponerror;
	char	*o_defaultboot;
	char	*o_ramdiskimage;
};

typedef	enum	{PF_OK, PF_ABORTED, PF_ERROR} pf_rv_t;
typedef pf_rv_t	(parse_fn_t) __P((char *args, struct mi_boot_options *));

struct parse_info {
	char		*pi_match;	/* This is the input we look for */
	parse_fn_t	*pi_func;	/* function to call */
	char		*pi_help;	/* print for help memu */
};

extern int bootdebug;

/* 
 * Functions provided by parser.c and define.c for use
 * by each other and machine dependent code.
 */
int	bootflags __P((char *, int *, char *));
void	bootloop __P((struct mi_boot_options *, int *cyloffset))
	    __attribute__((__volatile));
int	initdefines __P((char *));
int	loadexec __P((char *, struct mi_boot_options *));
int	loadrdsk __P((char *, struct mi_boot_options *));
char	*nextarg __P((char *));
int	resolveflags __P((char *name, u_long id, char *alias, u_long *result));
int	resolveid __P((char *name, char *id, u_long *result));
int	strtomemsize __P((char *, int rounddown));

void	addpath __P((char *path, void **vpl));
char *	getpath __P((void **vpl));

extern	void *default_list;
extern	void *define_list;


/*
 * Functions provide by machine dependent code for parser.c
 */
struct	bootparam* getbootparam __P((int, struct bootparam*));
void	addbootparam __P((struct bootparam *, void *));
int	load_prog __P((struct mi_boot_options *, caddr_t, int));
int	load_ramdisk __P((struct mi_boot_options *, caddr_t, int));
int	pause __P((int));
void	start_prog __P((struct mi_boot_options*));
void	md_show __P((struct mi_boot_options*));

#endif /* !_STAND_H_ */
