/*-
 * Copyright (c) 1994, 1996 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI mbuf.h,v 2.13.6.2 2001/08/27 13:49:11 dab Exp
 */

/*
 * Copyright (c) 1982, 1986, 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)mbuf.h	8.5 (Berkeley) 2/19/95
 */

#ifndef _SYS_MBUF_H_
#define _SYS_MBUF_H_

#ifndef M_WAITOK
#include <sys/malloc.h>
#endif

/*
 * Mbufs are of a single size, MSIZE (machine/param.h), which
 * includes overhead.  An mbuf may add a single "mbuf cluster" of size
 * MCLBYTES (also in machine/param.h), which has no additional overhead
 * and is used instead of the internal data area; this is done when
 * at least MINCLSIZE of data must be stored.
 */

#define	MLEN		(MSIZE - sizeof(struct m_hdr))	/* normal data len */
#define	MHLEN		(MLEN - sizeof(struct pkthdr))	/* data len w/pkthdr */

#define	MINCLSIZE	(MHLEN+1)	/* smallest amount to put in cluster */
#define	M_MAXCOMPRESS	(MHLEN / 2)	/* max amount to copy for compression */

/*
 * Macros for type conversion
 * mtod(m,t) -	convert mbuf pointer to data pointer of correct type
 * mtoe(m,t) -	convert mbuf pointer to beginning of private ext area
 * dtom(x) -	convert data pointer within mbuf to mbuf pointer (XXX)
 * mtocl(x) -	convert pointer within cluster to cluster index #
 * cltom(x) -	convert cluster # to ptr to beginning of cluster
 */
#define	mtod(m,t)	((t)((m)->m_data))
#define	mtoe(m,t)	((t)(&((m)->m_ext) + 1))
#define	dtom(x)		((struct mbuf *)((long)(x) & ~(MSIZE-1)))
#define	mtocl(x)	(((u_long)(x) - (u_long)mbutl) >> MCLSHIFT)
#define	cltom(x)	((caddr_t)((u_long)mbutl + ((u_long)(x) << MCLSHIFT)))

/* header at beginning of each mbuf: */
struct m_hdr {
	struct	mbuf *mh_next;		/* next buffer in chain */
	struct	mbuf *mh_nextpkt;	/* next chain in queue/record */
	caddr_t	mh_data;		/* location of data */
	int	mh_len;			/* amount of data in this mbuf */
	short	mh_type;		/* type of data in this mbuf */
	short	mh_flags;		/* flags; see below */
};

/* record/packet header in first mbuf of chain; valid if M_PKTHDR set */
struct	pkthdr {
	struct	ifnet *rcvif;		/* rcv interface */
	int	len;			/* total packet length */
	struct mbuf *aux;		/* extra data buffer; ipsec/others */
};

/*
 * Description of external storage mapped into mbuf, valid if M_EXT set.
 * External storage can be a "standard" mbuf cluster of size MCLBYTES,
 * or can be a buffer handled by a private manager.  If there is a private
 * manager, it is responsible for reference counting, etc, and the ext_func
 * field points to the manager's function.  ext_func is NULL for a standard
 * cluster, for which we use the parallel array of reference counts.
 * When we m_copym an mbuf with a private manager, we allocate a new mbuf,
 * then call
 *	(*(m_old->m_ext.ext_func))(struct mbuf *m_old, struct mbuf *m_new)
 * to increment the reference count and/or initialize private fields
 * in m_new.  When releasing an mbuf, any private manager is called with
 *	(*(m_old->m_ext.ext_func))(struct mbuf *m_old, NULL)
 * to release the reference.  In the current implementation, the latter
 * call MUST NOT call spl* functions; it is protected by splmem_fast.
 *
 * The manager can use ext_handle to refer to private data, and/or
 * can use mtoe() to overlay additional information in the internal
 * data area.
 */
struct m_ext {
	caddr_t	ext_buf;		/* start of buffer */
	size_t	ext_size;		/* size of buffer */
	void	(*ext_func) __P((struct mbuf *, struct mbuf *));
					/* free or dup if not the usual */
	void	*ext_handle;		/* private to manager of storage */
	/* mtoe() returns a pointer to the remaining internal space */
};

struct mbuf {
	struct	m_hdr m_hdr;
	union {
		struct {
			struct	pkthdr MH_pkthdr;	/* M_PKTHDR set */
			union {
				struct	m_ext MH_ext;	/* M_EXT set */
				char	MH_databuf[MHLEN];
			} MH_dat;
		} MH;
		char	M_databuf[MLEN];		/* !M_PKTHDR, !M_EXT */
	} M_dat;
};
#define	m_next		m_hdr.mh_next
#define	m_len		m_hdr.mh_len
#define	m_data		m_hdr.mh_data
#define	m_type		m_hdr.mh_type
#define	m_flags		m_hdr.mh_flags
#define	m_nextpkt	m_hdr.mh_nextpkt
#define	m_act		m_nextpkt
#define	m_pkthdr	M_dat.MH.MH_pkthdr
#define	m_ext		M_dat.MH.MH_dat.MH_ext
#define	m_pktdat	M_dat.MH.MH_dat.MH_databuf
#define	m_dat		M_dat.M_databuf

/* mbuf flags */
#define	M_EXT		0x0001	/* has associated external storage */
#define	M_PKTHDR	0x0002	/* start of record */
#define	M_EOR		0x0004	/* end of record */

#define	M_MIP6TUNNEL	0x0010	/* MIP6 temporary use */

/* mbuf pkthdr flags, also in m_flags */
#define	M_BCAST		0x0100	/* send/received as link-level broadcast */
#define	M_MCAST		0x0200	/* send/received as link-level multicast */
#define	M_FORW		0x0400	/* forwarding packet */
#define	M_LOCAL		0x0800	/* force as a local packet */

#if 0	/*NRL IPv6*/
/* IP security related mbuf flags */
#define	M_AUTHENTIC	0x1000	/* This packet has been authenticated. */
#define	M_DECRYPTED	0x2000	/* This packet was decrypted. */
/* IPv6 mbuf flag */
#define	M_DAD		0x4000	/* Used on outbound packets to indicate */
				/* that this is for duplicate address */
				/* detection. */
#endif

/* KAME IPv6 */
#if 1 /*KAME IPSEC*/
#define M_AUTHIPHDR	0x0010	/* data origin authentication for IP header */
#define M_DECRYPTED	0x0020	/* confidentiality */
#endif
#define M_LOOP		0x0040	/* for Mbuf statistics */
#if 1 /*KAME IPSEC*/
#define M_AUTHIPDGM     0x0080  /* data origin authentication */
#endif

/* flags copied when copying m_pkthdr */
#define	M_COPYFLAGS	(M_PKTHDR|M_EOR|M_BCAST|M_MCAST|M_FORW|\
			M_AUTHIPHDR|M_DECRYPTED|M_LOOP|M_AUTHIPDGM)

/* flags that a filter can modify */
#define	M_MODFLAGS	(M_LOCAL|M_FORW|M_MCAST|M_BCAST)

/* mbuf types */
#define	MT_FREE		0	/* should be on free list */
#define	MT_DATA		1	/* dynamic (data) allocation */
#define	MT_HEADER	2	/* packet header */
#define	MT_SOCKET	3	/* socket structure */
#define	MT_PCB		4	/* protocol control block */
#define	MT_RTABLE	5	/* routing tables */
#define	MT_HTABLE	6	/* IMP host tables */
#define	MT_ATABLE	7	/* address resolution tables */
#define	MT_SONAME	8	/* socket name */
#define	MT_SOOPTS	10	/* socket options */
#define	MT_FTABLE	11	/* fragment reassembly header */
#define	MT_RIGHTS	12	/* access rights */
#define	MT_IFADDR	13	/* interface address */
#define MT_CONTROL	14	/* extra-data protocol message */
#define MT_OOBDATA	15	/* expedited data  */

/* flags to m_get/MGET */
#define	M_DONTWAIT	M_NOWAIT
#define	M_WAIT		M_WAITOK

/*
 * mbuf utility macros:
 *
 *	MBUFLOCK(code)
 * prevents a section of code from from being interrupted by network
 * drivers.
 */
#define	MBUFLOCK(code) \
	{ int ms = splmem_fast(); \
	  { code } \
	  splxmem_fast(ms); \
	}

/*
 * mbuf allocation/deallocation macros:
 *
 *	MGET(struct mbuf *m, int how, int type)
 * allocates an mbuf and initializes it to contain internal data.
 *
 *	MGETHDR(struct mbuf *m, int how, int type)
 * allocates an mbuf and initializes it to contain a packet header
 * and internal data.
 */
#define MBALLOC(m, how, type) { \
	int ms = splmem_fast(); \
	if (mbuffree) { \
		(m) = mbuffree; \
		mbuffree = (m)->m_next; \
		mbstat.m_mtypes[MT_FREE]--; \
		splxmem_fast(ms); \
	} else { \
		splxmem_fast(ms); \
		MALLOC((m), struct mbuf *, MSIZE, M_MBUF, (how)); \
	} \
}

#define	MGET(m, how, type) { \
	MBALLOC((m), (how), (type)); \
	if (m) { \
		(m)->m_type = (type); \
		MBUFLOCK(mbstat.m_mtypes[type]++;) \
		(m)->m_next = (struct mbuf *)NULL; \
		(m)->m_nextpkt = (struct mbuf *)NULL; \
		(m)->m_data = (m)->m_dat; \
		(m)->m_flags = 0; \
	} else \
		(m) = m_retry((how), (type)); \
}

#define	MGETHDR(m, how, type) { \
	MBALLOC((m), (how), (type)); \
	if (m) { \
		(m)->m_type = (type); \
		MBUFLOCK(mbstat.m_mtypes[type]++;) \
		(m)->m_next = (struct mbuf *)NULL; \
		(m)->m_nextpkt = (struct mbuf *)NULL; \
		(m)->m_data = (m)->m_pktdat; \
		(m)->m_flags = M_PKTHDR; \
		(m)->m_pkthdr.aux = (struct mbuf *)NULL; \
	} else \
		(m) = m_retryhdr((how), (type)); \
}

/*
 * Mbuf cluster macros.
 * MCLALLOC(caddr_t p, int how) allocates an mbuf cluster.
 * MCLGET adds such clusters to a normal mbuf;
 * the flag M_EXT is set upon success.
 * MCLFREE releases a reference to a cluster allocated by MCLALLOC,
 * freeing the cluster if the reference count has reached 0.
 *
 * Normal mbuf clusters are normally treated as character arrays
 * after allocation, but use the first word of the buffer as a free list
 * pointer while on the free list.
 */
union mcluster {
	union	mcluster *mcl_next;
	char	mcl_buf[MCLBYTES];
};

#define	MCLALLOC(p, how) \
	{ int ms = splmem_fast(); \
	  if (mclfree == 0) { \
		splxmem_fast(ms); \
	  	ms = splimp(); \
		(void)m_clalloc(1, (how)); \
		if (((p) = (caddr_t)mclfree) != 0) { \
			++mclrefcnt[mtocl(p)]; \
			mbstat.m_clfree--; \
			mclfree = ((union mcluster *)(p))->mcl_next; \
		} \
		splx(ms); \
	  } else {\
		if (((p) = (caddr_t)mclfree) != 0) { \
			++mclrefcnt[mtocl(p)]; \
			mbstat.m_clfree--; \
			mclfree = ((union mcluster *)(p))->mcl_next; \
		} \
		splxmem_fast(ms); \
	  } \
	}

#define	MCLGET(m, how) \
	{ MCLALLOC((m)->m_ext.ext_buf, (how)); \
	  if ((m)->m_ext.ext_buf != NULL) { \
		(m)->m_data = (m)->m_ext.ext_buf; \
		(m)->m_flags |= M_EXT; \
		(m)->m_ext.ext_size = MCLBYTES;  \
		(m)->m_ext.ext_func = NULL;  \
	  } \
	}

/* MCLFREE must be used only at splmem/splimp. */
#define	MCLFREE(p) \
	{ if (--mclrefcnt[mtocl(p)] == 0) { \
		((union mcluster *)(p))->mcl_next = mclfree; \
		mclfree = (union mcluster *)(p); \
		mbstat.m_clfree++; \
	  } \
	}

/*
 * MFREE(struct mbuf *m, struct mbuf *n)
 * Free a single mbuf and associated external storage.
 * Place the successor, if any, in n.
 *
 * we do need to check non-first mbuf for m_aux, since some of existing
 * code does not call M_PREPEND properly.
 * (example: call to bpf_mtap from drivers)
 */
#define	MAX_MBFREE	32
#define	MFREE(m, nn) \
	{ int ms; \
	  if (((m)->m_flags & M_PKTHDR) != 0 && (m)->m_pkthdr.aux) { \
		m_freem((m)->m_pkthdr.aux); \
		(m)->m_pkthdr.aux = NULL; \
	  } \
	  if ((m)->m_flags & M_EXT) { \
		if ((m)->m_ext.ext_func) { \
			(*((m)->m_ext.ext_func))(m, (struct mbuf *)0); \
			ms = splmem_fast(); \
		} else { \
			ms = splmem_fast(); \
			MCLFREE((m)->m_ext.ext_buf); \
		} \
	  } else \
		ms = splmem_fast(); \
	  mbstat.m_mtypes[(m)->m_type]--; \
	  (nn) = (m)->m_next; \
	  if (mbstat.m_mtypes[MT_FREE] < MAX_MBFREE) { \
		(m)->m_next = mbuffree; \
		mbuffree = (m); \
		mbstat.m_mtypes[MT_FREE]++; \
		splxmem_fast(ms); \
	  } else { \
		splxmem_fast(ms); \
	  	FREE((m), M_MBUF); \
	  } \
	}

/*
 * Copy mbuf pkthdr from from to to.
 * from must have M_PKTHDR set, and to must be empty.
 * aux pointer will be moved to `to'.
 */
#define	M_COPY_PKTHDR(to, from) { \
	(to)->m_pkthdr = (from)->m_pkthdr; \
	(from)->m_pkthdr.aux = (struct mbuf *)NULL; \
	(to)->m_flags = (from)->m_flags & M_COPYFLAGS; \
	(to)->m_data = (to)->m_pktdat; \
}

/*
 * Set the m_data pointer of a newly-allocated mbuf (m_get/MGET) to place
 * an object of the specified size at the end of the mbuf, longword aligned.
 */
#define	M_ALIGN(m, len) \
	{ (m)->m_data += (MLEN - (len)) &~ (sizeof(long) - 1); }
/*
 * As above, for mbufs allocated with m_gethdr/MGETHDR
 * or initialized by M_COPY_PKTHDR.
 */
#define	MH_ALIGN(m, len) \
	{ (m)->m_data += (MHLEN - (len)) &~ (sizeof(long) - 1); }

/*
 * As above, for cluster mbufs (regardless of header or not).
 */
 
#define MCL_ALIGN(m, len) \
        { (m)->m_data += (MCLBYTES - (len)) &~ (sizeof(long) -1); }

/*
 * Compute the amount of space available
 * before the current start of data in an mbuf.
 * If there is a cluster, other mbufs might be using
 * that space, so return 0.
 */
#define	M_LEADINGSPACE(m) \
	((m)->m_flags & M_EXT ? /* (m)->m_data - (m)->m_ext.ext_buf */ 0 : \
	    (m)->m_flags & M_PKTHDR ? (m)->m_data - (m)->m_pktdat : \
	    (m)->m_data - (m)->m_dat)
/* as above, but allow leading space in a cluster to be used */
#define	M_LEADINGSPACE1(m) \
	((m)->m_flags & M_EXT ? (m)->m_data - (m)->m_ext.ext_buf : \
	    (m)->m_flags & M_PKTHDR ? (m)->m_data - (m)->m_pktdat : \
	    (m)->m_data - (m)->m_dat)

/*
 * Compute the amount of space available
 * after the end of data in an mbuf.
 * Note that if there are multiple references to a cluster,
 * it is the caller's responsibility to ensure that the trailing
 * space is not in use by another mbuf referencing the cluster.
 */
#define	M_TRAILINGSPACE(m) \
	((m)->m_flags & M_EXT ? (m)->m_ext.ext_buf + (m)->m_ext.ext_size - \
	    ((m)->m_data + (m)->m_len) : \
	    &(m)->m_dat[MLEN] - ((m)->m_data + (m)->m_len))

/*
 * Arrange to prepend space of size plen to mbuf m.
 * If a new mbuf must be allocated, how specifies whether to wait.
 * If how is M_DONTWAIT and allocation fails, the original mbuf chain
 * is freed and m is set to NULL.
 */
#define	M_PREPEND(m, plen, how) { \
	if (M_LEADINGSPACE(m) >= (plen)) { \
		(m)->m_data -= (plen); \
		(m)->m_len += (plen); \
	} else \
		(m) = m_prepend((m), (plen), (how)); \
	if ((m) && (m)->m_flags & M_PKTHDR) \
		(m)->m_pkthdr.len += (plen); \
}
/*
 * As above, but allows leading space in a cluster to be used.
 * It is up to the caller to ensure that no other reference
 * to a cluster can be using the leading space.
 */
#define	M_PREPEND1(m, plen, how) { \
	if (M_LEADINGSPACE1(m) >= (plen)) { \
		(m)->m_data -= (plen); \
		(m)->m_len += (plen); \
	} else \
		(m) = m_prepend((m), (plen), (how)); \
	if ((m) && (m)->m_flags & M_PKTHDR) \
		(m)->m_pkthdr.len += (plen); \
}

/* change mbuf to new type */
#define MCHTYPE(m, t) { \
	MBUFLOCK(mbstat.m_mtypes[(m)->m_type]--; mbstat.m_mtypes[t]++;) \
	(m)->m_type = t;\
}

/* length to m_copy to copy all */
#define	M_COPYALL	1000000000

/* compatiblity with 4.3 */
#define  m_copy(m, o, l)	m_copym((m), (o), (l), M_DONTWAIT)

/*
 * Mbuf statistics.
 */
struct mbstat {
	u_long	m_mbufs;	/* mbufs obtained from page pool */
	u_long	m_clusters;	/* clusters obtained from page pool */
	u_long	m_maxclusters;	/* upper bound on nmbclusters */
	u_long	m_clfree;	/* free clusters */
	u_long	m_drops;	/* times failed to find space */
	u_long	m_wait;		/* times waited for space */
	u_long	m_drain;	/* times drained protocols for space */
	u_short	m_mtypes[256];	/* type specific mbuf allocations */

	u_quad_t m_exthdrget;	/* # of calls to IP6_EXTHDR_GET */
	u_quad_t m_exthdrget0;	/* # of calls to IP6_EXTHDR_GET0 */
	u_quad_t m_pulldowns;	/* # of calls to m_pulldown */
	u_quad_t m_pulldown_copy; /* # of mbuf copies in m_pulldown */
	u_quad_t m_pulldown_alloc; /* # of mbuf allocs in m_pulldown */
	u_quad_t m_pullups;	/* # of calls to m_pullup */
	u_quad_t m_pullup_copy;	/* # of possible m_pullup copies */
	u_quad_t m_pullup_alloc; /* # of possible m_pullup mallocs */
	u_quad_t m_pullup_fail;	/* # of possible m_pullup failures */
	u_quad_t m_pullup2;	/* # of calls to m_pullup2 */
	u_quad_t m_pullup2_copy; /* # of possible m_pullup2 copies */
	u_quad_t m_pullup2_alloc; /* # of possible m_pullup2 mallocs */
	u_quad_t m_pullup2_fail; /* # of possible m_pullup2 failures */
};

/*
 * pkthdr.aux type tags.
 */
struct mauxtag {
	int af;
	int type;
	void* p;
};

#ifdef	KERNEL
extern	struct mbuf *mbutl;		/* virtual address of mclusters */
extern	u_int *mclrefcnt;		/* cluster reference counts */
struct	mbstat mbstat;
extern	int nmbclusters;
extern	int maxmbclusters;
union	mcluster *mclfree;
struct	mbuf *mbuffree;
int	max_linkhdr;			/* largest link-level header */
int	max_protohdr;			/* largest protocol header */
int	max_hdr;			/* largest link+protocol header */
int	max_datalen;			/* MHLEN - max_hdr */

struct	mbuf *m_copym __P((struct mbuf *, int, int, int));
struct  mbuf *m_copym2 __P((struct mbuf *, int, int, int));
struct  mbuf *m_pullup2 __P((struct mbuf *, int));
struct  mbuf *m_split __P((register struct mbuf *, int, int));
struct	mbuf *m_free __P((struct mbuf *));
struct	mbuf *m_get __P((int, int));
struct	mbuf *m_getclr __P((int, int));
struct	mbuf *m_gethdr __P((int, int));
struct	mbuf *m_prepend __P((struct mbuf *, int, int));
struct	mbuf *m_pulldown __P((struct mbuf *, int, int, int *));
struct	mbuf *m_pullup __P((struct mbuf *, int));
struct	mbuf *m_retry __P((int, int));
struct	mbuf *m_retryhdr __P((int, int));
struct	mbuf *mcl_get __P((int, int, int));
void	m_adj __P((struct mbuf *, int));
void	m_cat __P((struct mbuf *, struct mbuf *));
void	m_copydata __P((struct mbuf *, int, int, caddr_t));
int	m_clalloc __P((int, int));
void	m_copyback __P((struct mbuf *, int, int, caddr_t));
void	m_freem __P((struct mbuf *));
void	m_reclaim __P((void));

void	m_x128(struct mbuf *, int, int *, q128_t *);
u_long	m_x32(struct mbuf *, int, int *);
u_short	m_x16(struct mbuf *, int, int *);
u_char	m_x8(struct mbuf *, int, int *);
int	m_s128(struct mbuf *, int, q128_t *);
int	m_s32(struct mbuf *, int, unsigned long);
int	m_s16(struct mbuf *, int, unsigned int);
int	m_s8(struct mbuf *, int, unsigned int);

struct mbuf *m_aux_add2 __P((struct mbuf *, int, int, void *));
struct mbuf *m_aux_find2 __P((struct mbuf *, int, int, void *));
struct mbuf *m_aux_add __P((struct mbuf *, int, int));
struct mbuf *m_aux_find __P((struct mbuf *, int, int));
void m_aux_delete __P((struct mbuf *, struct mbuf *));
#endif

#endif /* !_SYS_MBUF_H_ */
