/*	Copyright (c) 1984, 1986, 1987, 1988, 1989 AT&T	*/
/*	  All Rights Reserved  	*/

/*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF AT&T	*/
/*	The copyright notice above does not evidence any   	*/
/*	actual or intended publication of such source code.	*/

/*
 * +++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *		PROPRIETARY NOTICE (Combined)
 *
 * This source code is unpublished proprietary information
 * constituting, or derived under license from AT&T's UNIX(r) System V.
 * In addition, portions of such source code were derived from Berkeley
 * 4.3 BSD under license from the Regents of the University of
 * California.
 *
 *
 *
 *		Copyright Notice
 *
 * Notice of copyright on this source code product does not indicate
 * publication.
 *
 * Copyright 2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 *
 *	Copyright (c) 1983-1989 by AT&T.
 *	All rights reserved.
 */

#ifndef	_VM_AS_H
#define	_VM_AS_H

#pragma ident	"@(#)as.h	1.88	05/06/20 SMI"

#include <sys/watchpoint.h>
#include <vm/seg.h>
#include <vm/faultcode.h>
#include <vm/hat.h>
#include <sys/avl.h>

#ifdef	__cplusplus
extern "C" {
#endif

/*
 * VM - Address spaces.
 */

/*
 * Each address space consists of a sorted list of segments
 * and machine dependent address translation information.
 *
 * All the hard work is in the segment drivers and the
 * hardware address translation code.
 *
 * The segment list is represented as an AVL tree.
 *
 * The address space lock (a_lock) is a long term lock which serializes
 * access to certain operations (as_map, as_unmap) and protects the
 * underlying generic segment data (seg.h) along with some fields in the
 * address space structure as shown below:
 *
 *	address space structure 	segment structure
 *
 *	a_segtree			s_base
 *	a_size				s_size
 *	a_lastgap			s_link
 *	a_seglast			s_ops
 *					s_as
 *					s_data
 *
 * The address space contents lock (a_contents) is a short term
 * lock that protects most of the data in the address space structure.
 * This lock is always acquired after the "a_lock" in all situations
 * except while dealing with AS_CLAIMGAP to avoid deadlocks.
 *
 * The following fields are protected by this lock:
 *
 *	a_flags (AS_PAGLCK, AS_CLAIMGAP, etc.)
 *	a_unmapwait
 *	a_seglast
 *
 * The address space lock (a_lock) is always held prior to any segment
 * operation.  Some segment drivers use the address space lock to protect
 * some or all of their segment private data, provided the version of
 * "a_lock" (read vs. write) is consistent with the use of the data.
 *
 * The following fields are protected by the hat layer lock:
 *
 *	a_vbits
 *	a_hat
 *	a_hrm
 */

struct as {
	kmutex_t a_contents;	/* protect certain fields in the structure */
	uchar_t  a_flags;	/* as attributes */
	uchar_t	a_vbits;	/* used for collecting statistics */
	kcondvar_t a_cv;	/* used by as_rangelock */
	struct	hat *a_hat;	/* hat structure */
	struct	hrmstat *a_hrm; /* ref and mod bits */
	caddr_t	a_userlimit;	/* highest allowable address in this as */
	struct seg *a_seglast;	/* last segment hit on the addr space */
	krwlock_t a_lock;	/* protects segment related fields */
	size_t	a_size;		/* size of address space */
	struct seg *a_lastgap;	/* last seg found by as_gap() w/ AS_HI (mmap) */
	struct seg *a_lastgaphl; /* last seg saved in as_gap() either for */
				/* AS_HI or AS_LO used in as_addseg() */
	avl_tree_t a_segtree;	/* segments in this address space. (AVL tree) */
	avl_tree_t a_wpage;	/* watched pages (procfs) */
	uchar_t	a_updatedir;	/* mappings changed, rebuild a_objectdir */
	timespec_t a_updatetime;	/* time when mappings last changed */
	vnode_t	**a_objectdir;	/* object directory (procfs) */
	size_t	a_sizedir;	/* size of object directory */
	struct as_callback *a_callbacks; /* callback list */
	void *a_xhat;		/* list of xhat providers */
};

#define	AS_PAGLCK		0x80
#define	AS_CLAIMGAP		0x40
#define	AS_UNMAPWAIT		0x20
#define	AS_NEEDSPURGE		0x10	/* mostly for seg_nf, see as_purge() */
#define	AS_BUSY			0x01	/* needed by XHAT framework */

#define	AS_ISPGLCK(as)		((as)->a_flags & AS_PAGLCK)
#define	AS_ISCLAIMGAP(as)	((as)->a_flags & AS_CLAIMGAP)
#define	AS_ISUNMAPWAIT(as)	((as)->a_flags & AS_UNMAPWAIT)
#define	AS_ISBUSY(as)		((as)->a_flags & AS_BUSY)


#define	AS_SETPGLCK(as)		((as)->a_flags |= AS_PAGLCK)
#define	AS_SETCLAIMGAP(as)	((as)->a_flags |= AS_CLAIMGAP)
#define	AS_SETUNMAPWAIT(as)	((as)->a_flags |= AS_UNMAPWAIT)
#define	AS_SETBUSY(as)		((as)->a_flags |= AS_BUSY)

#define	AS_CLRPGLCK(as)		((as)->a_flags &= ~AS_PAGLCK)
#define	AS_CLRCLAIMGAP(as)	((as)->a_flags &= ~AS_CLAIMGAP)
#define	AS_CLRUNMAPWAIT(as)	((as)->a_flags &= ~AS_UNMAPWAIT)
#define	AS_CLRBUSY(as)		((as)->a_flags &= ~AS_BUSY)

#define	AS_TYPE_64BIT(as)	\
	    (((as)->a_userlimit > (caddr_t)UINT32_MAX) ? 1 : 0)

/*
 * The as_callback is the basic structure which supports the ability to
 * inform clients of specific events pertaining to address space management.
 * A user calls as_add_callback to register an address space callback
 * for a range of pages, specifying the events that need to occur.
 * When as_do_callbacks is called and finds a 'matching' entry, the
 * callback is called once, and the callback function MUST call
 * as_delete_callback when all callback activities are complete.
 * The thread calling as_do_callbacks blocks until the as_delete_callback
 * is called.  This allows for asynchorous events to subside before the
 * as_do_callbacks thread continues.
 *
 * An example of the need for this is a driver which has done long-term
 * locking of memory.  Address space management operations (events) such
 * as as_free, as_umap, and as_setprot will block indefinitely until the
 * pertinent memory is unlocked.  The callback mechanism provides the
 * way to inform the driver of the event so that the driver may do the
 * necessary unlocking.
 *
 * The contents of this structure is protected by a_contents lock
 */
typedef void (*callback_func_t)(struct as *, void *, uint_t);
struct as_callback {
	struct as_callback	*ascb_next;		/* list link */
	uint_t			ascb_events;		/* event types */
	callback_func_t		ascb_func;   		/* callback function */
	void			*ascb_arg;		/* callback argument */
	caddr_t			ascb_saddr;		/* start address */
	size_t			ascb_len;		/* address range */
};
/*
 * Callback events
 */
#define	AS_FREE_EVENT		0x1
#define	AS_SETPROT_EVENT	0x2
#define	AS_UNMAP_EVENT		0x4
#define	AS_CALLBACK_CALLED	((uint_t)(1U << (8 * sizeof (uint_t) - 1U)))
#define	AS_UNMAPWAIT_EVENT				\
		(AS_FREE_EVENT | AS_SETPROT_EVENT | AS_UNMAP_EVENT)
#define	AS_ALL_EVENT					\
		(AS_FREE_EVENT | AS_SETPROT_EVENT | AS_UNMAP_EVENT)


/* Return code values for as_callback_delete */
enum as_cbdelete_rc {
	AS_CALLBACK_DELETED,
	AS_CALLBACK_NOTFOUND,
	AS_CALLBACK_DELETE_DEFERRED
};

#ifdef _KERNEL

/*
 * Flags for as_gap.
 */
#define	AH_DIR		0x1	/* direction flag mask */
#define	AH_LO		0x0	/* find lowest hole */
#define	AH_HI		0x1	/* find highest hole */
#define	AH_CONTAIN	0x2	/* hole must contain `addr' */

extern struct as kas;		/* kernel's address space */

/*
 * Macros for address space locking.
 */
#define	AS_LOCK_ENTER(as, lock, type)		rw_enter((lock), (type))
#define	AS_LOCK_EXIT(as, lock)			rw_exit((lock))
#define	AS_LOCK_DESTROY(as, lock)		rw_destroy((lock))
#define	AS_LOCK_TRYENTER(as, lock, type)	rw_tryenter((lock), (type))

/*
 * Macros to test lock states.
 */
#define	AS_LOCK_HELD(as, lock)		RW_LOCK_HELD((lock))
#define	AS_READ_HELD(as, lock)		RW_READ_HELD((lock))
#define	AS_WRITE_HELD(as, lock)		RW_WRITE_HELD((lock))

/*
 * macros to walk thru segment lists
 */
#define	AS_SEGFIRST(as)		avl_first(&(as)->a_segtree)
#define	AS_SEGNEXT(as, seg)	AVL_NEXT(&(as)->a_segtree, (seg))
#define	AS_SEGPREV(as, seg)	AVL_PREV(&(as)->a_segtree, (seg))

void	as_init(void);
void	as_avlinit(struct as *);
struct	seg *as_segat(struct as *as, caddr_t addr);
void	as_rangelock(struct as *as);
void	as_rangeunlock(struct as *as);
struct	as *as_alloc(void);
void	as_free(struct as *as);
int	as_dup(struct as *as, struct as **outas);
struct	seg *as_findseg(struct as *as, caddr_t addr, int tail);
int	as_addseg(struct as *as, struct seg *newseg);
struct	seg *as_removeseg(struct as *as, struct seg *seg);
faultcode_t as_fault(struct hat *hat, struct as *as, caddr_t addr, size_t size,
		enum fault_type type, enum seg_rw rw);
faultcode_t as_faulta(struct as *as, caddr_t addr, size_t size);
int	as_setprot(struct as *as, caddr_t addr, size_t size, uint_t prot);
int	as_checkprot(struct as *as, caddr_t addr, size_t size, uint_t prot);
int	as_unmap(struct as *as, caddr_t addr, size_t size);
int	as_map(struct as *as, caddr_t addr, size_t size, int ((*crfp)()),
		void *argsp);
void	as_purge(struct as *as);
int	as_gap(struct as *as, size_t minlen, caddr_t *basep, size_t *lenp,
		uint_t flags, caddr_t addr);
int	as_memory(struct as *as, caddr_t *basep, size_t *lenp);
size_t	as_swapout(struct as *as);
int	as_incore(struct as *as, caddr_t addr, size_t size, char *vec,
		size_t *sizep);
int	as_ctl(struct as *as, caddr_t addr, size_t size, int func, int attr,
		uintptr_t arg, ulong_t *lock_map, size_t pos);
int	as_exec(struct as *oas, caddr_t ostka, size_t stksz,
		struct as *nas, caddr_t nstka, uint_t hatflag);
int	as_pagelock(struct as *as, struct page ***ppp, caddr_t addr,
		size_t size, enum seg_rw rw);
void	as_pageunlock(struct as *as, struct page **pp, caddr_t addr,
		size_t size, enum seg_rw rw);
void	as_pagereclaim(struct as *as, struct page **pp, caddr_t addr,
		size_t size, enum seg_rw rw);
int	as_setpagesize(struct as *as, caddr_t addr, size_t size, uint_t szc,
		boolean_t wait);
void	as_setwatch(struct as *as);
void	as_clearwatch(struct as *as);
int	as_getmemid(struct as *, caddr_t, memid_t *);

int	as_add_callback(struct as *, void (*)(), void *, uint_t,
			caddr_t, size_t, int);
uint_t	as_delete_callback(struct as *, void *);

#endif	/* _KERNEL */

#ifdef	__cplusplus
}
#endif

#endif	/* _VM_AS_H */
