/*
 * Copyright 2007 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#ifndef	_PG_H
#define	_PG_H

#pragma ident	"@(#)pg.h	1.1	07/03/01 SMI"

/*
 * Processor Groups
 */

#ifdef	__cplusplus
extern "C" {
#endif

#if (defined(_KERNEL) || defined(_KMEMUSER))
#include <sys/cpuvar.h>
#include <sys/group.h>
#include <sys/processor.h>
#include <sys/bitset.h>
#include <sys/atomic.h>
#include <sys/types.h>
#include <sys/kstat.h>

typedef uint_t		pgid_t;		/* processor group id */
typedef uint_t		pg_cid_t;	/* processor group class id */

/*
 * Nature of CPU relationships
 */
typedef enum pg_relation {
	PGR_LOGICAL,
	PGR_PHYSICAL
} pg_relation_t;

/*
 * Processor group structure
 */
typedef struct pg {
	pgid_t		pg_id;		/* seq id */
	pg_relation_t	pg_relation;	/* grouping relationship */
	struct pg_class	*pg_class;	/* pg class */
	struct group	pg_cpus;	/* group of CPUs */
} pg_t;

/*
 * PG class callbacks
 */
struct pg_ops {
	struct pg	*(*alloc)();
	void		(*free)(struct pg *);
	void		(*cpu_init)(struct cpu *);
	void		(*cpu_fini)(struct cpu *);
	void		(*cpu_active)(struct cpu *);
	void		(*cpu_inactive)(struct cpu *);
	void		(*cpupart_in)(struct cpu *, struct cpupart *);
	void		(*cpupart_out)(struct cpu *, struct cpupart *);
	void		(*cpupart_move)(struct cpu *, struct cpupart *,
			    struct cpupart *);
	int		(*cpu_belongs)(struct pg *, struct cpu *);
};

#define	PG_CLASS_NAME_MAX 32

/*
 * PG class structure
 */
typedef struct pg_class {
	pg_cid_t	pgc_id;
	char		pgc_name[PG_CLASS_NAME_MAX];
	struct pg_ops	*pgc_ops;
	pg_relation_t	pgc_relation;
} pg_class_t;

/*
 * Per CPU processor group data
 */
typedef struct cpu_pg {
	struct group	pgs;		/* All the CPU's PGs */
	struct group	cmt_pgs;	/* CMT load balancing lineage */
					/* (Group hierarchy ordered) */
	struct pg	*cmt_lineage;	/* Ascending lineage chain */
} cpu_pg_t;

/*
 * PG cpu iterator cookie
 */
typedef struct	pg_cpu_itr {
	pg_t		*pg;
	group_iter_t	position;
} pg_cpu_itr_t;

/*
 * Initialize a PG CPU iterator cookie
 */
#define	PG_CPU_ITR_INIT(pgrp, itr)		\
{						\
	group_iter_init(&(itr).position);	\
	(itr).pg = ((pg_t *)pgrp);		\
}

/*
 * Return the first CPU in a PG
 */
#define	PG_CPU_GET_FIRST(pgrp)			\
	(GROUP_SIZE(&((pg_t *)pgrp)->pg_cpus) > 0 ?	\
	    GROUP_ACCESS(&((pg_t *)pgrp)->pg_cpus, 0) : NULL)

/*
 * Framework routines
 */
void		pg_init(void);
pg_cid_t	pg_class_register(char *, struct pg_ops *, pg_relation_t);

/*
 * PG CPU reconfiguration hooks
 */
void		pg_cpu0_init(void);
void		pg_cpu_init(cpu_t *);
void		pg_cpu_fini(cpu_t *);
void		pg_cpu_active(cpu_t *);
void		pg_cpu_inactive(cpu_t *);
void		pg_cpu_startup(cpu_t *);
void		pg_cpu_bootstrap(cpu_t *);

/*
 * PG cpupart service hooks
 */
void		pg_cpupart_in(cpu_t *, struct cpupart *);
void		pg_cpupart_out(cpu_t *, struct cpupart *);
void		pg_cpupart_move(cpu_t *, struct cpupart *, struct cpupart *);

/*
 * PG CPU utility routines
 */
pg_t		*pg_create(pg_cid_t);
void		pg_destroy(pg_t *);
void		pg_cpu_add(pg_t *, cpu_t *);
void		pg_cpu_delete(pg_t *, cpu_t *);
pg_t		*pg_cpu_find_pg(cpu_t *, group_t *);
cpu_t		*pg_cpu_next(pg_cpu_itr_t *);


#endif	/* !_KERNEL && !_KMEMUSER */

#ifdef	__cplusplus
}
#endif

#endif /* _PG_H */
