//////////////////////////////////////////////////////////////////////////////////
// Project Name:   [ CDX Class Library - CDX.lib ]
// Source File:    [ CDXTile Implementation ]
// Author:         [ Bil Simser - bsimser@home.com ]
// Contributions:  [ Lennart Steinke, Dan Farley ]
// Revision:       [ 1.99a ]
//////////////////////////////////////////////////////////////////////////////////
#include "CDX.h"

typedef struct
{
	BITMAPINFOHEADER bmiHeader;
	RGBQUAD bmiColors[256];
} BITMAPINFO_256;


//////////////////////////////////////////////////////////////////////////////////
// CDXTile Constructor
//////////////////////////////////////////////////////////////////////////////////
CDXTile::CDXTile(CDXScreen *pScreen, const char* szFilename, int w, int h, int num, BOOL memoryType)
{
	Create(pScreen, szFilename, w, h, num, memoryType);
}

//////////////////////////////////////////////////////////////////////////////////
// CDXTile Constructor
//////////////////////////////////////////////////////////////////////////////////
CDXTile::CDXTile()
{
	DestRect.top = 0;
	DestRect.left = 0;
	DestRect.bottom = 0;
	DestRect.right = 0;
}

//////////////////////////////////////////////////////////////////////////////////
// CDXTile Create
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXTile::Create(CDXScreen *pScreen, const char* szFilename, int w, int h, int num, BOOL memoryType)
{
	if(szFilename == NULL) 
		return FALSE;

	m_lpDDS = DDLoadSizeBitmap(pScreen->m_lpDD, szFilename, &m_PixelWidth, &m_PixelHeight, memoryType);
	if(m_lpDDS == NULL) 
		return FALSE;

	m_pFilename = szFilename;

	Screen = pScreen;

    m_BlockWidth = w;
	m_BlockHeight = h;
	m_BlockNum = num;
	
	DestRect.top = 0;
	DestRect.left = 0;
	DestRect.bottom = m_PixelHeight;
	DestRect.right = m_PixelWidth;

	SrcRect.top = 0;
	SrcRect.left = 0;
	SrcRect.bottom = m_PixelHeight;
	SrcRect.right = m_PixelWidth;

//<JJH>
	SetClipRect(&SrcRect);
	GetRGBFormat(m_lpDDS, &m_RGB);
	m_MMX = IsMMX();
//<JJH>

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////////////
// SetBitmapInfo
//////////////////////////////////////////////////////////////////////////////////
void SetBitmapInfo(long width, long height, BITMAPINFO_256 &info, PBYTE pal)
{
	BITMAPINFOHEADER &h = info.bmiHeader;
	
	h.biSize			= sizeof(info.bmiHeader);
	h.biWidth			= width;
	h.biHeight			= height;
	h.biPlanes			= 1;
	h.biBitCount		= 8;
	h.biCompression		= BI_RGB;
	h.biSizeImage		= ((width +3) & ~3) * height;
	h.biXPelsPerMeter	= 0;
	h.biYPelsPerMeter	= 0;
	h.biClrUsed			= 256; // Max. amount of colors
	h.biClrImportant	= 256; // All colors are important

	if (pal)
	{
		for (int col=0; col<256; col++)
		{
			
			info.bmiColors[col].rgbRed		= pal[0];
			info.bmiColors[col].rgbGreen	= pal[1];
			info.bmiColors[col].rgbBlue		= pal[2];
			info.bmiColors[col].rgbReserved	= 0;

			pal+=3;
		}
	}
	else
	{
		::ZeroMemory(info.bmiColors, sizeof(info.bmiColors));
	}
}

//////////////////////////////////////////////////////////////////////////////////
// CreateFromTLE
//
// Creates a CDX tile file from an autograb/MapMaker tle file
// pScreen points to the CDX main screen, szFilename contains the name
// of the TLE file to load.
//////////////////////////////////////////////////////////////////////////////////
BOOL CDXTile::CreateFromTLE(CDXScreen *pScreen, const char* szFilename)
{
#if DIRECTDRAW_VERSION >= CDX_DDVER
	DDSURFACEDESC2 ddsd;
#else
	DDSURFACEDESC ddsd;
#endif

	if(szFilename == NULL) 
		return FALSE;

	// So, let's try to load the tle file.
	FILE *tleFile = fopen(szFilename, "rb"); // open for binary read
	WORD count;    // number of bytes stored in the tle file
	WORD width;    // width of each tile
	WORD height;   // height of each tile
	BYTE pal[768]; // the palette used by the tiles

	if (!tleFile)
	{
		return FALSE;
	}
	
	// read the number of tiles and their dimensions
	fread(&count , sizeof(WORD), 1, tleFile);
	fread(&width , sizeof(WORD), 1, tleFile);
	fread(&height, sizeof(WORD), 1, tleFile);

	// read the (dos) palette
	fread(pal, 768, 1, tleFile);

	// read all of the tile data at once
	PBYTE buffer;
	DWORD size = count * width * height;
	DWORD count_read;

	// create a buffer which can handle "size" bytes
	buffer = (PBYTE) malloc(size);
	if (!buffer)
	{
		// uups, no memory
		fclose(tleFile);
		return FALSE;
	}

	// Read "size" bytes
	count_read = fread(buffer, 1, size, tleFile);

	if (count_read != size)
	{
		// Something went wrong. Either a read error,
		// or a wrong file format
		free(buffer);
		fclose(tleFile);
		return FALSE;
	}

	fclose(tleFile); // close TLE file
    
	DWORD image_width    = count * width;
    DWORD image_height   = height;
    LONG  count_per_line = count;
    LONG  count_per_col  = 1;

    DWORD maxWidth = pScreen->m_dwPixelWidth;
    if (image_width > maxWidth)
    {
        count_per_line = maxWidth / width;
        count_per_col  = (count / count_per_line) +1;
        
        image_width    = count_per_line * width;
        image_height   = count_per_col  * height;
    }

	// Since CDX awaits tile data in a DD surface, let's create one
	// with a height of "height" and a size of ("width" * "count")
    m_lpDDS = DDCreateSurface(pScreen->m_lpDD, image_width, image_height, &ddsd);
	if (m_lpDDS == NULL)
	{
		free(buffer);
		MessageBox(NULL, "CreateSurface failed", "Dbg", MB_OK);
		return FALSE;
	}
    
	// Now, we create a DIBSection to hold the tiles
	// First step is to set-up the information
	HDC				hDC = ::CreateDC("DISPLAY", NULL, NULL, NULL);
	BITMAPINFO_256	BitmapInfo;	
	UINT			iUsage= DIB_RGB_COLORS;

	SetBitmapInfo(image_width, image_height, BitmapInfo, pal);
	
	// Then to create the DIBSection
	BYTE *bmp_data; // Going to get the buffer
	HBITMAP hBitmap;
	hBitmap = CreateDIBSection(
					hDC,
					(BITMAPINFO*)&BitmapInfo, 
					iUsage, 
					(void **) &bmp_data, 
					NULL, 0);
	
	DeleteDC(hDC);

	// Fill with 0
	FillMemory(bmp_data, ((width +3) & ~3) * image_height, 0);
	
	// Since TLE files store 256 color tiles, we need to check
	// the format of the CDX- screen.
	// If we have less than 8 bpp, we return with an error.
	// If we have 8 bpp, we copy the data
	// If we have more than 8bpp, we use  the palette data to
	//    fill the surface with rgbx values.

	LONG dwLineWidth = (image_width +3) & ~3;
	LONG offset=0;
    LONG btm = (image_height-1) * dwLineWidth;
    LONG line_height = dwLineWidth *height;
    LONG x=0, y=0, foo = count_per_col-1;
	PBYTE surface; 
	PBYTE source = buffer;
    LONG  transfered =count;

	int line;

    for (y=0; y < (count_per_col-1); y++)
    {
        for (x=0, offset=0; x < count_per_line; x++, offset += width)
        {
            surface = bmp_data + btm + offset;
        
            for (line=0; line < height; line++)
		    {
			    memcpy(surface, source, width);
			    source  += width;
                surface -= dwLineWidth;
		    }
            transfered--;
        }
        btm -= line_height;
    }

    for (x=0, offset=0; transfered >0; x++, offset += width, transfered--)
    {
        surface = bmp_data + btm + offset;
    
        for (line=0; line < height; line++)
		{
			memcpy(surface, source, width);
			source  += width;
            surface -= dwLineWidth;
		}
    }
	free(buffer);

	// At this point, we have a DIB holding the
	// tile data, and a surface holding nothing, so
	// let's copy the DIB to the surface
	DDCopyBitmap(m_lpDDS, hBitmap, 0, 0, 0, 0);

	// Now, we can free the DIB
	::DeleteObject(hBitmap);

	m_pFilename = NULL;//szFilename;

	Screen = pScreen; 

	m_BlockWidth  = width;
	m_BlockHeight = height;
	m_BlockNum    = count;
	
	DestRect.top  = 0;
	DestRect.left = 0;
	DestRect.bottom = m_PixelHeight  = image_height;
	DestRect.right  = m_PixelWidth = image_width;

	SrcRect.top = 0;
	SrcRect.left = 0;
	SrcRect.bottom = m_PixelHeight;
	SrcRect.right = m_PixelWidth;

	return TRUE;
}

