/**
 * @(#)Theme.java
 *
 * Theme class is very similar to the Configurations class, 
 * however Themes don't have some important variables such 
 * as bookPath or currentLanguage or save and load methods. 
 * Instead, they have names and a method named applyTheme. 
 * These features renders possible the storage of the data 
 * about preset themes with the smallest unnecessary memory 
 * loss.
 *
 * @author Gokhan Capar
 * @version 1.00 21.12.2009
 */

package bin;

import java.awt.Font;
import java.awt.Color;

public class Theme {
	
	/**
	 * <code>name</code><br>
	 * Name of the theme.
	 */
	private String name;
	
	/**
	 * <code>font</code><br>
	 * The font object for the theme.
	 */
	private Font font;
	
	/**
	 * <code>fontColor</code><br>
	 * The font color of the theme.
	 */
	private Color fontColor;
	
	/**
	 * <code>backColor</code><br>
	 * The backColor of the theme.
	 */
	private Color backColor;
	
	/**
	 * <code>BGImagePath</code><br>
	 * If there is, the background image path.
	 */
	private String BGImagePath;
	
	/**
	 * <code>useBGImage</code><br>
	 * Whether the theme uses a BGImage.
	 * False by default.
	 */
	private boolean useBGImage = false;

	/**
	 * The constructor without an option of BGImagePath.
	 * This constructor is used when a background image 
	 * is not used and it sets useBGImage directly to false.
	 */
	public Theme(String name,Font chosenFont,Color chosenFontColor,Color chosenBackColor) {
		// Set the name
		this.name = name;
		// Set the font
		font = chosenFont;
		// Set the font color
		fontColor = chosenFontColor;
		// Set the back color
		backColor = chosenBackColor;
	}

    /**
	 * The constructor with an option of BGImagePath.
	 * This constructor is used when a background image 
	 * is used and it sets useBGImage directly to true.
	 */
	public Theme(String name,Font chosenFont,Color chosenFontColor,Color chosenBackColor,String chosenBGImagePath) {
		// Call the first constructor
    	this(name,chosenFont,chosenFontColor,chosenBackColor);
    	// Set useBGImage to true
    	useBGImage = true;
    	// Assign BGImagePath
    	BGImagePath = chosenBGImagePath;
    }
    
    /**
	 * <code>applyTheme</code><br>
	 * This method gets a Display and a Configurations
	 * object as variables and makes the necessary changes 
	 * on them to apply the theme.
	 * @param d The Display object.
	 * @param c The Configurations object.
	 */
	public void applyTheme (Display d,Configurations c) {
		// Set the font
    	d.changeFont(font);
    	// Set the font color
    	c.setFontColor(fontColor);
    	// Set the background color
    	c.setBackColor(backColor);
    	// Set the useBGImage boolean
    	c.setUseBGImage(useBGImage);
    	// If using a background image
    	if (useBGImage)
    		// Assign the background image
	    	d.setBGImage(BGImagePath);
	    // Repaint to see the changes
		d.repaint();
		// Update the configurations
		d.updateConfig();
    }
    
    /**
	 * <code>getName</code><br>
	 * Returns the name of the theme.
	 * @return The name of the theme.
	 */
	public String getName() {
		// Return the name
    	return name;
    }
}