package commands;

import com.inet.jortho.SpellChecker;
import gui.Application;
import javax.swing.text.JTextComponent;
import neutrino.dialogs.LineChooser;
import neutrino.dialogs.TextFinder;
import neutrino.text.ITextComponent;

/**
 * Gateway to editing commands.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
class EditingCommandGateway implements ICommandGateway {

	private Application application = null;
	
	public EditingCommandGateway(Application application) {
		this.application = application;
	}

    /**
     * Performs command appropriate to given type
     * @param command - CommandType
     */
    public void performCommand(CommandType command) {
        switch (command) {
            case UNDO:
                undo();
                break;
            case REDO:
                redo();
                break;
            case REVERT:
                revert();
                break;
            case CUT:
                cut();
                break;
            case COPY:
                copy();
                break;
            case PASTE:
                paste();
                break;
            case CLEAR:
                clear();
                break;
            case READ_ONLY_MODE:
                readOnlyMode();
                break;
            case GO_TO:
                goTo();
                break;
            case FIND:
                find();
                break;
            case REPLACE:
                replace();
                break;
            case CHECK_SPELLING:
                checkSpelling();
                break;
            case SELECT_ALL:
                selectAll();
                break;
            case DESELECT:
                deselect();
                break;
        }
    }

    /**
     * Returns true when the command gateway contains the command
     * @param command - CommandType
     * @return - boolean
     */
    public boolean containsCommand(CommandType command) {
        switch (command) {
            case UNDO:
            case REDO:
            case REVERT:
            case CUT:
            case COPY:
            case PASTE:
            case CLEAR:
            case READ_ONLY_MODE:
            case GO_TO:
            case FIND:
            case REPLACE:
            case CHECK_SPELLING:
            case SELECT_ALL:
            case DESELECT:
                return true;
            default:
                return false;
        }
    }

    /**
     * Returns true when can perform command appropriate to given type
     * @param command - CommandType
     * @return boolean
     */
    public boolean canPerformCommand(CommandType command) {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        switch (command) {
            case UNDO:
                return canUndo();
            case REDO:
                return canRedo();
            case REVERT:
                return canRevert();
            case CUT:
                return canCut();
            case COPY:
                return canCopy();
            case PASTE:
                return canPaste();
            case CLEAR:
                return canClear();
            case READ_ONLY_MODE:
                return canReadOnlyMode();
            case GO_TO:
                return canGoTo();
            case FIND:
                return canFind();
            case REPLACE:
                return canReplace();
            case CHECK_SPELLING:
                return canCheckSpelling();
            case SELECT_ALL:
                return canSelectAll();
            case DESELECT:
                return canDeselect();
            default:
                return true;
        }
    }

    private boolean canUndo() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canUndo();
        } else {
            return false;
        }
    }

	private void undo() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) ((ITextComponent) textComponent).undo();
	}

    private boolean canRedo() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canRedo();
        } else {
            return false;
        }
    }

    private void redo() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) ((ITextComponent) textComponent).redo();
	}

    private boolean canRevert() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canRevert();
        } else {
            return false;
        }
    }

    private void revert() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) ((ITextComponent) textComponent).revert();
    }

    private boolean canCut() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canCut();
        } else {
            return true;
        }
    }

    private void cut() {
		application.getTextEditor().getTextComponent().cut();
	}

    private boolean canCopy() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canCopy();
        } else {
            return true;
        }
    }

    private void copy() {
		application.getTextEditor().getTextComponent().copy();
	}

    private boolean canPaste() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canPaste();
        } else {
            return true;
        }
    }

    private void paste() {
		application.getTextEditor().getTextComponent().paste();
	}

    private boolean canClear() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canClear();
        } else {
            return false;
        }
    }

    private void clear() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            ((ITextComponent) textComponent).clear();
        }
    }

    private boolean canReadOnlyMode() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canToggleReadOnlyMode();
        } else {
            return false;
        }
    }

    private void readOnlyMode() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            ((ITextComponent) textComponent).toggleReadOnlyMode();
        }
    }

    private boolean canGoTo() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return !((ITextComponent) textComponent).isTextEmpty();
        } else {
            return true;
        }
    }

    private void goTo() {
		LineChooser.showGoToLineDialog(application, application.getTextEditor().getTextComponent());
	}

    private boolean canFind() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return !((ITextComponent) textComponent).isTextEmpty();
        } else {
            return true;
        }
    }

    private void find() {
		TextFinder.showFindDialog(application, application.getTextEditor().getTextComponent());
	}

    private boolean canReplace() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return !((ITextComponent) textComponent).isTextEmpty();
        } else {
            return true;
        }
    }

    private void replace() {
		TextFinder.showReplaceDialog(application, application.getTextEditor().getTextComponent());
	}

    private boolean canCheckSpelling() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return !((ITextComponent) textComponent).isTextEmpty();
        } else {
            return true;
        }
    }

    private void checkSpelling() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        SpellChecker.showSpellCheckerDialog(textComponent, SpellChecker.getOptions());
    }

    private boolean canSelectAll() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canSelectAll();
        } else {
            return true;
        }
    }

    private void selectAll() {
		application.getTextEditor().getTextComponent().selectAll();
	}

    private boolean canDeselect() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canDeselect();
        } else {
            return false;
        }
    }

    private void deselect() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            ((ITextComponent) textComponent).deselect();
        }
    }

}
