package neutrino.autocomplete;

import java.awt.Point;
import java.io.*;
import java.nio.charset.Charset;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.text.*;
import org.fife.ui.autocomplete.*;

/**
 * @author Oleh Radvanskyj
 * @version 1.0
 */

public class XMLCompletionProvider extends CompletionProviderBase {

	protected Segment seg;

    /**
     * The completions this provider is aware of.  Subclasses should ensure
     * that this list is sorted alphabetically (case-insensitively).
     */
    protected List<Completion> completions;

    /**
	 * Constructor.  The returned provider will not be aware of any completions.
	 *
	 * @see #addCompletion(Completion)
	 */
	public XMLCompletionProvider() {
		init();
        clearParameterizedCompletionParams();
        completions = new ArrayList<Completion>();
	}

    /**
     * Adds a single completion to this provider.  If you are adding multiple
     * completions to this provider, for efficiency reasons please consider
     * using {@link #addCompletions(List)} instead.
     *
     * @param c The completion to add.
     * @throws IllegalArgumentException If the completion's provider isn't
     *         this <tt>CompletionProvider</tt>.
     * @see #addCompletions(List)
     * @see #removeCompletion(Completion)
     * @see #clear()
     */
    public void addCompletion(Completion c) {
        checkProviderAndAdd(c);
    }


    /**
     * Adds {@link Completion}s to this provider.
     *
     * @param completions The completions to add.  This cannot be
     *        <code>null</code>.
     * @throws IllegalArgumentException If a completion's provider isn't
     *         this <tt>CompletionProvider</tt>.
     * @see #addCompletion(Completion)
     * @see #removeCompletion(Completion)
     * @see #clear()
     */
    public void addCompletions(List<Completion> completions) {
        for (Completion c : completions) {
            checkProviderAndAdd(c);
        }
    }


    protected void checkProviderAndAdd(Completion c) {
        if (c.getProvider()!=this) throw new IllegalArgumentException("Invalid CompletionProvider");
        if (!(c instanceof TagCompletion)) throw new IllegalArgumentException("Invalid Completion");
        completions.add(c);
    }


    /**
     * Removes all completions from this provider.  This does not affect
     * the parent <tt>CompletionProvider</tt>, if there is one.
     *
     * @see #addCompletion(Completion)
     * @see #addCompletions(List)
     * @see #removeCompletion(Completion)
     */
    public void clear() {
        completions.clear();
    }


    /**
     * Returns a list of <tt>Completion</tt>s in this provider with the
     * specified input text.
     *
     * @param inputText The input text to search for.
     * @return A list of {@link Completion}s, or <code>null</code> if there
     *         are no matching <tt>Completion</tt>s.
     */
    @SuppressWarnings("unchecked")
    public List<Completion> getCompletionByInputText(String inputText) {
        return null;
    }

    private String getPreviousTagOptimized(JTextComponent comp) {
        Document doc = comp.getDocument();
        int dot = comp.getCaretPosition();
        if (dot >= 100) {
            int start = dot - 100;
            int len = 100;
            try {
                doc.getText(start, len, seg);
            } catch (BadLocationException ble) {
                ble.printStackTrace();
                return EMPTY_STRING;
            }

            String text = seg.toString();
            Pattern pattern = Pattern.compile("</?[a-z_-]*( */>|[ >])");
            Matcher matcher = pattern.matcher(text);
            int startIndex = -1;
            int endIndex = -1;
            while (matcher.find()) {
                startIndex = matcher.start();
                endIndex = matcher.end();
            }
            if (startIndex >= 0 && endIndex >= 0) {
                return text.substring(startIndex, endIndex);
            }
        }
        return EMPTY_STRING;
    }

    private String getPreviousTag(JTextComponent comp) {
        String optimized = getPreviousTagOptimized(comp);
        if (!optimized.isEmpty()) return optimized;
        Document doc = comp.getDocument();
        int start = 0;
        int dot = comp.getCaretPosition();
        int len = dot - start;
        try {
            doc.getText(start, len, seg);
        } catch (BadLocationException ble) {
            ble.printStackTrace();
            return EMPTY_STRING;
        }

        String text = seg.toString();
        Pattern pattern = Pattern.compile("</?[a-z_-]*( */>|[ >])");
        Matcher matcher = pattern.matcher(text);
        int startIndex = -1;
        int endIndex = -1;
        while (matcher.find()) {
            startIndex = matcher.start();
            endIndex = matcher.end();
        }
        if (startIndex >= 0 && endIndex >= 0) {
            return text.substring(startIndex, endIndex);
        }
        return EMPTY_STRING;
    }

    private Boolean isPreviousTagCompletedOptimized(JTextComponent comp) {
        Document doc = comp.getDocument();
        int dot = comp.getCaretPosition();
        if (dot >= 40) {
            int start = dot - 40;
            int len = 40;
            try {
                doc.getText(start, len, seg);
            } catch (BadLocationException ble) {
                ble.printStackTrace();
                return null;
            }

            String text = seg.toString();
            Pattern pattern = Pattern.compile("[<>]");
            Matcher matcher = pattern.matcher(text);
            int startIndex = -1;
            int endIndex = -1;
            while (matcher.find()) {
                startIndex = matcher.start();
                endIndex = matcher.end();
            }
            if (startIndex >= 0 && endIndex >= 0) {
                if (text.substring(startIndex, endIndex).equals("<")) {
                    pattern = Pattern.compile("<[a-z_-]+ ");
                    matcher = pattern.matcher(text.substring(startIndex, text.length()));
                    if (matcher.find()) return new Boolean(false);
                    else return new Boolean(true);
                }
                else return new Boolean(true);
            }
        }
        return null;
    }

    private boolean isPreviousTagCompleted(JTextComponent comp) {
        Boolean optimized = isPreviousTagCompletedOptimized(comp);
        if (optimized != null) return optimized.booleanValue();
        Document doc = comp.getDocument();
        int start = 0;
        int dot = comp.getCaretPosition();
        int len = dot - start;
        try {
            doc.getText(start, len, seg);
        } catch (BadLocationException ble) {
            ble.printStackTrace();
            return true;
        }
        String text = seg.toString();
        Pattern pattern = Pattern.compile("[<>]");
        Matcher matcher = pattern.matcher(text);
        int startIndex = -1;
        int endIndex = -1;
        while (matcher.find()) {
            startIndex = matcher.start();
            endIndex = matcher.end();
        }
        if (startIndex >= 0 && endIndex >= 0) {
            if (text.substring(startIndex, endIndex).equals("<")) {
                pattern = Pattern.compile("<[a-z_-]+ ");
                matcher = pattern.matcher(text.substring(startIndex, text.length()));
                if (matcher.find()) return false;
            }
        }
        return true;
    }

    private Boolean isPreviousTagClosedOptimized(JTextComponent comp) {
        Document doc = comp.getDocument();
        int dot = comp.getCaretPosition();
        if (dot >= 40) {
            int start = dot - 40;
            int len = 40;
            try {
                doc.getText(start, len, seg);
            } catch (BadLocationException ble) {
                ble.printStackTrace();
                return null;
            }

            String text = seg.toString();
            Pattern pattern = Pattern.compile("[<>]");
            Matcher matcher = pattern.matcher(text);
            int startIndex = -1;
            int endIndex = -1;
            while (matcher.find()) {
                startIndex = matcher.start();
                endIndex = matcher.end();
            }
            if (startIndex >= 0 && endIndex >= 0) {
                if (text.substring(startIndex, endIndex).equals("<")) {
                    pattern = Pattern.compile("<[a-z_-]+.*/>.*");
                    matcher = pattern.matcher(text.substring(startIndex, text.length()));
                    if (matcher.find()) return new Boolean(true);
                    else return new Boolean(false);
                } else {
                    if (startIndex - 1 >= 0 && text.charAt(startIndex - 1) == '/') return new Boolean(true);
                }
            }
        }
        return null;
    }

    private boolean isPreviousTagClosed(JTextComponent comp) {
        Boolean optimized = isPreviousTagClosedOptimized(comp);
        if (optimized != null) return optimized.booleanValue();
        Document doc = comp.getDocument();
        int start = 0;
        int dot = comp.getCaretPosition();
        int len = dot - start;
        try {
            doc.getText(start, len, seg);
        } catch (BadLocationException ble) {
            ble.printStackTrace();
            return false;
        }
        String text = seg.toString();
        Pattern pattern = Pattern.compile("[<>]");
        Matcher matcher = pattern.matcher(text);
        int startIndex = -1;
        int endIndex = -1;
        while (matcher.find()) {
            startIndex = matcher.start();
            endIndex = matcher.end();
        }
        if (startIndex >= 0 && endIndex >= 0) {
            if (text.substring(startIndex, endIndex).equals("<")) {
                pattern = Pattern.compile("<[a-z_-]+.*/>.*");
                matcher = pattern.matcher(text.substring(startIndex, text.length()));
                if (matcher.find()) return true;
            } else {
                if (startIndex - 1 >= 0) return text.charAt(startIndex - 1) == '/';
            }
        }
        return false;
    }

    private Completion findCompletionForTagName(String tagName) {
        for (Completion completion : completions) {
            if (completion instanceof TagCompletion && ((TagCompletion) completion).getTagName().equals(tagName)) return completion;
        }
        return null;
    }

    private String getParentTag(JTextComponent comp) {
        Document doc = comp.getDocument();
        int start = 0;
        int dot = comp.getCaretPosition();
        int len = dot - start;
        try {
            doc.getText(start, len, seg);
        } catch (BadLocationException ble) {
            ble.printStackTrace();
            return EMPTY_STRING;
        }

        String text = seg.toString();
        Pattern pattern = Pattern.compile("(<.*>.*</.*>|<.*/>|<.*>)");
        Matcher matcher = pattern.matcher(text);
        int startIndex = -1;
        int endIndex = -1;
        ArrayList<String> list = new ArrayList<String>();
        while (matcher.find()) {
            startIndex = matcher.start();
            endIndex = matcher.end();
            list.add(text.substring(startIndex, endIndex));
        }
        filterSingleTagsInList(list);
        filterDoubledTagsInList(list);
        return getTagFromList(list);
    }

    private String getTagFromList(ArrayList<String> list) {
        if (list.size() == 0) return EMPTY_STRING;
        String tag = list.get(list.size() - 1);
        Pattern pattern = Pattern.compile("<[a-z_-]+[ />]");
        Matcher matcher = pattern.matcher(tag);
        if (!matcher.find()) return EMPTY_STRING;
        int startIndex = matcher.start();
        int endIndex = matcher.end();
        tag = tag.substring(startIndex + 1, endIndex - 1);
        return tag;
    }

    private void filterDoubledTagsInList(ArrayList<String> list) {
        int size = list.size();
        for (int i = 0; i < size; i++) {
            for (int n = list.size() - 1; n >= 1; n--) {
                String tag1 = list.get(n);
                Pattern pattern = Pattern.compile("</?[a-z_-]+[ />]");
                Matcher matcher = pattern.matcher(tag1);
                if (!matcher.find()) continue;
                int startIndex = matcher.start();
                int endIndex = matcher.end();
                tag1 = (tag1.charAt(1) == '/') ? tag1.substring(startIndex + 2, endIndex - 1) : tag1.substring(startIndex + 1, endIndex - 1);
                String tag2 = list.get(n-1);
                matcher = pattern.matcher(tag2);
                if (!matcher.find()) continue;
                startIndex = matcher.start();
                endIndex = matcher.end();
                tag2 = (tag2.charAt(1) == '/') ? tag2.substring(startIndex + 2, endIndex - 1) : tag2.substring(startIndex + 1, endIndex - 1);
                if (tag1.equals(tag2)) {
                    list.remove(n-1);
                    list.remove(n-1);
                    n--;
                }
            }
        }
    }

    private void filterSingleTagsInList(ArrayList<String> list) {
        for (int i = 0; i < list.size(); i++) {
            String tagName = list.get(i);
            if (tagName.matches("<.*>.*</.*>")) {
                list.remove(i);
                i--;
                continue;
            }
            if (tagName.matches("<.*/>")) {
                list.remove(i);
                i--;
                continue;
            }
        }
    }

    private List<Completion> getChildrenForCompletion(TagCompletion completion) {
        List<Completion> completions = new ArrayList<Completion>();
        if (completion == null) return completions;
        for (TagChild tagChild : completion.getChildren()) {
            Completion tagChildCompletion = findCompletionForTagName(tagChild.getName());
            if (tagChildCompletion != null) completions.add(tagChildCompletion);
        }
        return completions;
    }

    private List<Completion> filterChildrenForParent(TagCompletion parent, String previousTagName) {
        List<Completion> completions = new ArrayList<Completion>();
        if (parent == null) return completions;
        if (parent.getMultiplicity() == TagChild.ONE_OR_MORE || parent.getMultiplicity() == TagChild.ZERO_OR_MORE) {
            return getChildrenForCompletion(parent);
        }
        boolean isUsed = false;
        for (TagChild tagChild : parent.getChildren()) {
            if (tagChild.getName().equals(previousTagName)) {
                isUsed = true;
                if (tagChild.getMultiplicity() == TagChild.ZERO_OR_MORE || tagChild.getMultiplicity() == TagChild.ONE_OR_MORE) {
                    Completion tagChildCompletion = findCompletionForTagName(tagChild.getName());
                    completions.add(tagChildCompletion);
                }
                continue;
            }
            if (isUsed) {
                Completion tagChildCompletion = findCompletionForTagName(tagChild.getName());
                completions.add(tagChildCompletion);
            }
        }
        return completions;
    }

    private List<Completion> filterCompletionsForEnteredText(List<Completion> completions, String enteredText) {
        List<Completion> result = new ArrayList<Completion>();
        if (enteredText == null || enteredText.isEmpty() || enteredText.charAt(0) != '<' || enteredText.length() == 1) return completions;
        enteredText = enteredText.substring(1);
        for (Completion completion : completions) {
            String tagName = ((TagCompletion) completion).getTagName();
            if (tagName.startsWith(enteredText)) {
                result.add(completion);
            }
        }
        if (result.size() > 0) return result;
        else return completions;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @SuppressWarnings("unchecked")
    protected List<Completion> getCompletionsImpl(JTextComponent comp) {
        List<Completion> retVal = new ArrayList<Completion>();
        String text = getAlreadyEnteredText(comp);
        String previousTag = getPreviousTag(comp);
        if (previousTag.isEmpty()) { // no previous tag
            retVal.add(completions.get(0));
        } else if (previousTag.charAt(previousTag.length() - 2) == '/') { // previous tag is already closed
            previousTag = previousTag.substring(1, previousTag.length() - 2).trim();
            String parentTag = getParentTag(comp);
            TagCompletion parent = (TagCompletion) findCompletionForTagName(parentTag);
            retVal = filterCompletionsForEnteredText(filterChildrenForParent(parent, previousTag), text);
        } else if (Character.isLetter(previousTag.charAt(1))) { // previous tag is opened
            previousTag = previousTag.substring(1, previousTag.length() - 1);
            if (isPreviousTagCompleted(comp)) {
                if (isPreviousTagClosed(comp)) {
                    String parentTag = getParentTag(comp);
                    TagCompletion parent = (TagCompletion) findCompletionForTagName(parentTag);
                    retVal = filterCompletionsForEnteredText(filterChildrenForParent(parent, previousTag), text);
                } else {
                    TagCompletion completion = (TagCompletion) findCompletionForTagName(previousTag);
                    retVal = filterCompletionsForEnteredText(getChildrenForCompletion(completion), text);
                }
            } else { // previous tag is not completed
                TagCompletion completion = (TagCompletion) findCompletionForTagName(previousTag);
                retVal = completion.getAttributes();
            }
        } else if (previousTag.charAt(1) == '/') { // previous tag is closed
            previousTag = previousTag.substring(2, previousTag.length() - 1);
            String parentTag = getParentTag(comp);
            TagCompletion parent = (TagCompletion) findCompletionForTagName(parentTag);
            retVal = filterCompletionsForEnteredText(filterChildrenForParent(parent, previousTag), text);
        }
        return retVal;
    }

    @Override
    public List<Completion> getCompletions(JTextComponent comp) {
        return getCompletionsImpl(comp);
    }

    /**
     * Removes the specified completion from this provider.  This method
     * will not remove completions from the parent provider, if there is one.
     *
     * @param c The completion to remove.
     * @return <code>true</code> if this provider contained the specified
     *         completion.
     * @see #clear()
     * @see #addCompletion(Completion)
     * @see #addCompletions(List)
     */
    public boolean removeCompletion(Completion c) {
        // Don't just call completions.remove(c) as it'll be a linear search.
        int index = Collections.binarySearch(completions, c);
        if (index<0) {
            return false;
        }
        completions.remove(index);
        return true;
    }



    /**
	 * Returns the text just before the current caret position that could be
	 * the start of something auto-completable.<p>
	 *
	 * This method returns all characters before the caret that are matched
	 * by  {@link #isValidChar(char)}.
	 *
	 * {@inheritDoc}
	 */
	public String getAlreadyEnteredText(JTextComponent comp) {
		
		Document doc = comp.getDocument();

		int dot = comp.getCaretPosition();
		Element root = doc.getDefaultRootElement();
		int index = root.getElementIndex(dot);
		Element elem = root.getElement(index);
		int start = elem.getStartOffset();
		int len = dot-start;
		try {
			doc.getText(start, len, seg);
		} catch (BadLocationException ble) {
			ble.printStackTrace();
			return EMPTY_STRING;
		}

		int segEnd = seg.offset + len;
		start = segEnd - 1;
		while (start>=seg.offset && isValidChar(seg.array[start])) {
			start--;
		}
		start++;

		len = segEnd - start;
		return len==0 ? EMPTY_STRING : new String(seg.array, start, len);

	}


	/**
	 * {@inheritDoc}
	 */
	public List<Completion> getCompletionsAt(JTextComponent tc, Point p) {
        return null;
	}


	/**
	 * {@inheritDoc}
	 */
	public List<ParameterizedCompletion> getParameterizedCompletions(JTextComponent tc) {
        return null;
	}


	/**
	 * Initializes this completion provider.
	 */
	protected void init() {
		seg = new Segment();
	}


	/**
	 * Returns whether the specified character is valid in an auto-completion.
	 * The default implementation is equivalent to
	 * "<code>Character.isLetterOrDigit(ch) || ch=='_'</code>".  Subclasses
	 * can override this method to change what characters are matched.
	 *
	 * @param ch The character.
	 * @return Whether the character is valid.
	 */
	protected boolean isValidChar(char ch) {
		return Character.isLetterOrDigit(ch) || ch == '_' ||  ch == '<' ||  ch == '-';
	}

    private String getDocumentTagsDefinition(String fileName) {
        FileInputStream input = null;
        StringBuilder builder = new StringBuilder();
        try {
            input = new FileInputStream(fileName);
            byte[] buffer = new byte[4098];
            int length = 0;
            while ((length = input.read(buffer)) != -1) {
                builder.append(new String(buffer, 0, length, Charset.forName("utf-8")));
            }
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (input != null)
                try {
                    input.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
        }
        return builder.toString();
    }

    public void loadAttributes(TagCompletion completion, String source) {
        Pattern pattern = Pattern.compile("<!ATTLIST +" + completion.getTagName() + " +" + ".*>");
        Matcher matcher = pattern.matcher(source);
        while (matcher.find()) {
            String attribute = source.substring(matcher.start(), matcher.end());
            attribute = attribute.substring("<!ATTLIST".length()).trim().substring(completion.getTagName().length()).trim();
            String attributeName = attribute.substring(0, attribute.indexOf(' '));
            String description = attribute.substring(attribute.indexOf(' '), attribute.length() - 1).trim();
            AttributeCompletion attributeCompletion = new AttributeCompletion(this, attributeName, description);
            completion.getAttributes().add(attributeCompletion);
        }
    }

    public void loadFromDTD(String fileName) {
        List<Completion> completions = new ArrayList<Completion>();
        Pattern pattern = Pattern.compile("<!ELEMENT.*>");
        String source = getDocumentTagsDefinition(fileName);
        Matcher matcher = pattern.matcher(source);
        while (matcher.find()) {
            String element = source.substring(matcher.start(), matcher.end());
            String tagName = "";
            String description = "";
            if (element.indexOf("EMPTY") != -1) {
                tagName = element.substring("<!ELEMENT".length(), element.indexOf("EMPTY")).trim();
            } else if (element.indexOf("ANY") != -1) {       // to do all elements for any
                tagName = element.substring("<!ELEMENT".length(), element.indexOf("ANY")).trim();
            } else {
                tagName = element.substring("<!ELEMENT".length(), element.indexOf("(")).trim();
                description = element.substring(element.indexOf('('), element.indexOf('>'));
            }
            TagCompletion completion = new TagCompletion(this, tagName, description);
            loadAttributes(completion, source);
            completions.add(completion);
        }
        addCompletions(completions);
    }

}