package neutrino.dialogs;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.ListDataListener;
import static javax.swing.JOptionPane.*;

import neutrino.multitext.MultiTextComponent;
import neutrino.text.BackupManager;
import neutrino.text.TextEditor;

/**
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class DocumentRestorer {

	private class AbstractRestoreDialog extends JDialog {

        protected class RestoreListCellRenderer extends JLabel implements ListCellRenderer {

            public RestoreListCellRenderer() {
                setOpaque(true);
            }

            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                String text = value.toString();
                setText(text.substring(0, text.indexOf('|')));
                setToolTipText(text.substring(0, text.indexOf('|')) + " " + text.substring(text.indexOf('|') + 1));
                if (isSelected) {
                    setForeground(list.getSelectionForeground());
                    setBackground(list.getSelectionBackground());
                } else {
                    setForeground(list.getForeground());
                    setBackground(list.getBackground());
                }
                return this;
            }

        }

		protected JList lBackups = new JList();
		protected JButton bClear = new JButton("Clear");
		protected JButton bClearAll = new JButton("Clear All");
		protected JButton bCancel = new JButton("Cancel");
		protected int option = JOptionPane.CANCEL_OPTION;
		protected List<File> backups = null;
        protected TextEditor textEditor = null;
        protected MultiTextComponent multiTextComponent = null;
		protected ListModel listModel = new ListModel() {

			public void addListDataListener(ListDataListener l) { }

			public Object getElementAt(int index) {
				return getBackupInformation(backups.get(index));
			}

			public int getSize() {
				return backups.size();
			}

			public void removeListDataListener(ListDataListener l) { }
			
		};
		
		public AbstractRestoreDialog(JFrame owner) {
			super(owner);
		}
		

		protected String getBackupInformation(File backup) {
			ObjectInputStream stream = null;
			String filePath = null;
            Date date = null;
            try {
				stream = new ObjectInputStream(new FileInputStream(backup));
				try {
                    stream.readLong(); // read editor id
                    stream.readObject(); // read editor class name
					filePath = (String) stream.readObject();
                    date = (Date) stream.readObject();
				} catch (EOFException e) {
					e.printStackTrace();
				}
			} catch (FileNotFoundException e) {
				filePath = null;
			} catch (IOException e) {
				filePath = null;
			} catch (ClassNotFoundException e) {
				filePath = null;
			} finally {
				if (stream != null) {
					try {
						stream.close();
					} catch (IOException e) {
						filePath = null;
					}
				}
			}
            // refresh the model when element is missed
			if (filePath == null) refreshModel();
			return filePath + "|" + date.toGMTString();
		}

        protected void refreshModel() {
            if (textEditor != null) backups = getBackups(getOwner(), textEditor);
            else backups = getBackups(getOwner(), multiTextComponent);
            if (!backups.isEmpty()) {
                lBackups.setSelectedIndex(0);
            } else {
                this.option = JOptionPane.CANCEL_OPTION;
                setVisible(false);
            }
            lBackups.revalidate();
            lBackups.repaint();
        }
		
		public int getOption() {
			return this.option;
		}
		
	}
	
	private class RestoreDialog extends AbstractRestoreDialog implements ActionListener {
		
		private JButton bOk = new JButton("Ok");
		protected File selectedBackup = null;
		
		public RestoreDialog(JFrame owner, List<File> backups, TextEditor textEditor) {
			super(owner);
            this.textEditor = textEditor;
			this.backups = backups;
			setTitle("Restore");
			setModal(true);
			setResizable(false);
			getRootPane().setDefaultButton(bOk);
			setDefaultCloseOperation(DISPOSE_ON_CLOSE);
			
			lBackups.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			lBackups.setModel(listModel);
			Dimension size = new Dimension(400, 300);
			lBackups.setMinimumSize(size);
			lBackups.setPreferredSize(size);
			lBackups.setMaximumSize(size);
			lBackups.setSelectedIndex(0);
            lBackups.setCellRenderer(new RestoreListCellRenderer());

            JPanel pBackups = new JPanel();
			pBackups.setBorder(new CompoundBorder(new TitledBorder("Backup"), new EmptyBorder(0, 5, 5, 5)));
			pBackups.setLayout(new BorderLayout());
			pBackups.add(new JScrollPane(lBackups), BorderLayout.CENTER);
			
			
			bOk.setMnemonic(KeyEvent.VK_O);
			bClear.setMnemonic(KeyEvent.VK_C);
			bClearAll.setMnemonic(KeyEvent.VK_A);
			bCancel.setMnemonic(KeyEvent.VK_N);
			
			bOk.addActionListener(this);
			bClear.addActionListener(this);
			bClearAll.addActionListener(this);
			bCancel.addActionListener(this);
			
			
			GridBagLayout layout = new GridBagLayout();
			getContentPane().setLayout(layout);
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 5;
			c.insets = new Insets(5, 5, 5, 0);
			getContentPane().add(pBackups, c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(10, 5, 0, 5);
			getContentPane().add(bOk, c);
			c.gridy = 1;
			c.insets = new Insets(5, 5, 0, 5);
			getContentPane().add(bClear, c);
			c.gridy = 2;
			getContentPane().add(bClearAll, c);
			c.gridy = 3;
			getContentPane().add(bCancel, c);		
			pack();
		}
		
		public File getSelectedBackup() {
			return selectedBackup;
		}

		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == bOk) {
				int index = lBackups.getSelectedIndex();
				if (index == -1) return;
				selectedBackup = backups.get(index);
				this.option = JOptionPane.OK_OPTION;
				setVisible(false);
			} else if (e.getSource() == bClear) {
				int index = lBackups.getSelectedIndex();
				if (index == -1) return;
				backups.get(index).delete();
				backups.remove(index);
				if (backups.isEmpty()) {
					this.option = JOptionPane.CANCEL_OPTION;
					setVisible(false);
				}					
				lBackups.repaint();
			} else if (e.getSource() == bClearAll) {
				Iterator<File> iterator = backups.iterator();
				while (iterator.hasNext()) {
					iterator.next().delete();
				}
				this.option = JOptionPane.CANCEL_OPTION;
				setVisible(false);
			} else if (e.getSource() == bCancel) {
				this.option = JOptionPane.CANCEL_OPTION;
				setVisible(false);
			}
		}
		
	}
	
	private class MultiRestoreDialog extends AbstractRestoreDialog implements ActionListener {
		
		private JButton bRestore = new JButton("Restore");
		private JButton bRestoreAll = new JButton("Restore All");
		protected File[] selectedBackups = null;
		
		public MultiRestoreDialog(JFrame owner, List<File> backups, MultiTextComponent multiTextComponent) {
			super(owner);
			this.backups = backups;
            this.multiTextComponent = multiTextComponent;
			setTitle("Restore from backup");
			setModal(true);
			setResizable(false);
			getRootPane().setDefaultButton(bRestore);
			setDefaultCloseOperation(DISPOSE_ON_CLOSE);
			
			lBackups.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
			lBackups.setModel(listModel);
			Dimension size = new Dimension(400, 300);
			lBackups.setMinimumSize(size);
			lBackups.setPreferredSize(size);
			lBackups.setMaximumSize(size);
			lBackups.setSelectedIndex(0);
            lBackups.setCellRenderer(new RestoreListCellRenderer());

			JPanel pBackups = new JPanel();
			pBackups.setBorder(new CompoundBorder(new TitledBorder("Backup"), new EmptyBorder(0, 5, 5, 5)));
			pBackups.setLayout(new BorderLayout());
			pBackups.add(new JScrollPane(lBackups), BorderLayout.CENTER);
			
			
			bRestore.setMnemonic(KeyEvent.VK_R);
			bRestoreAll.setMnemonic(KeyEvent.VK_A);
			bClear.setMnemonic(KeyEvent.VK_C);
			bClearAll.setMnemonic(KeyEvent.VK_L);
			bCancel.setMnemonic(KeyEvent.VK_N);
			
			bRestore.addActionListener(this);
			bRestoreAll.addActionListener(this);
			bClear.addActionListener(this);
			bClearAll.addActionListener(this);
			bCancel.addActionListener(this);
			
			
			GridBagLayout layout = new GridBagLayout();
			getContentPane().setLayout(layout);
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 6;
			c.insets = new Insets(5, 5, 5, 0);
			getContentPane().add(pBackups, c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(10, 5, 0, 5);
			getContentPane().add(bRestore, c);
			c.gridy = 1;
			c.insets = new Insets(5, 5, 0, 5);
			getContentPane().add(bRestoreAll, c);
			c.gridy = 2;
			c.insets = new Insets(5, 5, 0, 5);
			getContentPane().add(bClear, c);
			c.gridy = 3;
			getContentPane().add(bClearAll, c);
			c.gridy = 4;
			getContentPane().add(bCancel, c);		
			pack();
		}
		
		public File[] getSelectedBackups() {
			return selectedBackups;
		}

		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == bRestore) {
				int[] indices = lBackups.getSelectedIndices();
				if (indices == null || indices.length == 0) return;
				selectedBackups = new File[indices.length];
				for (int i = 0; i < indices.length; i++) {
					selectedBackups[i] = backups.get(indices[i]);
				}
				this.option = JOptionPane.OK_OPTION;
				setVisible(false);
			} else if (e.getSource() == bRestoreAll) {
				selectedBackups = new File[backups.size()];
				for (int i = 0; i < selectedBackups.length; i++) {
					selectedBackups[i] = backups.get(i);
				}
				this.option = JOptionPane.OK_OPTION;
				setVisible(false);
			} else if (e.getSource() == bClear) {
				int[] indices = lBackups.getSelectedIndices();
				if (indices == null || indices.length == 0) return;
				ArrayList<File> selectedFiles = new ArrayList<File>(); 
				for (int i = 0; i < indices.length; i++) {
					backups.get(indices[i]).delete();
					selectedFiles.add(backups.get(indices[i]));
				}
				backups.removeAll(selectedFiles);
				if (backups.isEmpty()) {
					this.option = JOptionPane.CANCEL_OPTION;
					setVisible(false);
				}					
				lBackups.repaint();
			} else if (e.getSource() == bClearAll) {
				Iterator<File> iterator = backups.iterator();
				while (iterator.hasNext()) {
					iterator.next().delete();
				}
				this.option = JOptionPane.CANCEL_OPTION;
				setVisible(false);
			} else if (e.getSource() == bCancel) {
				this.option = JOptionPane.CANCEL_OPTION;
				setVisible(false);
			}
		}
		
	}
	
	private static void establishBounds(JDialog dialog, Window owner) {
	      Dimension d1 = dialog.getSize();
	      Dimension d2 = owner.getSize();
	      Dimension ds = dialog.getToolkit().getScreenSize();
	      int x = Math.max((d2.width-d1.width)/2, 0);
	      int y = Math.max((d2.height-d1.height)/2, 0);
	      int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
	      int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
	      dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
	}

    private static ArrayList<File> getBackups(Window owner, TextEditor textEditor) {
        File backupsDirectory = new File(BackupManager.BACKUPS_DIRECTORY);
        if (!backupsDirectory.isDirectory()) {
            showMessageDialog(owner, "Can not locate backups directory", "Error", ERROR_MESSAGE);
            return null;
        }
        ArrayList<File> list = new ArrayList<File>();
        File[] files = backupsDirectory.listFiles();
        for (int i = 0; i < files.length; i++) {
            File file = files[i];
            if (file.isFile() && file.getAbsolutePath().endsWith(BackupManager.BACKUP_FILE_EXTENSION)) {
                long id = getId(file);
                if (id >= 0 && id != textEditor.getBackupManager().getId()) {
                    list.add(file);
                }
            }
        }
        return list;
    }

    private static ArrayList<File> getBackups(Window owner, MultiTextComponent multiTextComponent) {
        File backupsDirectory = new File(BackupManager.BACKUPS_DIRECTORY);
        if (!backupsDirectory.isDirectory()) {
            showMessageDialog(owner, "Can not locate backups directory", "Error", ERROR_MESSAGE);
            return null;
        }
        ArrayList<File> list = new ArrayList<File>();
        File[] files = backupsDirectory.listFiles();
        for (int i = 0; i < files.length; i++) {
            File file = files[i];
            if (file.isFile() && file.getAbsolutePath().endsWith(BackupManager.BACKUP_FILE_EXTENSION)) {
                long id = getId(file);
                boolean canAdd = true;
                for (TextEditor textEditor : multiTextComponent.getTextEditors()) {
                    if (id >= 0 && id == textEditor.getBackupManager().getId()) {
                        canAdd = false;
                        break;
                    }
                }
                if (canAdd) list.add(file);
            }
        }
        return list;
    }
	
	/**
	 * Creates and shows restore dialog. Returns selected backup file or null othrewise.
	 * @return File
	 */
	public static File showRestoreDialog(JFrame owner, TextEditor textEditor) {
		ArrayList<File> list = getBackups(owner, textEditor);
		if (list.isEmpty()) {
			showMessageDialog(owner, "There is no backups", "Message", INFORMATION_MESSAGE);
			return null;
		}
		File backup = null;
		DocumentRestorer plainDocumentRestorer = new DocumentRestorer();
		RestoreDialog dialog = plainDocumentRestorer.new RestoreDialog(owner, list, textEditor);
		establishBounds(dialog, owner);
		dialog.setVisible(true);
		if (dialog.getOption() == JOptionPane.OK_OPTION) {
			backup = dialog.getSelectedBackup();
		}
		dialog.dispose();
		return backup;
	}
	
	/**
	 * Creates and shows restore dialog. Returns selected backup files or null othrewise.
	 * @return File
	 */
	public static File[] showMultiRestoreDialog(JFrame owner, MultiTextComponent multiTextComponent) {
		ArrayList<File> list = getBackups(owner, multiTextComponent);
		if (list.isEmpty()) {
			showMessageDialog(owner, "There is no backups", "Message", INFORMATION_MESSAGE);
			return null;
		}
		File[] backups = null;
		DocumentRestorer plainDocumentRestorer = new DocumentRestorer();
		MultiRestoreDialog dialog = plainDocumentRestorer.new MultiRestoreDialog(owner, list, multiTextComponent);
		establishBounds(dialog, owner);
		dialog.setVisible(true);
		if (dialog.getOption() == JOptionPane.OK_OPTION) {
			backups = dialog.getSelectedBackups();
		}
		dialog.dispose();
		return backups;
	}

    public static boolean hasBackup(TextEditor textEditor) {
        File backupsDirectory = new File(BackupManager.BACKUPS_DIRECTORY);
        if (!backupsDirectory.isDirectory()) {
            return false;
        }
        File[] files = backupsDirectory.listFiles();
        for (int i = 0; i < files.length; i++) {
            File file = files[i];
            if (file.isFile() && file.getAbsolutePath().endsWith(BackupManager.BACKUP_FILE_EXTENSION)) {
                long id = getId(file);
                if (id >= 0 && id != textEditor.getBackupManager().getId()) {
                    return true;
                }
            }
        }
        return false;
    }

    public static boolean hasBackup(MultiTextComponent multiTextComponent) {
        File backupsDirectory = new File(BackupManager.BACKUPS_DIRECTORY);
        if (!backupsDirectory.isDirectory()) {
            return false;
        }
        File[] files = backupsDirectory.listFiles();
        for (int i = 0; i < files.length; i++) {
            File file = files[i];
            if (file.isFile() && file.getAbsolutePath().endsWith(BackupManager.BACKUP_FILE_EXTENSION)) {
                long id = getId(file);
                boolean inSet = false;
                for (TextEditor textEditor : multiTextComponent.getTextEditors()) {
                    if (id >= 0 && id == textEditor.getBackupManager().getId()) {
                        inSet = true;
                    }
                }
                if (!inSet) return true;
            }
        }
        return false;
    }

    private static long getId(File backup) {
        ObjectInputStream stream = null;
        long id = -1;
        try {
            stream = new ObjectInputStream(new FileInputStream(backup));
            try {
                id = stream.readLong();
            } catch (EOFException e) {
                e.printStackTrace();
            }
        } catch (FileNotFoundException e) {
            id = -1;
        } catch (IOException e) {
            id = -1;
        } finally {
            if (stream != null) {
                try {
                    stream.close();
                } catch (IOException e) {
                    id = -1;
                }
            }
        }
        return id;
    }
	
	public static String getTextComponentClassName(File backup) {
		ObjectInputStream stream = null;
		String className = null;
		try {
			stream = new ObjectInputStream(new FileInputStream(backup));
			try {
				stream.readLong(); // read editor id
				className = (String) stream.readObject();
			} catch (EOFException e) {
				e.printStackTrace();
			}
		} catch (FileNotFoundException e) {
			className = null;
		} catch (IOException e) {
			className = null;
		} catch (ClassNotFoundException e) {
			className = null;
		} finally {
			if (stream != null) {
				try {
					stream.close();
				} catch (IOException e) {
					className = null;
				}
			}
		}
		return className;
	}
	
}
