package neutrino.dialogs;

import java.io.File;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

import neutrino.multitext.EditorAdapterFactory;
import neutrino.multitext.EditorType;
import neutrino.multitext.MultiTextComponent;
import neutrino.multitext.TextEditorAdapter;
import neutrino.text.TextEditor;

/**
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class EditorActivator {
	
	private class EditorActivationDialog extends JDialog implements ActionListener {
		
		protected JList lTexts = new JList();
		private JButton bActivate = new JButton("Activate");
		private JButton bCurrent = new JButton("Current");
		private JButton bSelect = new JButton("Select");
		private JButton bCancel = new JButton("Cancel");
        private int currentIndex = 0;
        private MultiTextComponent multiTextComponent = null;
		private ArrayList<TextEditor> texts = null;
		private ArrayList<TextEditor> selectedTexts = null;
        private JPopupMenu popupMenu = new JPopupMenu();
        private JMenuItem miHide = new JMenuItem("Hide");
		protected int option = JOptionPane.CANCEL_OPTION;
		protected ListModel listModel = new ListModel() {

			public void addListDataListener(ListDataListener l) { }

			public Object getElementAt(int index) {
                if (texts.get(index).isFileLoaded()) {
				    return texts.get(index).getFile().getName();
                } else {
                    return MultiTextComponent.createTitleForTextComponent(texts.get(index));
                }
			}

			public int getSize() {
				return texts.size();
			}

			public void removeListDataListener(ListDataListener l) { }
			
		};
        protected class TextsListCellRenderer extends JLabel implements ListCellRenderer {

            public TextsListCellRenderer() {
                setOpaque(true);
            }

            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                String text;
                if (texts.get(index).isFileLoaded()) {
                    File file = texts.get(index).getFile();
                    text = file.getName();
                    setToolTipText(file.getAbsolutePath());
                } else {
                    text = MultiTextComponent.createTitleForTextComponent(texts.get(index));
                    setToolTipText("The text is without the file");
                }
                setText(text);
                if (isSelected) {
                    setForeground(list.getSelectionForeground());
                    setBackground(list.getSelectionBackground());
                } else {
                    setForeground(list.getForeground());
                    setBackground(list.getBackground());
                }
                return this;
            }

        }

        public class TabMenuItem extends JMenuItem {

            private int index = 0;

            public TabMenuItem(String title, int index, String tooltipText) {
                super(Integer.toString(index + 1) + " " + title);
                this.index = index;
                setToolTipText(tooltipText);
                addActionListener(actionListener);
            }

            private ActionListener actionListener = new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    lTexts.setSelectedIndex(index);
                }
            };

        }

        private void buildPopupMenu() {
            TextEditorAdapter[] adapters = multiTextComponent.getTextEditorAdapters();
            for (int i = 0; i < adapters.length; i++) {
                EditorType type = EditorAdapterFactory.getEditorTypeForTextEditorAdapter(adapters[i]);
                String tooltipText;
                if (texts.get(i).isFileLoaded()) {
                    File file = texts.get(i).getFile();
                    tooltipText = file.getAbsolutePath();
                } else {
                    tooltipText = MultiTextComponent.createTitleForTextComponent(texts.get(i));
                }
                TabMenuItem menuItem = new TabMenuItem(type.getName(), i, tooltipText);
                popupMenu.add(menuItem);
            }
            popupMenu.addSeparator();
            popupMenu.add(miHide);
            miHide.setMnemonic(KeyEvent.VK_H);
            miHide.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent actionEvent) {
                    popupMenu.setVisible(false);
                }
            });
        }

        private void setCurrentIndex() {
            TextEditorAdapter[] adapters = multiTextComponent.getTextEditorAdapters();
            for (int i = 0; i < adapters.length; i++) {
                if (adapters[i] == multiTextComponent.getCurrentEditorAdapter()) {
                    currentIndex = i;
                    break;
                }
            }
        }

		public EditorActivationDialog(MultiTextComponent multiTextComponent) {
			super();
			setTitle("Editor activation");
			setModal(true);
			setResizable(false);
			getRootPane().setDefaultButton(bActivate);
			setDefaultCloseOperation(DISPOSE_ON_CLOSE);
            this.multiTextComponent = multiTextComponent;

            texts = new ArrayList<TextEditor>();
            TextEditor[] array = multiTextComponent.getTextEditors();
            for (int i = 0; i < array.length ; i++) texts.add(array[i]);

            buildPopupMenu();
            setCurrentIndex();

			lTexts.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			lTexts.setModel(listModel);
			Dimension size = new Dimension(400, 300);
			lTexts.setMinimumSize(size);
			lTexts.setPreferredSize(size);
			lTexts.setMaximumSize(size);
			lTexts.setSelectedIndex(currentIndex);
            lTexts.setCellRenderer(new TextsListCellRenderer());

			JPanel pEditors = new JPanel();
			pEditors.setBorder(new CompoundBorder(new TitledBorder("Editor"), new EmptyBorder(0, 5, 5, 5)));
			pEditors.setLayout(new BorderLayout());
			pEditors.add(new JScrollPane(lTexts), BorderLayout.CENTER);
			
			bActivate.setMnemonic(KeyEvent.VK_A);
			bCurrent.setMnemonic(KeyEvent.VK_U);
			bSelect.setMnemonic(KeyEvent.VK_S);
			bCancel.setMnemonic(KeyEvent.VK_C);
			
			bActivate.addActionListener(this);
			bCurrent.addActionListener(this);
			bSelect.addActionListener(this);
			bCancel.addActionListener(this);

            bActivate.setToolTipText("Activate the selected editor");
            bCurrent.setToolTipText("Select the current text editor");
            bSelect.setToolTipText("Select the editor from a menu");
            bCancel.setToolTipText("Abort editor activation dialog");
			
			GridBagLayout layout = new GridBagLayout();
			getContentPane().setLayout(layout);
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 5;
			c.insets = new Insets(5, 5, 5, 0);
			getContentPane().add(pEditors, c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(10, 5, 0, 5);
			getContentPane().add(bActivate, c);
			c.gridy = 1;
			c.insets = new Insets(5, 5, 0, 5);
			getContentPane().add(bCurrent, c);
			c.gridy = 2;
			c.insets = new Insets(5, 5, 0, 5);
			getContentPane().add(bSelect, c);
			c.gridy = 3;
			getContentPane().add(bCancel, c);	
			pack();
		}
		
		public int getOption() {
			return this.option;
		}
		
		public ArrayList<TextEditor> getTexts() {
			return selectedTexts;
		}

		@Override
		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == bActivate) {
				int index = lTexts.getSelectedIndex();
                multiTextComponent.activateTab(texts.get(index));
				setVisible(false);
                dispose();
			} else if (e.getSource() == bCurrent) {
                lTexts.setSelectedIndex(currentIndex);
			} else if (e.getSource() == bSelect) {
				popupMenu.show(bSelect, bSelect.getWidth() / 2, bSelect.getHeight() / 2);
			} if (e.getSource() == bCancel) {
				setVisible(false);
                dispose();
			}
		}
		
	}
	
	private static void establishBounds(JDialog dialog) {
        Dimension d1 = dialog.getSize();
        Dimension d2 = dialog.getToolkit().getScreenSize();
        int x = Math.max((d2.width-d1.width)/2, 0);
        int y = Math.max((d2.height-d1.height)/2, 0);
        dialog.setBounds(x + 0, y + 0, d1.width, d1.height);
	}
	
	public static void showEditorActivationDialog(MultiTextComponent multiTextComponent) {
		EditorActivationDialog dialog = new EditorActivator().new EditorActivationDialog(multiTextComponent);
		establishBounds(dialog);
		dialog.setVisible(true);
	}

}
