package neutrino.multitext.adapters.syntax;

import neutrino.multitext.BookmarkInformation;
import neutrino.multitext.BookmarkSupport;
import neutrino.multitext.TextEditorAdapter;
import neutrino.text.TextEditor;
import neutrino.text.components.syntax.*;
import org.fife.ui.rsyntaxtextarea.ErrorStrip;
import org.fife.ui.rtextarea.Gutter;
import org.fife.ui.rtextarea.GutterIconInfo;
import org.fife.ui.rtextarea.RTextScrollPane;
import javax.swing.*;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import java.awt.*;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;

import static javax.swing.JOptionPane.*;

/**
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class MakeEditorAdapter extends TextEditorAdapter implements BookmarkSupport {

    private MakeComponent textComponent = null;
    private TextEditor textEditor = null;
    private RTextScrollPane scrollPane;
    private ImageIcon bookmarkIcon = null;

    public MakeEditorAdapter() {
        super();
        textComponent = new MakeComponent();
        textComponent.setUndoable(true);
        textEditor = new TextEditor(textComponent);
        textEditor.setAcceptAllFileFilterUsed(true);
        scrollPane = new RTextScrollPane(textComponent, true);
        Gutter gutter = scrollPane.getGutter();
        gutter.setBookmarkingEnabled(true);
        URL url = getClass().getClassLoader().getResource("neutrino/assets/bookmark.png");
        bookmarkIcon = new ImageIcon(url);
        gutter.setBookmarkIcon(bookmarkIcon);
        setLayout(new BorderLayout());
        add(scrollPane, BorderLayout.CENTER);
        ErrorStrip errorStrip = new ErrorStrip(textComponent);
        add(errorStrip, BorderLayout.LINE_END);
        textEditor.setFileFilter(new FileNameExtensionFilter("Make Files (*.Makefile, *.makefile)", "Makefile", "makefile"));
    }

    /**
     * Sets whether the line numbers is visible in the text editor adapter
     * @param value - boolean
     */
    @Override
    public void setLineNumbersEnabled(boolean value) {
        scrollPane.setLineNumbersEnabled(value);
    }

    /**
     * Returns true when the line numbers is visible in the text editor adapter
     * @return boolean
     */
    @Override
    public boolean isLineNumbersEnabled() {
        return scrollPane.getLineNumbersEnabled();
    }

    /**
     * Sets whether the bookmark is enabled in the text editor adapter
     * @param value - boolean
     */
    @Override
    public void setBookmarksEnabled(boolean value) {
        scrollPane.setIconRowHeaderEnabled(value);
    }

    /**
     * Returns true when the bookmark is enabled in the text editor adapter
     * @return boolean
     */
    @Override
    public boolean isBookmarksEnabled() {
        return scrollPane.isIconRowHeaderEnabled();
    }

    /**
     * Toggles the bookmark at the caret position
     */
    @Override
    public void toggleBookmark() {
        Gutter gutter = scrollPane.getGutter();
        Element root = textComponent.getDocument().getDefaultRootElement();
        int line = root.getElementIndex(textComponent.getCaretPosition());
        try {
            gutter.toggleBookmark(line);
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when exists the bookmark before the caret position
     * @return boolean
     */
    @Override
    public boolean hasPreviousBookmark() {
        Gutter gutter = scrollPane.getGutter();
        int  caretPosition = textComponent.getCaretPosition();
        GutterIconInfo[] bookmarks = gutter.getBookmarks();
        for (GutterIconInfo bookmark : bookmarks) {
            if (bookmark.getMarkedOffset() < caretPosition) {
                return true;
            };
        }
        return false;
    }

    /**
     * Places the cursor at the previous bookmark to the cursor position
     */
    @Override
    public void previousBookmark() {
        Gutter gutter = scrollPane.getGutter();
        int  caretPosition = textComponent.getCaretPosition();
        GutterIconInfo[] bookmarks = gutter.getBookmarks();
        for (int i = bookmarks.length - 1; i >= 0; i--) {
            if (bookmarks[i].getMarkedOffset() < caretPosition) {
                textComponent.setCaretPosition(bookmarks[i].getMarkedOffset());
                break;
            };
        }
    }

    /**
     * Returns true when exists the bookmark next to the caret position
     * @return boolean
     */
    @Override
    public boolean hasNextBookmark() {
        Gutter gutter = scrollPane.getGutter();
        int  caretPosition = textComponent.getCaretPosition();
        GutterIconInfo[] bookmarks = gutter.getBookmarks();
        for (GutterIconInfo bookmark : bookmarks) {
            if (bookmark.getMarkedOffset() > caretPosition) {
                return true;
            };
        }
        return false;
    }

    /**
     * Places the cursor at the next bookmark to the cursor position
     */
    @Override
    public void nextBookmark() {
        Gutter gutter = scrollPane.getGutter();
        int  caretPosition = textComponent.getCaretPosition();
        GutterIconInfo[] bookmarks = gutter.getBookmarks();
        for (int i = 0; i < bookmarks.length; i++ ) {
            if (bookmarks[i].getMarkedOffset() > caretPosition) {
                textComponent.setCaretPosition(bookmarks[i].getMarkedOffset());
                break;
            };
        }
    }

    /**
     * Returns true when in the line where the cursor is placed the bookmark
     * @return boolean
     */
    @Override
    public boolean canRemoveBookmark() {
        Element root = textComponent.getDocument().getDefaultRootElement();
        int line = root.getElementIndex(textComponent.getCaretPosition());
        Element element = root.getElement(line);
        int start = element.getStartOffset();
        int end = element.getEndOffset();
        Gutter gutter = scrollPane.getGutter();
        GutterIconInfo[] bookmarks = gutter.getBookmarks();
        for (int i = 0; i < bookmarks.length; i++ ) {
            if (bookmarks[i].getMarkedOffset() >= start && bookmarks[i].getMarkedOffset() < end) {
                return true;
            };
        }
        return false;
    }

    /**
     * Removes the bookmark at the current line
     */
    @Override
    public void removeBookmark() {
        if (canRemoveBookmark()) toggleBookmark();
    }

    /**
     * Adds the named bookmark in the caret position. Asks fo a name in the dialog
     */
    @Override
    public void addNamedBookmark() {
        // get the name
        JPanel panel = new JPanel();
        BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
        panel.setLayout(layout);
        panel.add(new JLabel("Enter the name of bookmark"));
        panel.add(Box.createVerticalStrut(5));
        final JTextField textField = new JTextField(20);
        panel.add(textField);
        int option = showConfirmDialog(this, panel, "Bookmark name", YES_NO_OPTION, QUESTION_MESSAGE);
        if (option == NO_OPTION) return;
        // add the bookmark
        removeBookmark();
        Gutter gutter = scrollPane.getGutter();
        try {
            gutter.addOffsetTrackingIcon(textComponent.getCaretPosition(), bookmarkIcon, textField.getText());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when the document contains a bookmarks
     * @return boolean
     */
    @Override
    public boolean canRemoveAllBookmarks() {
        Gutter gutter = scrollPane.getGutter();
        if (gutter.getBookmarks() != null && gutter.getBookmarks().length > 0) return true;
        else return false;
    }

    /**
     * Removes all bookmarks
     */
    @Override
    public void removeAllBookmarks() {
        if (!canRemoveAllBookmarks()) return;
        Gutter gutter = scrollPane.getGutter();
        gutter.removeAllTrackingIcons();
    }

    /**
     * Returns the array of bookmarks
     * @return array list of BookmarkInformation
     */
    @Override
    public ArrayList<BookmarkInformation> getBookmarks() {
        ArrayList<BookmarkInformation> list = new ArrayList<BookmarkInformation>();
        Gutter gutter = scrollPane.getGutter();
        GutterIconInfo[] bookmarks = gutter.getBookmarks();
        for (GutterIconInfo bookmark : bookmarks) {
            BookmarkInformation information = new BookmarkInformation(bookmark.getToolTip(), bookmark.getMarkedOffset());
            list.add(information);
        }
        return list;
    }

    /**
     * Returns true when the editor has a bookmarks
     * @return boolean
     */
    @Override
    public boolean hasBookmarks() {
        return getBookmarks().size() > 0;
    }

    /**
     * Returns the array of named bookmarks
     * @return array list of BookmarkInformation
     */
    @Override
    public ArrayList<BookmarkInformation> getNamedBookmarks() {
        ArrayList<BookmarkInformation> list = getBookmarks();
        Iterator<BookmarkInformation> iterator = list.iterator();
        while (iterator.hasNext()) {
            BookmarkInformation information = iterator.next();
            if (information.getName() == null || information.getName().equals("")) iterator.remove();
        }
        return list;
    }

    /**
     * Returns true when the editor has a named bookmarks
     * @return boolean
     */
    @Override
    public boolean hasNamedBookmarks() {
        return getNamedBookmarks().size() > 0;
    }

    /**
     * Sets the bookmarks
     * @param bookmarks ArrayList of BookmarkInformation
     */
    @Override
    public void setBookmarks(ArrayList<BookmarkInformation> bookmarks) {
        if (bookmarks == null) return;
        try {
            Gutter gutter = scrollPane.getGutter();
            for (BookmarkInformation bookmark : bookmarks) {
                gutter.addOffsetTrackingIcon(bookmark.getPosition(), bookmarkIcon, bookmark.getName());
            }
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    @Override
    public TextEditor getTextEditor() {
        return textEditor;
    }

}
