package neutrino.text;

import javax.swing.*;

/**
 * By using for internal purposes instead of TextComponent class for redusing the complexity
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public interface ITextComponent {

    /**
     * Sets popup menu for text component
     * @param popupMenu - a popup menu
     */
    public void setPopupMenu(JPopupMenu popupMenu);

    /**
     * Returns true when all text may be selected
     * @return boolean
     */
    public boolean canSelectAll();

    /**
     * Returns true when text may be printed
     * @return boolean
     */
    public boolean canPrint();

    /**
     * Check is text component is empty.
     * @return true when text is empty.
     */
    public boolean isTextEmpty();

    /**
     * Returns true when all text is selected
     * @return boolean
     */
    public boolean isAllTextSelected();

    /**
     * Returns true if text is selected.
     */
    public boolean isTextSelected();

    /**
     * Return length of selection or 0 if text is not selected.
     */
    public int getSelectionLength();

    /**
     * Sets undoable for text component
     * @param value - boolean
     */
    public void setUndoable(boolean value);

    /**
     * Returns true when text component is undoable
     * @return boolean
     */
    public boolean isUndoable();

    /**
     * Check is text is changed.
     * @return true when text is changed.
     */
    public boolean isTextChanged();

    /**
     * Returns true when text changes may be reverted
     * @return boolean
     */
    public boolean canRevert();

    /**
     * Returns true if the text change may be undone
     */
    public boolean canUndo();

    /**
     * Returns true if the text change may be redone
     */
    public boolean canRedo();

    /**
     * Undoes the last text change if possible.
     */
    public void undo();

    /**
     * Redose the last text change if possible.
     */
    public void redo();

    /**
     * Returns true when compound edit is begun
     * @return boolean
     */
    public boolean isEditBegun();

    /**
     * Marks beginning of coalesce edit operation
     */
    public void beginEdit();

    /**
     * Marks end of coalesce edit operation
     */
    public void endEdit();

    /**
     * Reverts all text changes
     */
    public void revert();

    /**
     * Marks text as not changed
     */
    public void reset(boolean discardAllEdits);

    /**
     * Returns true if clipboard contain text.
     * @throws IllegalStateException
     * @return boolean value
     */
    public boolean isClipboardContainText() throws IllegalStateException;

    /**
     * Returns true when text fragment may be cut
     * @return boolean
     */
    public boolean canCut();

    /**
     * Returns true when text fragment may be copied
     * @return boolean
     */
    public boolean canCopy();

    /**
     * Returns true when text fragment may be pasted
     * @return boolean
     */
    public boolean canPaste();

    /**
     * Returns true when selection can be cleared
     * @return boolean
     */
    public boolean canClear();

    /**
     * Clears the selected text
     */
    public void clear();

    /**
     * Returns true when text can be deselected
     * @return boolean
     */
    public boolean canDeselect();

    /**
     * Deselects the text fragment
     */
    public void deselect();

    /**
     * Returns true when the editor in read only mode
     * @return - boolean
     */
    public boolean isReadOnlyMode();

    /**
     * Sets the read only mode
     * @param mode - boolean
     */
    public void setReadOnlyMode(boolean mode);

    /**
     * Toggles the reading and editing modes in the text editor
     */
    public void toggleReadOnlyMode();

    /**
     * Returns true when the read only mode can be set
     * @return - boolean
     */
    public boolean canToggleReadOnlyMode();

    /**
     * Adds the read mode listener to the list of of observes
     * @param listener - ReadModeListener
     */
    public void addReadModeListener(ReadModeListener listener);

    /**
     * Removes the listener from the list of observes
     * @param listener - ReadModeListener
     */
    public void removeReadModeListener(ReadModeListener listener);

}
