package neutrino.text;

import neutrino.dialogs.FileRenamer;
import neutrino.text.components.syntax.SyntaxTextComponent;
import static javax.swing.JOptionPane.*;

import java.awt.event.KeyEvent;
import java.io.*;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Iterator;
import javax.swing.*;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.text.*;

public class TextEditor {
	
	private JTextComponent textComponent = null;
	private File currentDirectory = null;
	private BackupManager backupManager = null;
	private File currentFile = null;
	private ArrayList<FileChangeListener> fileChangeListeners = null;
	private long lastModified = 0;
	private Charset currentEncoding = null;
	private ArrayList<FileFilter> choosableFileFilters = new ArrayList<FileFilter>(); 
	private FileFilter m_fileFilter = null;
	private boolean acceptAllFileFilterUsed = true;
    private boolean isEncodingChanged = false;
    private boolean isConfirmReloading = true;
    private boolean isConfirmRemoving = true;
    private boolean isConfirmOverwriting = true;

	public TextEditor(JTextComponent textComponent) {
		this.textComponent = textComponent;
		fileChangeListeners = new ArrayList<FileChangeListener>();
		currentEncoding = Charset.defaultCharset();
		backupManager = new BackupManager(this);
		textComponent.getDocument().addDocumentListener(textChangedDocumentListener);
		addFileChangeListener(fileChangeListener);
	}
	
	public JTextComponent getTextComponent() {
		return this.textComponent;
	}	
	
	/**
	 * Returns true if file is ready for opening. Otherwise shows message dialog and returns false.
	 * @param file - the file
	 * @return boolean 
	 */
	private boolean isFileReadyForOpening(File file) {
		if (file == null) {
			try {
				throw new NullPointerException("File parameter is not initialized");
			} catch (NullPointerException e) {
				e.printStackTrace();
			}
			return false;
		} else if (file.exists() == false) {
			showMessageDialog(textComponent.getParent(), "File is not exist. \nPlease verify the correct file name was given", "Error", ERROR_MESSAGE);
			return false;
		} else if (file.isFile() == false) {
			showMessageDialog(textComponent.getParent(), file.getAbsolutePath() + " is not a file", "Error", ERROR_MESSAGE);
			return false;
		} else if (file.canRead() == false) {
			showMessageDialog(textComponent.getParent(), "File is protected from reading", "Error", ERROR_MESSAGE);
			return false;
		}
		return true;
	}
	
	/**
	 * Load text from file into text component. Returns true when success.
	 * precondition - file is ready for opening
	 * @param file - loaded file.
	 * @param encoding - encoding for reading.
	 * @return boolean
	 */
	private boolean openFile(File file, Charset encoding) {
		FileInputStream stream = null;
		InputStreamReader reader = null;
		boolean isSuccess;
		try {
			stream = new FileInputStream(file);
			if (textComponent instanceof JEditorPane) {
				EditorKit editorKit = ((JEditorPane) textComponent).getEditorKit();
				Document document = editorKit.createDefaultDocument();
				editorKit.read(stream, document, 0);
				textComponent.setDocument(document);
			} else {
				reader = new InputStreamReader(stream, encoding);
				textComponent.read(reader, null);
			}
            if (textComponent instanceof SyntaxTextComponent) {
                textComponent.setDocument(textComponent.getDocument());
                textComponent.setCaretPosition(0);
            }
            isSuccess = true;
		} catch (FileNotFoundException e) {
			showMessageDialog(textComponent.getParent(), "File not found", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} catch (IOException e) {
			showMessageDialog(textComponent.getParent(), "Cannot read file", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} catch (BadLocationException e) {
			showMessageDialog(textComponent.getParent(), "Cannot read file", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
					showMessageDialog(textComponent.getParent(), "Cannot close file", "Error", ERROR_MESSAGE);
					e.printStackTrace();
					isSuccess = false;
				}
			}
		}
		return isSuccess;
	}

	/**
	 * Loads text file in text component using given encoding. Returns true when success
	 * @param file - file for loading
	 * @return boolean
	 */
	public boolean open(File file) {
		if (!isFileReadyForOpening(file)) {
			return false;
		}
		boolean isSuccess = openFile(file, currentEncoding);
		if (isSuccess) {
			setFile(file);
            isEncodingChanged = false;
			fireFileChanged(FileChangeEventType.OPENING);
		}
		return isSuccess;
	}
	
	/**
	 * Loads file selected by user using given encoding. Returns true when file is opened
	 * @return boolean
	 */
	public boolean open() {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setMultiSelectionEnabled(false);
		for (FileFilter fileFilter : choosableFileFilters) {
			fileChooser.addChoosableFileFilter(fileFilter);
		}
		if (m_fileFilter != null) fileChooser.setFileFilter(m_fileFilter);
		fileChooser.setAcceptAllFileFilterUsed(acceptAllFileFilterUsed);
		if (isFileLoaded()) 
			fileChooser.setCurrentDirectory(getFile().getParentFile());
		else 
			fileChooser.setCurrentDirectory(getCurrentDirectory());
		while (fileChooser.showOpenDialog(textComponent.getParent()) == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();
			if (!isFileReadyForOpening(file)) continue;
			if (!openFile(file, currentEncoding)) continue;
			setFile(file);
            isEncodingChanged = false;
			fireFileChanged(FileChangeEventType.OPENING);
			return true;
		}
		return false;
	}
	
	/**
	 * Returns true if file is ready for saving. Otherwise show message dialog and return false.
	 * @param file - the file
	 * @return boolean
	 */
	private boolean isFileReadyForSaving(File file) {
		if (file == null) {
			try {
				throw new NullPointerException("File parameter is not initialized");
			} catch (NullPointerException e) {
				e.printStackTrace();
			}
			return false;
		} else if (file.exists() == true && file.isFile() == false) {
			showMessageDialog(textComponent.getParent(), file.getAbsoluteFile() + " is not a file. \nPlease verify the correct file name was given", "Error", ERROR_MESSAGE);
			return false;
		} else if (file.exists() == true && file.canWrite() == false) {
			showMessageDialog(textComponent.getParent(), "File is protected from writing", "Error", ERROR_MESSAGE);
			return false;
		}
		return true;
	}
	
	/**
	 * Save text in file. Returns true when success.
	 * precondition - file is ready for saving
	 * @param file - file for saving.
	 * @param encoding - encoding for writing.
	 * @return boolean
	 */ 
	private boolean saveFile(File file, Charset encoding) {
		FileOutputStream stream = null;
		OutputStreamWriter writer = null;
		boolean isSuccess;
		try {
			stream = new FileOutputStream(file);
			if (textComponent instanceof JEditorPane) {
				writer = new OutputStreamWriter(stream);
				EditorKit editorKit = ((JEditorPane) textComponent).getEditorKit();
				Document document = textComponent.getDocument();
				editorKit.write(stream, document, 0, document.getLength());
			} else {
				writer = new OutputStreamWriter(stream, encoding);
				textComponent.write(writer);
			}
			isSuccess = true;
		} catch (IOException e) {
			showMessageDialog(textComponent.getParent(), "Cannot write file", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} catch (BadLocationException e) {
			showMessageDialog(textComponent.getParent(), "Cannot write file", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} finally {
			if (writer != null) {
				try {
					writer.close();
				} catch (IOException e) {
					showMessageDialog(textComponent.getParent(), "Cannot close file", "Error", ERROR_MESSAGE);
					e.printStackTrace();
					isSuccess = false;
				}
			}
		}
		return isSuccess;
	}

	/**
	 * Saves text in file using given encoding. Returns true when success
	 * @param file - file for saving
	 * @return boolean
	 */
	public boolean save(File file) {
		if (!isFileReadyForSaving(file)) {
			return false;
		}
		boolean isSuccess = saveFile(file, currentEncoding);
		if (isSuccess) {
			setFile(file);
			resetTextComponent();
            isEncodingChanged = false;
			fireFileChanged(FileChangeEventType.SAVING);
		}
		return isSuccess;
	}
	
	private void resetTextComponent() {
        if (textComponent instanceof ITextComponent) {
			((ITextComponent) textComponent).reset(false);
		}
	}
	
	/**
	 * Save text in selected file. Returns true when file is saved
	 * @return boolean
	 */
	public boolean saveAs() {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setMultiSelectionEnabled(false);
		for (FileFilter fileFilter : choosableFileFilters) {
			fileChooser.addChoosableFileFilter(fileFilter);
		}
		if (m_fileFilter != null) fileChooser.setFileFilter(m_fileFilter);
		fileChooser.setAcceptAllFileFilterUsed(acceptAllFileFilterUsed);
		fileChooser.setDialogTitle("Save As");
		if (isFileLoaded()) 
			fileChooser.setSelectedFile(getFile());
		else
			fileChooser.setCurrentDirectory(getCurrentDirectory());
		while (fileChooser.showSaveDialog(textComponent.getParent()) == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();
			if (!isFileReadyForSaving(file)) continue;
			if (file.exists() && file.isFile()) {
				if (showConfirmDialog(textComponent.getParent(), "Do want to replace file", "Saving confirmation", YES_NO_OPTION) == NO_OPTION) {
					continue;
				}
			}
			if (!saveFile(file, currentEncoding)) continue;
			setFile(file);
			resetTextComponent();
            isEncodingChanged = false;
			fireFileChanged(FileChangeEventType.SAVING);
			return true;
		}
		return false;
	}
	
	/**
	 * Saves current file or selected file. Returns true when file is saved
	 * @return boolean
	 */
	public boolean save() {
		if (isFileLoaded()) {
			return save(getFile());
		} else {
			return saveAs();
		}
	}
	
	/**
	 * Clear text component.
	 */
	public void clear() {
        if (textComponent instanceof SyntaxTextComponent) {
            try {
                textComponent.getDocument().remove(0, textComponent.getDocument().getLength());
                textComponent.setDocument(textComponent.getDocument());
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        } else if (textComponent instanceof JTextArea) {
			PlainDocument document = new PlainDocument();
			textComponent.setDocument(document);
		} else if (textComponent instanceof JEditorPane) {
			EditorKit editorKit = ((JEditorPane) textComponent).getEditorKit();
			Document document = editorKit.createDefaultDocument();
			textComponent.setDocument(document);
		} else {
			textComponent.setText("");
		}
		setFile(null);
		fireFileChanged(FileChangeEventType.CLEARING);
	}

    /**
     * Reload the text from file
     */
	public void reload() {
		if (!canReload()) return;
        if (isConfirmReloading()) {
            JPanel panel = new JPanel();
            BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
            panel.setLayout(layout);
            panel.add(new JLabel("Do you want to reload current file?"));
            panel.add(Box.createVerticalStrut(5));
            JCheckBox checkBox = new JCheckBox("Always reload without prompt");
            panel.add(checkBox);
            int option = showConfirmDialog(textComponent.getParent(), panel, "Confirm reloading", YES_NO_OPTION, QUESTION_MESSAGE);
            setConfirmationReloading(!checkBox.isSelected());
            if (option == NO_OPTION) return;
        }
        if (!isFileReadyForOpening(getFile())) return;
        boolean isSuccess = openFile(getFile(), currentEncoding);
        if (isSuccess) {
            isEncodingChanged = false;
            lastModified = getFile().lastModified();
            fireFileChanged(FileChangeEventType.RELOADING);
        }
	}

    /**
     * Copy file to destination file. Returns false when fail or true when success
     * @param file - file for coping
     * @param destination - destination file
     * @return true when success
     */
    private boolean copyFile(File file, File destination) {
        FileInputStream in = null;
        FileOutputStream out = null;
        try {
            in = new FileInputStream(file);
            out = new FileOutputStream(destination);
            final int SIZE = 4096;
            byte[] buffer = new byte[SIZE];
            int i = 0;
            while ((i = in.read(buffer)) >= 0) {
                out.write(buffer, 0, i);
            }
        } catch (NullPointerException e) {
            e.printStackTrace();
            return false;
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            return false;
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        } finally {
            if (in != null)
                try {
                    in.close();
                } catch (IOException e) {
                    e.printStackTrace();
                    return false;
                }
            if (out != null)
                try {
                    out.close();
                } catch (IOException e) {
                    e.printStackTrace();
                    return false;
                }
        }
        return true;
    }

    /**
     * Copy the file to another place in the file system
     */
    public void copy() {
        JFileChooser fileChooser = new JFileChooser();
        fileChooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
        fileChooser.setMultiSelectionEnabled(false);
        fileChooser.setDialogTitle("Copy");
        fileChooser.setApproveButtonMnemonic(KeyEvent.VK_C);
        fileChooser.setApproveButtonToolTipText("Copy File");
        for (FileFilter fileFilter : choosableFileFilters) {
            fileChooser.addChoosableFileFilter(fileFilter);
        }
        if (m_fileFilter != null) fileChooser.setFileFilter(m_fileFilter);
        fileChooser.setAcceptAllFileFilterUsed(acceptAllFileFilterUsed);
        if (isFileLoaded())
            fileChooser.setCurrentDirectory(getFile().getParentFile());
        else
            fileChooser.setCurrentDirectory(getCurrentDirectory());
        fileChooser.setSelectedFile(getFile());
        while (fileChooser.showDialog(textComponent.getParent(), "Copy") == JFileChooser.APPROVE_OPTION) {
            File file = fileChooser.getSelectedFile();
            if (file == null) continue;
            if (file.isDirectory()) {
                file = new File(file.getAbsolutePath() + "/" + getFile().getName());
            } else if (file.equals(getFile())) {
                showMessageDialog(textComponent.getParent(), "You can not copy a file to itself", "Error message", ERROR_MESSAGE);
                continue;
            } else if (file.exists()) {
                if (isConfirmOverwriting()) {
                    JPanel panel = new JPanel();
                    BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
                    panel.setLayout(layout);
                    panel.add(new JLabel("File already exists! Overwrite?"));
                    panel.add(Box.createVerticalStrut(5));
                    JCheckBox checkBox = new JCheckBox("Always overwrite without prompt");
                    panel.add(checkBox);
                    int option = showConfirmDialog(textComponent.getParent(), panel, "Confirm overwriting", YES_NO_OPTION, QUESTION_MESSAGE);
                    setConfirmationOverwriting(!checkBox.isSelected());
                    if (option == NO_OPTION) continue;
                }
            }
            if (!copyFile(getFile(), file)) {
                showMessageDialog(textComponent.getParent(), "Can not copy the file", "Error message", ERROR_MESSAGE);
                continue;
            }
            fireFileChanged(FileChangeEventType.COPING);
            return;
        }
    }

    /**
     * Move the file to another place in the file system
     */
    public void move() {
        JFileChooser fileChooser = new JFileChooser();
        fileChooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
        fileChooser.setMultiSelectionEnabled(false);
        fileChooser.setDialogTitle("Move");
        fileChooser.setApproveButtonMnemonic(KeyEvent.VK_M);
        fileChooser.setApproveButtonToolTipText("Move File");
        for (FileFilter fileFilter : choosableFileFilters) {
            fileChooser.addChoosableFileFilter(fileFilter);
        }
        if (m_fileFilter != null) fileChooser.setFileFilter(m_fileFilter);
        fileChooser.setAcceptAllFileFilterUsed(acceptAllFileFilterUsed);
        if (isFileLoaded())
            fileChooser.setCurrentDirectory(getFile().getParentFile());
        else
            fileChooser.setCurrentDirectory(getCurrentDirectory());
        fileChooser.setSelectedFile(getFile());
        while (fileChooser.showDialog(textComponent.getParent(), "Move") == JFileChooser.APPROVE_OPTION) {
            File file = fileChooser.getSelectedFile();
            if (file == null) continue;
            if (file.isDirectory()) {
                file = new File(file.getAbsolutePath() + "/" + getFile().getName());
            } else if (file.equals(getFile())) {
                showMessageDialog(textComponent.getParent(), "You can not move a file to itself", "Error message", ERROR_MESSAGE);
                continue;
            } else if (file.exists()) {
                if (isConfirmOverwriting()) {
                    JPanel panel = new JPanel();
                    BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
                    panel.setLayout(layout);
                    panel.add(new JLabel("File already exists! Overwrite?"));
                    panel.add(Box.createVerticalStrut(5));
                    JCheckBox checkBox = new JCheckBox("Always overwrite without prompt");
                    panel.add(checkBox);
                    int option = showConfirmDialog(textComponent.getParent(), panel, "Confirm overwriting", YES_NO_OPTION, QUESTION_MESSAGE);
                    setConfirmationOverwriting(!checkBox.isSelected());
                    if (option == NO_OPTION) continue;
                }
            }
            if (!copyFile(getFile(), file)) {
                showMessageDialog(textComponent.getParent(), "Can not move the file", "Error message", ERROR_MESSAGE);
                continue;
            }
            boolean isDeleted;
            String errorMessage = "Can not move the file";
            try {
                isDeleted = getFile().delete();
            } catch (SecurityException e) {
                isDeleted = false;
                errorMessage = e.getMessage();
            }
            if (!isDeleted) {
                showMessageDialog(textComponent.getParent(), errorMessage, "Error", ERROR_MESSAGE);
                continue;
            }
            setFile(file);
            fireFileChanged(FileChangeEventType.MOVING);
            return;
        }
    }

    /**
     * Remove the file from file system
     */
    public void remove() {
        if (!canRemove()) return;
        if (isConfirmRemoving()) {
            JPanel panel = new JPanel();
            BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
            panel.setLayout(layout);
            panel.add(new JLabel("Do you want to delete file " + getFile().getName() + " from file system?"));
            panel.add(Box.createVerticalStrut(5));
            JCheckBox checkBox = new JCheckBox("Always delete without prompt");
            panel.add(checkBox);
            int option = showConfirmDialog(textComponent.getParent(), panel, "Confirm deleting", YES_NO_OPTION, QUESTION_MESSAGE);
            setConfirmationRemoving(!checkBox.isSelected());
            if (option == NO_OPTION) return;
        }
        File file = getFile();
        boolean isDeleted;
        String errorMessage = "Can not delete the file";
        try {
            isDeleted = file.delete();
        } catch (SecurityException e) {
            isDeleted = false;
            errorMessage = e.getMessage();
        }
        if (!isDeleted) {
            showMessageDialog(textComponent.getParent(), errorMessage, "Error", ERROR_MESSAGE);
            return;
        }
        setFile(null);
        fireFileChanged(FileChangeEventType.REMOVING);
    }

    /**
     * Rename the file with new name entered from the dialog
     */
    public void rename() {
        if (!canRename()) return;
        File file = FileRenamer.showFileRenameDialog(getFile());
        if (file != null) {
            setFile(file);
            fireFileChanged(FileChangeEventType.RENAMING);
        }
    }

	/**
	 * Returns true when text may be cleared
	 * @return boolean
	 */
	public boolean canClear() {
		if (textComponent instanceof ITextComponent) {
			return isFileLoaded() || !((ITextComponent) textComponent).isTextEmpty() || ((ITextComponent) textComponent).isTextChanged();
		} else {
			return true;
		}
	}
	
	/**
	 * Returns true when text may be saved
	 * @return boolean
	 */
	public boolean canSave() {
		if (textComponent instanceof ITextComponent) {
			return ((ITextComponent) textComponent).isTextChanged() || !isFileLoaded() || isFileChanged() || isEncodingChanged();
		} else {
			return true;
		}
	}
	
	/**
	 * Returns true when text may be saved in another file
	 * @return boolean
	 */
	public boolean canSaveAs() {
		return isFileLoaded();
	}
	
	/**
	 * Returns true when file may be reloaded
	 * @return boolean
	 */
	public boolean canReload() {
		if (textComponent instanceof ITextComponent) {
			return isFileChanged() || (isFileLoaded() && ((ITextComponent) textComponent).isTextChanged()) || (isFileLoaded() && isEncodingChanged());
		} else {
			return true;
		}
	}
	
	/**
	 * Returns true when need to save file
	 * @return boolean
	 */
	public boolean conditionOfSaving() {
		if (textComponent instanceof ITextComponent) {
			return (isFileLoaded() && ((ITextComponent) textComponent).isTextChanged())
				|| (!isFileLoaded() && !((ITextComponent) textComponent).isTextEmpty())
				|| isFileChanged()
                || (isFileLoaded() && isEncodingChanged());
		} else {
			return true;
		}
	}

    /**
     * Returns true when the file can be removed
     * @return boolean
     */
    public boolean canRemove() {
        return isFileLoaded();
    }

    /**
     * Returns true when the file can be renamed
     * @return boolean
     */
    public boolean canRename() {
        return isFileLoaded();
    }

    /**
     * Returns true when the file can be copied
     * @return boolean
     */
    public boolean canCopy() {
        return isFileLoaded();
    }

    /**
     * Returns true when the file can be moved
     * @return boolean
     */
    public boolean canMove() {
        return isFileLoaded();
    }

	/**
	 * Returns true when text is saved. Causes saving when need
	 * @return boolean
	 */
	public boolean isTextSaved() {
		if (conditionOfSaving()) {
			int option;
			if (isFileChanged()) {
				option = showConfirmDialog(textComponent.getParent(), "The file has been changed on the file system.\nDo you want to overwrite the changes?", "Update conflict", YES_NO_CANCEL_OPTION);
			} else {
				option = showConfirmDialog(textComponent.getParent(), "Do you want to save file?", "Saving confirmation", YES_NO_CANCEL_OPTION); 
			}
			switch (option) {
			case YES_OPTION:
				if (!save()) return false;
				break;
			case NO_OPTION:
				break;
			case CANCEL_OPTION:
				return false;
			}
		}
		return true;		
	}
			
	/**
	 * Get current edited file
	 * @return file
	 */
	public File getFile() {
		return this.currentFile;
	}
	
	private void setFile(File file) {
		currentFile = file;
		if (currentFile == null) { 
			lastModified = 0;
		} else {
			lastModified = file.lastModified();
		}
	}
	
	/**
	 * Returns true when current file modified outside of the program
	 * @return boolean
	 */
	public boolean isFileChanged() {
		if (!isFileLoaded()) return false;
		return currentFile.lastModified() != lastModified; 
	}
	
	/**
	 * Check is file for editing is loaded
	 * @return boolean
	 */
	public boolean isFileLoaded() {
		return currentFile != null;
	}
	
	/**
	 * Return current encoding used for io operations.
	 * @return
	 */
	public Charset getEncoding() {
		return this.currentEncoding;
	}
	
	/**
	 * Set current encoding used for io operations.
	 * @param encoding
	 */
	public void setEncoding(Charset encoding) {
		if (encoding == null) return;
        if (currentEncoding.compareTo(encoding) != 0) {
            isEncodingChanged = true;
		    currentEncoding = encoding;
        }
	}

    /**
     * Returns true when encoding is changed since last io operation
     * @return boolean
     */
    public boolean isEncodingChanged() {
        return isEncodingChanged;
    }
	
	/**
	 * Returns the backup manager for saving text changes
	 * @return BackupManager
	 */
	public BackupManager getBackupManager() {
		return this.backupManager;
	}
	
	/**
	 * Restores document from backup
	 * @param backup - File
	 */
	public void restore(File backup) {
		if (backup == null) return;
		if (!backup.isFile()) {
			showMessageDialog(textComponent.getParent(), backup.getAbsolutePath() + " - is not a file", "Error massage", ERROR_MESSAGE);
			return;
		}
		String fileName = null;
		Object doc = null;
		boolean isBackupReaded = true;
		ObjectInputStream stream = null;
		try {
			stream = new ObjectInputStream(new FileInputStream(backup));
			try {
                stream.readLong();   // editor id
                stream.readObject(); // editor class name
				fileName = (String) stream.readObject();
                stream.readObject(); // date
				doc = stream.readObject();
			} catch (EOFException e) {
				isBackupReaded = false;
			} 
		} catch (FileNotFoundException e) {
			isBackupReaded = false;
			e.printStackTrace();
		} catch (IOException e) {
			isBackupReaded = false;
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			isBackupReaded = false;
			e.printStackTrace();
		} finally {
			if (stream != null) {
				try {
					stream.close();
				} catch (IOException e) {
					isBackupReaded = false;
					e.printStackTrace();
				}
			}
		}
		if (!isBackupReaded) {
			showMessageDialog(textComponent.getParent(), "Error of reading backup file", "Error massage", ERROR_MESSAGE);
			return;
		}
		if (fileName.equalsIgnoreCase(BackupManager.NONAME_FILE)) {
			setFile(null);
		} else {
			currentFile = new File(fileName);
			lastModified = 0;
		}
        if (doc instanceof Document) textComponent.setDocument((Document) doc);
		else textComponent.setText((String) doc);
		backup.delete();
		fireFileChanged(FileChangeEventType.RESTORING);
	}
	
	private FileChangeListener fileChangeListener = new FileChangeListener() {

		@Override
		public void fileChanged(FileChangeEvent event) {
			switch (event.getType()) {
			case OPENING:
			case CLEARING:
			case RESTORING:
            case RELOADING:
			textComponent.getDocument().addDocumentListener(textChangedDocumentListener);
			getBackupManager().persistBackup();
			}
		}
		
	};
	
	private DocumentListener textChangedDocumentListener = new DocumentListener() {

		public void changedUpdate(DocumentEvent e) {
			getBackupManager().persistBackup();
		}

		public void insertUpdate(DocumentEvent e) {
			getBackupManager().persistBackup();
		}

		public void removeUpdate(DocumentEvent e) {
			getBackupManager().persistBackup();
		}
		
	};
	
	/**
	 * Returns current directory for save and open operations. 
	 * @return directory
	 */
	public File getCurrentDirectory() {
		return this.currentDirectory;
	}
	
	/**
	 * Sets current directory for save and open operations.
	 * @param dir - current directory
	 */
	public void setCurrentDirectory(File dir) {
		this.currentDirectory = dir;
	}

    /**
     * Returns true when the read only attribute of the file can be toggled
     * @return - boolean
     */
    public boolean canToggleReadOnly() {
        File file = getFile();
        return file != null && file.exists() && file.isFile();
    }

    /**
     * Returns true when the file is read only
     * @return - boolean
     */
    public boolean isReadOnly() {
        File file = getFile();
        return file != null && file.exists() && file.isFile() && file.canRead();
    }

    /**
     * Toggles read only attribute of the file
     */
    public void toggleReadOnly() {
        if (!canToggleReadOnly()) {
            showMessageDialog(getTextComponent(), "Can not set read only attribute", "Error message", OK_OPTION);
            return;
        }
        File file = getFile();
        boolean canRead = file.canRead();
        file.setReadable(!canRead);
    }

    /**
     * Returns true when the write only attribute of the file can be toggled
     * @return - boolean
     */
    public boolean canToggleWriteOnly() {
        File file = getFile();
        return file != null && file.exists() && file.isFile();
    }

    /**
     * Returns true when the file is write only
     * @return - boolean
     */
    public boolean isWriteOnly() {
        File file = getFile();
        return file != null && file.exists() && file.isFile() && file.canWrite();
    }

    /**
     * Toggles write only attribute of the file
     */
    public void toggleWriteOnly() {
        if (!canToggleWriteOnly()) {
            showMessageDialog(getTextComponent(), "Can not set write only attribute", "Error message", OK_OPTION);
            return;
        }
        File file = getFile();
        boolean canWrite = file.canWrite();
        file.setWritable(!canWrite);
    }

    /**
     * Returns true when the hidden attribute of the file can be toggled
     * @return - boolean
     */
    public boolean canToggleHidden() {
        File file = getFile();
        return file != null && file.exists() && file.isFile();
    }

    /**
     * Returns true when the file is hidden
     * @return - boolean
     */
    public boolean isHidden() {
        File file = getFile();
        return file != null && file.exists() && file.isFile() && file.getName().charAt(0) == '.';
    }

    /**
     * Toggles hidden attribute of the file
     */
    public void toggleHidden() {
        if (!canToggleHidden()) {
            showMessageDialog(getTextComponent(), "Can not set hidden attribute", "Error message", OK_OPTION);
            return;
        }
        File file = getFile();
        File newFile = null;
        boolean isHidden = file.isHidden();
        boolean isChanged;
        try {
            String path = file.getAbsolutePath().substring(0, file.getAbsolutePath().length() - file.getName().length());
            if (isHidden) newFile = new File(path + file.getName().substring(1));
            else newFile = new File(path + "." + file.getName());
            isChanged = file.renameTo(newFile) ;
        } catch (IndexOutOfBoundsException e) {
            isChanged = false;
        }
        if (!isChanged) showMessageDialog(getTextComponent(), "Can not set hidden attribute", "Error message", OK_OPTION);
        else {
            setFile(newFile);
            fireFileChanged(FileChangeEventType.RENAMING);
        }
    }

    /**
     * Returns true when the execute attribute of the file can be toggled
     * @return - boolean
     */
    public boolean canToggleExecute() {
        File file = getFile();
        return file != null && file.exists() && file.isFile();
    }

    /**
     * Returns true when the file is execute
     * @return - boolean
     */
    public boolean isExecute() {
        File file = getFile();
        return file != null && file.exists() && file.isFile() && file.canExecute();
    }

    /**
     * Toggles execute attribute of the file
     */
    public void toggleExecute() {
        if (!canToggleExecute()) {
            showMessageDialog(getTextComponent(), "Can not set execute attribute", "Error message", OK_OPTION);
            return;
        }
        File file = getFile();
        boolean isExecute = file.canExecute();
        file.setExecutable(!isExecute);
    }

	/**
	 * Adds the specify file change listener to intercept file loading events.
	 * If paramater is null no action is perfomed. 
	 * @param listener - the file change listener.
	 */
	public void addFileChangeListener(FileChangeListener listener) {
		if (listener == null) return; 
		fileChangeListeners.add(listener);
	}
	
	/**
	 * Removes the specify file change listener.
	 * If paramater is null no action is perfomed. 
	 * @param listener - the file change listener.
	 */
	public void removeFileChangeListener(FileChangeListener listener) {
		if (listener == null) return; 
		fileChangeListeners.remove(listener);
	}
		
	private void fireFileChanged(FileChangeEventType type) {
		Iterator<FileChangeListener> iterator = fileChangeListeners.iterator();
		while (iterator.hasNext()) {
			FileChangeListener listener = iterator.next();
			FileChangeEvent event = new FileChangeEvent(this, type);
			listener.fileChanged(event);
		}
	} 
	
	public void addChoosableFileFilter(FileFilter fileFilter) {
		choosableFileFilters.add(fileFilter);
	}
	
	public boolean removeChoosableFileFilter(FileFilter fileFilter) {
		return choosableFileFilters.remove(fileFilter);
	}
	
	public void setFileFilter(FileFilter fileFilter) {
		this.m_fileFilter = fileFilter;
	}
	
	public void setAcceptAllFileFilterUsed(boolean value) {
		acceptAllFileFilterUsed = value;
	}
	
	public boolean isAcceptAllFileFilterUsed() {
		return acceptAllFileFilterUsed;
	}

    /**
     * Sets confirming until reloading
     * @param value - boolean
     */
    public void setConfirmationReloading(boolean value) {
        this.isConfirmReloading = value;
    }

    /**
     * Returns whether to confirm reloading
     * @return boolean
     */
    public boolean isConfirmReloading() {
        return this.isConfirmReloading;
    }

    /**
     * Sets confirming until removing
     * @param value - boolean
     */
    public void setConfirmationRemoving(boolean value) {
        this.isConfirmRemoving = value;
    }

    /**
     * Returns whether to confirm removing
     * @return boolean
     */
    public boolean isConfirmRemoving() {
        return this.isConfirmRemoving;
    }

    /**
     * Sets confirming until overwriting
     * @param value - boolean
     */
    public void setConfirmationOverwriting(boolean value) {
        this.isConfirmOverwriting = value;
    }

    /**
     * Returns whether to confirm overwriting
     * @return boolean
     */
    public boolean isConfirmOverwriting() {
        return this.isConfirmOverwriting;
    }
}
