package neutrino.text.components.plain;

import com.inet.jortho.SpellChecker;
import neutrino.dialogs.SpellingOptionsChooser;
import neutrino.text.IPlainTextComponent;
import neutrino.text.ITextComponent;
import neutrino.text.ReadModeEvent;
import neutrino.text.ReadModeListener;

import java.awt.*;
import java.awt.datatransfer.DataFlavor;
import java.awt.event.*;
import java.util.*;
import java.util.List;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.undo.*;

/**
 * Incapsulates text component.
 * Also supports undo and redo operations.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class PlainTextArea extends JTextArea implements ITextComponent, IPlainTextComponent {
	
	private UndoManager undoManager = null;
 	private Stack<CompoundEdit> edits = null;  	
	private int numberOfCurrentEdit = 0;	
	private boolean autoIndentMode = true;
	private JPopupMenu popupMenu = null;
	private boolean undoable = true;
    private ArrayList<ReadModeListener> readModeListeners = new ArrayList<ReadModeListener>();

	public PlainTextArea() {
		super();
		setTabSize(4);
		undoManager = new UndoManager();
		undoManager.setLimit(-1);
		edits = new Stack<CompoundEdit>();
		getDocument().addUndoableEditListener(undoableEditListener);
		addKeyListener(startLineKeyListener);
		final PlainTextArea instance = this;
		addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				if (popupMenu != null && e.getButton() == e.BUTTON3) {
					popupMenu.show(instance, e.getX(), e.getY());
				}
			}
		});
        SpellChecker.register(this, false, false, false, SpellingOptionsChooser.isAutoSpellingMode());
		createDefaultPopupMenu();
	}
	
	protected void createDefaultPopupMenu() {
		// create menu items
		final JMenuItem pmiUndo = new JMenuItem("Undo");
		final JMenuItem pmiRedo = new JMenuItem("Redo");
		final JMenuItem pmiCut = new JMenuItem("Cut");
		final JMenuItem pmiCopy = new JMenuItem("Copy");
		final JMenuItem pmiPaste = new JMenuItem("Paste");
        final JMenuItem pmiClear = new JMenuItem("Clear");
        final JMenu pmCheckSpelling = SpellChecker.createCheckerMenu();
        final JMenu pmLanguages = SpellChecker.createLanguagesMenu();
        final JMenuItem pmiSelectAll = new JMenuItem("Select all");
        final JMenuItem pmiDeselect = new JMenuItem("Deselect");
		// create popup menu
		popupMenu = new JPopupMenu();
		popupMenu.addPopupMenuListener(new PopupMenuListener() {
			@Override
			public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
				pmiUndo.setEnabled(canUndo());
				pmiRedo.setEnabled(canRedo());
				pmiCut.setEnabled(canCut());
				pmiCopy.setEnabled(canCopy());
				pmiPaste.setEnabled(canPaste());
                pmiClear.setEnabled(canClear());
				pmiSelectAll.setEnabled(canSelectAll());
                pmiDeselect.setEnabled(canDeselect());
			}
			@Override
			public void popupMenuWillBecomeInvisible(PopupMenuEvent e) { }
			@Override
			public void popupMenuCanceled(PopupMenuEvent e) { }
		});
		// build popup menu
		popupMenu.add(pmiUndo);
		popupMenu.add(pmiRedo);
		popupMenu.addSeparator();		
		popupMenu.add(pmiCut);
		popupMenu.add(pmiCopy);
		popupMenu.add(pmiPaste);
        popupMenu.add(pmiClear);
        popupMenu.addSeparator();
        popupMenu.add(pmCheckSpelling);
        popupMenu.add(pmLanguages);
		popupMenu.addSeparator();
		popupMenu.add(pmiSelectAll);
        popupMenu.add(pmiDeselect);
		// build mnemonics
		pmiUndo.setMnemonic(KeyEvent.VK_U);
		pmiRedo.setMnemonic(KeyEvent.VK_R);
		pmiCut.setMnemonic(KeyEvent.VK_C);
		pmiCopy.setMnemonic(KeyEvent.VK_O);
		pmiPaste.setMnemonic(KeyEvent.VK_P);
        pmiClear.setMnemonic(KeyEvent.VK_L);
        pmCheckSpelling.setMnemonic(KeyEvent.VK_H);
        pmLanguages.setMnemonic(KeyEvent.VK_N);
		pmiSelectAll.setMnemonic(KeyEvent.VK_S);
        pmiDeselect.setMnemonic(KeyEvent.VK_D);
		// build accelerators
		pmiUndo.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, InputEvent.CTRL_DOWN_MASK));
		pmiRedo.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Y, InputEvent.CTRL_DOWN_MASK));
		pmiCut.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, InputEvent.CTRL_DOWN_MASK));
		pmiCopy.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_DOWN_MASK));
		pmiPaste.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V, InputEvent.CTRL_DOWN_MASK));
        pmiClear.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0));
		pmiSelectAll.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, InputEvent.CTRL_DOWN_MASK));
        pmiDeselect.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0));
		// build actions
		ActionListener listener = new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (e.getSource() == pmiUndo) {
					undo();
				} else if (e.getSource() == pmiRedo) {
					redo();
				} else if (e.getSource() == pmiCut) {
					cut();
				} else if (e.getSource() == pmiCopy) {
					copy();
				} else if (e.getSource() == pmiPaste) {
					paste();
				} else if (e.getSource() == pmiClear) {
                    clear();
                } else if (e.getSource() == pmiSelectAll) {
					selectAll();
				} else if (e.getSource() == pmiDeselect) {
                    deselect();
                }
            }
		};
		pmiUndo.addActionListener(listener);
		pmiRedo.addActionListener(listener);
		pmiCut.addActionListener(listener);
		pmiCopy.addActionListener(listener);
		pmiPaste.addActionListener(listener);
        pmiClear.addActionListener(listener);
		pmiSelectAll.addActionListener(listener);
        pmiDeselect.addActionListener(listener);
	}
	
	/**
	 * Sets popup menu for text component
	 * @param popupMenu - a popup menu
	 */
	public void setPopupMenu(JPopupMenu popupMenu) {
		this.popupMenu = popupMenu;
	}
	
	public void setFont(Font font) {
		int tabSize = super.getTabSize();
		super.setFont(font);
		super.setTabSize(tabSize);
	}

    public void setForeground(Color foreground) {
        super.setForeground(foreground);
        super.setCaretColor(foreground);
        super.setSelectionColor(foreground);
    }

    public void setBackground(Color background) {
        super.setBackground(background);
        super.setSelectedTextColor(background);
    }
	
	/**
	 * Notices that text is not changed and read only mode disabled
     * when new document is set. Prepares undo manager for set document
	 */
	public void setDocument(Document doc) {
		int tabSize = super.getTabSize();
		super.setDocument(doc);
		super.setTabSize(tabSize);
		doc.addUndoableEditListener(undoableEditListener);
		reset(true);
        setReadOnlyMode(false);
		System.gc();
	}

	/**
	 * Returns true when all text may be selected
	 * @return boolean
	 */
	public boolean canSelectAll() {
		return !isTextEmpty() && !isAllTextSelected() && !isReadOnlyMode();
	}
	
	/**
	 * Returns true when text may be printed 
	 * @return boolean
	 */
	public boolean canPrint() {
		return !isTextEmpty();
	}
	
	/**
	 * Chack is text component is empty.
	 * @return true when text is empty.
	 */
	public boolean isTextEmpty() {
		return getDocument().getLength() == 0;
	}
	
	/**
	 * Returns true when all text is selected
	 * @return boolean
	 */
	public boolean isAllTextSelected() {
		if (!isTextSelected()) return false;
		else return (getSelectionStart() == 0) && (getSelectionLength() == getDocument().getLength());
	}
	
	/**
	 * Set type of line start. If true then line start as previous. 
	 * If false then line start from beginning. 
	 * @param flag - boolean value
	 */
	public void setAutoIndentMode(boolean flag) {
		this.autoIndentMode = flag;
	}

	/**
	 * Return true if line starts as previous line.
	 * @return boolean value
	 */
	public boolean isAutoIndentMode() {
		return this.autoIndentMode;
	}
	
	/**
	 * Returns true if text is selected.
	 */
	public boolean isTextSelected() {
		return getSelectionStart() != getSelectionEnd();
	}
	
	/**
	 * Return length of selection or 0 if text is not selected. 
	 */
	public int getSelectionLength() {
		return getSelectionEnd() - getSelectionStart();
	}

    /**
     * Returns true when the editor in read only mode
     * @return - boolean
     */
    public boolean isReadOnlyMode() {
        return !isEditable();
    }

    /**
     * Sets the read only mode
     * @param mode - boolean
     */
    public void setReadOnlyMode(boolean mode) {
        if (!canToggleReadOnlyMode()) return;
        setEditable(!mode);
        fireReadModeChanged();
    }

    /**
     * Toggles the reading and editing modes in the text editor
     */
    public void toggleReadOnlyMode() {
        if (!canToggleReadOnlyMode()) return;
        setEditable(!isEditable());
        fireReadModeChanged();
    }

    /**
     * Returns true when the read only mode can be set
     * @return - boolean
     */
    public boolean canToggleReadOnlyMode() {
        return !isTextEmpty();
    }

    /**
     * Fires the read mode listeners on change of read only mode
     */
    protected void fireReadModeChanged() {
        if (readModeListeners == null || readModeListeners.size() == 0) return;
        ReadModeEvent event = new ReadModeEvent(this);
        for (ReadModeListener listener : readModeListeners) {
            listener.readModeChanged(event);
        }
    }

    /**
     * Adds the read mode listener to the list of of observes
     * @param listener - ReadModeListener
     */
    public void addReadModeListener(ReadModeListener listener) {
        readModeListeners.add(listener);
    }

    /**
     * Removes the listener from the list of observes
     * @param listener - ReadModeListener
     */
    public void removeReadModeListener(ReadModeListener listener) {
        readModeListeners.remove(listener);
    }

    /**
     * Returns true when the current character or selection can be made to upper case
     * @return boolean
     */
    @Override
    public boolean canMakeUpperCase() {
        if (isReadOnlyMode()) return false;
        try {
            if (isTextSelected()) {
                String selectedText = getSelectedText();
                for (int i = 0; i < selectedText.length(); i++) {
                    if (Character.isLowerCase(selectedText.charAt(i))) return true;
                }
            }
            char currentCharacter = getDocument().getText(getCaretPosition() - 1, 1).charAt(0);
            return Character.isLowerCase(currentCharacter);
        } catch (BadLocationException e) {
            return false;
        } catch (IndexOutOfBoundsException e) {
            return false;
        } catch (NullPointerException e) {
            return false;
        }
    }
	
	/**
	 * Make selection or current character upper case
	 */
    @Override
	public void makeUpperCase() {
		if (!canMakeUpperCase()) return;
        if (isTextSelected()) {
            int startPosition = getSelectionStart();
            int length = getSelectionLength();
            String sourceText = getSelectedText();
            String upperCaseText = sourceText.toUpperCase();
            try {
                beginEdit();
                getDocument().remove(startPosition, length);
                getDocument().insertString(startPosition, upperCaseText, null);
                endEdit();
                select(startPosition, startPosition + length);
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        } else {
            try {
                int position = getCaretPosition() - 1;
                char currentCharacter = getDocument().getText(position, 1).charAt(0);
                char upperCaseCharacter = Character.toUpperCase(currentCharacter);
                beginEdit();
                getDocument().remove(position, 1);
                getDocument().insertString(position, new String(new char[] { upperCaseCharacter }), null);
                endEdit();
            } catch (BadLocationException e) {
                e.printStackTrace();
            } catch (IndexOutOfBoundsException e) {
                e.printStackTrace();
            } catch (NullPointerException e) {
                e.printStackTrace();
            }
        }
	}

    /**
     * Returns true when the current character or selection can be made to lower case
     * @return boolean
     */
    @Override
    public boolean canMakeLowerCase() {
        if (isReadOnlyMode()) return false;
        try {
            if (isTextSelected()) {
                String selectedText = getSelectedText();
                for (int i = 0; i < selectedText.length(); i++) {
                    if (Character.isUpperCase(selectedText.charAt(i))) return true;
                }
            }
            char currentCharacter = getDocument().getText(getCaretPosition() - 1, 1).charAt(0);
            return Character.isUpperCase(currentCharacter);
        } catch (BadLocationException e) {
            return false;
        } catch (IndexOutOfBoundsException e) {
            return false;
        } catch (NullPointerException e) {
            return false;
        }
    }

	/**
	 * Make selection or current character lower case
	 */
    @Override
	public void makeLowerCase() {
		if (!canMakeLowerCase()) return;
        if (isTextSelected()) {
            int startPosition = getSelectionStart();
            int length = getSelectionLength();
            String sourceText = getSelectedText();
            String lowercaseText = sourceText.toLowerCase();
            try {
                beginEdit();
                getDocument().remove(startPosition, length);
                getDocument().insertString(startPosition, lowercaseText, null);
                endEdit();
                select(startPosition, startPosition + length);
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        } else {
            try {
                int position = getCaretPosition() - 1;
                char currentCharacter = getDocument().getText(position, 1).charAt(0);
                char lowerCaseCharacter = Character.toLowerCase(currentCharacter);
                beginEdit();
                getDocument().remove(position, 1);
                getDocument().insertString(position, new String(new char[] { lowerCaseCharacter }), null);
                endEdit();
            } catch (BadLocationException e) {
                e.printStackTrace();
            } catch (IndexOutOfBoundsException e) {
                e.printStackTrace();
            } catch (NullPointerException e) {
                e.printStackTrace();
            }
        }
	}

    /**
     * Returns true when the current character or selection can be made to title case
     * @return boolean
     */
    @Override
    public boolean canMakeTitleCase() {
        if (isReadOnlyMode()) return false;
        try {
            if (isTextSelected()) return true;
            char currentCharacter = getDocument().getText(getCaretPosition() - 1, 1).charAt(0);
            if (Character.isSpaceChar(currentCharacter)) return false;
            return !Character.isTitleCase(currentCharacter);
        } catch (BadLocationException e) {
            return false;
        } catch (IndexOutOfBoundsException e) {
            return false;
        } catch (NullPointerException e) {
            return false;
        }
    }

    /**
     * Make selection or current character title case
     */
    @Override
    public void makeTitleCase() {
        if (!canMakeTitleCase()) return;
        if (isTextSelected()) {
            int startPosition = getSelectionStart();
            int length = getSelectionLength();
            String sourceText = getSelectedText();
            char[] array = sourceText.toCharArray();
            for (int i = 0; i < array.length; i++) {
                if (!Character.isTitleCase(array[i])) array[i] = Character.toTitleCase(array[i]);
            }
            String titleCaseText = new String(array);
            try {
                beginEdit();
                getDocument().remove(startPosition, length);
                getDocument().insertString(startPosition, titleCaseText, null);
                endEdit();
                select(startPosition, startPosition + length);
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        } else {
            try {
                int position = getCaretPosition() - 1;
                char currentCharacter = getDocument().getText(position, 1).charAt(0);
                char titleCaseCharacter = Character.toTitleCase(currentCharacter);
                beginEdit();
                getDocument().remove(position, 1);
                getDocument().insertString(position, new String(new char[] { titleCaseCharacter }), null);
                endEdit();
            } catch (BadLocationException e) {
                e.printStackTrace();
            } catch (IndexOutOfBoundsException e) {
                e.printStackTrace();
            } catch (NullPointerException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Returns the current line at the cursor position
     * @return String
     */
    public String getCurrentLine() {
        String line = "";
        try {
            int index = getDocument().getDefaultRootElement().getElementIndex(getCaretPosition());
            Element element = getDocument().getDefaultRootElement().getElement(index);
            line = getDocument().getText(element.getStartOffset(), element.getEndOffset() - element.getStartOffset() - 1);
        } catch (Exception e) { }
        return line;
    }

    /**
     * Returns the current word at the cursor position or null when the cursor placed at the white space
     * @return String
     */
    public String getCurrentWord() {
        String word = null;
        try {
            if (getCaretPosition() == 0) return null;
            int index = getDocument().getDefaultRootElement().getElementIndex(getCaretPosition() - 1);
            Element element = getDocument().getDefaultRootElement().getElement(index);
            String line = getDocument().getText(element.getStartOffset(), element.getEndOffset() - element.getStartOffset());
            int position = getCaretPosition() - 1 - element.getStartOffset();
            int beginIndex = position;
            while (beginIndex > 0) {
                if (Character.isWhitespace(line.charAt(beginIndex))) {
                    beginIndex++;
                    break;
                } else
                    beginIndex--;
            }
            int endIndex = position;
            while (endIndex < line.length()) {
                if (Character.isWhitespace(line.charAt(endIndex))) {
                    break;
                } else
                    endIndex++;
            }
            if (endIndex > beginIndex)
                word = line.substring(beginIndex, endIndex);
        } catch (Exception e) { }
        return word;
    }

    /**
     * Returns true when the selected text or current word can be capitalized
     * @return boolean
     */
    public boolean canCapitalize() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) {
            String selectedText = getSelectedText();
            for (int index = 0; index < selectedText.length(); index++) {
                if (index == 0) {
                    try {
                        if (getSelectionStart() == 0) return Character.isLowerCase(selectedText.charAt(index));
                        char previousCharacter = getDocument().getText(getSelectionStart() - 1, 1).charAt(0);
                        if (Character.isLowerCase(selectedText.charAt(index))
                                && Character.isWhitespace(previousCharacter))
                            return true;
                    } catch (BadLocationException e) {
                        // do nothing
                    } catch (IndexOutOfBoundsException e) {
                        // do nothing
                    }
                } else if (Character.isWhitespace(selectedText.charAt(index - 1))
                        && Character.isLowerCase(selectedText.charAt(index)))
                    return true;
            }
            return false;
        } else {
            String currentWord = getCurrentWord();
            if (currentWord == null) return false;
            return Character.isLowerCase(currentWord.charAt(0));
        }
    }

    /**
     * Capitalizes the selected text or current word
     */
    public void capitalize() {
        if (!canCapitalize()) return;
        if (isTextSelected()) {
            try {
                int beginSelection = getSelectionStart();
                int endSelection = getSelectionEnd();
                beginEdit();
                for (int index = beginSelection; index <= endSelection; index++) {
                    char previousCharacter = (index == 0) ? ' ' : getDocument().getText(index - 1, 1).charAt(0);
                    char currentCharacter = getDocument().getText(index, 1).charAt(0);
                    if (Character.isWhitespace(previousCharacter) && Character.isLowerCase(currentCharacter)) {
                        char character = Character.toUpperCase(currentCharacter);
                        getDocument().remove(index, 1);
                        getDocument().insertString(index, new String(new char[] { character }), null);
                    }
                }
                endEdit();
                select(beginSelection, endSelection);
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else {
            try {
                int position = getCaretPosition() - 1;
                int elementIndex = getDocument().getDefaultRootElement().getElementIndex(position);
                Element element = getDocument().getDefaultRootElement().getElement(elementIndex);
                int beginCurrentLine =  element.getStartOffset();
                for (int index = position; index >= beginCurrentLine; index--) {
                    char previousCharacter = (index == 0) ? ' ' : getDocument().getText(index - 1, 1).charAt(0);
                    char currentCharacter = getDocument().getText(index, 1).charAt(0);
                    if (Character.isWhitespace(previousCharacter) && Character.isLowerCase(currentCharacter)) {
                        char character = Character.toUpperCase(currentCharacter);
                        beginEdit();
                        getDocument().remove(index, 1);
                        getDocument().insertString(index, new String(new char[] { character }), null);
                        endEdit();
                        break;
                    }
                }
            } catch (Exception e) { }
        }
    }

    /**
     * Returns whether the capitalization of current word or selection can be inverted
     * @return boolean
     */
    public boolean canInvertCapitalization() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) {
            String selectedText = getSelectedText();
            for (int index = 0; index < selectedText.length(); index++) {
                if (index == 0) {
                    try {
                        if (getSelectionStart() == 0) return Character.isLetter(selectedText.charAt(index));
                        char previousCharacter = getDocument().getText(getSelectionStart() - 1, 1).charAt(0);
                        if (Character.isLetter(selectedText.charAt(index))
                                && Character.isWhitespace(previousCharacter))
                            return true;
                    } catch (BadLocationException e) {
                        // do nothing
                    } catch (IndexOutOfBoundsException e) {
                        // do nothing
                    }
                } else if (Character.isWhitespace(selectedText.charAt(index - 1))
                        && Character.isLetter(selectedText.charAt(index)))
                    return true;
            }
            return false;
        } else {
            String currentWord = getCurrentWord();
            if (currentWord == null) return false;
            return Character.isLetter(currentWord.charAt(0));
        }
    }

    /**
     * Inverts the capitalization of current word or selected text
     */
    public void invertCapitalization() {
        if (!canInvertCapitalization()) return;
        if (isTextSelected()) {
            try {
                int beginSelection = getSelectionStart();
                int endSelection = getSelectionEnd();
                beginEdit();
                for (int index = beginSelection; index <= endSelection; index++) {
                    char previousCharacter = (index == 0) ? ' ' : getDocument().getText(index - 1, 1).charAt(0);
                    char currentCharacter = getDocument().getText(index, 1).charAt(0);
                    if (Character.isWhitespace(previousCharacter) && Character.isLetter(currentCharacter)) {
                        char character = (Character.isLowerCase(currentCharacter)) ? Character.toUpperCase(currentCharacter) : Character.toLowerCase(currentCharacter);
                        getDocument().remove(index, 1);
                        getDocument().insertString(index, new String(new char[] { character }), null);
                    }
                }
                endEdit();
                select(beginSelection, endSelection);
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else {
            try {
                int position = getCaretPosition() - 1;
                int elementIndex = getDocument().getDefaultRootElement().getElementIndex(position);
                Element element = getDocument().getDefaultRootElement().getElement(elementIndex);
                int beginCurrentLine =  element.getStartOffset();
                for (int index = position; index >= beginCurrentLine; index--) {
                    char previousCharacter = (index == 0) ? ' ' : getDocument().getText(index - 1, 1).charAt(0);
                    char currentCharacter = getDocument().getText(index, 1).charAt(0);
                    if (Character.isWhitespace(previousCharacter) && Character.isLetter(currentCharacter)) {
                        char character = (Character.isLowerCase(currentCharacter)) ? Character.toUpperCase(currentCharacter) : Character.toLowerCase(currentCharacter);
                        beginEdit();
                        getDocument().remove(index, 1);
                        getDocument().insertString(index, new String(new char[] { character }), null);
                        endEdit();
                        break;
                    }
                }
            } catch (Exception e) { }
        }
    }

    /**
     * Returns true when operation of inverting the case of current character or selection can be done
     * @return boolean
     */
    public boolean canInvertCase() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) {
            String selectedText = getSelectedText();
            for (int i = 0; i < selectedText.length(); i++) {
                if (Character.isLetter(selectedText.charAt(i))) return true;
            }
        } else {
            int position = getCaretPosition() - 1;
            if (position < 0) return false;
            try {
                char currentCharacter = getDocument().getText(position, 1).charAt(0);
                if (Character.isLetter(currentCharacter)) return true;
            } catch (BadLocationException e) {
                e.printStackTrace();
            } catch (IndexOutOfBoundsException e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    /**
     * Invert the case of current letter or selected text
     */
    public void invertCase() {
        if (!canInvertCase()) return;
        if (isTextSelected()) {
            int selectionStart = getSelectionStart();
            int selectionEnd = getSelectionEnd();
            beginEdit();
            try {
                for (int index = selectionStart; index < selectionEnd; index++) {
                    char character = getDocument().getText(index, 1).charAt(0);
                    if (Character.isLetter(character)) {
                        if (Character.isUpperCase(character)) character = Character.toLowerCase(character);
                        else if (Character.isLowerCase(character)) character = Character.toUpperCase(character);
                        getDocument().remove(index, 1);
                        getDocument().insertString(index, new String(new char[] { character }), null);
                    }
                }
            } catch (BadLocationException e) {
                e.printStackTrace();
            } catch (IndexOutOfBoundsException e) {
                e.printStackTrace();
            }
            endEdit();
            select(selectionStart, selectionEnd);
        } else {
            int position = getCaretPosition() - 1;
            if (position < 0) return;
            try {
                char character = getDocument().getText(position, 1).charAt(0);
                if (Character.isLetter(character)) {
                    if (Character.isUpperCase(character)) character = Character.toLowerCase(character);
                    else if (Character.isLowerCase(character)) character = Character.toUpperCase(character);
                    beginEdit();
                    getDocument().remove(position, 1);
                    getDocument().insertString(position, new String(new char[] { character }), null);
                    endEdit();
                }
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        }
    }

	/**
	 * Shift text on one symbol in left.
	 * @param text - text to be shifted
	 * @return shifted text
	 */
	private String shiftTextInLeft(String text) {
		if (text == null) return "";
		StringBuffer buffer = new StringBuffer();
		StringTokenizer tokenizer = new StringTokenizer(text, "\n", true);
		boolean isPreviousTokenNotEmpty = false;
		while (tokenizer.hasMoreTokens()) {
			String row = tokenizer.nextToken();
			if (row.equals("\n")) {
				if (isPreviousTokenNotEmpty) {
					isPreviousTokenNotEmpty = false;
				} else {
					buffer.append('\n');
				}
				continue;
			}
			if (row.length() > 0 && Character.isSpace(row.charAt(0))) {
				buffer.append(row.substring(1));
			} else {
				buffer.append(row);
			}
			isPreviousTokenNotEmpty = true;
			if (tokenizer.hasMoreTokens()) { 
				buffer.append('\n');
			}
		}
		return buffer.toString();
	}
	
	/**
	 * Shift text on one symbol in right.
	 * @param text - text to be shifted
	 * @return shifted text
	 */
	private String shiftTextInRight(String text) {
		if (text == null) return "";
		StringBuffer buffer = new StringBuffer();
		StringTokenizer tokenizer = new StringTokenizer(text, "\n", true);
		boolean isPreviousTokenNotEmpty = false;
		while (tokenizer.hasMoreTokens()) {
			String row = tokenizer.nextToken();
			if (row.equals("\n")) {
				if (isPreviousTokenNotEmpty) {
					isPreviousTokenNotEmpty = false;
				} else {
					buffer.append('\n');
				}
				continue;
			}
			buffer.append(' ');
			buffer.append(row);
			isPreviousTokenNotEmpty = true;
			if (tokenizer.hasMoreTokens()) { 
				buffer.append('\n');
			}
		}
		return buffer.toString();
	}

    /**
     * Returns true when the selection or current line can be shifted in left
     * @return - boolean
     */
    public boolean canShiftInLeft() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) return true;
        else {
            String currentLine = getCurrentLine();
            return currentLine.length() > 0 && Character.isSpaceChar(currentLine.charAt(0));
        }
    }

	/**
	 * Shift selected text in left on one character.
	 * If text is not selected shift current line in left.
	 */
	public void shiftInLeft() {
		if (isTextSelected()) {
			String shiftedText = shiftTextInLeft(getSelectedText());
			int startPosition = getSelectionStart();
			int length = getSelectionLength();
			try {
				beginEdit();
				getDocument().remove(startPosition, length);
				getDocument().insertString(startPosition, shiftedText, null);
				endEdit();
				select(startPosition, startPosition + shiftedText.length());
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		} else if (!isTextEmpty()) {
			try {
				// get current caret position
				char c;
				int position = getCaretPosition();
				// find first character position of current line
				c = getDocument().getText(position, 1).charAt(0);
				if (c == '\n') position--;
				if (position < 0) return;
				for (; position > 0; position--) {
					c = getDocument().getText(position, 1).charAt(0);
					if (c == '\n') {
						position++;
						break;
					}
				}
				// remove first space
				c = getDocument().getText(position, 1).charAt(0);
				if (c == ' ' || c == '\t') {
					getDocument().remove(position, 1);
				}
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		}
	}

    /**
     * Returns true when the selection or current line can be shifted in right
     * @return - boolean
     */
    public boolean canShiftInRight() {
        if (isReadOnlyMode()) return false;
        return true;
    }

	/**
	 * Shift selected text in right on one character.
	 * If text is not selected shift current line in right.
	 */
	public void shiftInRight() {
		if (isTextSelected()) {
			String shiftedText = shiftTextInRight(getSelectedText());
			int startPosition = getSelectionStart();
			int length = getSelectionLength();
			try {
				beginEdit();
				getDocument().remove(startPosition, length);
				getDocument().insertString(startPosition, shiftedText, null);
				endEdit();
				select(startPosition, startPosition + shiftedText.length());
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		} else if (!isTextEmpty()) {
			try {
				// get current caret position
				char c;
				int position = getCaretPosition();
				// find first character position of current line
				c = getDocument().getText(position, 1).charAt(0);
				if (c == '\n') position--;
				if (position < 0) return;
				for (; position > 0; position--) {
					c = getDocument().getText(position, 1).charAt(0);
					if (c == '\n') {
						position++;
						break;
					}
				}
				// add space
				getDocument().insertString(position, " ", null);
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		}
	}
	
	/**
	 * Replaces groups of spaces with tabs in string. Returns result of replacement
	 * @param source - string for replacement
	 * @return String
	 */
	private String tabifyString(String source) {
		int tabSize = getTabSize();
		StringBuffer standard = new StringBuffer();
		for (int i = 0; i < tabSize; i++) {
			standard.append(' ');
		}
		return source.replaceAll(standard.toString(), "\t");
	}

    /**
     * Returns true when the selected text can be tabified
     * @return - boolean
     */
    public boolean canTabifySelectedLines() {
        if (isReadOnlyMode()) return false;
        return isTextSelected();
    }
	
	/*
	 * Replaces groups of spaces with tabs in selection. 
	 * If text is not selected do nothing.
	 */
	public void tabifySelectedLines() {
        if (!canTabifySelectedLines()) return;
        int selectionStart = getSelectionStart();
        int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
        selectionStart = startElement.getStartOffset();
        int selectionEnd = getSelectionEnd();
        int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
        Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
        selectionEnd = endElement.getEndOffset();
        if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            String selectedText = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            String result = tabifyString(selectedText);
            beginEdit();
            getDocument().remove(selectionStart, selectedText.length());
            getDocument().insertString(selectionStart, result, null);
            endEdit();
            select(selectionStart, selectionStart + result.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
	}
	
	/**
	 * Replaces tabs with groups of spaces in string. 
	 * Returns result of replacement
	 * @param source - string for replacement
	 * @return String
	 */
	private String untabifyString(String source) {
		int tabSize = getTabSize();
		StringBuffer standard = new StringBuffer();
		for (int i = 0; i < tabSize; i++) {
			standard.append(' ');
		}
		return source.replace("\t", standard.toString());
	}

    /**
     * Returns true when the selected text can be untabified
     * @return - boolean
     */
    public boolean canUntabifySelectedLines() {
        if (isReadOnlyMode()) return false;
        return isTextSelected();
    }

	/**
	 * Replaces tabs with groups of spaces in selection. 
	 * If text is not selected do nothing 
	 */
	public void untabifySelectedLines() {
        if (!canUntabifySelectedLines()) return;
        int selectionStart = getSelectionStart();
        int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
        selectionStart = startElement.getStartOffset();
        int selectionEnd = getSelectionEnd();
        int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
        Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
        selectionEnd = endElement.getEndOffset();
        if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            String selectedText = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            String result = untabifyString(selectedText);
            beginEdit();
            getDocument().remove(selectionStart, selectedText.length());
            getDocument().insertString(selectionStart, result, null);
            endEdit();
            select(selectionStart, selectionStart + result.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
	}
	
	/**
	 * Returns result of alignment source in left
	 * @param source - string to alignment
	 * @return String
	 */
	private String alignStringInLeft(String source) {
		StringTokenizer tokenizer = new StringTokenizer(source, "\n", true);
		StringBuffer result = new StringBuffer();
		while (tokenizer.hasMoreTokens()) {
			String token = tokenizer.nextToken();
			int startPosition = 0;
			while (startPosition < token.length() 
					&& (token.charAt(startPosition) == ' ' || token.charAt(startPosition) == '\t')) {
				startPosition++;
			}
			result.append(token.substring(startPosition));
		}
		return result.toString();
	}

    /**
     * Returns true when the horizontal white space can be deleted from the selection or current line
     * @return - boolean
     */
    public boolean canDeleteHorizontalWhiteSpace() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) return true;
        else {
            String currentLine = getCurrentLine();
            return currentLine.length() > 0 && Character.isSpaceChar(currentLine.charAt(0));
        }
    }
	
	/**
	 * Removes start spaces from current line or from selection.
	 */
	public void deleteHorizontalWhiteSpace() {
		String source, result;
		int startPosition;
		if (isTextSelected()) {
            int selectionStart = getSelectionStart();
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
            Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
            selectionStart = startElement.getStartOffset();
            int selectionEnd = getSelectionEnd();
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
            Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
            selectionEnd = endElement.getEndOffset();
            if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
			try {
                source = getDocument().getText(selectionStart, selectionEnd - selectionStart);
                result = alignStringInLeft(source);
                beginEdit();
				getDocument().remove(selectionStart, source.length());
				getDocument().insertString(selectionStart, result, null);
                endEdit();
                select(selectionStart, selectionStart + result.length());
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
		} else if (!isTextEmpty()) {
			try {
				Element rootElement = getDocument().getDefaultRootElement();
				Element currentLine = rootElement.getElement(rootElement.getElementIndex(getCaretPosition()));
				startPosition = currentLine.getStartOffset();
				source = getText(startPosition, currentLine.getEndOffset() - startPosition);
				result = alignStringInLeft(source);
				beginEdit();
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
				endEdit();
				setCaretPosition(startPosition);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
		}
	}
	
	/**
	 * Removes trailing whitespaces from source and returns result
	 * @param source - String
	 * @return String
	 */
	private String removeTrailingWhitespacesFromString(String source) {
		StringTokenizer tokenizer = new StringTokenizer(source, "\n", true);
		StringBuffer result = new StringBuffer();
		while (tokenizer.hasMoreTokens()) {
			String token = tokenizer.nextToken();
			int endPosition = token.length() - 1;
			while (endPosition >= 0 
					&& (token.charAt(endPosition) == ' ' || token.charAt(endPosition) == '\t')) {
				endPosition--;
			}
			result.append(token.substring(0, endPosition + 1));
		}
		return result.toString();
	}

    /**
     * Returns true when the trailing white space can be deleted from the selection or current line
     * @return - boolean
     */
    public boolean canRemoveTrailingWhitespaces() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) return true;
        else {
            String currentLine = getCurrentLine();
            return currentLine.length() > 0 && Character.isSpaceChar(currentLine.charAt(currentLine.length() - 1));
        }
    }
	
	/**
	 * Removes trailing whitespaces from current line or selected text
	 */
	public void removeTrailingWhitespaces() {
		String source, result;
		int startPosition;
		if (isTextSelected()) {
            int selectionStart = getSelectionStart();
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
            Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
            selectionStart = startElement.getStartOffset();
            int selectionEnd = getSelectionEnd();
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
            Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
            selectionEnd = endElement.getEndOffset();
            if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
			try {
                source = getDocument().getText(selectionStart, selectionEnd - selectionStart);
                result = removeTrailingWhitespacesFromString(source);
                beginEdit();
				getDocument().remove(selectionStart, source.length());
				getDocument().insertString(selectionStart, result, null);
                endEdit();
                select(selectionStart, selectionStart + result.length());
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
		} else if (!isTextEmpty()) {
			try {
				Element rootElement = getDocument().getDefaultRootElement();
				Element currentLine = rootElement.getElement(rootElement.getElementIndex(getCaretPosition()));
				startPosition = currentLine.getStartOffset();
				source = getText(startPosition, currentLine.getEndOffset() - startPosition);
				result = removeTrailingWhitespacesFromString(source);
				beginEdit();
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
				endEdit();
				if (getCaretPosition() >= startPosition + result.length()) {
					setCaretPosition( startPosition + result.length() - 1);
				}
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
		}
	}

    /**
     * Returns true when the given string contains no characters
     * @param line - string
     * @return boolean
     */
    private boolean isLineEmpty(String line) {
        if (line == null) return true;
        if (line.length() == 0) return true;
        for (int i = 0; i < line.length(); i++) {
            if (!Character.isWhitespace(line.charAt(i))) return false;
        }
        return true;
    }

    /**
     * Returns true when the current line is empty or the selected text contains empty lines
     * @return boolean
     */
    public boolean canRemoveEmptyLines() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) {
            String text = getSelectedText();
            if (text.indexOf("\n\n") != -1) return true;
            StringTokenizer tokenizer  = new StringTokenizer(text, "\n");
            while (tokenizer.hasMoreTokens()) {
                String token = tokenizer.nextToken();
                if (isLineEmpty(token)) return true;
            }
            return false;
        } else {
            if (isTextEmpty()) return false;
            String line = getCurrentLine();
            return isLineEmpty(line);
        }
    }

    /**
     * Removes the empty lines in selection or current line when is empty
     */
    public void removeEmptyLines() {
        if (!canRemoveEmptyLines()) return;
        if (isTextSelected()) {
            int selectionStart = getSelectionStart();
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
            Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
            selectionStart = startElement.getStartOffset();
            int selectionEnd = getSelectionEnd();
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
            Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
            selectionEnd = endElement.getEndOffset();
            if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
            try {
                String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
                StringTokenizer tokenizer  = new StringTokenizer(text, "\n");
                StringBuffer buffer = new StringBuffer();
                while (tokenizer.hasMoreTokens()) {
                    String token = tokenizer.nextToken();
                    if (!isLineEmpty(token)) {
                        buffer.append(token);
                        if (tokenizer.hasMoreTokens()) buffer.append("\n");
                    }
                }
                beginEdit();
                getDocument().remove(selectionStart, selectionEnd - selectionStart);
                getDocument().insertString(selectionStart, buffer.toString(), null);
                endEdit();
                select(selectionStart, buffer.toString().length());
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        } else {
            if (isTextEmpty()) return;
            int index = getDocument().getDefaultRootElement().getElementIndex(getCaretPosition());
            Element element = getDocument().getDefaultRootElement().getElement(index);
            try {
                int length = element.getEndOffset() - element.getStartOffset() - 1;
                if (index == 0) getDocument().remove(element.getStartOffset(), length);
                else getDocument().remove(element.getStartOffset() - 1, length + 1);
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Returns true when can duplicate the current line or selection
     * @return - boolean
     */
    public boolean canDuplicateSelectedLines() {
        if (isReadOnlyMode()) return false;
        return !isTextEmpty();
    }

    /**
     * Duplicates the current line or selection
     */
    public void duplicateSelectedLines() {
        if (!canDuplicateSelectedLines()) return;
        if (isTextSelected()) {
            int selectionStart = getSelectionStart();
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
            Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
            selectionStart = startElement.getStartOffset();
            int selectionEnd = getSelectionEnd();
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
            Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
            selectionEnd = endElement.getEndOffset();
            if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
            try {
                String selectedText = getDocument().getText(selectionStart, selectionEnd - selectionStart);
                boolean hasNoNewLineCharacter = selectedText.charAt(selectedText.length() - 1) != '\n';
                beginEdit();
                if (hasNoNewLineCharacter) {
                    getDocument().insertString(selectionEnd, "\n", null);
                    selectionEnd++;
                }
                getDocument().insertString(selectionEnd, selectedText, null);
                if (hasNoNewLineCharacter) {
                    getDocument().insertString(selectionEnd + selectedText.length(), "\n", null);
                }
                endEdit();
                int selectionLength = selectedText.length();
                if (hasNoNewLineCharacter) selectionLength++;
                select(selectionEnd, selectionEnd + selectionLength);
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        } else {
            String currentLine = getCurrentLine();
            int index = getDocument().getDefaultRootElement().getElementIndex(getCaretPosition());
            Element element = getDocument().getDefaultRootElement().getElement(index);
            int lineEnd = element.getEndOffset();
            try {
                boolean isLastLine = getDocument().getDefaultRootElement().getElementIndex(index) == getDocument().getDefaultRootElement().getElementCount();
                beginEdit();
                if (isLastLine) {
                    getDocument().insertString(lineEnd, "\n", null);
                    lineEnd++;
                }
                getDocument().insertString(lineEnd, currentLine, null);
                if (!isLastLine) {
                    getDocument().insertString(lineEnd + currentLine.length(), "\n", null);
                }
                endEdit();
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Returns true when can sort lines ascendant
     * @return - boolean
     */
    public boolean canSortLinesAscendant() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) {
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(getSelectionStart());
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(getSelectionEnd());
            return startIndex != endIndex;
        } else return false;
    }

    /**
     * Sorts the selected lines ascendant
     */
    public void sortLinesAscendant() {
        if (!canSortLinesAscendant()) return;
        int selectionStart = getSelectionStart();
        int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
        selectionStart = startElement.getStartOffset();
        int selectionEnd = getSelectionEnd();
        int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
        Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
        selectionEnd = endElement.getEndOffset();
        if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            StringTokenizer tokenizer = new StringTokenizer(text, "\n");
            List<String> list = new ArrayList<String>();
            while (tokenizer.hasMoreTokens()) {
                list.add(tokenizer.nextToken());
            }
            Collections.sort(list);
            StringBuffer buffer = new StringBuffer();
            for (String line : list) {
                buffer.append(line);
                buffer.append("\n");
            }
            beginEdit();
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            getDocument().insertString(selectionStart, buffer.toString(), null);
            endEdit();
            select(selectionStart, selectionEnd);
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when can sort lines descendant
     * @return - boolean
     */
    public boolean canSortLinesDescendant() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) {
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(getSelectionStart());
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(getSelectionEnd());
            return startIndex != endIndex;
        } else return false;
    }

    /**
     * Sorts the selected lines descendant
     */
    public void sortLinesDescendant() {
        if (!canSortLinesDescendant()) return;
        int selectionStart = getSelectionStart();
        int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
        selectionStart = startElement.getStartOffset();
        int selectionEnd = getSelectionEnd();
        int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
        Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
        selectionEnd = endElement.getEndOffset();
        if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            StringTokenizer tokenizer = new StringTokenizer(text, "\n");
            List<String> list = new ArrayList<String>();
            while (tokenizer.hasMoreTokens()) {
                list.add(tokenizer.nextToken());
            }
            Collections.sort(list);
            Collections.reverse(list);
            StringBuffer buffer = new StringBuffer();
            for (String line : list) {
                buffer.append(line);
                buffer.append("\n");
            }
            beginEdit();
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            getDocument().insertString(selectionStart, buffer.toString(), null);
            endEdit();
            select(selectionStart, selectionEnd);
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when can move the selected or current line up
     * @return - boolean
     */
    public boolean canMoveLinesUp() {
        if (isReadOnlyMode()) return false;
        int position;
        if (isTextSelected()) position = getSelectionStart();
        else position = getCaretPosition();
        int index = getDocument().getDefaultRootElement().getElementIndex(position);
        return index != 0;
    }

    /**
     * Moves the selected or current line up
     */
    public void moveLinesUp() {
        if (!canMoveLinesUp()) return;
        int selectionStart;
        int selectionEnd;
        int newPosition;
        boolean isTextSelected = isTextSelected();
        if (isTextSelected()) {
            selectionStart = getSelectionStart();
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
            Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
            Element previousElement = getDocument().getDefaultRootElement().getElement(startIndex - 1);
            newPosition = previousElement.getStartOffset();
            selectionStart = startElement.getStartOffset();
            selectionEnd = getSelectionEnd();
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
            Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
            selectionEnd = endElement.getEndOffset();
            if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        } else {
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(getCaretPosition());
            Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
            Element previousElement = getDocument().getDefaultRootElement().getElement(startIndex - 1);
            selectionStart = startElement.getStartOffset();
            selectionEnd = startElement.getEndOffset();
            if (startIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
            newPosition = previousElement.getStartOffset();
        }
        try {
            beginEdit();
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            if (!text.endsWith("\n")) text = text + "\n";
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            getDocument().insertString(newPosition, text, null);
            endEdit();
            if (isTextSelected) select(newPosition, newPosition + text.length());
            else setCaretPosition(newPosition);
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when the lines can be moved down
     * @return - boolean
     */
    public boolean canMoveLinesDown() {
        if (isReadOnlyMode()) return false;
        int position;
        if (isTextSelected()) position = getSelectionEnd();
        else position = getCaretPosition();
        int index = getDocument().getDefaultRootElement().getElementIndex(position);
        int count = getDocument().getDefaultRootElement().getElementCount();
        return index != count - 1;
    }

    /**
     * Moves the selected text or current line down
     */
    public void moveLinesDown() {
        if (!canMoveLinesDown()) return;
        int selectionStart;
        int selectionEnd;
        boolean isTextSelected = isTextSelected();
        if (isTextSelected()) {
            selectionStart = getSelectionStart();
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
            Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
            selectionStart = startElement.getStartOffset();
            selectionEnd = getSelectionEnd();
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
            Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
            selectionEnd = endElement.getEndOffset();
        } else {
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(getCaretPosition());
            Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
            selectionStart = endElement.getStartOffset();
            selectionEnd = endElement.getEndOffset();
        }
        try {
            beginEdit();
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            int nextIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
            Element nextElement = getDocument().getDefaultRootElement().getElement(nextIndex);
            int newPosition = nextElement.getEndOffset();
            getDocument().insertString(newPosition, text, null);
            endEdit();
            if (isTextSelected) select(newPosition, newPosition + text.length());
            else setCaretPosition(newPosition);
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    public void setColumns(int columns) {
        setSize(columns * getFont().getSize(), getSize().height);
    }

    public int getColumns() {
        return getSize().width / getFont().getSize();
    }

    /**
     * Returns true when the text can be aligned to left
     * @return - boolean
     */
    public boolean canAlignTextLeft() {
        if (isReadOnlyMode()) return false;
        return isTextSelected();
    }

    /**
     * Aligns the text to left
     */
    public void alignTextLeft() {
        if (!canAlignTextLeft()) return;
        int columns = getColumns();
        int selectionStart = getSelectionStart();
        int selectionEnd = getSelectionEnd();
        int index = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element element = getDocument().getDefaultRootElement().getElement(index);
        selectionStart = element.getStartOffset();
        index = getDocument().getDefaultRootElement().getElementIndex(selectionEnd);
        element = getDocument().getDefaultRootElement().getElement(index);
        selectionEnd = element.getEndOffset();
        if (index == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            StringBuffer buffer = new StringBuffer();
            StringBuffer line;
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            StringTokenizer tokenizer = new StringTokenizer(text, "\n");
            while (tokenizer.hasMoreTokens()) {
                line = new StringBuffer();
                String token = tokenizer.nextToken();
                StringTokenizer lineTokenizer = new StringTokenizer(token, " \t\r\f");
                while (lineTokenizer.hasMoreTokens()) {
                    String lineToken = lineTokenizer.nextToken();
                    if (line.length() + lineToken.length() > columns) {
                        if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
                        buffer.append(line.toString());
                        buffer.append("\n");
                        line = new StringBuffer();
                    }
                    line.append(lineToken);
                    line.append(" ");
                }
                if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
                buffer.append(line.toString());
                buffer.append("\n");
            }
            beginEdit();
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            getDocument().insertString(selectionStart, buffer.toString(), null);
            endEdit();
            select(selectionStart, selectionStart + buffer.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when the text can be aligned to right
     * @return - boolean
     */
    public boolean canAlignTextRight() {
        if (isReadOnlyMode()) return false;
        return isTextSelected();
    }

    /**
     * Aligns the text to right
     */
    public void alignTextRight() {
        if (!canAlignTextRight()) return;
        int columns = getColumns();
        int selectionStart = getSelectionStart();
        int selectionEnd = getSelectionEnd();
        int index = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element element = getDocument().getDefaultRootElement().getElement(index);
        selectionStart = element.getStartOffset();
        index = getDocument().getDefaultRootElement().getElementIndex(selectionEnd);
        element = getDocument().getDefaultRootElement().getElement(index);
        selectionEnd = element.getEndOffset();
        if (index == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            StringBuffer buffer = new StringBuffer();
            StringBuffer line;
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            StringTokenizer tokenizer = new StringTokenizer(text, "\n");
            while (tokenizer.hasMoreTokens()) {
                line = new StringBuffer();
                String token = tokenizer.nextToken();
                StringTokenizer lineTokenizer = new StringTokenizer(token, " \t\r\f");
                while (lineTokenizer.hasMoreTokens()) {
                    String lineToken = lineTokenizer.nextToken();
                    if (line.length() + lineToken.length() > columns) {
                        if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
                        StringBuffer prefix = new StringBuffer();
                        for (int i = columns; i > line.length(); i--) {
                            prefix.append(' ');
                        }
                        buffer.append(prefix.toString());
                        buffer.append(line.toString());
                        buffer.append("\n");
                        line = new StringBuffer();
                    }
                    line.append(lineToken);
                    line.append(" ");
                }
                if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
                StringBuffer prefix = new StringBuffer();
                for (int i = columns; i > line.length(); i--) {
                    prefix.append(' ');
                }
                buffer.append(prefix.toString());
                buffer.append(line.toString());
                buffer.append("\n");
            }
            beginEdit();
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            getDocument().insertString(selectionStart, buffer.toString(), null);
            endEdit();
            select(selectionStart, selectionStart + buffer.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when the text can be centered in editor
     * @return - boolean
     */
    public boolean canCenterText() {
        if (isReadOnlyMode()) return false;
        return isTextSelected();
    }

    /**
     * Centers the text in editor
     */
    public void centerText() {
        if (!canCenterText()) return;
        int columns = getColumns();
        int selectionStart = getSelectionStart();
        int selectionEnd = getSelectionEnd();
        int index = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element element = getDocument().getDefaultRootElement().getElement(index);
        selectionStart = element.getStartOffset();
        index = getDocument().getDefaultRootElement().getElementIndex(selectionEnd);
        element = getDocument().getDefaultRootElement().getElement(index);
        selectionEnd = element.getEndOffset();
        if (index == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            StringBuffer buffer = new StringBuffer();
            StringBuffer line;
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            StringTokenizer tokenizer = new StringTokenizer(text, "\n");
            while (tokenizer.hasMoreTokens()) {
                line = new StringBuffer();
                String token = tokenizer.nextToken();
                StringTokenizer lineTokenizer = new StringTokenizer(token, " \t\r\f");
                while (lineTokenizer.hasMoreTokens()) {
                    String lineToken = lineTokenizer.nextToken();
                    if (line.length() + lineToken.length() > columns) {
                        if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
                        StringBuffer prefix = new StringBuffer();
                        for (int i = columns / 2; i > line.length() / 2; i--) {
                            prefix.append(' ');
                        }
                        buffer.append(prefix.toString());
                        buffer.append(line.toString());
                        buffer.append("\n");
                        line = new StringBuffer();
                    }
                    line.append(lineToken);
                    line.append(" ");
                }
                if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
                StringBuffer prefix = new StringBuffer();
                for (int i = columns / 2; i > line.length() / 2; i--) {
                    prefix.append(' ');
                }
                buffer.append(prefix.toString());
                buffer.append(line.toString());
                buffer.append("\n");
            }
            beginEdit();
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            getDocument().insertString(selectionStart, buffer.toString(), null);
            endEdit();
            select(selectionStart, selectionStart + buffer.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when the text can be justified in editor
     * @return - boolean
     */
    public boolean canJustifyText() {
        if (isReadOnlyMode()) return false;
        return isTextSelected();
    }

    /**
     * Centers the text in editor
     */
    public void justifyText() {
        if (!canJustifyText()) return;
        int columns = getColumns();
        int selectionStart = getSelectionStart();
        int selectionEnd = getSelectionEnd();
        int index = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element element = getDocument().getDefaultRootElement().getElement(index);
        selectionStart = element.getStartOffset();
        index = getDocument().getDefaultRootElement().getElementIndex(selectionEnd);
        element = getDocument().getDefaultRootElement().getElement(index);
        selectionEnd = element.getEndOffset();
        if (index == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            StringBuffer buffer = new StringBuffer();
            StringBuffer line = new StringBuffer();
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            StringTokenizer tokenizer = new StringTokenizer(text, " \t\r\f\n");
            while (tokenizer.hasMoreTokens()) {
                String token = tokenizer.nextToken();
                if (line.length() + token.length() > columns) {
                    if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
                    while ((line.length() < columns) && (line.indexOf(" ") >= 0)) {
                        for (int i = 0; i < line.length(); i++) {
                            if (Character.isSpaceChar(line.charAt(i))) {
                                line.insert(i, ' ');
                                while (Character.isSpaceChar(line.charAt(i))) i++;
                            }
                        }
                    }
                    buffer.append(line.toString());
                    buffer.append("\n");
                    line = new StringBuffer();
                }
                line.append(token);
                line.append(' ');
            }
            if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
            while ((line.length() < columns) && (line.indexOf(" ") >= 0)) {
                for (int i = 0; i < line.length(); i++) {
                    if (Character.isSpaceChar(line.charAt(i))) {
                        line.insert(i, ' ');
                        while (Character.isSpaceChar(line.charAt(i))) i++;
                    }
                }
            }
            buffer.append(line.toString());
            buffer.append("\n");
            beginEdit();
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            getDocument().insertString(selectionStart, buffer.toString(), null);
            endEdit();
            select(selectionStart, selectionStart + buffer.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when the latin paragraph can be made
     * @return -- boolean
     */
    public boolean canMakeLatinParagraph() {
        if (isReadOnlyMode()) return false;
        return isTextSelected();
    }

    /**
     * Makes the latin paragraph
     */
    public void makeLatinParagraph() {
        if (!canMakeLatinParagraph()) return;
        int columns = getColumns();
        int selectionStart = getSelectionStart();
        int selectionEnd = getSelectionEnd();
        int index = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element element = getDocument().getDefaultRootElement().getElement(index);
        selectionStart = element.getStartOffset();
        index = getDocument().getDefaultRootElement().getElementIndex(selectionEnd);
        element = getDocument().getDefaultRootElement().getElement(index);
        selectionEnd = element.getEndOffset();
        if (index == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            StringBuffer buffer = new StringBuffer();
            StringBuffer line = new StringBuffer();
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            StringTokenizer tokenizer = new StringTokenizer(text, " \t\r\f\n");
            while (tokenizer.hasMoreTokens()) {
                String token = tokenizer.nextToken();
                if (line.length() + token.length() > columns) {
                    if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
                    while ((line.length() < columns) && (line.indexOf(" ") >= 0)) {
                        for (int i = 0; i < line.length(); i++) {
                            if (Character.isSpaceChar(line.charAt(i))) {
                                line.insert(i, ' ');
                                while (Character.isSpaceChar(line.charAt(i))) i++;
                            }
                        }
                    }
                    buffer.append(line.toString());
                    buffer.append("\n");
                    line = new StringBuffer();
                }
                line.append(token);
                line.append(' ');
            }
            if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
            buffer.append(line.toString());
            buffer.append("\n");
            buffer.append("\n");
            beginEdit();
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            getDocument().insertString(selectionStart, buffer.toString(), null);
            endEdit();
            select(selectionStart, selectionStart + buffer.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when the cyrillic paragraph can be made
     * @return -- boolean
     */
    public boolean canMakeCyrillicParagraph() {
        if (isReadOnlyMode()) return false;
        return isTextSelected();
    }

    /**
     * Makes the cyrillic paragraph
     */
    public void makeCyrillicParagraph() {
        if (!canMakeCyrillicParagraph()) return;
        int columns = getColumns();
        int selectionStart = getSelectionStart();
        int selectionEnd = getSelectionEnd();
        int index = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element element = getDocument().getDefaultRootElement().getElement(index);
        selectionStart = element.getStartOffset();
        index = getDocument().getDefaultRootElement().getElementIndex(selectionEnd);
        element = getDocument().getDefaultRootElement().getElement(index);
        selectionEnd = element.getEndOffset();
        if (index == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            StringBuffer buffer = new StringBuffer();
            StringBuffer line = new StringBuffer();
            line.append("    ");
            String text = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            StringTokenizer tokenizer = new StringTokenizer(text, " \t\r\f\n");
            while (tokenizer.hasMoreTokens()) {
                String token = tokenizer.nextToken();
                if (line.length() + token.length() > columns) {
                    if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
                    outer: while ((line.length() < columns) && (line.indexOf(" ") >= 0)) {
                        for (int i = 0; i < line.length(); i++) {
                            if (i == 0 && line.charAt(0) == ' ') {
                                i += 4;
                                if (line.lastIndexOf(" ") <= 4) break outer;
                                else continue;
                            }
                            if (Character.isSpaceChar(line.charAt(i))) {
                                line.insert(i, ' ');
                                while (Character.isSpaceChar(line.charAt(i))) i++;
                            }
                        }
                    }
                    buffer.append(line.toString());
                    buffer.append("\n");
                    line = new StringBuffer();
                }
                line.append(token);
                line.append(' ');
            }
            if (line.length() > 0 && line.charAt(line.length() - 1) == ' ') line.deleteCharAt(line.length() - 1);
            buffer.append(line.toString());
            buffer.append("\n");
            beginEdit();
            getDocument().remove(selectionStart, selectionEnd - selectionStart);
            getDocument().insertString(selectionStart, buffer.toString(), null);
            endEdit();
            select(selectionStart, selectionStart + buffer.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Trims lines in string
     * @param source - string
     * @return string
     */
    private String trimString(String source) {
        StringTokenizer tokenizer = new StringTokenizer(source, "\n");
        StringBuffer result = new StringBuffer();
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            result.append(token.trim());
            if (tokenizer.hasMoreTokens()) result.append("\n");
        }
        return result.toString();
    }

    /**
     * Returns true when the selection or current line can be trimmed
     * @return - boolean
     */
    public boolean canTrimSelectedLines() {
        if (isReadOnlyMode()) return false;
        if (isTextSelected()) return true;
        else {
            String currentLine = getCurrentLine();
            return currentLine.length() > 0
                    && (Character.isSpaceChar(currentLine.charAt(0))
                    || Character.isSpaceChar(currentLine.charAt(currentLine.length() - 1)));
        }
    }

    /**
     * Trims the selected lines
     */
    public void trimSelectedLines() {
        if (!canTrimSelectedLines()) return;
        if (isTextSelected()) {
            int selectionStart = getSelectionStart();
            int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
            Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
            selectionStart = startElement.getStartOffset();
            int selectionEnd = getSelectionEnd();
            int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
            Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
            selectionEnd = endElement.getEndOffset();
            if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
            try {
                String selectedText = getDocument().getText(selectionStart, selectionEnd - selectionStart);
                String result = trimString(selectedText);
                beginEdit();
                getDocument().remove(selectionStart, selectedText.length());
                getDocument().insertString(selectionStart, result, null);
                endEdit();
                select(selectionStart, selectionStart + result.length());
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        } else {
            if (isTextEmpty()) return;
            int index = getDocument().getDefaultRootElement().getElementIndex(getCaretPosition());
            Element element = getDocument().getDefaultRootElement().getElement(index);
            int offset = element.getStartOffset();
            try {
                String line = getDocument().getText(offset, element.getEndOffset() - offset);
                String result = trimString(line);
                result += "\n";
                beginEdit();
                getDocument().remove(offset, line.length());
                getDocument().insertString(offset, result, null);
                endEdit();
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Joins lines in parameter. Omits an empty lines. Returns the result
     * @param source - string
     * @return string
     */
    private String joinLinesInString(String source) {
        StringTokenizer tokenizer = new StringTokenizer(source, "\n");
        StringBuffer result = new StringBuffer();
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            if (!token.isEmpty()) {
                result.append(token);
                result.append(' ');
            }
        }
        if (result.length() > 0 && result.charAt(result.length() - 1) == ' ') result.deleteCharAt(result.length() - 1);
        return result.toString();
    }

    /**
     * Returns true when the lines can be joined
     * @return - boolean
     */
    public boolean canJoinSelectedLines() {
        if (isReadOnlyMode()) return false;
        return isTextSelected();
    }

    /**
     * Joins selected lines
     */
    public void joinSelectedLines() {
        if (!canJoinSelectedLines()) return;
        int selectionStart = getSelectionStart();
        int startIndex = getDocument().getDefaultRootElement().getElementIndex(selectionStart);
        Element startElement = getDocument().getDefaultRootElement().getElement(startIndex);
        selectionStart = startElement.getStartOffset();
        int selectionEnd = getSelectionEnd();
        int endIndex = getDocument().getDefaultRootElement().getElementIndex(selectionEnd - 1);
        Element endElement = getDocument().getDefaultRootElement().getElement(endIndex);
        selectionEnd = endElement.getEndOffset();
        if (endIndex == getDocument().getDefaultRootElement().getElementCount() - 1) selectionEnd--;
        try {
            String source = getDocument().getText(selectionStart, selectionEnd - selectionStart);
            String result = joinLinesInString(source);
            beginEdit();
            getDocument().remove(selectionStart, source.length());
            getDocument().insertString(selectionStart, result, null);
            endEdit();
            select(selectionStart, selectionStart + result.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when the current line or selected text can be splitted into lines by words
     * @return boolean
     */
    public boolean canSplitIntoLines() {
        if (isReadOnlyMode()) return false;
        String text = "";
        if (isTextSelected()) text = getSelectedText();
        else text = getCurrentLine();
        return text.indexOf(' ') != -1 || text.indexOf('\t') != -1;
    }

    /**
     * Spits each word in the current or selected lines into a separate lines
     */
    public void splitIntoLines() {
        if (!canSplitIntoLines()) return;
        String line = "";
        int startOffset = 0;
        int endOffset = 0;
        boolean isTextSelected = isTextSelected();
        if (isTextSelected) {
            line = getSelectedText();
            startOffset = getSelectionStart();
            endOffset = getSelectionEnd();
        } else {
            int index = getDocument().getDefaultRootElement().getElementIndex(getCaretPosition());
            Element element = getDocument().getDefaultRootElement().getElement(index);
            startOffset = element.getStartOffset();
            endOffset = element.getEndOffset() - 1;
            try {
                line = getDocument().getText(startOffset, endOffset - startOffset);
            } catch (BadLocationException e) {
                e.printStackTrace();
            }
        }
        StringTokenizer tokenizer = new StringTokenizer(line, " \t");
        StringBuffer buffer = new StringBuffer();
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            buffer.append(token);
            if (tokenizer.hasMoreTokens()) buffer.append('\n');
        }
        beginEdit();
        try {
            getDocument().remove(startOffset, endOffset - startOffset);
            getDocument().insertString(startOffset, buffer.toString(), null);
            if (isTextSelected) select(startOffset, buffer.length());
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
        endEdit();
    }

	private KeyListener startLineKeyListener = new KeyListener() {
		
		public void keyPressed(KeyEvent e) { }

		public void keyReleased(KeyEvent e) { }

		public void keyTyped(KeyEvent e) {
			if (isAutoIndentMode() && e.getKeyChar() == KeyEvent.VK_ENTER) {
				int startPosition = 0;
				int length = 0;
				String text;
				int i = 0;
				do {
					i++;
					try {
						text = getDocument().getText(getCaretPosition() - 1 - i, 1);
					} catch (BadLocationException e1) {
						break;
					}
				} while(text.charAt(0) != KeyEvent.VK_ENTER);
				startPosition = getCaretPosition() - 1 - i + 1;
				do {
					try {
						text = getDocument().getText(startPosition + length, 1);
						if (Character.isSpace(text.charAt(0)) && text.charAt(0) != KeyEvent.VK_ENTER) {
							length++;
						}
					} catch (BadLocationException e1) {
						break;
					}
				} while(Character.isSpace(text.charAt(0)) && text.charAt(0) != KeyEvent.VK_ENTER);
				
				try {
					String spaceLine = getDocument().getText(startPosition, length);
					getDocument().insertString(getCaretPosition(), spaceLine, null);
				} catch (BadLocationException e1) {
					e1.printStackTrace();
				}
			}
		}
		
	};

	// undoable capability
	
	/**
	 * Sets undoable for text component
	 * @param value - boolean
	 */
	public void setUndoable(boolean value) {
		this.undoable = value;
	}
	
	/**
	 * Returns true when text component is undoable
	 * @return boolean
	 */
	public boolean isUndoable() {
		return this.undoable;
	}
	
	/**
	 * Chack is text is changed.
	 * @return true when text is changed.
	 */
	public boolean isTextChanged() {
		return numberOfCurrentEdit != 0;
	}	

	/**
	 * Returns true when text changes may be reverted
	 * @return boolean
	 */
	public boolean canRevert() {
		return undoable && isTextChanged() && !isReadOnlyMode();
	}	
	
	/**
	 * Returns true if the text change may be undone
	 */
	public boolean canUndo() {
		return undoable && undoManager.canUndo() && !isReadOnlyMode();
	}
	
	/**
	 * Returns true if the text change may be redone
	 */
	public boolean canRedo() {
		return undoable && undoManager.canRedo() && !isReadOnlyMode();
	}
	
	/**
	 * Undoes the last text change if possible. 
	 */
	public void undo() {
		if (canUndo()) {
			undoManager.undo();
			numberOfCurrentEdit--;
		}
	}

	/**
	 * Redose the last text change if possible.
	 */
	public void redo() {
		if (canRedo()) {
			undoManager.redo();
			numberOfCurrentEdit++;
		}
	}
	
 	/**
 	 * Returns true when compound edit is begun
 	 * @return boolean
 	 */
	public boolean isEditBegun() {
		return !edits.isEmpty();
	}
	
	/**
	 * Marks beginning of coalesce edit operation
	 */
	public void beginEdit() {
		if (!undoable) return;
		edits.push(new CompoundEdit());
	}
	
	/**
	 * Marks end of coalesce edit operation
	 */
	public void endEdit() {
		if (undoable) {
			if (!isEditBegun()) {
				return;
			}
			CompoundEdit currentEdit = edits.pop();
			if (isEditBegun()) {
				edits.peek().addEdit(currentEdit);
			} else {
				undoManager.addEdit(currentEdit);
			}
			currentEdit.end();
		}
		numberOfCurrentEdit++;
	}
	
	public void revert() {
		if (!canRevert()) return;
		try {
			while (isTextChanged()) {
				undoManager.undo();
				numberOfCurrentEdit--;
			}
		} catch (CannotUndoException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Marks text as not changed
	 */
	public void reset(boolean discardAllEdits) {
		if (discardAllEdits) {
			if (undoManager != null) {
				undoManager.discardAllEdits();
			}
			if (edits != null) {
				edits.clear();
			}
		}
		numberOfCurrentEdit = 0;
	}
	
	private UndoableEditListener undoableEditListener = new UndoableEditListener() {
		
		public void undoableEditHappened(UndoableEditEvent e) {
			if (isEditBegun()) {
				if (undoable) edits.peek().addEdit(e.getEdit());
			} else {
				if (undoable) undoManager.addEdit(e.getEdit());
				numberOfCurrentEdit++;
			}
		}
		
	};

	/**
	 * Returns true if clipboard contain text.
	 * @throws IllegalStateException
	 * @return boolean value
	 */
	public boolean isClipboardContainText() throws IllegalStateException {
		for (DataFlavor dataFlavor : getToolkit().getSystemClipboard().getAvailableDataFlavors()) {
			if (dataFlavor.isFlavorTextType()) {
				return true;
			}
		}
		return false;
	}
	
	public void cut() {
		if (canCut()) super.cut();
	}
	
	public void copy() {
		if (canCopy()) super.copy();
	}
	
	public void paste() {
		if (canPaste()) super.paste();
	}
	
	/**
	 * Returns true when text fragment may be cut
	 * @return boolean
	 */
	public boolean canCut() {
		return isTextSelected() && !isReadOnlyMode();
	}
	
	/**
	 * Returns true when text fragment may be copied
	 * @return boolean
	 */
	public boolean canCopy() {
		return isTextSelected() && !isReadOnlyMode();
	}
	
	/**
	 * Returns true when text fragment may be pasted
	 * @return boolean
	 */
	public boolean canPaste() {
		try {
			return isClipboardContainText() && !isReadOnlyMode();
		} catch (IllegalStateException e1) {
			return true;
		}
	}

    /**
     * Returns true when selection can be cleared
     * @return boolean
     */
    public boolean canClear() {
         return isTextSelected() && !isReadOnlyMode();
    }

    /**
     * Clears the selected text
     */
    public void clear() {
        if (!canClear()) return;
        int start = getSelectionStart();
        int end = getSelectionEnd();
        try {
            getDocument().remove(start, end - start);
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when text can be deselected
     * @return boolean
     */
    public boolean canDeselect() {
        return isTextSelected();
    }

    /**
     * Deselects the text fragment
     */
    public void deselect() {
        int caretPosition = getSelectionEnd();
        select(caretPosition, caretPosition);
    }
	
	public int getNumberOfCurrentLine() {
		Element root = getDocument().getDefaultRootElement();
		return root.getElementIndex(getCaretPosition()) + 1;
	}
	
	public int getNumberOfCurrentColumn() {
		Element root = getDocument().getDefaultRootElement();
		Element element = root.getElement(root.getElementIndex(getCaretPosition()));		
		return getCaretPosition() - element.getStartOffset() + 1;
	}
	
}
