/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text.syntax;

import java.awt.Color;
import java.awt.Font;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.logging.Level;
import java.util.logging.Logger;

import jnpad.JNPad;
import jnpad.config.Config;
import jnpad.util.Utilities;

/**
 * The Class CPPScheme.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class CPPScheme extends CScheme {
  static Color                exceptionColor;
  static Color                operatorColor_;

  Font                        exceptionFont;
  Font                        operatorFont_;

  /** Logger. */
  private static final Logger LOGGER           = Logger.getLogger(CPPScheme.class.getName());

  /** UID */
  private static final long   serialVersionUID = 8959231363486824139L;

  static {
    if (LOGGER.isLoggable(Level.CONFIG)) 
      LOGGER.config("C++ Scheme - init."); //$NON-NLS-1$

    BufferedReader in = null;
    try {
      String dir = JNPad.PROPS_DIR + Utilities.DIR_SEPARATOR + "schemes"; //$NON-NLS-1$
      String file = dir + Utilities.DIR_SEPARATOR + "cpp.words"; //$NON-NLS-1$
      in = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-8")); //$NON-NLS-1$
      String line;
      // Ignore anything before "KEYWORDS"
      while ((line = in.readLine()) != null && !line.equals(":KEYWORDS")); //$NON-NLS-1$
      // Everything is KEYWORDS until we run into "MODIFIERS"
      while ((line = in.readLine()) != null && !line.equals(":MODIFIERS")) //$NON-NLS-1$
        read(line, WordType.KEYWORD);
      // Everything is MODIFIERS until we run into "EXCEPTIONS"
      while ((line = in.readLine()) != null && !line.equals(":EXCEPTIONS")) //$NON-NLS-1$
        read(line, WordType.MODIFIER);
      // Everything is EXCEPTIONS until we run into "DATA_TYPES"
      while ((line = in.readLine()) != null && !line.equals(":DATA_TYPES")) //$NON-NLS-1$
        read(line, WordType.EXCEPTION);
      // Everything is DATA_TYPES until we run into "OPERATORS"
      while ((line = in.readLine()) != null && !line.equals(":OPERATORS")) //$NON-NLS-1$
        read(line, WordType.DATA_TYPE);
      // The rest of the file is OPERATORS
      while ((line = in.readLine()) != null)
        read(line, WordType.OPERATOR);
    }
    catch (IOException ex) {
      LOGGER.log(Level.WARNING, ex.getMessage(), ex);
    }
    finally {
      try {
        if (in != null) {
          in.close();
        }
      }
      catch (IOException ex) {
        LOGGER.log(Level.WARNING, ex.getMessage(), ex);
      }
    }
  }

  /**
   * Instantiates a new cPP scheme.
   *
   * @param mini the mini
   */
  public CPPScheme(boolean mini) {
    super(mini);
    doUpdateColors();
    doUpdateFonts();
  }

  /**
   * Update colors.
   */
  private void doUpdateColors() {
    exceptionColor   = Config.SYNTAX_KEYWORD3_COLOR.getValue();
    operatorColor_   = Config.SYNTAX_KEYWORD7_COLOR.getValue();
  }

  /**
   * Update fonts.
   */
  private void doUpdateFonts() {
    exceptionFont   = textFont.deriveFont(Config.SYNTAX_KEYWORD3_STYLE.getValue());
    operatorFont_   = textFont.deriveFont(Config.SYNTAX_KEYWORD7_STYLE.getValue());
  }

  /**
   * Sets the text font.
   *
   * @param f the new text font
   * @see jnpad.text.syntax.CScheme#setTextFont(java.awt.Font)
   */
  @Override
  public void setTextFont(Font f) {
    super.setTextFont(f);
    doUpdateFonts();
  }
  
  /**
   * Configure.
   * 
   * @param cfg the cfg
   * @see jnpad.text.syntax.SyntaxScheme#configure(int)
   */
  @Override
  public void configure(final int cfg) {
    super.configure(cfg);
    if ((cfg & CFG_COLOR) != 0) {
      doUpdateColors();
    }
    if ((cfg & CFG_FONT) != 0) {
      doUpdateFonts();
    }
  }

  /**
   * Gets the word color.
   *
   * @param type the type
   * @return the word color
   * @see jnpad.text.syntax.CScheme#getWordColor(jnpad.text.syntax.CScheme.WordType)
   */
  @Override
  public Color getWordColor(WordType type) {
    if (type == null) return textColor;
    switch (type) { 
      case OPERATOR : return classify ? operatorColor_ : keywordColor;
      case EXCEPTION: return classify ? exceptionColor : keywordColor;
      default       : return super.getWordColor(type);
    }
  }

  /**
   * Gets the word font.
   *
   * @param type the type
   * @return the word font
   * @see jnpad.text.syntax.CScheme#getWordFont(jnpad.text.syntax.CScheme.WordType)
   */
  @Override
  public Font getWordFont(WordType type) {
    if (type == null) return textFont;
    switch (type) {
      case OPERATOR : return classify ? operatorFont_ : keywordFont;
      case EXCEPTION: return classify ? exceptionFont : keywordFont;
      default       : return super.getWordFont(type);
    }
  }

  /**
   * Gets the content type.
   * 
   * @return the content type
   * @see jnpad.text.Scheme#getContentType()
   */
  @Override
  public String getContentType() {
    return ContentTypes.CPP;
  }
  
}
