/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui.table;

import java.io.Serializable;
import java.util.Collections;
import java.util.Comparator;
import java.util.Vector;

import javax.swing.table.DefaultTableModel;

/**
 * The Class JNPadDefaultTableModel.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class JNPadDefaultTableModel extends DefaultTableModel implements JNPadTableModel {
  private int               sortedColumnIndex = -1;
  private boolean           isAscending;
  private boolean           isEditable        = true;

  /** UID */
  private static final long serialVersionUID  = -8677302633092266110L;

  /**
   * Instantiates a new jNPad default table model.
   */
  public JNPadDefaultTableModel() {
    //empty
  }

  /**
   * Instantiates a new jNPad default table model.
   * 
   * @param isEditable boolean
   */
  public JNPadDefaultTableModel(boolean isEditable) {
    this.isEditable = isEditable;
  }

  /**
   * Instantiates a new jNPad default table model.
   * 
   * @param rowCount int
   * @param columnCount int
   */
  public JNPadDefaultTableModel(int rowCount, int columnCount) {
    super(rowCount, columnCount);
  }

  /**
   * Instantiates a new jNPad default table model.
   * 
   * @param data Object[][]
   * @param columnNames Object[]
   */
  public JNPadDefaultTableModel(Object[][] data, Object[] columnNames) {
    super(data, columnNames);
  }

  /**
   * Instantiates a new jNPad default table model.
   * 
   * @param columnNames Object[]
   * @param rowCount int
   */
  public JNPadDefaultTableModel(Object[] columnNames, int rowCount) {
    super(columnNames, rowCount);
  }

  /**
   * Instantiates a new jNPad default table model.
   * 
   * @param columnNames Vector
   * @param rowCount int
   */
  public JNPadDefaultTableModel(Vector<?> columnNames, int rowCount) {
    super(columnNames, rowCount);
  }

  /**
   * Instantiates a new jNPad default table model.
   *
   * @param data Vector
   * @param columnNames Vector
   */
  public JNPadDefaultTableModel(Vector<?> data, Vector<?> columnNames) {
    super(data, columnNames);
  }

  /**
   * Gets the column class.
   *
   * @param column the column
   * @return the column class
   * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
   */
  @Override
  public Class<?> getColumnClass(int column) {
    try {
      if (getRowCount() > 0)
        return getValueAt(0, column).getClass();
    }
    catch (Exception ex) {
      //ignored
    }
    return super.getColumnClass(column);
  }

  /**
   * Removes the rows.
   *
   * @param rows the rows
   */
  public void removeRows(int[] rows) {
    for (int i = 0; i < rows.length; i++)
      removeRow(rows[i] - i);
  }

  /**
   * Checks if is cell editable.
   *
   * @param row the row
   * @param column the column
   * @return true, if is cell editable
   * @see javax.swing.table.DefaultTableModel#isCellEditable(int, int)
   */
  @Override
  public boolean isCellEditable(int row, int column) {
    return isEditable;
  }

  /**
   * Checks if is sortable.
   *
   * @param column the column
   * @return true, if is sortable
   * @see jnpad.ui.table.JNPadTableModel#isSortable(int)
   */
  @Override
  public boolean isSortable(int column) {
    return true;
  }

  /**
   * Sets the sort by column.
   *
   * @param columnIndex the column index
   * @param isAscending is ascending
   * @see jnpad.ui.table.JNPadTableModel#setSortByColumn(int, boolean)
   */
  @Override
  @SuppressWarnings("unchecked")
  public void setSortByColumn(int columnIndex, boolean isAscending) {
    Vector data = getDataVector();
    Collections.sort(data, new UComparator(columnIndex, isAscending));
    this.isAscending = isAscending;
    sortedColumnIndex = columnIndex;

    super.fireTableDataChanged(); // va?
  }

  /**
   * Checks if is ascending.
   *
   * @return true, if is ascending
   * @see jnpad.ui.table.JNPadTableModel#isAscending()
   */
  @Override
  public boolean isAscending() {
    return isAscending;
  }

  /**
   * Gets the sorted column index.
   *
   * @return the sorted column index
   * @see jnpad.ui.table.JNPadTableModel#getSortedColumnIndex()
   */
  @Override
  public int getSortedColumnIndex() {
    return sortedColumnIndex;
  }

  //////////////////////////////////////////////////////////////////////////////
  /**
   * The Class UComparator.
   */
  @SuppressWarnings("unchecked")
  static class UComparator implements Comparator, Serializable {
    private int               sortedColumnIndex;
    private boolean           isAscending;

    /** UID */
    private static final long serialVersionUID = 6785677239317289558L;

    /**
     * Instantiates a new u comparator.
     *
     * @param sortedColumnIndex int
     * @param isAscending boolean
     */
    UComparator(int sortedColumnIndex, boolean isAscending) {
      this.sortedColumnIndex = sortedColumnIndex;
      this.isAscending = isAscending;
    }

    /**
     * Compare.
     *
     * @param obj1 the obj1
     * @param obj2 the obj2
     * @return the int
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     */
    @Override
    public int compare(Object obj1, Object obj2) {
      if (obj1 instanceof Vector && obj2 instanceof Vector) {
        Object o1 = ((Vector) obj1).elementAt(sortedColumnIndex);
        Object o2 = ((Vector) obj2).elementAt(sortedColumnIndex);
        if (o1 instanceof Comparable && o2 instanceof Comparable) {
          Comparable c1 = (Comparable) o1;
          Comparable c2 = (Comparable) o2;
          if (isAscending)
            return c1.compareTo(c2);
          return c2.compareTo(c1);
        }
      }
      return 0;
    }
  }
  //////////////////////////////////////////////////////////////////////////////
}
