/**
 * $Id: HLProtocol.java,v 1.20 2001/10/08 22:03:28 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.OutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Calendar;
import java.util.StringTokenizer;
import java.text.SimpleDateFormat;
import java.text.NumberFormat;

import redlight.utils.SignUtils;
import redlight.utils.TextUtils;
import redlight.utils.DebuggerOutput;
import redlight.utils.ToArrayConverters;
import redlight.utils.Sortable;

/**
 * This class specifies Hotline protocol constants and 
 * encapsulates Hotline protocol packets using inner 
 * classes.<p>
 *
 * This file started out as a straightforward rip from hx.
 * And although it's evolved since then, props where props are
 * due.
 */
public class HLProtocol {

    private SimpleDateFormat dateFormatter;

    public static final int SOCK_SEND_BUF_SIZE = 16 * 1024;
    public static final int SOCK_RECEIVE_BUF_SIZE = 16 * 1024;

    /**
     * Whether or not to dump hexdumps from the packet payloads to
     * standard output as they are read or written.  
     */
    public static final boolean hexdumpPayload = false;

    /**
     * If this is false, the file send routines will wait
     * for the receiving end to acknowledge reception of the
     * last block. This is the default, and the way Hotline
     * does it. It's the safest, and pretty much guarantees that
     * a transfer was actually completed. It's also very slow, 
     * especially when transferring many little files. 
     *
     * So Red Light supports asynchronous file transfers. This way,
     * the transfer routines just assume that everything is OK when
     * the last block has been written. This is slightly less safe,
     * but much faster. The problem with this is that it's
     * incompatible with Hotline software, and you'll get "Transfer
     * timeout!" errors from Hotline servers that you upload to. Or,
     * if you are running a Red Light server with this set to true,
     * your users will get them when they download.  
     */
    public static boolean asynchronousFileTransfers = false;

    /**
     * The default TCP port for a Hotline tracker.
     */
    public static final short HTRK_TCPPORT = 5498;
    public static final short HTRK_UDPPORT = 5499;
    /**
     * The default TCP port for a Hotline server.
     */
    public static final short HTLS_TCPPORT = 5500;
    static final short HTXF_TCPPORT = 5501;

    /**
     * The Hotline dir separator; usually :
     */
    public static final char DIR_SEPARATOR = ':';

    static final String HTLC_MAGIC = "TRTPHOTL\0\1\0\2";
    static final short HTLC_MAGIC_LEN = 12;
    static final String HTLS_MAGIC = "TRTP\0\0\0\0";
    static final short HTLS_MAGIC_LEN = 8;
    static final String HTRK_MAGIC = "HTRK\0\1";
    static final short HTRK_MAGIC_LEN = 6;
    static final String HTXF_MAGIC  = "HTXF";
    static final short HTXF_MAGIC_LEN = 4;
    static final int HTXF_MAGIC_INT = 0x48545846;
 
    static final short SIZEOF_PACKETHEADER = (22);
    static final short SIZEOF_DATACOMPONENT = (4);

    /* Should sort this out properly some time. */

    static final int HTLC_HDR_USER_CHANGE = 0x00000130;
    static final int HTLC_HDR_CHAT = 0x00000069;
    static final int HTLC_HDR_LOGIN = 0x0000006b;
    static final int HTLC_HDR_MSG = 0x0000006c;
    static final int HTLC_HDR_NEWS_POST = 0x00000067;
    static final int HTLC_HDR_USER_GETINFO = 0x0000012f;
    static final int HTLC_HDR_USER_KICK = 0x0000006e;
    static final int HTLC_HDR_DIR_LIST = 0x000000c8;
    static final int HTLC_HDR_ACCOUNT_CREATE = 0x0000015e;
    static final int HTLC_HDR_ACCOUNT_DELETE = 0x0000015f;
    static final int HTLC_HDR_ACCOUNT_READ = 0x00000160;
    static final int HTLC_HDR_ACCOUNT_MODIFY = 0x00000161;
    static final int HTLC_HDR_FILE_GET = 0x000000ca;
    static final int HTLC_HDR_FILE_PUT = 0x000000cb;
    static final int HTLC_HDR_DIR_CREATE = 0x000000cd;
    static final int HTLC_HDR_FILE_GETINFO = 0x000000ce;
    static final int HTLC_HDR_USER_LIST = 0x0000012c;
    static final int HTLC_HDR_NEWS_GET = 0x00000065;
    static final int HTLC_HDR_FILE_SETINFO = 0x000000cf;
    static final int HTLC_HDR_FILE_DELETE = 0x000000cc;
    static final int HTLC_HDR_FILE_MOVE = 0x000000d0;
    static final int HTLC_HDR_FILE_MAKE_ALIAS = 0x000000d1;
    static final int HTLC_HDR_PRIVCHAT_CREATE = 0x00000070;
    static final int HTLC_HDR_PRIVCHAT_INVITE = 0x00000071;
    static final int HTLC_HDR_PRIVCHAT_DECLINE = 0x00000072;
    static final int HTLC_HDR_PRIVCHAT_JOIN = 0x00000073;
    static final int HTLC_HDR_PRIVCHAT_LEAVE = 0x00000074;
    static final int HTLC_HDR_PRIVCHAT_SUBJECT = 0x00000078;
    static final int HTLC_HDR_AGREE = 0x0079;

    static final short HTLC_DATA_ICON = 0x0068;
    static final short HTLC_DATA_NICK = 0x0066;
    static final short HTLC_DATA_OPTION = 0x006d;
    static final short HTLC_DATA_LOGIN = 0x0069;
    static final short HTLC_DATA_PASSWORD = 0x006a;
    static final short HTLC_DATA_SOCKET = 0x0067;
    static final short HTLC_DATA_CHAT = 0x0065;
    static final short HTLC_DATA_MSG = 0x0065;
    static final short HTLC_DATA_NEWS_POST = 0x0065;
    static final short HTLC_DATA_LISTDIR = 0x00ca;
    static final short HTLC_DATA_FILE = 0x00c9;
    static final short HTLC_DATA_DIR = 0x00ca;
    static final short HTLC_DATA_RFLT = 0x00cb;
    static final short HTLC_DATA_XFERSIZE = 0x006c;
    static final short HTLC_DATA_BAN = 0x0071;
    static final short HTLC_DATA_CHAT_REF = 0x0072;
    static final short HTLC_DATA_CHAT_SUBJECT = 0x0073;
    static final short HTLC_DATA_FILENAME = 0x0000007b;
    static final short HTLC_DATA_RESUME = 0x000000cc;
    static final short HTLC_DATA_FILE_RENAME = 0x00d3;
    static final short HTLC_DATA_DIR_RENAME = 0x00d4;

    static final int HTLS_HDR_USER_LEAVE = 0x0000012e;
    static final int HTLS_HDR_USER_CHANGE = 0x0000012d;
    static final int HTLS_HDR_TASK = 0x00010000;
    static final int HTLS_HDR_NEWS_POST = 0x00000066;
    static final int HTLS_HDR_MSG = 0x00000068;
    static final int HTLS_HDR_CHAT = 0x0000006a;
    static final int HTLS_HDR_AGREEMENT = 0x0000006d;
    static final int HTLS_HDR_POLITEQUIT = 0x0000006f;
    static final int HTLS_HDR_CHAT_INVITE = 0x00000071;
    static final int HTLS_HDR_CHAT_USER_CHANGE = 0x00000075;
    static final int HTLS_HDR_CHAT_USER_LEAVE = 0x00000076;
    static final int HTLS_HDR_CHAT_SUBJECT = 0x00000077;
    static final int HTLS_HDR_USERDISCONNECT = 0x0000012e;
    static final int HTLS_HDR_QUEUEPOS = 0x00d3;

    static final short HTLS_DATA_SOCKET = 0x0067;
    static final short HTLS_DATA_LOGIN = 0x0069;
    static final short HTLS_DATA_PASSWORD = 0x006a;
    static final short HTLS_DATA_PRIVILEGES = 0x006e;
    static final short HTLS_DATA_TASKERROR = 0x0064;
    static final short HTLS_DATA_OPTION = 0x006d;
    static final short HTLS_DATA_ICON = 0x0068;
    static final short HTLS_DATA_COLOUR = 0x0070;
    static final short HTLS_DATA_NICK = 0x0066;
    static final short HTLS_DATA_USER_LIST = 0x012c;
    static final short HTLS_DATA_NEWS = 0x0065;
    static final short HTLS_DATA_AGREEMENT = 0x0065;
    static final short HTLS_DATA_USER_INFO = 0x0065;
    static final short HTLS_DATA_CHAT = 0x0065;
    static final short HTLS_DATA_MSG = 0x0065;
    static final short HTLS_DATA_FILE_LIST = 0x00c8;
    static final short HTLS_DATA_FILE_ICON = 0x00d5;
    static final short HTLS_DATA_FILE_TYPE = 0x00cd;
    static final short HTLS_DATA_FILE_CREATOR = 0x00ce;
    static final short HTLS_DATA_FILE_SIZE = 0x00cf;
    static final short HTLS_DATA_FILE_NAME = 0x00c9;
    static final short HTLS_DATA_FILE_COMMENT = 0x00d2;
    static final short HTLS_DATA_FILE_DATE_CREATED = 0x00d0;
    static final short HTLS_DATA_FILE_DATE_MODIFIED = 0x00d1;
    static final short HTLS_DATA_HTXF_REF = 0x006b;
    static final short HTLS_DATA_HTXF_SIZE = 0x006c;
    static final short HTLS_DATA_HTXF_QUEUEPOS = 0x0074;
    static final short HTLS_DATA_HTXF_RFLT = 0x00cb;
    static final short HTLS_DATA_CHAT_REF = 0x0072;
    static final short HTLS_DATA_CHAT_SUBJECT = 0x0073;

    static final short HTLS_DATA_VERSION = 0x00a0;
    static final short HTLS_DATA_AUTO_AGREE = 0x009a;

    static final short DL_WHOLE_FILE = 0;
    static final short DL_DATA_FORK = 2;

    /* HOPE (Hotline Open Protocol Extensions) */

    static final short HTLX_DATA_APP_ID = 0xe01;
    static final short HTLX_DATA_APP_STRING = 0xe02;
    static final short HTLX_DATA_SESSION_KEY = 0xe03;
    static final short HTLX_DATA_MAC_ALGORITHM = 0xe04;

    public HLProtocol() {

        dateFormatter = new SimpleDateFormat ("EEE, MMM dd, yyyy, kk:mm:ss");

    }

    /**
     * Objectified Hotline packet. A Hotline packet consists of:<p>
     *
     * - A {@link HLProtocol.PacketHeader}.<br>
     * - Zero or more {@link HLProtocol.DataComponent}'s.<br>
     *
     * The packet header is accessible via the 'header' member.
     * DataComponents are accessible via the 'hasMoreComponents'
     * and 'nextComponent' methods of the Packet class.
     */
    class Packet {

        /* The header of this packet. */

        PacketHeader header;

        /* Zero or more DataComponents. */

        DataComponent[] dataComponents;

        /** 
         * Reads a Hotline packet from the given DataInputStream
         * and initializes the header and dataComponents fields.
         * @param input the DataInputStream to read from.
         */
        Packet(DataInputStream input) throws IOException {

            header = new PacketHeader(input);

            if(header.len >= 2) {

                byte[] payload = new byte[(int) header.len - 2];
                input.readFully(payload, 0, payload.length);

                if(hexdumpPayload) {

                    DebuggerOutput.debug("Dumping payload for incoming packet:");
                    DebuggerOutput.dumpBytes(payload, 8);

                }

                ByteArrayInputStream bis = new ByteArrayInputStream(payload);
                DataInputStream dis = new DataInputStream(bis);

                /* The following code incorporates a workaround for a
                   Tube 1.0b2 bug. The problem is that Tube 1.0b2
                   sometimes sends packets where the header.hc field
                   does not match the actual number of components. So
                   we simply read until EOF and allow the creation of
                   null components, then prune that stuff afterwards. */

                DataComponent[] specifiedComponents = 
                    new DataComponent[header.hc];

                short actualComponentCount = 0;

                for(int i = 0; i < specifiedComponents.length; i++) {
                    
                    specifiedComponents[i] = 
                        ComponentFactory.createComponent(dis);

                    if(specifiedComponents[i] != null)
                        actualComponentCount++;

                }

                dataComponents = new DataComponent[actualComponentCount];

                int j = 0;

                for(int i = 0; i < specifiedComponents.length; i++)
                    if(specifiedComponents[i] != null)
                        dataComponents[j++] = specifiedComponents[i];
                
                header.hc = actualComponentCount;

            }

        }

        /**
         * Creates a Packet with the given values for later 
         * writing with the write() method.
         * @param type the type of this packet.
         * @param trans the transaction number of this packet.
         * @param dataComponents the data components in this
         * packet.
         */
        Packet(int type, int trans, DataComponent[] dataComponents) {

            if(dataComponents == null)
                dataComponents = new DataComponent[0];

            header = new PacketHeader(0, type, trans, 0, 2 + (SIZEOF_DATACOMPONENT * dataComponents.length), 0);
            
            /* Compute the total length of this packet. */

            for(int k = 0; k < dataComponents.length; k++)
                if(dataComponents[k].data != null)
                    header.len += dataComponents[k].data.length;
            
            header.len2 = header.len;
            header.hc = (short) dataComponents.length;
         
            this.dataComponents = dataComponents;
            
        }

        /**
         * Returns true while there are more DataComponent's in this
         * packet.         
         */
        boolean hasMoreComponents() {

            return(header.hc > 0 ? true : false);

        }

        /**
         * Returns the next DataComponent for this packet.
         */
        DataComponent nextComponent() {

            return dataComponents[--header.hc];

        }

        /**
         * Writes the packet to the given DataOutputStream.
         * @param output the DataOutputStream to write to.
         */
        void write(DataOutputStream output) throws IOException {

            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            DataOutputStream dos = new DataOutputStream(bos);

            for(int k = 0; k < dataComponents.length; k++)
                dataComponents[k].write(dos);

            dos.flush();

            byte[] payload = bos.toByteArray();

            if(hexdumpPayload) {

                DebuggerOutput.debug("Dumping payload for outgoing packet:");
                DebuggerOutput.dumpBytes(payload, 8);
                
            }

            synchronized(output) {

                header.write(output);
                output.write(payload);                
                output.flush();

            }

        }

        /**
         * Returns a String representation of this packet 
         * (for debugging purposes). Slow.
         * @return String.
         */
        public String toString() {

            if(!DebuggerOutput.on)
                return "Packet[debugging output disabled]";
            
            String s = "Packet[" + header.toString() + ", ";

            if(dataComponents == null) {

                s += "null]";

            } else {

                s += "[";
                for(int i = 0; i < dataComponents.length; i++) {

                    if(dataComponents[i] != null) {

                        s += dataComponents[i].toString();
                        
                        if(i + 1 < dataComponents.length)
                            s += ", ";

                    }

                }
                s += "]";

            }

            s += "]";

            return s;

        }

    }

    /**
     * The header of a Hotline packet, describing the length
     * and the kind of packet among other things.
     */
    class PacketHeader {

        /**
         * Whether this is a reply to a request or not.
         */
	public int cls;

	/**
	 * Kind of packet.
	 */
	public int id;

	/**
	 * Transaction ID.
	 */
	public int trans;

	/**
	 * Error flag.
	 */
	public int isError;

	/**
	 * Length of packet.
	 */
	public long len;

	/**
	 * Length of packet.
	 */
	public long len2;

	/**
	 * Number of data components in this packet.
	 */
	public short hc;

	/**
	 * Constructs empty.
	 */
	public PacketHeader() {}

	/**
	 * Constructs with specified values.
	 * @param t cls
	 * @param i type of packet
	 * @param tr transaction ID of packet
	 * @param isError error flag
	 * @param l length of packet
	 * @param hc number of data packets
	 */
	public PacketHeader(int t, int i, int tr, int isError, int l, int c) {

	    cls = t; id = i; trans = tr; this.isError = isError; len = len2 = l; hc = (short) c;

	}

	/**
	 * Constructs from specified DataInputStream.
	 * @param din DataInputStream to read from.
	 */
	public PacketHeader(DataInputStream din) throws IOException {

	    read(din);

	}

        /**
         * Reads from a DataInputStream.
         * @param din DataInputStream to read from.
         * @throws IOException if the packet does not appear to
         * be a Hotline packet.
         */
	public void read(DataInputStream din) throws IOException {

		if (din == null) DebuggerOutput.debug("DataInputStream called with NULL");
	    int temp = din.readInt();
	    cls = temp >> 16;
	    id = (temp == HTLS_HDR_TASK ? HTLS_HDR_TASK : (temp & 0xffff));
	    trans = din.readInt();
	    isError = din.readInt();
	    len = din.readInt();
	    len2 = din.readInt();

            hc = 0;

            if(len != len2)
                throw new IOException("Not a Hotline packet");

            if (len >= 2)
                hc = din.readShort();

	}

	/**
	 * Writes to a DataOutputStream.
	 * @param dos DataOutputStream to write to.
	 */
	public void write(DataOutputStream dos) throws IOException {

            int temp = ((int) cls << 16) | id;
	    dos.writeInt(temp);
	    dos.writeInt(trans);
	    dos.writeInt(isError);
	    dos.writeInt((int) len);
	    dos.writeInt((int) len2);
	    dos.writeShort(hc);
	    dos.flush();

	}

        public String toString() {

            return "PacketHeader[cls = 0x" + Integer.toHexString(this.cls) + ", id = 0x" + Integer.toHexString(this.id) + ", trans = " + this.trans + ", isError = " + this.isError + ", len = " + this.len + ", len2 = " + this.len2 + ", hc = " + this.hc + "]";
            
        }

    }

    /**
     * Most Hotline packets will contain one or more of these.
     * They are essentially typed data packets, consisting of 
     * a type, a length, and the data itself.<p>
     * 
     * For some types of data there are specialized subclasses of this
     * class, e.g. {@link HLProtocol.UserListComponent} and {@link
     * DateComponent} providing an easier interface to the data
     * then a byte array. 
     */
    class DataComponent implements Cloneable {

	/**
	 * The type of this data component.
	 */
	public short type;

	/**
	 * The data in this data component.
	 */
	public byte[] data;

	/**
	 * Constructs a DataComponent with specified type and no contents.
         * @param t the type of this DataComponent.
	 */
        DataComponent(short t) {

            type = t;
            data = new byte[0];

        }

	/**
	 * Constructs a DataComponent with specified type and contents.
         * @param t the type of this DataComponent.
         * @param d the data in this DataComponent.
	 */
	DataComponent(short t, byte[] d) {

            type = t;
            data = (d != null ? d : new byte[0]);

	}

        /**
         * Constructs a DataComponent of the given type from the given
         * DataInputStream. The input stream should point at the beginning
         * of the 16-bit size value for this component, ie. skipping the
         * 16-bit type value.
         * @param t the type of this DataComponent.
         * @param din DataInputStream to read from.  
         */
	DataComponent(short t, DataInputStream din) throws IOException {

            type = t;
	    data = new byte[(int) din.readChar()];
	    din.readFully(data, 0, data.length);

        }

	/**
	 * Constructs by reading from a DataInputStream.
	 * @param din DataInputStream to read the hx_data_hdr from.
	 */
	DataComponent(DataInputStream din) throws IOException {

	    this(din.readShort(), din);

	}

	/**
	 * Writes this DataComponent to a DataOutputStream.
	 * @param dos DataOutputStream to write hx_data_hdr to.
	 */ 
	void write(DataOutputStream dos) throws IOException {

	    dos.writeChar(type);
	    dos.writeChar((char) data.length);
	    dos.write(data, 0, data.length);
	    dos.flush();

	}

        public String toString() {

            String dataString = new String("");

            switch(type) {

            case HTLS_DATA_FILE_SIZE:
            case HTLS_DATA_HTXF_SIZE:
            case HTLS_DATA_OPTION:
            case HTLS_DATA_ICON:
            case HTLS_DATA_COLOUR:
            case HTLS_DATA_SOCKET:
            case HTLS_DATA_VERSION:
                dataString = new Integer(ToArrayConverters.byteArrayToInt(data)).toString();
                break;

            case HTLS_DATA_FILE_DATE_CREATED:
            case HTLS_DATA_FILE_DATE_MODIFIED:

                try {

                    ByteArrayInputStream bis = new ByteArrayInputStream(data);
                    DataInputStream dis = new DataInputStream(bis);
                    dataString = new DateComponent(type, dis).toString();

                } catch(IOException e) {}

                break;

            case HTLS_DATA_CHAT_REF:
            case HTLS_DATA_HTXF_REF:
                dataString = "0x" + Integer.toHexString(ToArrayConverters.byteArrayToInt(data));
                break;

            default:
                dataString = new String(data);

                dataString = TextUtils.findAndReplace(dataString, "\r", "\\r");
                dataString = TextUtils.findAndReplace(dataString, "\n", "\\n");
                
                if(dataString.length() > 40) {
                    
                    int restLen = dataString.length();
                    dataString = dataString.substring(0, 40);
                    dataString += "... (" + restLen + ")";

                }
                break;
                
            }
            
            return "DataComponent[type = 0x" + Integer.toHexString(type) + ", data = " + dataString + "]";

        }

    }

    /**
     * Describes a user as a part of a user list.
     */
    public class UserListComponent extends DataComponent {

	/**
	 * The sock of the user.
	 */
	public int sock;

	/**
	 * The icon number of the user.
	 */
	public int icon;

	/**
	 * The color (status) of the user.
	 */
	public short clr;

	/** 
	 * The nick of the user.
	 */
	public String nick = null;

        /**
         * Constants to mask the {@link #clr} field.
         */
        public static final short HAS_BEEN_IDLE        = 0x01;
        public static final short CAN_DISCONNECT_USERS = 0x02;

        /**
         * Constructs a UserListComponent with the given values.
         * @param s the sock of the user.
         * @param i the icon number of the user.
         * @param c the color of the user.
         * @param n the nickname of the user.
         */
        UserListComponent(int s, int i, short c, String n) {

            super(HTLS_DATA_USER_LIST);

            sock = s;
            icon = i;
            clr = c;
            nick = n;

            construct();

        }

        /**
         * Constructs a UserListComponent from an input stream
         * from which the type short has already been read.
         * @param in the input stream to read from.
         */
        UserListComponent(DataInputStream is) throws IOException {

            super(HTLS_DATA_USER_LIST, is);

	    DataInputStream din = 
                new DataInputStream(new ByteArrayInputStream(data));

	    sock = (int) din.readChar();
	    icon = (int) din.readChar();
	    clr = din.readShort();
	    byte[] n = new byte[(int) din.readChar()];
	    din.readFully(n, 0, n.length);

            nick = new String(n);

        }

        /**
         * Writes this component to the given output stream.
         * @param os the output stream to write to.
         */
        void write(DataOutputStream os) throws IOException {

            construct();

            super.write(os);

        }

        private void construct() {

            try {

                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                DataOutputStream dos = new DataOutputStream(bos);

                dos.writeShort(sock);
                dos.writeShort(icon);
                dos.writeShort(clr);

                if(nick != null) {

                    dos.writeShort(nick.length());
                    dos.write(nick.getBytes());

                }

                data = bos.toByteArray();

            } catch(IOException e) {}

        }

        /**
         * Returns a string representation of this object (the user's
         * nick name).
         * @return nick name.
         */
	public String toString() {

            String s = "UserListComponent[icon: " + icon + ", sock: " + sock + ", name: " + new String(nick);
            String state = null;

            if((clr & CAN_DISCONNECT_USERS) == CAN_DISCONNECT_USERS)
                state = "CAN_DISCONNECT_USERS";

            if((clr & HAS_BEEN_IDLE) == HAS_BEEN_IDLE) {

                if(state != null)
                    state += ", ";
                else
                    state = "";

                state += "HAS_BEEN_IDLE";

            }

            if(state != null)
                s += ", state: " + state;

            s += "]";

	    return s;

	}

    }

    /**
     * Resume file transfer component.
     */
    class ResumeTransferComponent extends DataComponent {

        /**
         * The chunks described by this transfer block.
         * (e.g. key = DATA, value = <size of data chunk>).
         */
        Hashtable chunks;
        
        /**
         * Constructs a ResumeTransferComponent with default
         * values.
         */
	ResumeTransferComponent() {
            
            this(0, 0);

        }

        /**
         * Constructs a ResumeTransferComponent with the
         * given values.
         */
	ResumeTransferComponent(long dataSize, long macrSize) {

            super(HTLS_DATA_HTXF_RFLT);

            chunks = new Hashtable();
            chunks.put("DATA", new Long(dataSize));
            chunks.put("MACR", new Long(macrSize));

            construct();

        }

	/**
	 * Interprets a given DataComponent as a ResumeTransferComponent
         * and initializes the object with the resulting values.
         * @throws IllegalArgumentException if the supplied 
         * input stream does not point at a ResumeTransferComponent.
	 */
	ResumeTransferComponent(DataInputStream is) throws IOException {

            super(HTLS_DATA_HTXF_RFLT, is);

            chunks = new Hashtable();
            chunks.put("DATA", new Long(0));
            chunks.put("MACR", new Long(0));
            
	    DataInputStream din = 
                new DataInputStream(new ByteArrayInputStream(data));

            byte[] id = new byte[4];
            din.readFully(id, 0, 4);

            if(new String(id).equals("RFLT")) {

                byte[] not_interesting = new byte[38];
                din.readFully(not_interesting, 0, 38);

                int available = data.length - (4 + 38);

                while(available > 0) {

                    /* Use 64 bit long for chunk sizes here. This is a
                       protocol extension (or I haven't seen it
                       yet). */

                    din.readFully(id, 0, 4);
                    int low = din.readInt();
                    int high = din.readInt();
                    long chunkSize = (long) high << 32 | (long) low ;
                    din.readInt();
                    chunks.put(new String(id), new Long(chunkSize));
                    available -= 16;
                    
                }

                construct();

            } else {

                throw new IllegalArgumentException("not a ResumeTransferComponent");

            }

	}

        private void construct() {

            try {

                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                DataOutputStream dos = new DataOutputStream(bos);
                
                dos.write(new String("RFLT").getBytes());
                dos.writeShort(1);
                dos.write(new byte[34], 0, 34);
                dos.writeShort(2);
                dos.write(new String("DATA").getBytes());
                long ds = ((Long) chunks.get("DATA")).longValue();
                int low = (int) (ds & 0xffffffffL);
                int high = (int) (ds >> 32);
                dos.writeInt(low);
                dos.writeInt(high);
                dos.writeInt(0);
                dos.write(new String("MACR").getBytes());
                long rs = ((Long) chunks.get("MACR")).longValue();
                low = (int) (rs & 0xffffffffL);
                high = (int) (rs >> 32);
                dos.writeInt(low);
                dos.writeInt(high);
                dos.writeInt(0);
                dos.flush();
                
                data = bos.toByteArray();
                
            } catch(IOException e) {}

        }

        public String toString() {

            return "ResumeTransferComponent[DATA = " + 
                (Long) chunks.get("DATA") + ", MACR = " +
                (Long) chunks.get("MACR") +
                "]";

        }

    }

    /**
     * Describes a file as a part of a file list.
     */
    public class FileListComponent extends DataComponent implements Cloneable, Sortable {

	/**
	 * The (MacOS) file type of the file.
	 */
	public String fileType;

	/**
	 * The (MacOS) file creator of the file.
	 */
	public String fileCreator;

	/**
	 * The size of the file.
	 */
	public long fileSize;

        /**
         * Unknown.
         */
	public int unknown;

	/**
	 * The name of the file.
	 */
	public String fileName;

        /**
         * Creates a file list component with the specified parameters.
         * @param fName the file name.
         * @param fType 4 byte file type info.
         * @param fCreator 4 byte file creator info.
         * @param fSize the file size.
         * @param fExtra (unknown).
         */
        public FileListComponent(String fName,
                                 String fType, 
                                 String fCreator,
                                 long fSize,
                                 int fExtra) {

            super(HTLS_DATA_FILE_LIST);

            if(fType.length() != 4)
                throw new IllegalArgumentException("File type must be 4 bytes.");

            if(fCreator.length() != 4)
                throw new IllegalArgumentException("File creator must be 4 bytes.");

            fileName = fName;
            fileType = fType;
            fileCreator = fCreator;
            fileSize = fSize;
            unknown = fExtra;

            construct();

        }

	/**
	 * Interprets the given DataComponent as a FileListComponent
         * and initializes the FileListComponent with those values.
         * @param dh the DataComponent to interpret.
	 */
	public FileListComponent(DataInputStream is) throws IOException {

            super(HTLS_DATA_FILE_LIST, is);

	    DataInputStream din = 
                new DataInputStream(new ByteArrayInputStream(data));

            byte[] ftype = new byte[4];
            byte[] fcreator = new byte[4];
            din.read(ftype, 0, 4);
            din.read(fcreator, 0, 4);
            fileSize = din.readInt();
            unknown = din.readInt();
            byte[] fname = new byte[din.readInt()];
            din.readFully(fname, 0, fname.length);

            fileType = new String(ftype);
            fileCreator = new String(fcreator);
            fileName = new String(fname);

	}

        private void construct() {

            try {

                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                DataOutputStream dos = new DataOutputStream(bos);
                
                byte[] ftype = fileType.getBytes();
                byte[] fcreator = fileCreator.getBytes();
                dos.write(ftype, 0, 4);
                dos.write(fcreator, 0, 4);
                dos.writeInt((int) fileSize);
                dos.writeInt(unknown);
                dos.writeInt(fileName.length());
                dos.write(fileName.getBytes());
                dos.flush();
                
                data = bos.toByteArray();
                
            } catch(IOException e) {}
            
        }

	public Object clone() throws CloneNotSupportedException {

	    return super.clone();

	}

	public String toString() {

	    return "FileListComponent[" + fileType + ", " + (fileType.equals("fldr") ? "" : fileCreator + ", ") + fileSize + ", " + fileName + "]";

	}

        public boolean smallerThan(Sortable other) {

            if(other instanceof HLProtocol.FileListComponent) {
                
                HLProtocol.FileListComponent otherFile = 
                    (HLProtocol.FileListComponent) other;

                /* Directories precede files. */
                
                if(fileType.equals("fldr") && !otherFile.fileType.equals("fldr"))
                    return true;
                
                if(!fileType.equals("fldr") && otherFile.fileType.equals("fldr"))
                    return false;
                
                return fileName.compareToIgnoreCase(otherFile.fileName) < 0;

            } else {

                return fileName.compareToIgnoreCase(other.toString()) < 0;

            }
        
        }

    }

    /**
     * Encodes a Hotline date.
     */
    class DateComponent extends DataComponent { 

        Calendar date;

        /**
         * @param type one of HTLS_DATA_FILE_DATE_CREATED or
         * HTLS_DATA_FILE_DATE_MODIFIED.
         * @param when the date to encode.
         */
        public DateComponent(short type, Calendar when) {

            super(type);
            date = when;
            data = HLProtocol.calendarToHotlineDate(date);
            
        }
        
        /**
         * @param type one of HTLS_DATA_FILE_DATE_CREATED or
         * HTLS_DATA_FILE_DATE_MODIFIED.
         * @param in the DataInputStream to construct this object from.
         */
        public DateComponent(short type, DataInputStream in) throws IOException {

            super(type, in);
            
            date = HLProtocol.hotlineDateToCalendar(data);

        }

        public String toString() {

            return "DateComponent[type = 0x" + Integer.toHexString(type) + ", " + dateFormatter.format(date.getTime()) + "]";

        }

    }

    /**
     * The header which precedes a file transfer, for handshaking
     * purposes among other things. Is followed by a FileTransferInfo
     * block.  */
    class FileTransferHeader {

        /**
         * The ID for this transfer.
         */
	public int ref;

        /**
         * The length of this transfer header.
         */
        public long len;

        /**
         * Unknown.
         */
        public int unknown = 0;

        /**
         * Constructs a transfer header with the specified
         * transfer ID and header length.
         * @param r the transfer ID.
         * @param l the transfer header length.
         */
        FileTransferHeader(int r, long l) {

            ref = r;
            len = l;

        }

        FileTransferHeader(DataInputStream din) throws IOException {

            read(din);

        }

        /**
         * Writes the transfer header to the given DataOutputStream.
         * @param dos the DataOutputStream to write to.
         */
	void write(DataOutputStream dos) throws IOException {

	    dos.writeInt(HLProtocol.HTXF_MAGIC_INT);
	    dos.writeInt(ref);

            /* This is for 63 bit file size support. Don't know
               if it clobbers anything important. */

            int low = (int) (len & 0xffffffffL);
            int high = (int) (len >> 32);
            dos.writeInt(low);
            dos.writeInt(high);
	    dos.flush();

	}  

        void read(DataInputStream din) throws IOException {

            int magic = din.readInt();

            if(magic != HLProtocol.HTXF_MAGIC_INT)
                throw new IOException("not a FileTransferHeader");

            ref = din.readInt();
            int low = din.readInt();
            int high = din.readInt();
            len = (long) high << 32 | (long) low ;

        }

        public String toString() {

            return "FileTransferHeader[ref = 0x" + Integer.toHexString(ref) + ", len = " + len + ", unknown = " + unknown + "]";

        }

    }

    /**
     * This block describes some meta-data about the file being transferred.
     * Quite a bit of data is ignored / assumed zero when reading /
     * writing this info packet, because I don't know it's purpose.
     */
    class FileTransferInfo {

        String fileName, fileType, fileCreator, fileComment;
        Calendar creationDate, modificationDate;
        int numberOfBlocks;
        int finderFlags;
        byte[] data;

        public FileTransferInfo(String fName, 
                                String fType,
                                String fCreator,
                                String fComment,
                                Calendar created,
                                Calendar modified,
                                int flags) throws IOException {

            if(fComment.length() > 255)
                throw new IllegalArgumentException("comment length must be <= 255.");

            if(fType.length() != 4)
                throw new IllegalArgumentException("file type must be 4 bytes.");

            if(fCreator.length() != 4)
                throw new IllegalArgumentException("file creator must be 4 bytes.");

            fileName = fName;
            fileType = fType;
            fileCreator = fCreator;
            fileComment = fComment;
            creationDate = created;
            modificationDate = modified;
            finderFlags = flags;

            construct();

        }

        public FileTransferInfo(DataInputStream originalInput) throws IOException {

            DataInputStream input = originalInput;
            ByteArrayOutputStream monitorBos = new ByteArrayOutputStream();
            DataOutputStream monitorDos = new DataOutputStream(monitorBos);

            byte[] filp = new byte[40];
            originalInput.readFully(filp, 0, 40);
            monitorDos.write(filp);
            ByteArrayInputStream bis = new ByteArrayInputStream(filp);
            input = new DataInputStream(bis);

            /* Get the FILP magic. */

            byte[] magic = new byte[4];
            input.readFully(magic, 0, 4);

            if(!new String(magic).equals("FILP"))
                throw new IOException("Not a FileTransferInfo - bad magic.");

            /* Get some unknown parameter. */

            int parameter = input.readShort();

            if(parameter != 1)
                throw new IOException("Scared of unknown parameter 1 (is " + parameter + ", expected 1).");

            /* Skip some unknown stuff. */

            input.skipBytes(14);

            /* Get another unknown parameter. Number of blocks?? 
               I think this is the number of blocks
               that are being sent (e.g. 2 for FILP + DATA or 3 for
               FILP + DATA + MACR). */

            numberOfBlocks = input.readInt();

            if(!(numberOfBlocks == 3 || numberOfBlocks == 2))
                throw new IOException("Scared of numberOfBlocks value (is " + numberOfBlocks + ", expected 2 or 3).");

            /* Get the INFO magic. */

            input.readFully(magic, 0, 4);

            if(!new String(magic).equals("INFO"))
                throw new IOException("Got " + new String(magic) + ", expected INFO.");

            /* Skip some unknown stuff. */

            input.skipBytes(8);

            /* Get the remaining header size. */

            int headerRemaining = input.readInt();

            if(headerRemaining > 64 * 1024)
                throw new IOException("Scared of too large FileTransferInfo size (" + headerRemaining + ").");

            /* Read the remaining header, up until the DATA block. */

            byte[] amac = new byte[headerRemaining];
            originalInput.readFully(amac, 0, headerRemaining);
            monitorDos.write(amac);
            monitorDos.flush();
            data = monitorBos.toByteArray();

            if(hexdumpPayload) {

                DebuggerOutput.debug("Dumping incoming file transfer info:");
                DebuggerOutput.dumpBytes(data, 8);

            }
            
            bis = new ByteArrayInputStream(amac);
            input = new DataInputStream(bis);

            /* Get the AMAC / MWIN magic. */

            input.readFully(magic, 0, 4);

            if(!(new String(magic).equals("AMAC") ||
                 new String(magic).equals("MWIN")))
                throw new IOException("Got " + new String(magic) + ", expected AMAC or MWIN.");

            /* Get the type and creator info. */

            input.readFully(magic, 0, 4);
            fileType = new String(magic);
            input.readFully(magic, 0, 4);
            fileCreator = new String(magic);

            /* Skip some unknown stuff. */

            input.skipBytes(6);

            /* Skip an unknown parameter (perhaps Finder flags?
               21 for an SimpleText, 0 and 1 for some other files).*/

            finderFlags = input.readUnsignedByte();

            /* Skip more unknown stuff. */

            input.skipBytes(33);

            /* Get the creation and modification date. */

            byte[] date = new byte[8];
            input.readFully(date, 0, 8);
            creationDate = HLProtocol.hotlineDateToCalendar(date);
            input.readFully(date, 0, 8);
            modificationDate = HLProtocol.hotlineDateToCalendar(date);

            /* Get the file name. */

            headerRemaining -= 60;
            int filenameLength = input.readInt();

            if(filenameLength > headerRemaining)
                throw new IOException("FileTransferInfo is inconsistent, filename exceeds total header length.");

            byte[] fName = new byte[filenameLength];
            input.readFully(fName, 0, filenameLength);
            fileName = new String(fName);
            input.skipBytes(1);

            /* Get the comment. */

            headerRemaining -= filenameLength + 2;
            char commentLength = (char) SignUtils.toUnsigned(input.readByte());

            if(commentLength > headerRemaining)
                throw new IOException("FileTransferInfo is inconsistent, comment length (" + (int) commentLength + ") exceeds total header length (" + headerRemaining + ").");

            byte[] fComment = new byte[commentLength];
            input.readFully(fComment, 0, commentLength);
            fileComment = new String(fComment);

            input.skipBytes(1);

        }

        public void write(DataOutputStream output) throws IOException {

            output.write(data);
            output.flush();

        }

        private void construct() throws IOException {

            ByteArrayOutputStream monitorBos = new ByteArrayOutputStream();
            DataOutputStream output = new DataOutputStream(monitorBos);

            byte[] buf = new byte[33];
            
            output.write(new String("FILP").getBytes());
            output.writeShort(1);
            output.write(buf, 0, 14);
            output.writeInt(3);
            output.write(new String("INFO").getBytes());
            output.write(buf, 0, 8);
            
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            DataOutputStream dos = new DataOutputStream(bos);
            
            dos.write(new String("AMAC").getBytes());
            dos.write(fileType.getBytes());
            dos.write(fileCreator.getBytes());
            dos.write(buf, 0, 6);
            dos.writeByte((byte) finderFlags);
            dos.write(buf, 0, 33);

            byte[] created = HLProtocol.calendarToHotlineDate(creationDate);
            byte[] modified = HLProtocol.calendarToHotlineDate(modificationDate);
            dos.write(created);
            dos.write(modified);

            dos.writeInt(fileName.length());
            dos.write(fileName.getBytes());
            dos.writeByte(0);
            dos.writeByte((short) fileComment.length());
            dos.write(fileComment.getBytes());
            dos.writeByte(0);
            dos.flush();
            
            output.writeInt(bos.toByteArray().length);
            output.write(bos.toByteArray());
            output.flush();
            
            if(hexdumpPayload) {

                DebuggerOutput.debug("Dumping outgoing file transfer header:");
                DebuggerOutput.dumpBytes(monitorBos.toByteArray(), 8);

            }

            data = monitorBos.toByteArray();

        }

    }

    class TrackerPacketHeader {
	public byte[] unknown;
	public short pack;
	public short nservers;
 
	public TrackerPacketHeader(DataInputStream din) throws IOException {
	    read(din);
	}
 
	void read(DataInputStream din) throws IOException {
	    byte[] unknown = new byte[4];
	    din.readFully(unknown, 0, 4);
	    nservers = din.readShort();
	    pack = din.readShort();
	}
    }

    /**
     * Describes a server as gotten from a tracker.
     */
    public class ServerInfo {
	/**
	 * Address of server.
	 */
	public int address;
	/**
	 * Port of server.
	 */
	public char port;
	/**
	 * Number of users at server.
	 */
	public short nusers; 
        /**
         * Unknown. 
         */
	public short unknown;
	/**
	 * The server name.
	 */
	public String name;
	/**
	 * The server description.
	 */
	public String desc;
	/** 
	 * Constructs empty.
	 */
	public ServerInfo() {
	}
	/**
	 * Constructs from a DataInputStream.
	 */
	public ServerInfo(DataInputStream din) throws IOException {

	    read(din);
	}

	void read(DataInputStream din) throws IOException {

	    address = din.readInt();
	    port = (char) din.readShort();
	    nusers = din.readShort();
	    unknown = (short)din.readShort();
	    byte[] n = new byte[(short) (SignUtils.toUnsigned(din.readByte()) & 0xff)];
	    din.readFully(n, 0, n.length);
            
            /* Explicitly specify ISO-8859-1 encoding or we may run
               into converter errors, crashing the thread. Probably
               Sun's or MS's bug ... */

            name = new String(n, "ISO-8859-1");
	    byte[] d = new byte[(short) (SignUtils.toUnsigned(din.readByte()) & 0xff)];
	    din.readFully(d, 0, d.length);
            desc = new String(d, "ISO-8859-1");

	} 

	public String getAddress() {

	    char[] b = ToArrayConverters.intToCharArray(address);
	    return new String((int) b[0] + "." + (int) b[1] + "." + (int) b[2] + "." + (int) b[3]);

	}

	public String toString() {

	    return "ServerInfo[name = " + name + ", description = " + desc + "]";

	}

    }

    /**
     * Describes a Hotline user account.
     */
    public class AccountInfo {

	/**
	 * The nickname for this account.
	 */
	public String nick;

	/**
	 * The login for this account.
	 */
        public String login;

	/**
	 * The password for this account.
	 * Included for completeness only, the Hotline server
	 * never actually sends the password.
	 */
	public String password;

	/**
	 * The privileges for this account:
         * <pre>
         * bit 1: can rename folders
         * bit 2: can delete folders
         * bit 3: can create folders
         * bit 4: can move files
         * bit 5: can rename files
         * bit 6: can download files
         * bit 7: can upload files
         * bit 8: can delete files
         * bit 9: can delete users
         * bit 10: can create users
         * bit 11: reserved
         * bit 12: reserved
         * bit 13: reserved
         * bit 14: can send chat
         * bit 15: can read chat
         * bit 16: can move folders
         * bit 17: cannot be disconnected
         * bit 18: can disconnect users
         * bit 19: can post news
         * bit 20: can read news
         * bit 21: reserved
         * bit 22: reserved
         * bit 23: can modify users
         * bit 24: can read users
         * bit 25: can make aliases
         * bit 26: can view drop boxes
         * bit 27: can comment folders
         * bit 28: can comment files
         * bit 29: don't show agreement
         * bit 30: can use any name      
         * bit 31: can upload anywhere
         * bit 32: can get user info
         * </pre>
	 * {@see redlight.hotline.HLProtocol.AccountInfo#CAN_RENAME_FOLDERS}
	 */
	public long privileges;

        /**
         * The home directory for this account (not strictly a part of
         * the Hotline protocol, and may be null).  
         */
        public String homeDirectory;
        
        /**
         * Constants for masking the {@link #privileges} field.
         */
        public static final long CAN_RENAME_FOLDERS     = 0x00000001L;
        public static final long CAN_DELETE_FOLDERS     = 0x00000002L;
        public static final long CAN_CREATE_FOLDERS     = 0x00000004L;
        public static final long CAN_MOVE_FILES         = 0x00000008L;
        public static final long CAN_RENAME_FILES       = 0x00000010L;
        public static final long CAN_DOWNLOAD_FILES     = 0x00000020L;
        public static final long CAN_UPLOAD_FILES       = 0x00000040L;
        public static final long CAN_DELETE_FILES       = 0x00000080L;
        public static final long CAN_DELETE_ACCOUNTS    = 0x00000100L;
        public static final long CAN_CREATE_ACCOUNTS    = 0x00000200L;
        public static final long CAN_SEND_CHAT          = 0x00002000L;
        public static final long CAN_READ_CHAT          = 0x00004000L;
        public static final long CAN_MOVE_FOLDERS       = 0x00008000L;
        public static final long CANNOT_BE_DISCONNECTED = 0x00010000L;
        public static final long CAN_DISCONNECT_USERS   = 0x00020000L;
        public static final long CAN_POST_NEWS          = 0x00040000L;
        public static final long CAN_READ_NEWS          = 0x00080000L;
        public static final long CAN_MODIFY_ACCOUNTS    = 0x00400000L;
        public static final long CAN_READ_ACCOUNTS      = 0x00800000L;
        public static final long CAN_MAKE_ALIASES       = 0x01000000L;
        public static final long CAN_VIEW_DROPBOXES     = 0x02000000L;
        public static final long CAN_COMMENT_FOLDERS    = 0x04000000L;
        public static final long CAN_COMMENT_FILES      = 0x08000000L;
        public static final long DONT_SHOW_AGREEMENT    = 0x10000000L;
        public static final long CAN_USE_ANY_NAME       = 0x20000000L;
        public static final long CAN_UPLOAD_ANYWHERE    = 0x40000000L;
        public static final long CAN_GET_USER_INFO      = 0x80000000L;

        /**
         * Creates the object with the specified values.
         * @param l login.
         * @param n nickname.
         * @param p password.
         * @param pr privileges.
         * @param h home directory for this account.
         */
        public AccountInfo(String l, String n, String p, long pr, String h) {
            
            login = l; nick = n; password = p; privileges = pr; homeDirectory = h;

        }

        /**
         * Creates the object with empty fields.
         */
        public AccountInfo() {

            this("", "", "", 0, "");

        }

        public String toString() {

            return "HLProtocol.AccountInfo[login = " + login + ", password = <not shown>, nick = " + nick + ", privileges = " + privileges + ", homeDirectory = " + homeDirectory + "]";

        }

    }

    /**
     * Describes info for a file.
     */
    public class FileInfo {

	/**
	 * The name of the file.
	 */
	public String name;

	/**
	 * The creator of the file.
	 */
	public String creator;

	/**
	 * The type of the file.
	 */
	public String type;

	/**
	 * The "icon" of the file (usually 4 character code).
	 */
	public String icon;

	/**
	 * The comment of the file.
	 */
	public String comment;

	/**
	 * The size of the file.
	 */
	public long size;

	/**
	 * The creation date of the file.
	 */
	public Calendar created;

	/**
	 * The date the file was last modified.
	 */
	public Calendar modified;

        public String toString() {

            return "FileInfo[" + name + ", " + creator + ", " + type + ", " + icon + ", " + comment + ", " + size + ", " + dateFormatter.format(created.getTime()) + ", " + dateFormatter.format(modified.getTime()) + "]";

        }

    }

    /**
     * Describes a file. {@see ComponentFactory#createPathComponents}.
     */
    class FileComponent extends DataComponent {

        String file;

        public FileComponent(String file) {

            super(HTLC_DATA_FILE);
            this.file = file;
            construct();

        }

        public FileComponent(DataInputStream din) throws IOException {

            super(HTLC_DATA_FILE, din);
            this.file = new String(data);

        }

        private void construct() {

            data = file.getBytes();

        }

    }

    /**
     * Describes a Hotline path. {@see ComponentFactory#createPathComponents}.
     */
    class PathComponent extends DataComponent {

        public String path;

        public PathComponent(short dataType, String path) {

            super(dataType);
            this.path = path;
            construct();

        }
        
	/**
	 * Interprets a given DataComponent as a PathComponent
         * and initializes the object with the resulting values.
         * @throws IllegalArgumentException if the supplied 
         * input stream does not point at a ResumeTransferComponent.
	 */
	PathComponent(short dataType, DataInputStream dis) throws IOException {

            super(dataType, dis);

            ByteArrayInputStream bis = new ByteArrayInputStream(data);
            DataInputStream is = new DataInputStream(bis);
            
            int dc = (int) is.readChar();
            
            /* Extract all the components from the path and use them
               to initialize the 'path' member field. */
           
            path = "";

            if(dc == 0)
                path = String.valueOf(DIR_SEPARATOR);
           
            while(dc-- > 0) {
                
                short z = is.readShort();  /* Skip, unknown. */
                byte[] el = new byte[is.readUnsignedByte()];
                is.read(el, 0, el.length);
                
                path += DIR_SEPARATOR + new String(el);
                
            }
            
            /* Recreate the bytestream from the just initialized path
               field. Wasteful, might just use the already given
               bytestream. */
            
            construct();
            
        }

        private void construct() {

            try {

                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                DataOutputStream dos = new DataOutputStream(bos);

                StringBuffer buf = new StringBuffer(path);
                StringTokenizer st = new StringTokenizer(path, 
                                                         String.valueOf(DIR_SEPARATOR));

                dos.writeChar((char) st.countTokens());

                /* Write the path piece by piece. */

                while(st.hasMoreTokens()) {
                    
                    String f = st.nextToken();
                    
                    dos.writeShort(0);
                    dos.writeByte((byte) f.getBytes().length);
                    dos.write(f.getBytes(), 0, f.getBytes().length);

                }
                
                dos.flush();
                data = bos.toByteArray();
                
            } catch(IOException e) {}
            
        }

        public String toString() {

            return "PathComponent[type = 0x" + Integer.toHexString(type) + ", " + path + "]";

        }

    }

    /**
     * From an 8 byte array signifying a Hotline date stamp,
     * calculates the date and returns it.<P>
     * The format of a Hotline date stamp is:<P>
     * <PRE>
     * bytes    description
     * 0 - 1    epoch (e.g. 1904)
     * 2 - 3    unknown (zero)
     * 4 - 7    seconds since epoch</PRE>
     * @param hotlineDate an 8 byte Hotline date stamp.
     * @return a Calendar date.
     * @throws IllegalArgumentException if hotlineDate is not an 
     * 8 byte array.
         */
    public static Calendar hotlineDateToCalendar(byte[] hotlineDate) {

        if(hotlineDate.length != 8)
            throw new IllegalArgumentException("Hotline date is not 8 bytes");

        Calendar calendar = Calendar.getInstance();
        byte[] epoch = new byte[2];
        epoch[0] = hotlineDate[0];
        epoch[1] = hotlineDate[1];
            
        /* Don't know what hotlineDate[2] and hotlineDate[3] represent. */
            
        byte[] seconds = new byte[4];
        seconds[0] = hotlineDate[4];
        seconds[1] = hotlineDate[5];
        seconds[2] = hotlineDate[6];
        seconds[3] = hotlineDate[7];
        int nEpoch = ToArrayConverters.byteArrayToInt(epoch);
        long nSeconds = ToArrayConverters.byteArrayToLong(seconds);
        int days = (int) (nSeconds / 86400); /* 86400 = 24 * 3600 */

        calendar.setLenient(true);
        calendar.clear();
        calendar.set(Calendar.YEAR, nEpoch);
        calendar.add(Calendar.DAY_OF_MONTH, days);
        calendar.add(Calendar.SECOND, (int) nSeconds - (days * 86400));

        return calendar;

    }

    /**
     * Translates a Calendar into the corresponding Hotline 
     * date stamp.
     * @param calendar the date to translate.
     * @return 8 byte Hotline date stamp.
     */
    public static byte[] calendarToHotlineDate(Calendar calendar) {
        
        byte[] hotlineDate = new byte[8];
        
        /* The 1.2.3 (Macintosh) version of the Hotline client
           seems broken in how it handles dates, but strangely the
           server is not. In any case this will work for Tube (and
           Tube also works correctly with the 1.2.3 server). */
        
        byte[] epochBytes = ToArrayConverters.shortToByteArray(1970);
        byte[] secondsBytes = ToArrayConverters.intToByteArray((int) (calendar.getTime().getTime() / 1000));
        
        hotlineDate = new byte[8];
        hotlineDate[0] = epochBytes[0];
        hotlineDate[1] = epochBytes[1];
        hotlineDate[2] = 0;
        hotlineDate[3] = 0;
        hotlineDate[4] = secondsBytes[0];
        hotlineDate[5] = secondsBytes[1];
        hotlineDate[6] = secondsBytes[2];
        hotlineDate[7] = secondsBytes[3];
        
        return hotlineDate;
        
    }
        
    static byte[] invert(byte[] b) {
	byte[] encoded = b;

        for(int i = 0; i < b.length; i++)
            encoded[i] = (byte) (255 - encoded[i]);
	
	return encoded;

    }

}

/**
 * Factory methods for obtaining and creating DataComponents.
 */
class ComponentFactory {

    static HLProtocol hlp = new HLProtocol();

    /**
     * Constructs a DataComponent from the given DataInputStream.  For
     * any given DataComponent on the input, this method creates and
     * returns the most specific subclass of DataComponent that it
     * knows about.
     * @param din the input stream to construct from.
     * @return most specific subclass of DataComponent that is possible.
     */
    static HLProtocol.DataComponent createComponent(DataInputStream din) {

        short type;
        
        try {

            /* Read the type of this component, then pass object
               creation on to the appropriate constructors, or create
               a generic DataComponent if there is no specialized 
               object to represent the given type. */
        
            type = din.readShort();
            
            switch(type) {

            case HLProtocol.HTLS_DATA_HTXF_RFLT:
                return hlp.new ResumeTransferComponent(din);

            case HLProtocol.HTLS_DATA_FILE_LIST:
                return hlp.new FileListComponent(din);

            case HLProtocol.HTLS_DATA_USER_LIST:
                return hlp.new UserListComponent(din);

            case HLProtocol.HTLC_DATA_DIR:
            case HLProtocol.HTLC_DATA_DIR_RENAME:
                return hlp.new PathComponent(type, din);

            case HLProtocol.HTLC_DATA_FILE:
                return hlp.new FileComponent(din);

            case HLProtocol.HTLS_DATA_FILE_DATE_CREATED:
            case HLProtocol.HTLS_DATA_FILE_DATE_MODIFIED:
                return hlp.new DateComponent(type, din);

            default:
                return hlp.new DataComponent(type, din);
                
            }

        } catch(EOFException e) {

            /* This is to compensate for a bug in Tube 1.0b2. Tube
               1.0b2 doesn't get the PacketHeader.hc field right. See
               also the PacketHeader constructor. */

            return null;

        } catch(IOException e) {

            e.printStackTrace();

        }
        
        return null;

    }

    /** 
     * Returns an array containing one or two DataComponents which
     * describing the requested path. The number of DataComponents
     * returns depends on the value of isFile: if it is true,
     * 2 components are returned.
     * @param path the Hotline path.
     * @param dataType {@link HLProtocol.HTLC_DATA_DIR} or 
     *                 {@link HLProtocol.HTLC_DATA_DIR_RENAME}.
     * @param isFile true if the path refers to a file.
     */
    static HLProtocol.DataComponent[] createPathComponents(String path,
                                                           short dataType,
                                                           boolean isFile) throws IOException {
        
        HLProtocol.DataComponent[] dataComponents = new HLProtocol.DataComponent[isFile ? 2 : 1];
        
        if(isFile) {
            
            String file = path.substring(path.lastIndexOf(HLProtocol.DIR_SEPARATOR) + 1);
            dataComponents[1] = hlp.new FileComponent(file);
            
            path = path.substring(0, path.length() - file.length());
            
        }
        
        dataComponents[0] = hlp.new PathComponent(dataType, path);
        
        return dataComponents;
        
    }
        
}
