/**
 * $Id: HLServerBanTable.java,v 1.1.1.1 2001/07/22 02:44:43 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.hotline;

import java.util.Hashtable;
import java.util.Enumeration;
import java.net.InetAddress;

import redlight.utils.TimeFormat;
import redlight.utils.DebuggerOutput;

/**
 * A bantable maintains a list of IP addresses that are not allowed
 * to login to the server.
 */
public class HLServerBanTable extends HLServerComponent {
    
    protected Hashtable table;

    /**
     * Creates an empty bantable.
     */
    public HLServerBanTable(HLServer h) {

        super(h);        
        table = new Hashtable();
        
    }
    
    /**
     * Returns an enumeration of the elements in the bantable.
     * Each element has type {@link HLServerBanTable.BanRecord}.
     */
    public Enumeration elements() {
        
        checkModified();
        return table.elements();
        
    }
    
    /**
     * Checks whether the bantable contains a given IP address.
     * @param ip the IP address.
     * @return true if the address is in the bantable.
     */
    public boolean contains(InetAddress ip) {
        
        checkModified();
        return table.containsKey(ip.getHostAddress());
        
    }

    /**
     * Inserts / replaces an address to the bantable.
     * @param ip the address to insert / replace.
     * @param expires the timestamp in seconds since epoch when the
     * ban expires. If this is 0, the ban never expires.
     * @throws IllegalArgumentException if ip is null.  
     */
    public void put(InetAddress ip, long expires) {
        
        if(ip == null)
            throw new IllegalArgumentException("ip == null");
        
        if(expires < 0)
            throw new IllegalArgumentException("expires < 0");
        
        BanRecord b = new BanRecord(ip, expires);
        table.put(ip.getHostAddress(), b);
        hls.log("Banning " + ip.getHostAddress() + " for" + (expires == 0 ? "ever." : " " + TimeFormat.format(expires - (System.currentTimeMillis() / 1000))));
        hasChanged();
        
    }
    
    /**
     * Removes an address from the bantable.
     * @param ip the address to remove.
     * @throws IllegalArgumentException if ip is null.
     */
    public void remove(InetAddress ip) {
    
        if(ip == null)
            throw new IllegalArgumentException("ip == null");

        if(table.containsKey(ip.getHostAddress())) {

            table.remove(ip.getHostAddress());
            hasChanged();
            
        }
        
    }

    /**
     * Removes entries that have been banned for long enough
     * from the bantable.
     */
    void expire() {
        boolean changed = false;

        checkModified();

        for(Enumeration en = table.elements(); en.hasMoreElements(); ) {
            
            BanRecord b = (BanRecord) en.nextElement();
            
            if(b.expires != 0 && 
               (System.currentTimeMillis() / 1000) > b.expires) {

                hls.log("Expired ban for " + b.address.getHostAddress());
                table.remove(b.address.getHostAddress());
                changed = true;

            }
            
        }

        if(changed)
            hasChanged();

    }

    /**
     * Called whenever the bantable changes. Subclasses can override
     * this method to implement e.g. persistent storage.  
     */
    public void hasChanged() {}

    /** 
     * Called whenever the latest version of the agreement is required.
     * Can be overridden by subclasses to implement persistent storage.
     */    
    public void checkModified() {}

    /**
     * Encapsulates IP address and time.
     */
    public class BanRecord {
       
        /* The IP address (read-only). */

        public InetAddress address;

        /* Time in seconds since the epoch (read-only). */

        public long expires;
        
        public BanRecord(InetAddress a, long e) {
            
            address = a;
            expires = e;
            
        }
        
    }

}
